package ru.yandex.zora.proxy;

import java.io.Closeable;
import java.io.IOException;
import java.lang.Thread.UncaughtExceptionHandler;
import java.net.MalformedURLException;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.logger.PrefixedLogger;

public class MissfitsImagesDownloader
    implements Runnable, Closeable, UncaughtExceptionHandler
{
    private final ConcurrentHashMap<String, String> pending;
    private final Map<String, BlockingQueue<SusspiciousTask>> queues;
    private volatile boolean shutdown = false;
    private final ZoraProxy server;
    private final PrefixedLogger logger;
    private final Thread thread;

    public MissfitsImagesDownloader(final ZoraProxy server) {
        this.server = server;
        this.queues = new ConcurrentHashMap<>();
        this.pending = new ConcurrentHashMap<>();
        this.thread = new Thread(new ThreadGroup("SuspiciousDownloader"), this);
        this.thread.setUncaughtExceptionHandler(this);
        this.logger = server.logger().addPrefix("Missfit");
    }

    public void start() {
        this.thread.start();
    }

    public int queues() {
        return queues.size();
    }

    public void add(
        final HttpHost host,
        final ProxySession session,
        final MissfitsWaitingLoadCallback callback)
    {
        ArrayBlockingQueue<SusspiciousTask> queue =
            new ArrayBlockingQueue<>(server.config().suspiciousQueueSize());

        BlockingQueue<SusspiciousTask> current =
            queues.putIfAbsent(host.getHostName(), queue);

        if (current == null) {
            current = queue;
        }

        session.logger().info("Queue size is " + current.size());

        if (current.remainingCapacity() > 0) {
            try {
                current.put(new SusspiciousTask(host, callback, session));
            } catch (InterruptedException ie) {
                session.logger().log(
                    Level.WARNING,
                    "Unable to put in suspicious queue",
                    ie);
            }

            return;
        }

        session.logger().info(
            "Queue is full, skipping "
                + host + ' ' + callback.task().url());
        callback.failed(new Image(-1, -1));
    }

    @Override
    public void close() throws IOException {
        this.thread.interrupt();
    }

    @Override
    public void uncaughtException(final Thread t, final Throwable e) {
        this.logger.log(
            Level.WARNING,
            "Missfit download thread exception " + t.toString(),
            e);
    }

    @Override
    public void run() {
        try {
            while (!shutdown) {
                Iterator<Entry<String, BlockingQueue<SusspiciousTask>>>
                    iterator =
                    queues.entrySet().iterator();

                this.logger.info("Hosts " + queues.size());
                while (iterator.hasNext()) {
                    Map.Entry<String, BlockingQueue<SusspiciousTask>> host
                        = iterator.next();

                    SusspiciousTask task = host.getValue().peek();
                    if (task == null) {
                        continue;
                    }

                    if (pending.putIfAbsent(host.getKey(), host.getKey())
                        != null)
                    {
                        continue;
                    }

                    task = host.getValue().poll();
                    if (task == null) {
                        pending.remove(host.getKey());
                        continue;
                    }

                    SuspiciousCallback callback = new SuspiciousCallback(task);

                    this.logger.info("Scheduling " + task.url());
                    try {
                        server.suspiciousRequests().accept(1);
                        server.scheduler().schedule(
                            task.session(),
                            task.url(),
                            callback);
                    } catch (MalformedURLException mue) {
                        callback.failed(mue);
                    }
                }

                Thread.sleep(server.config().timerResolution());
            }
        } catch (InterruptedException ie) {
            server.logger().warning("Downloader interrupted");
        }
    }

    private final class SuspiciousCallback
        implements FutureCallback<Image>
    {
        private final SusspiciousTask task;

        private SuspiciousCallback(final SusspiciousTask task) {
            this.task = task;
        }

        @Override
        public void completed(final Image image) {
            pending.remove(task.host());
            task.callback.completed(image);
        }

        @Override
        public void failed(final Exception e) {
            pending.remove(task.host());
            task.callback.failed(e);
        }

        @Override
        public void cancelled() {
            pending.remove(task.host());
            task.callback.cancelled();
        }
    }

    private static final class SusspiciousTask {
        private final HttpHost host;
        private final MissfitsWaitingLoadCallback callback;
        private final ProxySession session;

        private SusspiciousTask(
            final HttpHost host,
            final MissfitsWaitingLoadCallback callback,
            final ProxySession session)
        {
            this.host = host;
            this.callback = callback;
            this.session = session;
        }

        public MissfitsWaitingLoadCallback callback() {
            return callback;
        }

        public String host() {
            return host.getHostName();
        }

        public String url() {
            return callback.task().url();
        }

        public ProxySession session() {
            return session;
        }
    }
}
