package ru.yandex.zora.proxy;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.io.BufferedWriter;
import java.io.File;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.LinkedHashMap;
import java.util.Map;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import ru.yandex.zora.proxy.config.ImmutableZoraProxyConfig;

public class ZoraCache extends TimerTask {
    private static final int PENDING_CAPACITY = 10000;
    private static final long TIMER_DELAY = 10000;
    private static final long TIMER_PERIOD = 600000;

    private final ConcurrentMap<String, DownloadTask> pending;
    private final Cache<String, Image> downloaded;
    private final ZoraProxy proxy;
    private final Timer timer;
    private final String path;

    public ZoraCache(final ZoraProxy proxy) {
        this.proxy = proxy;

        ImmutableZoraProxyConfig config = proxy.config();
        this.downloaded =
            CacheBuilder.newBuilder()
                .maximumSize(config.maxCacheSize())
                .initialCapacity(PENDING_CAPACITY)
                .concurrencyLevel(config.workers())
                .expireAfterWrite(
                    config.expirePeriod(),
                    TimeUnit.SECONDS).build();
        this.pending = new ConcurrentHashMap<>(PENDING_CAPACITY);

        timer = new Timer("CacheDumpTask", true);
        if (config.cacheDumpLocation() != null) {
            this.path = config.cacheDumpLocation();
            final int parts = 3;
            try {
                Files.lines(new File(config.cacheDumpLocation()).toPath())
                    .forEach(l -> {
                        String[] split = l.split("\\t");
                        if (split.length == parts) {
                            downloaded.put(
                                split[0],
                                new Image(
                                    Integer.parseInt(split[1]),
                                    Integer.parseInt(split[2])));
                        }
                    });
            } catch (Exception e) {
                proxy.logger().log(Level.WARNING, "Failed to load cache", e);
            }

            timer.schedule(this, TIMER_DELAY, TIMER_PERIOD);
        } else {
            path = null;
        }
    }

    @Override
    public void run() {
        File tmpFile = new File(path + ".dumping");
        File dmpFile = new File(path);

        proxy.logger().info("Cache dump start");

        try {
            Files.delete(tmpFile.toPath());
        } catch (Exception nfe) {
            proxy.logger().info("Previous dump do not exists");
        }

        try (BufferedWriter bw =
                new BufferedWriter(
                    new OutputStreamWriter(
                        new FileOutputStream(tmpFile),
                        StandardCharsets.UTF_8)))
        {
            final char sep = '\t';
            for (Map.Entry<String, Image> item: downloaded.asMap().entrySet()) {
                bw.write(
                    item.getKey() + sep
                        + item.getValue().height()
                        + sep + item.getValue().width());
                bw.newLine();
            }
        } catch (IOException e) {
            proxy.logger().warning("Cache dump failed");
            System.err.println("Failed to cache dump");
            e.printStackTrace();
        }

        try {
            Files.move(
                tmpFile.toPath(),
                dmpFile.toPath(),
                StandardCopyOption.REPLACE_EXISTING);

            proxy.logger().info("cache dump finished");
        } catch (IOException | SecurityException e) {
            proxy.logger().log(Level.WARNING, "Failed to move cache dump", e);
        }
    }

    public void shutdown() {
        this.timer.cancel();
    }

    public ConcurrentMap<String, DownloadTask> pending() {
        return pending;
    }

    public Cache<String, Image> downloaded() {
        return downloaded;
    }

    public Map<String, Object> status() {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("downloaded", downloaded.size());
        map.put("pending", pending.size());
        return map;
    }
}
