package ru.yandex.zora.proxy;

import java.io.IOException;

import java.net.URISyntaxException;

import java.security.GeneralSecurityException;

import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;

import ru.yandex.collection.Pattern;

import ru.yandex.concurrent.TimeFrameQueue;

import ru.yandex.http.proxy.HttpProxy;

import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;

import ru.yandex.json.parser.JsonException;

import ru.yandex.parser.config.ConfigException;

import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.ImmutableStatersConfig;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.StaterConfigBuilder;
import ru.yandex.stater.StatersConfigBuilder;

import ru.yandex.zora.proxy.config.ImmutableZoraProxyConfig;

public class ZoraProxy extends HttpProxy<ImmutableZoraProxyConfig> {
    private static final String ZORA = "zora";
    private static final String CACHE = "cache";

    private final AsyncClient zoraClient;
    private final TimeFrameQueue<Long> imageParseStat;
    private final TimeFrameQueue<Integer> malformedUrls;
    private final TimeFrameQueue<Integer> cacheHit;
    private final TimeFrameQueue<Integer> suspiciousRequests;

    private final ZoraCache cache;
    private final ZoraScheduler scheduler;
    private final MissfitsImagesDownloader missfitsImagesDownloader;
    private final Tvm2TicketRenewalTask tvm2RenewalTask;

    public ZoraProxy(final ImmutableZoraProxyConfig config)
        throws GeneralSecurityException,
        HttpException,
        IOException,
        JsonException,
        URISyntaxException
    {
        super(config);

        tvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("zoraTvm2"),
            serviceContextRenewalTask,
            config.tvm2Config());

        try {
            this.zoraClient =
                adjustClient(
                    client("ZoraClient", config.zoraclConfig()),
                    ZORA);
        } catch (ConfigException ce) {
            throw new IOException(ce);
        }

        this.cache = new ZoraCache(this);
        this.scheduler = new ZoraScheduler(this);
        this.missfitsImagesDownloader = new MissfitsImagesDownloader(this);

        imageParseStat = new TimeFrameQueue<>(config.metricsTimeFrame());
        this.registerStater(
            new PassiveStaterAdapter<>(
                imageParseStat,
                new NamedStatsAggregatorFactory<>(
                    "image_parsing_time_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        malformedUrls = new TimeFrameQueue<>(config.metricsTimeFrame());
        this.registerStater(
            new PassiveStaterAdapter<>(
                malformedUrls,
                new NamedStatsAggregatorFactory<>(
                    "malformed_urls_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        cacheHit = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                cacheHit,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "cache_hits_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "image_requests_ammm",
                        CountAggregatorFactory.INSTANCE))));

        suspiciousRequests = new TimeFrameQueue<>(config.metricsTimeFrame());
        this.registerStater(
            new PassiveStaterAdapter<>(
                suspiciousRequests,
                new NamedStatsAggregatorFactory<>(
                    "suspicious_reqs_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        this.register(
            new Pattern<>("/image", false),
            new ImageHandler(this),
            RequestHandlerMapper.GET);
    }

    public String tvmTicket() {
        return tvm2RenewalTask.ticket();
    }

    @Override
    public void start() throws IOException {
        super.start();

        this.tvm2RenewalTask.start();
        this.missfitsImagesDownloader.start();
    }

    @Override
    public void close() throws IOException {
        this.missfitsImagesDownloader.close();
        this.tvm2RenewalTask.cancel();

        super.close();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put(ZORA, zoraClient.status(verbose));
        status.put(CACHE, cache.status());
        status.put("suspicious", missfitsImagesDownloader.queues());
        return status;
    }

    protected <T extends AbstractAsyncClient<T>> T adjustClient(
        final T client,
        final String prefix)
        throws ConfigException
    {
        StatersConfigBuilder stsb = new StatersConfigBuilder();
        final String uri = '/' + prefix;

        StaterConfigBuilder stb = new StaterConfigBuilder();
        stb.prefix(prefix);
        stsb.staters().put(new Pattern<>(uri, true), stb);

        HttpContext context = HttpCoreContext.create();

        context.setAttribute(HttpCoreContext.HTTP_REQUEST, new HttpGet(uri));

        ImmutableStatersConfig staters = stsb.build();
        registerStater(
            staters.preparedStaters().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        uri))));

        return client.adjustStater(staters, context);
    }

    public AsyncClient zoraClient() {
        return zoraClient;
    }

    public TimeFrameQueue<Long> imageParseStat() {
        return imageParseStat;
    }

    public void cacheHit(final boolean cache) {
        if (cache) {
            cacheHit.accept(1);
        } else {
            cacheHit.accept(0);
        }
    }

    public ZoraCache cache() {
        return cache;
    }

    public ZoraScheduler scheduler() {
        return scheduler;
    }

    public MissfitsImagesDownloader suspiciousImagesDownloader() {
        return missfitsImagesDownloader;
    }

    public TimeFrameQueue<Integer> suspiciousRequests() {
        return suspiciousRequests;
    }
}
