package ru.yandex.zora.proxy;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.function.Consumer;

import javax.imageio.ImageIO;

import org.apache.http.ContentTooLongException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;

import org.apache.http.entity.ContentType;

import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.IOControl;

import org.apache.http.nio.entity.ContentInputStream;
import org.apache.http.nio.protocol.AbstractAsyncResponseConsumer;

import org.apache.http.nio.util.HeapByteBufferAllocator;
import org.apache.http.nio.util.SimpleInputBuffer;

import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import org.apache.http.util.Asserts;

import ru.yandex.logger.PrefixedLogger;

public class ZoraResultConsumer extends AbstractAsyncResponseConsumer<Image> {
    private static final long DEFAULT_LENGTH = 4096L;

    private final long maxSize;
    private final Consumer<Long> imageProcessingTime;
    private volatile SimpleInputBuffer buf;
    private final PrefixedLogger logger;

    public ZoraResultConsumer(final ZoraProxy proxy) {
        this.imageProcessingTime = proxy.imageParseStat();
        this.maxSize = proxy.config().maxImageSize();
        this.logger = proxy.logger();
        this.buf = null;
    }

    @Override
    protected Image buildResult(final HttpContext context) throws Exception {
        long startTime = System.currentTimeMillis();
        BufferedImage image = ImageIO.read(new ContentInputStream(this.buf));
        imageProcessingTime.accept(System.currentTimeMillis() - startTime);
        if (image == null) {
            String message = "Broken or unsupported image format ";
            HttpRequest request =
                (HttpRequest) context.getAttribute(
                    HttpCoreContext.HTTP_REQUEST);
            if (request != null) {
                message += request.toString();
            }

            this.logger.warning(message);

            return new Image(-1, -1);
        }

        return new Image(image.getHeight(), image.getWidth());
    }

    @Override
    protected void onEntityEnclosed(
        final HttpEntity entity,
        final ContentType contentType)
        throws IOException
    {
        long len = entity.getContentLength();
        if (len > maxSize) {
            throw new ContentTooLongException(
                "Entity content is too long: " + len);
        } else {
            if (len < 0L) {
                len = DEFAULT_LENGTH;
            }

            this.buf =
                new SimpleInputBuffer((int) len, new HeapByteBufferAllocator());
        }
    }

    @Override
    protected void onResponseReceived(final HttpResponse httpResponse)
        throws HttpException, IOException
    {
    }

    @Override
    protected void onContentReceived(
        final ContentDecoder decoder,
        final IOControl ioControl)
        throws IOException
    {
        Asserts.notNull(this.buf, "Content buffer");
        this.buf.consumeContent(decoder);
    }

    @Override
    protected void releaseResources() {
        this.buf = null;
    }
}
