package ru.yandex.zora.proxy;

import com.google.common.util.concurrent.RateLimiter;

import java.net.MalformedURLException;
import java.net.URL;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.HttpStatusPredicates;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;

public class ZoraScheduler {
    // zora have special header
    public static final String ZORA_TVM2_TICKET = "X-Ya-Service-Ticket";
    public static final String ZORA_SOURCE_HEADER = "X-Yandex-Sourcename";
    public static final String ZORA_HTTPS_HEADER_KEY = "X-Yandex-Use-Https";
    public static final String ZORA_RESPONSE_TIMEOUT =
        "X-Yandex-Response-Timeout";

    private static final String ZORA_HTTPS_HEADER_VALUE = "true";

    private final ZoraProxy server;
    private final RateLimiter rateLimiter;
    private final StatusCheckAsyncResponseConsumerFactory<Image>
        zoraConsumerFactory;

    public ZoraScheduler(final ZoraProxy zoraProxy) {
        this.server = zoraProxy;

        if (server.config().zoraRpsLimit() > 0) {
            this.rateLimiter =
                RateLimiter.create(server.config().zoraRpsLimit());
        } else {
            this.rateLimiter = null;
        }

        this.zoraConsumerFactory =
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new ZoraResultConsumerFactory(server));
    }

    public void schedule(
        final ProxySession session,
        final URL url,
        final FutureCallback<Image> callback)
    {
        session.logger().info(url + " scheduling");
        AsyncClient client =
            server.zoraClient().adjust(session.context());

        BasicAsyncRequestProducerGenerator requestGenerator =
            new BasicAsyncRequestProducerGenerator(url.toString());
        int hostResponseTimeout = server.config().zoraResponseTimeout();
        if (hostResponseTimeout > 0) {
            requestGenerator.addHeader(
                ZORA_RESPONSE_TIMEOUT,
                String.valueOf(hostResponseTimeout));
        }
        requestGenerator.addHeader(
            ZORA_SOURCE_HEADER,
            server.config().source());
        requestGenerator.addHeader(ZORA_TVM2_TICKET, server.tvmTicket());

        if (url.getProtocol().equalsIgnoreCase("https")) {
            requestGenerator.addHeader(
                ZORA_HTTPS_HEADER_KEY,
                ZORA_HTTPS_HEADER_VALUE);
        }

        if (rateLimiter != null) {
            rateLimiter.acquire();
        }

        client.execute(
            new HttpHost(url.getHost(), url.getPort()),
            requestGenerator,
            zoraConsumerFactory,
            session.listener().createContextGeneratorFor(client),
            callback
        );
    }

    public void schedule(
        final ProxySession session,
        final String urlStr,
        final FutureCallback<Image> callback)
        throws MalformedURLException
    {
        schedule(session, new URL(urlStr), callback);
    }
}
