package ru.yandex.zora.proxy.config;

import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;

import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;

import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractZoraProxyConfigBuilder
    <T extends AbstractZoraProxyConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements ZoraProxyConfig
{
    private HttpTargetConfigBuilder zoraclConfig;
    private Tvm2ClientConfigBuilder tvm2Config;
    private String source;
    private long expirePeriod;
    private long maxImageSize;
    private int maxCacheSize;
    private int zoraResponseTimeout;
    private int zoraRpsLimit;
    private int suspiciousSize;
    private String cacheDumpLocation;

    protected AbstractZoraProxyConfigBuilder(final ZoraProxyConfig config) {
        super(config);

        expirePeriod(config.expirePeriod());
        zoraclConfig(config.zoraclConfig());
        maxImageSize(config.maxImageSize());
        maxCacheSize(config.maxCacheSize());
        source(config.source());
        cacheDumpLocation(config.cacheDumpLocation());
        zoraResponseTimeout(config.zoraResponseTimeout());
        zoraRpsLimit(config.zoraRpsLimit());
        suspiciousQueueSize(config.suspiciousQueueSize());
        tvm2Config(config.tvm2Config());
    }

    protected AbstractZoraProxyConfigBuilder(
        final IniConfig config,
        final ZoraProxyConfig defaults)
        throws ConfigException
    {
        super(config, defaults);

        this.zoraclConfig =
            new HttpTargetConfigBuilder(
                config.section("zoracl"),
                defaults.zoraclConfig());

        this.expirePeriod =
            config.getLong("expire-period", defaults.expirePeriod());

        this.maxImageSize =
            config.getLong("max-image-size", defaults.maxImageSize());

        this.source = config.getString("source", defaults.source());
        this.maxCacheSize =
            config.getInt("max-cache-size", defaults.maxCacheSize());

        this.cacheDumpLocation =
            config.getString("cache-dump-location", null);

        this.zoraResponseTimeout =
            config.getInt(
                "zora-response-timeout",
                defaults.zoraResponseTimeout());

        this.zoraRpsLimit =
            config.getInt("zora-rps-limit", defaults.zoraRpsLimit());

        this.tvm2Config =
            new Tvm2ClientConfigBuilder(
                config.section("tvm2"),
                defaults.tvm2Config());

        this.suspiciousSize =
            config.getInt(
                "suspicious-size-limit",
                defaults.suspiciousQueueSize());
    }

    @Override
    public long expirePeriod() {
        return expirePeriod;
    }

    public T expirePeriod(final long expirePeriod) {
        this.expirePeriod = expirePeriod;
        return self();
    }

    @Override
    public HttpTargetConfigBuilder zoraclConfig() {
        return zoraclConfig;
    }

    public T zoraclConfig(final HttpTargetConfig config) {
        this.zoraclConfig = new HttpTargetConfigBuilder(config);
        return self();
    }

    @Override
    public long maxImageSize() {
        return maxImageSize;
    }

    public T maxImageSize(final long imageSize) {
        this.maxImageSize = imageSize;
        return self();
    }

    @Override
    public String source() {
        return source;
    }

    public T source(final String source) {
        this.source = source;
        return self();
    }

    @Override
    public int maxCacheSize() {
        return maxCacheSize;
    }

    public T maxCacheSize(final int maxCacheSize) {
        this.maxCacheSize = maxCacheSize;
        return self();
    }

    @Override
    public String cacheDumpLocation() {
        return cacheDumpLocation;
    }

    public T cacheDumpLocation(final String location) {
        this.cacheDumpLocation = location;
        return self();
    }

    @Override
    public int zoraResponseTimeout() {
        return zoraResponseTimeout;
    }

    public T zoraResponseTimeout(final int timeout) {
        this.zoraResponseTimeout = timeout;
        return self();
    }

    @Override
    public int zoraRpsLimit() {
        return zoraRpsLimit;
    }

    public T zoraRpsLimit(final int limit) {
        this.zoraRpsLimit = limit;
        return self();
    }

    @Override
    public int suspiciousQueueSize() {
        return suspiciousSize;
    }

    public T suspiciousQueueSize(final int limit) {
        this.suspiciousSize = limit;
        return self();
    }

    @Override
    public Tvm2ClientConfigBuilder tvm2Config() {
        return tvm2Config;
    }

    public T tvm2Config(
        final Tvm2ClientConfig tvm2Config)
    {
        this.tvm2Config = new Tvm2ClientConfigBuilder(tvm2Config);
        return self();
    }
}
