#!/usr/bin/python

import argparse
import glob
import os
import re

import xml.etree.ElementTree as ET

import sys

MODULE_DEPENDENCY_ENTRY = '<orderEntry type="module" module-name="{dep}" />'
MODULE_PLACEHOLDER = '<moduleplace />'
DEFAULT_MODULE_CONTENT = '''<?xml version="1.0" encoding="UTF-8"?>
<module type="JAVA_MODULE" version="4">
  <component name="NewModuleRootManager" inherit-compiler-output="true">
    <exclude-output />
    <content url="file://$MODULE_DIR$/git/src/{project_name}">
      <sourceFolder url="file://$MODULE_DIR$/git/src/{project_name}/main/java" isTestSource="false" />
      <sourceFolder url="file://$MODULE_DIR$/git/src/{project_name}/test/java" isTestSource="true" />
    </content>
    <orderEntry type="inheritedJdk" />
    <orderEntry type="sourceFolder" forTests="false" />\n
    <orderEntry type="library" name="PSLIB-local" level="application" />
  </component>
</module>'''

DEFAULT_LIBRARY_CONTENT = '''<component name="libraryTable">
  <library name="PS_THIRDPARTY_LIBS">
    <CLASSES>
    </CLASSES>
    <JAVADOC />
    <SOURCES />
  </library>
</component>'''


DEFAULT_MODULES_XML = '''<?xml version="1.0" encoding="UTF-8"?>\n<project version="4">\n<component name="ProjectModuleManager"><modules>'''
DEFAULT_MODULES_XML += '''</modules>\n</component>\n</project>'''


def parse_projects(data):
    prjs_pattern = 'projects :='
    start = unicode(data).index('projects := ')
    start = start + len(prjs_pattern)
    return set(parse_var(data, start))


def parse_var(data, start):
    was_escaped = False
    projects = []
    part = ''
    for i in range(start, len(data)):
        ch = data[i]
        if ch == '\\':
            was_escaped = True
        elif ch == '\n':
            for item in part.strip().split():
                if item:
                    projects.append(item)

            part = ''
            if not was_escaped:
                break

            was_escaped = False
        else:
            part += ch

    return projects


def resolve_deps(data, projects):
    deps = {}
    for project in projects:
        get_dependencies(deps, data, project, False)

    for project in projects:
        get_dependencies(deps, data, project, True)

    return deps


def get_dependencies(res, data, name, test=False, depth=0):
    deps = []
    result = set()
    index_main = -1
    index_test = -1

    if not test:
        for match in re.finditer(r'[\n\s]+' + name + r'-main-deps :=', data):
            index_main = match.span(0)[1]

        if index_main >= 0:
            deps.extend(parse_var(data, index_main))
    else:
        for match in re.finditer(r'[\n\s]+' + name + r'-test-deps :=', data):
            index_test = match.span(0)[1]

        if index_test >= 0:
            deps.extend(parse_var(data, index_test))

    if not deps:
        if name not in res:
            res[name] = []
        return res[name]

    for dep in deps:
        result.add(dep)
        if dep in res:
            source = res[dep]
        else:
            source = get_dependencies(res, data, dep, test, depth+1)

        for resolved in source:
            result.add(resolved)

    if name in result:
        result.remove(name)

    if name not in res:
        res[name] = []
        
    res[name].extend(result)
    return res[name]


class ProjectLibrary(object):
    def __init__(self, idea_project_dir, projects_mk_file_path):
        self.idea_lib_file = idea_project_dir + '/.idea/libraries/PS_THIRDPARTY_LIBS.xml'
        self.idea_project_dir = idea_project_dir
        self.lib_dir = os.path.dirname(projects_mk_file_path) + '/lib'
        self.xml, self.jars_root = self.load_or_create()

    def load_or_create(self):
        if not os.path.exists(self.idea_lib_file):
            mes = 'WARNING: Idea project Library not attached to project {library_file} not exists, create brand new? [Y/N] '.format(
                library_file=self.idea_lib_file)

            answer = raw_input(mes)
            if 'Y' == answer.strip().upper():
                file_dir = os.path.dirname(self.idea_lib_file)
                if not os.path.exists(file_dir):
                    os.makedirs(file_dir)

                with open(self.idea_lib_file, 'w') as f:
                    f.write(DEFAULT_LIBRARY_CONTENT)
            else:
                print 'Aborting'
                sys.exit(1)

        return self.load()

    def update(self):
        for jar_file in glob.glob(self.lib_dir + '/*.jar'):
            ET.SubElement(self.jars_root, 'root',  {'url': 'jar://{jar}!/'.format(jar=os.path.abspath(jar_file))})

    def save(self):
        self.xml.write(self.idea_lib_file)

    def load(self):
        try:
            parsed_xml = ET.parse(self.idea_lib_file)
        except Exception as e:
            print 'Failed to parse', self.idea_lib_file, 'try to manually remove it and rerun'
            sys.exit(1)

        jars_root = parsed_xml.find('library').find('CLASSES')

        idea_jar_custom_deps = set()
        to_delete = []
        for item in jars_root.findall('root'):
            jar_raw_url = item.attrib.get('url').rstrip('/!')
            jar_url = jar_raw_url.replace('jar://$PROJECT_DIR$', self.idea_project_dir)
            jar_dir = os.path.dirname(jar_url)
            if os.path.normpath(jar_dir) != self.lib_dir:
                idea_jar_custom_deps.add(jar_url)
            else:
                to_delete.append(item)

            dep = os.path.basename(jar_raw_url)
            idea_jar_custom_deps.add(dep)

        for item in to_delete:
            jars_root.remove(item)

        return parsed_xml, jars_root


class IdeaModule(object):
    def __init__(self, path, name, deps, xml, deps_root):
        self.path = path
        self.name = name
        self.deps = set(deps)
        self.xml = xml
        self.deps_root = deps_root
        self.changed = False

    def set_changed(self):
        self.changed = True

    def add_deps(self, deps):
        if deps:
            self.set_changed()

        for dep in deps:
            self.deps.add(dep)

    def save(self, write_down=True):
        if not self.changed:
            return 0

        if write_down:
            for dep in self.deps:
                ET.SubElement(self.deps_root, 'orderEntry', {'type': 'module', 'module-name': dep})

            self.xml.write(self.path)
        else:
            pass

        return 1

    @staticmethod
    def load(path, name, parsed_xml):
        root = parsed_xml.getroot()
        modules_deps = set()

        deps_root = None
        library = False
        for component in root.findall('component'):
            if component.attrib.get('name') != 'NewModuleRootManager':
                continue

            deps_root = component
            to_delete = []
            for dep in deps_root.iter('orderEntry'):
                attrs = dep.attrib
                module_type = attrs.get('type')
                if module_type == 'module':
                    dep_name = attrs.get('module-name')
                    if dep_name:
                        to_delete.append(dep)
                        modules_deps.add(dep_name)
                elif module_type == 'library' \
                        and attrs.get('level') == 'project' \
                        and attrs.get('name') == 'PS_THIRDPARTY_LIBS':
                    library = True

            if to_delete:
                for item in to_delete:
                    deps_root.remove(item)

        module = IdeaModule(path, name, modules_deps, parsed_xml, deps_root)
        if not library:
            e = ET.SubElement(
                deps_root,
                'orderEntry',
                {'type': 'library', 'name': 'PS_THIRDPARTY_LIBS', 'level': 'project'})

            e.tail = '\n'
            module.set_changed()

        return module

    @staticmethod
    def load_from_file(path, name):
        return IdeaModule.load(path, name, ET.parse(path))

    @staticmethod
    def get_default(path, name):
        iml_file = path + '/' + name + '.iml'
        with open(iml_file, 'w') as f:
            f.write(DEFAULT_MODULE_CONTENT.format(project_name=name))

        return IdeaModule.load_from_file(iml_file, name)


class IdeaModules(object):
    def __init__(self, idea_project_dir, idea_internal_dir):
        self.idea_project_dir = idea_project_dir
        self.modules_xml_path = os.path.join(idea_internal_dir, 'modules.xml')
        self.modules_xml = None
        self.modules_xml_modules_root = None
        self.modules = {}

        self.load_modules()

    def load_modules(self):
        try:
            self.modules_xml = ET.parse(self.modules_xml_path)
            root = self.modules_xml.getroot()
            self.modules_xml_modules_root = root.find('component').find('modules')
            to_delete = []
            for item in self.modules_xml_modules_root.findall('module'):
                filepath = item.attrib['filepath']
                filename = os.path.basename(filepath)
                module_iml_path = os.path.join(self.idea_project_dir, filename)
                try:
                    module = IdeaModule.load_from_file(
                        module_iml_path,
                        filename.split('.iml')[0])
                    to_delete.append(item)
                except Exception as e:
                    print '[ERROR] Failed to load', module_iml_path, e, 'Skipping'
                    continue

                self.modules[module.name] = module

            # for item in to_delete:
            #     self.modules_xml_modules_root.remove(item)

        except Exception as e:
            print 'Malformed modules.xml file', self.modules_xml_path
            raise e

    def update_deps(self, module_name, mk_deps):
        module = self.modules[module_name]
        diff = mk_deps.difference(module.deps)
        for item in diff:
            module.deps.add(item)
            module.set_changed()

        return diff

    def modules_names(self):
        return self.modules.keys()

    def exists(self, name):
        return name in self.modules.keys()

    def add_new(self, name, deps):
        idea_module = IdeaModule.get_default(self.idea_project_dir, name)
        idea_module.add_deps(deps)

        self.modules[idea_module.name] = idea_module
        xml_url = '$PROJECT_DIR$/' + name + '.iml'
        e = ET.SubElement(self.modules_xml_modules_root, 'module', {'fileurl': 'file://' + xml_url, 'filepath': xml_url})
        e.tail = '\n'

    def save(self):
        self.modules_xml.write(self.modules_xml_path)

        changed = 0
        for module in self.modules.values():
            changed += module.save()

        return changed


class IdeaProjectDepsConverter(object):
    def __init__(self, idea_project_dir, projects_mk_file_path, write_down=False):
        self.idea_project_dir = idea_project_dir
        self.idea_internal_dir = os.path.join(self.idea_project_dir, '.idea')
        self.projects_mk_file_path = projects_mk_file_path
        self.write_down = write_down
        print 'Idea project dir', self.idea_project_dir
        print 'projects.mk', self.projects_mk_file_path

    def load_projects_mk_deps(self):
        with open(self.projects_mk_file_path, 'r') as projects_mk:
            data = projects_mk.read()

        projects = parse_projects(data)
        return resolve_deps(data, projects)

    def precheck(self):
        idea_dir = self.idea_project_dir + '/.idea'
        if not os.path.exists(idea_dir):
            mes = 'WARNING: Idea folder {idea_dir} not exists, create brand new? [Y/N] '.format(
                idea_dir=idea_dir)

            answer = raw_input(mes)
            if 'Y' == answer.strip().upper():
                os.makedirs(idea_dir)

                with open(os.path.join(idea_dir, 'modules.xml'), 'w') as mf:
                    mf.write(DEFAULT_MODULES_XML)
            else:
                sys.exit(1)

    def new_module(self, name, bases=[]):
        idea_modules = IdeaModules(self.idea_project_dir, self.idea_internal_dir)
        projects_mk_modules = self.load_projects_mk_deps()
        deps = []
        for base in bases:
            deps.append(base)
            deps += projects_mk_modules[base]

        idea_modules.add_new(name, deps=set(deps))
        main_java_dir = os.path.dirname(self.projects_mk_file_path) + '/src/' + name + '/main/java'
        if not os.path.exists(main_java_dir):
            os.makedirs(main_java_dir)
        test_java_dir = os.path.dirname(self.projects_mk_file_path) + '/src/' + name + '/test/java'
        if not os.path.exists(test_java_dir):
            os.makedirs(test_java_dir)
        bundle_dir = os.path.dirname(self.projects_mk_file_path) + '/src/' + name + '/bundle'
        if not os.path.exists(bundle_dir):
            os.makedirs(bundle_dir)

        with open(bundle_dir + '/' + name + '.conf', 'w') as f:
            f.write('\n')

        idea_modules.save()

    def convert(self):
        self.precheck()
        # first load modules list from modules file

        idea_modules = IdeaModules(self.idea_project_dir, self.idea_internal_dir)
        projects_mk_modules = self.load_projects_mk_deps()
        new_modules = set(projects_mk_modules.keys()).difference(set(idea_modules.modules_names()))
        if new_modules:
            print 'Found {cnt} new modules'.format(cnt=len(new_modules)), new_modules

        for module_name in new_modules:
            idea_modules.add_new(module_name, projects_mk_modules[module_name])

        deps_changed_cnt = 0
        # now check deps for existing modules
        for module_name in projects_mk_modules.keys():
            mk_deps = set(projects_mk_modules[module_name])
            added = idea_modules.update_deps(module_name, mk_deps)
            if added:
                deps_changed_cnt += 1
                print 'For module {module} new deps {deps}'.format(module=module_name, deps=added)

        total_changed = idea_modules.save()
        # checking library
        library = ProjectLibrary(self.idea_project_dir, self.projects_mk_file_path)
        library.update()
        library.save()

        print 'Update finished'
        print 'New modules: ', len(new_modules)
        print 'Dependencies updated: ', deps_changed_cnt
        print 'Total changed: ', total_changed


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    default_idea_dir = os.path.abspath(os.path.join(os.getcwd(), '..'))
    parser.add_argument(
        '-i',
        '--idea-path',
        help='Idea modules folder (.idea .iml files). We suppose structure is {idea_dir}/git/projects.mk, {idea_dir}/.idea/{..}',
        default=default_idea_dir,
        dest='idea_dir')
    parser.add_argument(
        '-p',
        '--projectsmk-path',
        help='Directory where projects.mk stored',
        default=os.getcwd(),
        dest='mk_dir')
    parser.add_argument(
        '-n',
        '--new-module',
        help='Creates new idea module',
        default='',
        dest='new_module')

    parser.add_argument(
        '-b',
        '--new-module-base',
        help='Base deps for new module, comma separated',
        default='http-proxy',
        dest='new_module_base')

    args = parser.parse_args()
    if not args.idea_dir:
        print 'Idea directory arg is empty, fail'
        sys.exit(1)

    if not os.path.exists(args.mk_dir):
        print 'Error projects.mk directory do not exists', args.mk_dir
        sys.exit(1)

    converter = IdeaProjectDepsConverter(args.idea_dir, os.path.join(args.mk_dir, 'projects.mk'), True)
    if args.new_module:
        print 'New module', args.new_module
        converter.new_module(args.new_module, args.new_module_base.strip().split(','))
        print 'Created, reload idea'
    else:
        print 'Updating idea project structure'
        converter.convert()
