package ru.yandex.client.producer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Future;
import java.util.function.IntPredicate;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableMap;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.collection.LazyList;
import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.http.util.BasicFuture;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.xpath.PrimitiveHandler;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapRow;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.searchmap.ZooKeeperAddress;
import ru.yandex.util.timesource.TimeSource;

/**
 * Slowly get your hands from keyboard and carefully read this.
 * Producer client guarantee that it returns sorted list of hosts,
 * sorted by queue position. Keep it in mind
 */
public class ProducerClient extends AbstractAsyncClient<ProducerClient> {
    private static final QueueHostInfo BAD_HOST =
        new QueueHostInfo(new HttpHost("bad:80"));
    private static final HttpRequest FAKE_REQUEST =
        new BasicHttpRequest("GET", "/fake");

    private final ProducerClientContext context;

    public ProducerClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableProducerClientConfig producerClientConfig,
        final SearchMap hostsResolver)
    {
        super(reactor, producerClientConfig);
        context = new ProducerClientContext(
            this,
            producerClientConfig,
            hostsResolver);
    }

    protected ProducerClient(
        final CloseableHttpAsyncClient client,
        final ProducerClient sample)
    {
        super(client, sample);
        context = sample.context;
    }

    @Override
    protected ProducerClient adjust(final CloseableHttpAsyncClient client) {
        return new ProducerClient(client, this);
    }

    public Future<List<HttpHost>> execute(final User user) {
        return execute(user, EmptyFutureCallback.INSTANCE);
    }

    public Future<List<HttpHost>> execute(
        final User user,
        final FutureCallback<? super List<HttpHost>> callback)
    {
        return execute(user, httpClientContextGenerator(), callback);
    }

    public Future<List<HttpHost>> execute(
        final User user,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super List<HttpHost>> callback)
    {
        final HostListCallbackProxy proxy = new HostListCallbackProxy(callback);
        executeWithInfo(user, contextGenerator, proxy);
        return proxy;
    }

    public Future<List<QueueHostInfo>> executeWithInfo(final User user) {
        return executeWithInfo(user, EmptyFutureCallback.INSTANCE);
    }

    /**
     * Slowly get your hands from keyboard and carefully read this.
     * This method (Actually all execute methods of this client)
     * guarantee: it will return sorted list of hosts,
     * sorted by queue position. Keep it in mind
     */
    public Future<List<QueueHostInfo>> executeWithInfo(
        final User user,
        final FutureCallback<? super List<QueueHostInfo>> callback)
    {
        return executeWithInfo(user, httpClientContextGenerator(), callback);
    }

    public Future<List<QueueHostInfo>> executeWithInfo(
        final User user,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super List<QueueHostInfo>> callback)
    {
        TopHostsCallbackProxy proxy = new TopHostsCallbackProxy(callback);
        executeAllWithInfo(user, contextGenerator, proxy);
        return proxy;
    }

    public Future<List<QueueHostInfo>> executeAllWithInfo(
        final User user,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super List<QueueHostInfo>> callback)
    {
        if (context.streaming) {
            checkStream(user.service());
        }
        Future<List<QueueHostInfo>> result;
        if (context.updateCacheDaemon != null || context.streaming) {
            HostsEntry hostsEntry = context.getHostsEntry(user);
            long now = TimeSource.INSTANCE.currentTimeMillis();
            if (hostsEntry != null
                && hostsEntry.lastUpdate() + context.cacheTtl > now)
            {
                List<QueueHostInfo> hosts = hostsEntry.hosts();
                if (hosts.size() > 0) {
                    fakeListenerEntry(
                        contextGenerator,
                        "cache." + hostsEntry.lastUpdate());
                    hostsEntry.lastUsed = now;
                    callback.completed(hosts);
                    result = new CompletedFuture<>(hosts);
                } else {
                    UpdateCacheCallback updateCacheCallback =
                        new UpdateCacheCallback(context, user, callback);
                    result = processExecute(
                        user,
                        contextGenerator,
                        updateCacheCallback,
                        false);
                }
            } else {
                UpdateCacheCallback updateCacheCallback =
                    new UpdateCacheCallback(context, user, callback);
                result = processExecute(
                    user,
                    contextGenerator,
                    updateCacheCallback,
                    false);
            }
        } else {
            result = processExecute(user, contextGenerator, callback, false);
        }
        return result;
    }

    private void fakeListenerEntry(
        final Supplier<? extends HttpClientContext> contextGenerator,
        final String entry)
    {
        HttpClientContext httpContext = contextGenerator.get();
        RequestsListener listener =
            (RequestsListener) httpContext.getAttribute(
                RequestsListener.LISTENER);
        if (listener != null) {
            FutureCallback<Object> fakeCallback =
                listener.<Object>createCallbackFor(
                    new HttpRoute(new HttpHost(entry)),
                    FAKE_REQUEST,
                    httpContext);
            fakeCallback.completed(new Object());
        }
    }

    private void checkStream(final String service) {
        Object running = context.streams.get(service);
        if (running == null) {
            running =
                context.streams.putIfAbsent(service, new Object());
            if (running == null) {
                startStreams(context, service);
            }
        }
    }

    private Future<List<QueueHostInfo>> processExecute(
        final User user,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super List<QueueHostInfo>> callback,
        final boolean noTimeout)
    {
        return processExecute(
            user.service(),
            (int) user.shard(),
            contextGenerator,
            callback,
            noTimeout);
    }

    private Future<List<QueueHostInfo>> processExecute(
        final String service,
        final int shardId,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super List<QueueHostInfo>> callback,
        final boolean noTimeout)
    {
        BasicFuture<List<QueueHostInfo>> future =
            new BasicFuture<>(callback);
        StringBuilder sb = new StringBuilder("/_status?service=");
        sb.append(service);
        sb.append("&prefix=");
        sb.append(shardId);
        SearchMapRow searchMapRow = context.hostsResolver.row(service);
        if (searchMapRow == null) {
            return new CompletedFuture<>(Collections.emptyList());
        }
        SearchMapShard searchMapShard = searchMapRow.get(shardId);
        int size = searchMapShard.size();
        Map<String, QueueHostInfo> searchHosts = new HashMap<>(size << 1);
        for (int i = 0; i < size; ++i) {
            HttpHost searchHost = searchMapShard.get(i).searchHost();
            if (searchHost != null) {
                searchHosts.put(
                    searchHost.getHostName(),
                    new QueueHostInfo(searchHost));
            }
        }
        List<HttpHost> hosts;
        HttpAsyncResponseConsumerFactory<List<QueueHostInfo>> consumerFactory;
        final boolean executeWithDelay;
        if (context.maxTotalTime == 0L || noTimeout) {
            executeWithDelay = true;
        } else {
            executeWithDelay = false;
        }
        if (size == 0 || searchMapShard.zk().isEmpty()) {
            hosts = Collections.singletonList(context.host);
            if (context.allowCached) {
                sb.append("&allow_cached");
            }
        } else {
            List<ZooKeeperAddress> zkHosts = searchMapShard.zk();
            if (executeWithDelay) {
                zkHosts = new ArrayList<>(zkHosts);
                Collections.rotate(zkHosts, context.hostsRotate[0]++);
            }
            hosts = new LazyList<>(zkHosts, x -> x.host());
        }
        consumerFactory = new StatusCheckAsyncResponseConsumerFactory<>(
            context.statusCodePredicate,
            new ZoolooserJsonStatusConsumerFactory(searchHosts));
        sb.append("&all&json-type=dollar");

        FutureCallback<? super List<QueueHostInfo>> realCallback;
        if (context.fallbackToSearchMap) {
            realCallback =
                new ErrorSuppressingFutureCallback<List<QueueHostInfo>>(
                    future,
                    ErrorSuppressingFutureCallback.ALL_ERRORS_AS_HTTP_CLASSIFIER,
                    () -> new ArrayList<>(searchHosts.values()));
        } else {
            realCallback = future;
        }
        if (executeWithDelay) {
            executeWithDelay(
                hosts,
                new BasicAsyncRequestProducerGenerator(new String(sb)),
                context.failoverDelay,
                consumerFactory,
                contextGenerator,
                realCallback);
        } else {
            execute(
                hosts,
                new BasicAsyncRequestProducerGenerator(new String(sb)),
                TimeSource.INSTANCE.currentTimeMillis() + context.maxTotalTime,
                context.failoverDelay,
                consumerFactory,
                contextGenerator,
                realCallback);
        }
        return future;
    }

    private static void startStreams(
        final ProducerClientContext context,
        final String service)
    {
        Set<List<ZooKeeperAddress>> zkHosts =
            context.hostsResolver.zkHosts(service);
        System.err.println("Starting streams for service: " + service
            + ", hosts: " + zkHosts);
        for (List<ZooKeeperAddress> zkShard: zkHosts) {
            StatusStreamCallback callback =
                new StatusStreamCallback(context, service, zkShard);
            startStream(context, zkShard, service, callback);
        }
    }

    private static void startStream(
        final ProducerClientContext context,
        final List<ZooKeeperAddress> zkShard,
        final String service,
        final StatusStreamCallback callback)
    {
        System.err.println("Starting stream for service: " + service
            + ", shard: " + zkShard);
        HttpAsyncResponseConsumerFactory<Object> consumerFactory;
        List<HttpHost> hosts = new LazyList<>(zkShard, x -> x.host());
        consumerFactory =
            new ZoolooserJsonStatusStreamConsumerFactory(callback);
        StringBuilder sb = new StringBuilder("/_statusStream?service=");
        sb.append(service);
        sb.append("&json-type=dollar");
        context.client.execute(
            hosts,
            new BasicAsyncRequestProducerGenerator(new String(sb)),
            Long.MAX_VALUE,
            consumerFactory,
            new StatusStreamRestartCallback(context, service, callback));
    }

    @Override
    public void start() {
        super.start();
        if (context.updateCacheDaemon != null) {
            context.updateCacheDaemon.setDaemon(true);
            context.updateCacheDaemon.start();
        }
    }

    @Override
    public void close() throws IOException {
        super.close();
        if (context.updateCacheDaemon != null) {
            context.updateCacheDaemon.setStopped();
            context.updateCacheDaemon.interrupt();
        }
    }

    private static class ProducerClientContext {
        private final Map<String, String> hostsUpdateInternMap =
            new ConcurrentHashMap<>();
        private final ProducerClient client;
        private final SearchMap hostsResolver;
        private final HttpHost host;
        private final boolean allowCached;
        private final IntPredicate statusCodePredicate;
        private final boolean fallbackToSearchMap;
        private final long maxTotalTime;
        private final long failoverDelay;
        private final long cacheTtl;
        private final long cacheUpdateInterval;
        private final boolean streaming;
        private final ConcurrentHashMap<String, Object> streams;
        //this should not be volatile nor Atomic for speed
        //precise compute is not required here
        private final int[] hostsRotate = new int[1];
        private final Map<String, String> hostsBaseInternMap;
        // Map from service name to service shards hosts entries
        private final Map<String, HostsEntry[]> shardIdToHosts;
        private final UpdateCache updateCacheDaemon;

        ProducerClientContext(
            final ProducerClient client,
            final ImmutableProducerClientConfig producerClientConfig,
            final SearchMap hostsResolver)
        {
            this.client = client;
            this.hostsResolver = hostsResolver;
            host = producerClientConfig.host();
            allowCached = producerClientConfig.allowCached();
            if (allowCached) {
                statusCodePredicate =
                    x -> x == HttpStatus.SC_OK
                    || x == HttpStatus.SC_NON_AUTHORITATIVE_INFORMATION;
            } else {
                statusCodePredicate = HttpStatusPredicates.OK;
            }
            fallbackToSearchMap = producerClientConfig.fallbackToSearchMap();
            maxTotalTime = producerClientConfig.maxTotalTime();
            failoverDelay = producerClientConfig.failoverDelay();
            cacheTtl = producerClientConfig.cacheTtl();
            cacheUpdateInterval = producerClientConfig.cacheUpdateInterval();
            streaming = producerClientConfig.streaming();
            streams = new ConcurrentHashMap<>();
            Set<String> hostnames = new HashSet<>();
            ImmutableMap.Builder<String, String> internMapBuilder =
                ImmutableMap.builder();
            for (HttpHost host: hostsResolver.searchHosts()) {
                String hostname = host.getHostName();
                if (hostnames.add(hostname)) {
                    internMapBuilder.put(hostname, hostname);
                }
            }
            hostsBaseInternMap = internMapBuilder.build();

            ImmutableMap.Builder<String, HostsEntry[]> shardIdToHostsBuilder =
                ImmutableMap.builder();
            for (String service: hostsResolver.names()) {
                HostsEntry[] hostsEntries =
                    new HostsEntry[(int) SearchMap.SHARDS_COUNT];
                for (int i = 0; i < SearchMap.SHARDS_COUNT; ++i) {
                    hostsEntries[i] = new HostsEntry(0L);
                }
                shardIdToHostsBuilder.put(service, hostsEntries);
            }
            shardIdToHosts = shardIdToHostsBuilder.build();

            if (cacheUpdateInterval > 0 && cacheTtl > 0) {
                updateCacheDaemon = new UpdateCache(this);
            } else {
                updateCacheDaemon = null;
            }
        }

        public HostsEntry getHostsEntry(final User user) {
            return getHostsEntry(user.service(), (int) user.shard());
        }

        public HostsEntry getHostsEntry(
            final String service,
            final int shardId)
        {
            HostsEntry[] hostsEntries = shardIdToHosts.get(service);
            if (hostsEntries == null) {
                return null;
            } else {
                return hostsEntries[shardId];
            }
        }

        private String internHostName(final String string) {
            String interned = hostsBaseInternMap.get(string);
            if (interned == null) {
                interned = hostsUpdateInternMap.putIfAbsent(string, string);
                if (interned == null) {
                    interned = string;
                }
            }
            return interned;
        }

        public void updateStatus(
            final String service,
            final int shardId,
            final List<QueueHostInfo> hosts)
        {
            long lastUpdate = TimeSource.INSTANCE.currentTimeMillis();
            HostsEntry hostsEntry = getHostsEntry(service, shardId);
            if (hostsEntry != null) {
                synchronized (hostsEntry) {
                    for (QueueHostInfo newInfo: hosts) {
                        hostsEntry.put(
                            internHostName(newInfo.host().getHostName()),
                            newInfo);
                    }
                }
                hostsEntry.lastUpdate(lastUpdate);
            }
        }

        @SuppressWarnings("ReferenceEquality")
        public void updateStatus(
            final String service,
            final int shardId,
            final String hostname,
            final long pos,
            final long lastUpdate)
        {
            HostsEntry hostsEntry = getHostsEntry(service, shardId);
            if (hostsEntry != null) {
                String host = internHostName(hostname);
                synchronized (hostsEntry) {
                    QueueHostInfo info = hostsEntry.get(host);
                    if (info == null) {
                        info = getInfo(service, shardId, host);
                        if (info != BAD_HOST) {
                            info.queueId(pos);
                        }
                        hostsEntry.put(host, info);
                    } else if (info != BAD_HOST) {
                        info.queueId(pos);
                    }
                }
                hostsEntry.lastUpdate(lastUpdate);
            }
        }

        private QueueHostInfo getInfo(
            final String service,
            final int shardId,
            final String host)
        {
            SearchMapRow searchMapRow = hostsResolver.row(service);
            if (searchMapRow != null) {
                SearchMapShard searchMapShard = searchMapRow.get(shardId);
                int size = searchMapShard.size();
                for (int i = 0; i < size; ++i) {
                    HttpHost searchHost = searchMapShard.get(i).searchHost();
                    if (searchHost != null
                        && searchHost.getHostName().equals(host))
                    {
                        return new QueueHostInfo(searchHost);
                    }
                }
            }
            return BAD_HOST;
        }
    }

    public static class UpdateCacheCallback
        implements FutureCallback<List<QueueHostInfo>>
    {
        private final ProducerClientContext context;
        private final String service;
        private final int shardId;
        private final FutureCallback<? super List<QueueHostInfo>> nextCallback;

        public UpdateCacheCallback(
            final ProducerClientContext context,
            final User user,
            final FutureCallback<? super List<QueueHostInfo>> nextCallback)
        {
            this(context, user.service(), (int) user.shard(), nextCallback);
        }

        public UpdateCacheCallback(
            final ProducerClientContext context,
            final String service,
            final int shardId,
            final FutureCallback<? super List<QueueHostInfo>> nextCallback)
        {
            this.context = context;
            this.service = service;
            this.shardId = shardId;
            this.nextCallback = nextCallback;
        }

        @Override
        public void completed(final List<QueueHostInfo> hosts) {
            if (hosts instanceof PrimitiveHandler) {
                context.updateStatus(service, shardId, hosts);
                if (!context.streaming) {
                    context.updateCacheDaemon.addToCacheToUpdate(
                        new CacheInfo(
                            service,
                            shardId,
                            TimeSource.INSTANCE.currentTimeMillis()));
                }
            }
            if (nextCallback != null) {
                nextCallback.completed(hosts);
            }
        }

        @Override
        public void cancelled() {
            if (context.updateCacheDaemon != null) {
                context.updateCacheDaemon.addToCacheToUpdate(
                    new CacheInfo(
                        service,
                        shardId,
                        TimeSource.INSTANCE.currentTimeMillis()));
            }

            if (nextCallback != null) {
                nextCallback.cancelled();
            }
        }

        @Override
        public void failed(final Exception e) {
            if (context.updateCacheDaemon != null) {
                context.updateCacheDaemon.addToCacheToUpdate(
                    new CacheInfo(
                        service,
                        shardId,
                        TimeSource.INSTANCE.currentTimeMillis()));
            }

            if (nextCallback != null) {
                nextCallback.failed(e);
            }
        }
    }

    private static class UpdateCache extends Thread {
        private final ConcurrentLinkedQueue<CacheInfo> cacheToUpdate =
                new ConcurrentLinkedQueue<CacheInfo>();
        private final ProducerClientContext context;
        private volatile boolean stopped;

        UpdateCache(final ProducerClientContext context) {
            this.context = context;
        }

        private boolean shouldUpdateShard(
            final String service,
            final int shardId)
        {
            HostsEntry[] hostsEntries = context.shardIdToHosts.get(service);
            if (hostsEntries == null) {
                return true;
            } else {
                HostsEntry hostsEntry = hostsEntries[shardId];
                long now = TimeSource.INSTANCE.currentTimeMillis();
                return now - hostsEntry.lastUsed() <= context.cacheTtl;
            }
        }

        @Override
        public void run() {
            while (!stopped) {
                boolean needDelay = true;
                while (!cacheToUpdate.isEmpty()) {
                    needDelay = false;
                    CacheInfo cacheInfo = cacheToUpdate.peek();
                    long now = TimeSource.INSTANCE.currentTimeMillis();
                    if (cacheInfo.lastModified()
                        + context.cacheUpdateInterval > now)
                    {
                        long sleepFor = (cacheInfo.lastModified()
                            + context.cacheUpdateInterval) - now;
                        try {
                            Thread.sleep(sleepFor);
                        } catch (InterruptedException e) {
                        }
                    } else {
                        boolean removed;
                        do {
                            removed = cacheToUpdate.remove(cacheInfo);
                        } while (removed);

                        String service = cacheInfo.service();
                        int shardId = cacheInfo.shardId();
                        if (context.streaming
                            || shouldUpdateShard(service, shardId))
                        {
                            context.client.processExecute(
                                service,
                                shardId,
                                context.client.httpClientContextGenerator(),
                                new UpdateCacheCallback(
                                    context,
                                    service,
                                    shardId,
                                    null),
                                true);
                        }
                    }
                }
                if (needDelay) {
                    try {
                        Thread.sleep(context.cacheUpdateInterval);
                    } catch (InterruptedException e) {
                    }
                }
            }
        }

        public void addToCacheToUpdate(final CacheInfo info) {
            cacheToUpdate.add(info);
        }

        public ConcurrentLinkedQueue<CacheInfo> cacheToUpdate() {
            return cacheToUpdate;
        }

        public void setStopped() {
            stopped = true;
            interrupt();
        }
    }

    private static class HostsEntry extends HashMap<String, QueueHostInfo> {
        private static final long serialVersionUID = 0L;

        private volatile long lastUsed;
        private volatile long lastUpdate;

        HostsEntry(final long lastUpdate) {
            this.lastUpdate = lastUpdate;
            lastUsed = TimeSource.INSTANCE.currentTimeMillis();
        }

        @SuppressWarnings("ReferenceEquality")
        public List<QueueHostInfo> hosts() {
            List<QueueHostInfo> hosts;
            synchronized (this) {
                int size = size();
                if (size == 0) {
                    return Collections.emptyList();
                } else {
                    hosts = new ArrayList<>(size);
                    for (QueueHostInfo info: values()) {
                        if (info != BAD_HOST) {
                            hosts.add(new QueueHostInfo(info));
                        }
                    }
                }
            }

            Collections.sort(hosts);
            return hosts;
        }

        public void lastUpdate(final long lastUpdate) {
            this.lastUpdate = lastUpdate;
        }

        public long lastUpdate() {
            return lastUpdate;
        }

        public long lastUsed() {
            return lastUsed;
        }
    }

    private static class CacheInfo {
        private final String service;
        private final int shardId;
        private final long lastModified;

        CacheInfo(
            final String service,
            final int shardId,
            final long lastModified)
        {
            this.service = service;
            this.shardId = shardId;
            this.lastModified = lastModified;
        }

        public long lastModified() {
            return lastModified;
        }

        public String service() {
            return service;
        }

        public int shardId() {
            return shardId;
        }

        @Override
        public int hashCode() {
            return service.hashCode() + shardId;
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof CacheInfo) {
                CacheInfo other = (CacheInfo) o;
                return shardId == other.shardId
                    && service.equals(other.service);
            }
            return false;
        }
    }

    private static class HostListCallbackProxy
        extends CallbackFutureBase<List<HttpHost>, List<QueueHostInfo>>
    {
        HostListCallbackProxy(
            final FutureCallback<? super List<HttpHost>> callback)
        {
            super(callback);
        }

        @Override
        public List<HttpHost> convertResult(final List<QueueHostInfo> infos) {
            final ArrayList<HttpHost> result = new ArrayList<>(infos.size());
            for (QueueHostInfo info : infos) {
                result.add(info.host());
            }
            return result;
        }
    }

    private static class TopHostsCallbackProxy
        extends BasicFuture<List<QueueHostInfo>>
    {
        TopHostsCallbackProxy(
            final FutureCallback<? super List<QueueHostInfo>> callback)
        {
            super(callback);
        }

        @Override
        public List<QueueHostInfo> convertResult(
            final List<QueueHostInfo> infos)
        {
            if (!infos.isEmpty()) {
                long topPos = infos.get(0).queueId();
                for (int i = infos.size(); i-- > 0;) {
                    if (infos.get(i).queueId() != topPos) {
                        infos.remove(i);
                    } else {
                        break;
                    }
                }
            }
            return infos;
        }
    }

    private static class StatusStreamCallback implements StatusCallback {
        private final ProducerClientContext context;
        private final List<ZooKeeperAddress> zkShard;
        private final String service;

        StatusStreamCallback(
            final ProducerClientContext context,
            final String service,
            final List<ZooKeeperAddress> zkShard)
        {
            this.context = context;
            this.service = service;
            this.zkShard = zkShard;
        }

        @Override
        public void status(final String host, final int shard, final long pos) {
            if (host != null) {
                context.updateStatus(
                    service,
                    shard,
                    host,
                    pos,
                    TimeSource.INSTANCE.currentTimeMillis());
            }
        }
    }

    private static class StatusStreamRestartCallback
        implements FutureCallback<Object>
    {
        private final ProducerClientContext context;
        private final String service;
        private final StatusStreamCallback callback;

        StatusStreamRestartCallback(
            final ProducerClientContext context,
            final String service,
            final StatusStreamCallback callback)
        {
            this.context = context;
            this.service = service;
            this.callback = callback;
        }

        @Override
        public void completed(final Object result) {
            startStream(context, callback.zkShard, service, callback);
        }

        @Override
        public void failed(final Exception e) {
            e.printStackTrace();
            startStream(context, callback.zkShard, service, callback);
        }

        @Override
        public void cancelled() {
        }
    }
}
