package ru.yandex.search.request.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.regex.Pattern;

import ru.yandex.function.CharArrayProcessor;
import ru.yandex.util.string.NormalizingProcessor;
import ru.yandex.util.string.StringUtils;

public class SearchRequestText {
    public static final UnaryOperator<String> DEFAULT_WORDS_MODIFIER =
        UnaryOperator.identity();
    public static final UnaryOperator<String> DEFAULT_PHRASES_MODIFIER =
        phrase -> StringUtils.concat('"', phrase, '"');

    private static final String OR = " OR ";
    private static final String B_AND_B = ") AND (";
    private static final String ESCAPE = "\\\\$1";
    private static final String ASTERISK = "*";
    private static final String SPACE = " ";
    private static final String BASIC_CHARS = "/\\\\(){}\\[\\]'?~+:!^&|";
    private static final String NON_WORD_CHARS = BASIC_CHARS + ASTERISK;
    private static final Pattern DOUBLE_QUOTE = Pattern.compile("\"");
    private static final Pattern SPACES = Pattern.compile(" +");
    private static final Pattern QUOTE_ESCAPE = Pattern.compile("([\\\\\"])");
    private static final Pattern FULL_ESCAPE =
        Pattern.compile("([" + BASIC_CHARS + " \"-])");
    private static final Pattern NON_WORD_CHARS_PATTERN =
        Pattern.compile('[' + NON_WORD_CHARS + ']' + '+');
    // Separate pattern for removing surrogate pairs because of
    // https://bugs.openjdk.java.net/browse/JDK-8149446
    private static final Pattern SURROGATE =
        Pattern.compile("[^\\u0000-\\uffff]");
    private static final Pattern SUGGEST_CLEANER =
        Pattern.compile("[^\\p{Graph}]+", Pattern.UNICODE_CHARACTER_CLASS);
    private static final Pattern NON_WORDS =
        Pattern.compile("[^\\p{Alnum}]+", Pattern.UNICODE_CHARACTER_CLASS);
    private static final CharArrayProcessor<String, RuntimeException> CLEANER =
            (buf, off, len) -> {
                StringBuilder sb = new StringBuilder(len);
                for (int i = 0; i < len; ++i) {
                    char c = buf[off + i];
                    if (Character.isISOControl(c)
                            || Character.isSpaceChar(c)
                            || Character.isWhitespace(c)) {
                        sb.append(' ');
                    } else {
                        sb.append(c);
                    }
                }
                return new String(sb);
            };

    private final String text;
    private final Collection<String> words;
    private final Collection<String> mentions;
    private final Collection<String> negations;
    private final Collection<String> phrases;

    public SearchRequestText(final String text) {
        this(text, DEFAULT_WORDS_MODIFIER);
    }

    public SearchRequestText(
        final String text,
        final UnaryOperator<String> wordsModifier)
    {
        this(text, wordsModifier, DEFAULT_PHRASES_MODIFIER);
    }

    public SearchRequestText(
        final String text,
        final UnaryOperator<String> wordsModifier,
        final UnaryOperator<String> phrasesModifier)
    {
        this.text = text;
        SearchCollector collector =
            new SearchCollector(wordsModifier, phrasesModifier);
        parse(normalize(text), collector);
        words = collector.words();
        negations = collector.negations();
        phrases = collector.phrases();
        mentions = Collections.emptyList();
    }

    // CSOFF: ParameterNumber
    public SearchRequestText(
        final String text,
        final Collection<String> words,
        final Collection<String> negations,
        final Collection<String> phrases,
        final Collection<String> mentions)
    {
        this.text = text;
        this.words = words;
        this.negations = negations;
        this.phrases = phrases;
        this.mentions = mentions;
    }
    // CSON: ParameterNumber

    public String text() {
        return text;
    }

    public Collection<String> words() {
        return words;
    }

    public Collection<String> mentions() {
        return mentions;
    }

    public Collection<String> negations() {
        return negations;
    }

    public Collection<String> phrases() {
        return phrases;
    }

    public static String normalize(final String text) {
        NormalizingProcessor processor = new NormalizingProcessor(true);
        processor.process(text.toCharArray());
        return processor.processWith(CLEANER);
    }

    public static String normalizeSuggest(final String text) {
        NormalizingProcessor processor = new NormalizingProcessor(true);
        processor.process(text.toCharArray());
        String withoutSurrogate = SURROGATE.matcher(processor.toString())
            .replaceAll("");
        return SUGGEST_CLEANER.matcher(withoutSurrogate)
            .replaceAll(SPACE).trim();
    }

    public String quoteEscape() {
        return quoteEscape(text);
    }

    public static String quoteEscape(final String text) {
        return QUOTE_ESCAPE.matcher(text).replaceAll(ESCAPE);
    }

    public String fullEscape(final boolean escapeForWildcardQuery) {
        return fullEscape(text, escapeForWildcardQuery);
    }

    public static String fullEscape(
        final String text,
        final boolean escapeForWildcardQuery)
    {
        String escaped = FULL_ESCAPE.matcher(text).replaceAll(ESCAPE);
        String asteriskReplacement;
        if (escapeForWildcardQuery) {
            asteriskReplacement = "\\\\\\\\*";
        } else {
            asteriskReplacement = "\\*";
        }
        return escaped.replace(ASTERISK, asteriskReplacement);
    }

    public boolean isEmpty() {
        return
            words.isEmpty()
            && negations.isEmpty()
            && phrases.isEmpty()
            && mentions.isEmpty();
    }

    public boolean isEmptyIgnoreMentions() {
        return
            words.isEmpty()
            && negations.isEmpty()
            && phrases.isEmpty();
    }

    public boolean hasWords() {
        return words.size() + phrases.size() > 0;
    }

    public boolean hasNegations() {
        return !negations.isEmpty();
    }

    public boolean singleWord() {
        boolean ret = false;
        if (words.size() + phrases.size() == 1) {
            ret = true;
        } else {
            //if expandLast (ex: art + art*)
            if (words.size() == 2) {
                Iterator<String> iter = words.iterator();
                String first = iter.next();
                String second = iter.next();
                if (second.startsWith(first)
                    && second.charAt(second.length() - 1) == '*'
                    && second.length() - first.length() == 1)
                {
                    ret = true;
                }
            }
        }
        return ret;
    }

    public boolean hasMentions() {
        return mentions.size() > 0;
    }

    public boolean singleMentionWord() {
        boolean ret = false;
        if (mentions.size() == 1) {
            ret = true;
        } else {
            //if expandLast (ex: art + art*)
            if (mentions.size() == 2) {
                Iterator<String> iter = mentions.iterator();
                String first = iter.next();
                String second = iter.next();
                if (second.startsWith(first)
                    && second.charAt(second.length() - 1) == '*'
                    && second.length() - first.length() == 1)
                {
                    ret = true;
                }
            }
        }
        return ret;
    }

    private static boolean appendQueries(
        final StringBuilder sb,
        final Supplier<? extends QueryAppender> queries,
        final String braceSeparator)
    {
        QueryAppender query = queries.get();
        boolean hasQueries = query != null;
        while (query != null) {
            query.appendTo(sb);
            sb.append(OR);
            query = queries.get();
        }
        if (hasQueries) {
            sb.setLength(sb.length() - OR.length());
            sb.append(braceSeparator);
        }
        return hasQueries;
    }

    // CSOFF: ParameterNumber
    private static boolean appendWords(
        final Collection<String> words,
        final StringBuilder sb,
        final Function<
            ? super String,
            ? extends Supplier<? extends QueryAppender>>
        queriesFactory,
        final String braceSeparator)
    {
        boolean hasQueries = false;
        for (String word: words) {
            Supplier<? extends QueryAppender> queries =
                queriesFactory.apply(word);
            if (appendQueries(sb, queries, braceSeparator)) {
                hasQueries = true;
            }
        }
        return hasQueries;
    }
    // CSON: ParameterNumber

    public String mentionsQuery(final String... fields) {
        StringBuilder sb = new StringBuilder();
        mentionsQuery(sb, fields);
        return new String(sb);
    }

    public void mentionsQuery(
        final StringBuilder sb,
        final String... fields)
    {
        mentionsQuery(sb, Arrays.asList(fields));
    }

    public void mentionsQuery(
        final StringBuilder sb,
        final Collection<String> fields)
    {
        mentionsQuery(sb, new FieldsTermsSupplierFactory(fields));
    }

    //CSOFF: ParameterNumber
    public void mentionsQuery(
        final StringBuilder sb,
        final List<String> fields,
        final String braceSeparator,
        final float initialBoost)
    {
        mentionsQuery(
            sb,
            new BoostByOrderFieldsTermsSupplierFactory(initialBoost, fields),
            braceSeparator);
    }
    //CSON: ParameterNumber

    public void mentionsQuery(
        final StringBuilder sb,
        final Function<
            ? super String,
            ? extends Supplier<? extends QueryAppender>>
        queriesFactory)
    {
        mentionsQuery(sb, queriesFactory, B_AND_B);
    }

    //CSOFF: ParameterNumber
    public void mentionsQuery(
        final StringBuilder sb,
        final Function<
            ? super String,
            ? extends Supplier<? extends QueryAppender>>
        queriesFactory,
        final String braceSeparator)
    {
        sb.append('(');
        boolean empty =
            !appendWords(mentions, sb, queriesFactory, braceSeparator);
        if (empty) {
            sb.setLength(sb.length() - 1);
        } else {
            sb.setLength(sb.length() + 1 - braceSeparator.length());
        }
    }
    //CSON: ParameterNumber

    public String fieldsQuery(final String... fields) {
        StringBuilder sb = new StringBuilder();
        fieldsQuery(sb, fields);
        return new String(sb);
    }

    public void fieldsQuery(final StringBuilder sb, final String... fields) {
        fieldsQuery(sb, Arrays.asList(fields));
    }

    public void fieldsQuery(
        final StringBuilder sb,
        final Collection<String> fields)
    {
        fieldsQuery(sb, new FieldsTermsSupplierFactory(fields));
    }

    //CSOFF: ParameterNumber
    public void fieldsQuery(
        final StringBuilder sb,
        final List<String> fields,
        final String braceSeparator,
        final float initialBoost)
    {
        fieldsQuery(
            sb,
            new BoostByOrderFieldsTermsSupplierFactory(initialBoost, fields),
            braceSeparator);
    }
    //CSON: ParameterNumber

    public void fieldsQuery(
        final StringBuilder sb,
        final Function<
            ? super String,
            ? extends Supplier<? extends QueryAppender>>
        queriesFactory)
    {
        fieldsQuery(sb, queriesFactory, B_AND_B);
    }

    public void fieldsQuery(
        final StringBuilder sb,
        final Function<
            ? super String,
            ? extends Supplier<? extends QueryAppender>>
        queriesFactory,
        final String braceSeparator)
    {
        sb.append('(');
        boolean empty =
            !appendWords(words, sb, queriesFactory, braceSeparator);
        if (appendWords(phrases, sb, queriesFactory, braceSeparator)) {
            empty = false;
        }
        if (empty) {
            sb.setLength(sb.length() - 1);
        } else {
            sb.setLength(sb.length() + 1 - braceSeparator.length());
        }
    }

    public void negationsQuery(
        final StringBuilder sb,
        final String... fields)
    {
        negationsQuery(sb, Arrays.asList(fields));
    }

    public void negationsQuery(
        final StringBuilder sb,
        final Collection<String> fields)
    {
        negationsQuery(sb, new FieldsTermsSupplierFactory(fields));
    }

    public void negationsQuery(
        final StringBuilder sb,
        final Function<
            ? super String,
            ? extends Supplier<? extends QueryAppender>>
        queriesFactory)
    {
        for (String negation: negations) {
            Supplier<? extends QueryAppender> queries =
                queriesFactory.apply(negation);
            QueryAppender query = queries.get();
            while (query != null) {
                sb.append(" AND NOT ");
                query.appendTo(sb);
                query = queries.get();
            }
        }
    }

    public static void parse(
        final String text,
        final SearchRequestTextCollector collector)
    {
        parse(text, NON_WORD_CHARS_PATTERN, collector);
    }

    @SuppressWarnings("StringSplitter")
    public static void parse(
        final String text,
        final Pattern wordSplitPattern,
        final SearchRequestTextCollector collector)
    {
        String[] parts =
            DOUBLE_QUOTE.split(
                wordSplitPattern.matcher(text).replaceAll(SPACE));
        for (int i = 0; i < parts.length; ++i) {
            String part = parts[i].trim();
            if (!part.isEmpty()) {
                if ((i & 1) == 0) {
                    for (String token: SPACES.split(part)) {
                        if (token.chars()
                            .anyMatch(Character::isLetterOrDigit))
                        {
                            if (token.charAt(0) == '-') {
                                if (token.charAt(1) != '-') {
                                    collector.acceptNegation(
                                        token.substring(1));
                                } else {
                                    collector.acceptWord(
                                        StringUtils.concat('\\', token));
                                }
                            } else if (token.charAt(0) == '@') {
                                if (token.charAt(1) != '@') {
                                    collector.acceptMention(token.substring(1));
                                } else {
                                    collector.acceptWord(
                                        StringUtils.concat('\\', token));
                                }
                            } else {
                                collector.acceptWord(token);
                            }
                        }
                    }
                } else if (part.chars().anyMatch(Character::isLetterOrDigit)) {
                    collector.acceptPhrase(part);
                }
            }
        }
    }

    public static SearchRequestText parse(
        final String text,
        final UnaryOperator<String> wordsModifier,
        final UnaryOperator<String> phrasesModifier)
    {
        SearchCollector collector =
            new SearchCollector(wordsModifier, phrasesModifier);
        parse(normalize(text), collector);
        return new SearchRequestText(
            text,
            collector.words(),
            collector.negations(),
            collector.phrases(),
            Collections.emptyList());
    }

    public static boolean incompleteRequest(final String text) {
        int len = text.length();
        return len != 0 && Character.isLetterOrDigit(text.charAt(len - 1));
    }

    public static boolean specialKeywordRequest(final String text) {
        int len = text.length();
        return len != 0 && !Character.isWhitespace(text.charAt(len - 1));
    }

    private static Set<String> modify(
        final Collection<String> words,
        final UnaryOperator<String> wordsModifier)
    {
        Set<String> result = new LinkedHashSet<>(words.size() << 1);
        for (String word: words) {
            result.add(wordsModifier.apply(word));
        }
        return result;
    }

    public static SearchRequestText parseSuggest(
        final String text,
        final Locale locale)
    {
        return parseSuggest(text, locale, false);
    }

    public static SearchRequestText parseSuggest(
        final String text,
        final boolean expandLast)
    {
        return parseSuggest(text, Locale.ROOT, expandLast);
    }

    public static SearchRequestText parseSuggest(
        final String text,
        final Locale locale,
        final boolean expandLast)
    {
        return parseSuggest(
            text,
            new SuggestWordsModifier(locale),
            new SuggestPhrasesModifier(locale),
            expandLast);
    }

    //CSOFF: ParameterNumber
    @SuppressWarnings("StringSplitter")
    public static SearchRequestText parseSuggest(
        final String text,
        final UnaryOperator<String> wordsModifier,
        final UnaryOperator<String> phrasesModifier,
        final boolean expandLast)
    {
        String normalized = normalizeSuggest(text);
        SuggestCollector collector = new SuggestCollector();
        parse(normalized, collector);
        List<String> last = collector.last();
        if (last != null) {
            if (incompleteRequest(text) && last != collector.phrases()) {
                int pos = last.size() - 1;
                if (expandLast) {
                    last.add(StringUtils.concat(last.get(pos), '*'));
                } else {
                    String lastWord = last.remove(pos);
                    String[] words = NON_WORDS.split(lastWord);
                    for (String word: words) {
                        if (word.chars().anyMatch(Character::isLetterOrDigit)) {
                            last.add(word);
                        }
                    }
                    last.add(
                        StringUtils.concat(last.remove(last.size() - 1), '*'));
                }
            } else if (expandLast && specialKeywordRequest(text)) {
                int pos = last.size() - 1;
                last.add(StringUtils.concat(last.get(pos), '*'));
            }
        }

        return new SearchRequestText(
            normalized,
            modify(collector.words(), wordsModifier),
            modify(collector.negations(), wordsModifier),
            modify(collector.phrases(), phrasesModifier),
            modify(collector.mentions(), wordsModifier));
    }

    public static class SearchCollector
        implements SearchRequestTextCollector
    {
        private final Set<String> words = new LinkedHashSet<>();
        private final Set<String> negations = new LinkedHashSet<>();
        private final Set<String> phrases = new LinkedHashSet<>();
        private final UnaryOperator<String> wordsModifier;
        private final UnaryOperator<String> phrasesModifier;

        public SearchCollector(
            final UnaryOperator<String> wordsModifier,
            final UnaryOperator<String> phrasesModifier)
        {
            this.wordsModifier = wordsModifier;
            this.phrasesModifier = phrasesModifier;
        }

        public Set<String> words() {
            return words;
        }

        public Set<String> negations() {
            return negations;
        }

        public Set<String> phrases() {
            return phrases;
        }

        @Override
        public void acceptWord(final String word) {
            words.add(wordsModifier.apply(word));
        }

        @Override
        public void acceptNegation(final String word) {
            negations.add(wordsModifier.apply(word));
        }

        @Override
        public void acceptPhrase(final String word) {
            phrases.add(phrasesModifier.apply(word));
        }

        @Override
        public void acceptMention(final String mention) {
            words.add(mention);
        }
    }

    public static class SuggestCollector
        implements SearchRequestTextCollector
    {
        private final List<String> words = new ArrayList<>();
        private final List<String> negations = new ArrayList<>();
        private final List<String> phrases = new ArrayList<>();
        private final List<String> mentions = new ArrayList<>();
        private List<String> last = null;

        public List<String> words() {
            return words;
        }

        public List<String> negations() {
            return negations;
        }

        public List<String> phrases() {
            return phrases;
        }

        public List<String> mentions() {
            return mentions;
        }

        public List<String> last() {
            return last;
        }

        @Override
        public void acceptWord(final String word) {
            words.add(word);
            last = words;
        }

        @Override
        public void acceptNegation(final String word) {
            negations.add(word);
            last = negations;
        }

        @Override
        public void acceptPhrase(final String word) {
            phrases.add(word);
            last = phrases;
        }

        @Override
        public void acceptMention(final String mention) {
            mentions.add(mention);
            last = mentions;
        }
    }

    public static class SuggestWordsModifier implements UnaryOperator<String> {
        private final Locale locale;

        public SuggestWordsModifier(final Locale locale) {
            this.locale = locale;
        }

        @Override
        public String apply(final String word) {
            return word.toLowerCase(locale).replace('ё', 'е');
        }
    }

    public static class SuggestPhrasesModifier extends SuggestWordsModifier {
        public SuggestPhrasesModifier(final Locale locale) {
            super(locale);
        }

        @Override
        public String apply(final String phrase) {
            return StringUtils.concat('"', super.apply(phrase), '"');
        }
    }
}

