package org.apache.lucene.analysis;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.Reader;
import java.io.IOException;
import java.util.Map;
import java.util.HashMap;

/**
 * This analyzer is used to facilitate scenarios where different
 * fields require different analysis techniques.  Use {@link #addAnalyzer}
 * to add a non-default analyzer on a field name basis.
 * 
 * <p>Example usage:
 * 
 * <pre>
 *   PerFieldAnalyzerWrapper aWrapper =
 *      new PerFieldAnalyzerWrapper(new StandardAnalyzer());
 *   aWrapper.addAnalyzer("firstname", new KeywordAnalyzer());
 *   aWrapper.addAnalyzer("lastname", new KeywordAnalyzer());
 * </pre>
 * 
 * <p>In this example, StandardAnalyzer will be used for all fields except "firstname"
 * and "lastname", for which KeywordAnalyzer will be used.
 * 
 * <p>A PerFieldAnalyzerWrapper can be used like any other analyzer, for both indexing
 * and query parsing.
 */
public class LengthFilterAnalyzerWrapper extends Analyzer {
//  private Analyzer defaultAnalyzer;
//  private Map analyzerMap = new HashMap();
    private Analyzer origAnalyzer;
    private ReusableLengthFilter filter = null;
    final int min;
    final int max;


  /**
   * Constructs with default analyzer.
   *
   * @param defaultAnalyzer Any fields not specifically
   * defined to use a different analyzer will use the one provided here.
   */
  public LengthFilterAnalyzerWrapper(Analyzer orig, int min, int max) {
    this.origAnalyzer = orig;
    this.min = min;
    this.max = max;
  }

  public TokenStream tokenStream(String fieldName, Reader reader) throws IOException {
    return new LengthFilter( origAnalyzer.tokenStream(fieldName, reader), min, max );
  }
  
  public TokenStream reusableTokenStream(String fieldName, Reader reader) throws IOException {
    TokenStream ts = origAnalyzer.reusableTokenStream( fieldName, reader );
    if( filter == null )
	filter = new ReusableLengthFilter( ts, min, max );
    else
	filter.setTokenStream( ts );
    return filter;
  }
  
  /** Return the positionIncrementGap from the analyzer assigned to fieldName */
  public int getPositionIncrementGap(String fieldName) {
    return origAnalyzer.getPositionIncrementGap(fieldName);
  }
  
  public String toString() {
    return "LengthFilterAnalyzerWrapper(" + origAnalyzer.toString() + ")";
  }
}
