package org.apache.lucene.document;
/**
 * Copyright 2006 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.search.PhraseQuery; // for javadocs
import org.apache.lucene.search.spans.SpanQuery; // for javadocs
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.index.FieldInvertState;
import org.apache.lucene.util.StringHelper; // for javadocs


/**
 *
 *
 **/
public abstract class AbstractField implements Fieldable {

  protected String name = "body";
  private static final int STORE_TERM_VECTOR_MASK = 1;
  private static final int STORE_OFFSET_WITH_TERM_VECTOR_MASK = 1 << 1;
  private static final int STORE_POSITION_WITH_TERM_VECTOR_MASK = 1 << 2;
  private static final int OMIT_NORMS_MASK = 1 << 3;
  private static final int IS_STORED_MASK = 1 << 4;
  private static final int IS_INDEXED_MASK = 1 << 5;
  private static final int IS_TOKENIZED_MASK = 1 << 6;
  private static final int IS_BINARY_MASK = 1 << 7;
  private static final int LAZY_MASK = 1 << 8;
  private static final int OMIT_TERM_FREQ_AND_POSITIONS_MASK = 1 << 9;

  private short flags;
//  private static final 
//  protected boolean storeTermVector = false;
//  protected boolean storeOffsetWithTermVector = false;
//  protected boolean storePositionWithTermVector = false;
//  protected boolean omitNorms = false;
//  protected boolean isStored = false;
  //protected boolean isIndexed = true;
//  protected boolean isTokenized = true;
  //protected boolean isBinary = false;
//  protected boolean lazy = false;
//  protected boolean omitTermFreqAndPositions = false;
  protected float boost = 1.0f;
  // the data object for all different kind of field values
  protected Object fieldsData = null;
  // pre-analyzed tokenStream for indexed fields
  protected TokenStream tokenStream;
  // length/offset for all primitive types
  protected int binaryLength;
  protected int binaryOffset;

  protected AbstractField()
  {
  }

  protected AbstractField(String name, Field.Store store, Field.Index index, Field.TermVector termVector) {
    if (name == null)
      throw new NullPointerException("name cannot be null");
    this.name = StringHelper.intern(name);        // field names are interned

    setIsStored(store.isStored());
    setIsIndexed(index.isIndexed());
    setIsTokenized(index.isAnalyzed());
    setOmitNorms(index.omitNorms());

    setIsBinary(false);

    setStoreTermVector(termVector);
  }

    public final void setIsStored(final boolean isStored) {
        if (isStored) {
            flags |= IS_STORED_MASK;
        } else {
            flags &= ~IS_STORED_MASK;
        }
    }

    public final void setIsIndexed(final boolean isIndexed) {
        if (isIndexed) {
            flags |= IS_INDEXED_MASK;
        } else {
            flags &= ~IS_INDEXED_MASK;
        }
    }

    public final void setIsTokenized(final boolean isTokenized) {
        if (isTokenized) {
            flags |= IS_TOKENIZED_MASK;
        } else {
            flags &= ~IS_TOKENIZED_MASK;
        }
    }

  /** Expert:
   *
   * If set, omit normalization factors associated with this indexed field.
   * This effectively disables indexing boosts and length normalization for this field.
   */
    public final void setOmitNorms(final boolean omitNorms) {
        if (omitNorms) {
            flags |= OMIT_NORMS_MASK;
        } else {
            flags &= ~OMIT_NORMS_MASK;
        }
    }

    public final void setIsBinary(final boolean binary) {
        if (binary) {
            flags |= IS_BINARY_MASK;
        } else {
            flags &= ~IS_BINARY_MASK;
        }
    }


    public final void setStoreTermVector(final boolean store) {
        if (store) {
            flags |= STORE_TERM_VECTOR_MASK;
        } else {
            flags &= ~STORE_TERM_VECTOR_MASK;
        }
    }


    private final void setStoreOffsetWithTermVector(final boolean store) {
        if (store) {
            flags |= STORE_OFFSET_WITH_TERM_VECTOR_MASK;
        } else {
            flags &= ~STORE_OFFSET_WITH_TERM_VECTOR_MASK;
        }
    }

    private final void setStorePositionWithTermVector(final boolean store) {
        if (store) {
            flags |= STORE_POSITION_WITH_TERM_VECTOR_MASK;
        } else {
            flags &= ~STORE_POSITION_WITH_TERM_VECTOR_MASK;
        }
    }

  /** Sets the boost factor hits on this field.  This value will be
   * multiplied into the score of all hits on this this field of this
   * document.
   *
   * <p>The boost is multiplied by {@link org.apache.lucene.document.Document#getBoost()} of the document
   * containing this field.  If a document has multiple fields with the same
   * name, all such values are multiplied together.  This product is then
   * used to compute the norm factor for the field.  By
   * default, in the {@link
   * org.apache.lucene.search.Similarity#computeNorm(FieldInvertState)} method, the boost value is multiplied
   * by the length normalization factor and then
   * rounded by {@link org.apache.lucene.search.Similarity#encodeNormValue(float)} before it is stored in the
   * index.  One should attempt to ensure that this product does not overflow
   * the range of that encoding.
   *
   * @see org.apache.lucene.document.Document#setBoost(float)
   * @see org.apache.lucene.search.Similarity#computeNorm(FieldInvertState)
   * @see org.apache.lucene.search.Similarity#encodeNormValue(float)
   */
  public void setBoost(float boost) {
    this.boost = boost;
  }

  /** Returns the boost factor for hits for this field.
   *
   * <p>The default value is 1.0.
   *
   * <p>Note: this value is not stored directly with the document in the index.
   * Documents returned from {@link org.apache.lucene.index.IndexReader#document(int)} and
   * {@link org.apache.lucene.search.IndexSearcher#doc(int)} may thus not have the same value present as when
   * this field was indexed.
   *
   * @see #setBoost(float)
   */
  public float getBoost() {
    return boost;
  }


  /** Returns the name of the field as an interned string.
   * For example "date", "title", "body", ...
   */
  public String name()    { return name; }

  protected void setStoreTermVector(Field.TermVector termVector) {
    setStoreTermVector(termVector.isStored());
    setStorePositionWithTermVector(termVector.withPositions());
    setStoreOffsetWithTermVector(termVector.withOffsets());
  }

  /** True iff the value of the field is to be stored in the index for return
    with search hits.  It is an error for this to be true if a field is
    Reader-valued. */
    public final boolean  isStored() {
        return (flags & IS_STORED_MASK) != 0;
    }

  /** True iff the value of the field is to be indexed, so that it may be
    searched on. */
    public final boolean  isIndexed() {
        return (flags & IS_INDEXED_MASK) != 0;
    }

  /** True iff the value of the field should be tokenized as text prior to
    indexing.  Un-tokenized fields are indexed as a single word and may not be
    Reader-valued. */
    public final boolean  isTokenized() {
        return (flags & IS_TOKENIZED_MASK) != 0;
    }

  /** True iff the term or terms used to index this field are stored as a term
   *  vector, available from {@link org.apache.lucene.index.IndexReader#getTermFreqVector(int,String)}.
   *  These methods do not provide access to the original content of the field,
   *  only to terms used to index it. If the original content must be
   *  preserved, use the <code>stored</code> attribute instead.
   *
   * @see org.apache.lucene.index.IndexReader#getTermFreqVector(int, String)
   */
    public final boolean isTermVectorStored() {
        return (flags & STORE_TERM_VECTOR_MASK) != 0;
    }

  /**
   * True iff terms are stored as term vector together with their offsets 
   * (start and end position in source text).
   */
    public final boolean isStoreOffsetWithTermVector() {
        return (flags & STORE_OFFSET_WITH_TERM_VECTOR_MASK) != 0;
    }

  /**
   * True iff terms are stored as term vector together with their token positions.
   */
    public final boolean isStorePositionWithTermVector() {
        return (flags & STORE_POSITION_WITH_TERM_VECTOR_MASK) != 0;
    }

  /** True iff the value of the filed is stored as binary */
    public final boolean  isBinary() {
        return (flags & IS_BINARY_MASK) != 0;
    }


  /**
   * Return the raw byte[] for the binary field.  Note that
   * you must also call {@link #getBinaryLength} and {@link
   * #getBinaryOffset} to know which range of bytes in this
   * returned array belong to the field.
   * @return reference to the Field value as byte[].
   */
  public byte[] getBinaryValue() {
    return getBinaryValue(null);
  }
  
  public byte[] getBinaryValue(byte[] result){
    if (isBinary() || fieldsData instanceof byte[])
      return (byte[]) fieldsData;
    else
      return null;
  }

  /**
   * Returns length of byte[] segment that is used as value, if Field is not binary
   * returned value is undefined
   * @return length of byte[] segment that represents this Field value
   */
  public int getBinaryLength() {
    if (isBinary()) {
      return binaryLength;
    } else if (fieldsData instanceof byte[])
      return ((byte[]) fieldsData).length;
    else
      return 0;
  }

  /**
   * Returns offset into byte[] segment that is used as value, if Field is not binary
   * returned value is undefined
   * @return index of the first character in byte[] segment that represents this Field value
   */
  public int getBinaryOffset() {
    return binaryOffset;
  }

  /** True if norms are omitted for this indexed field */
    public final boolean getOmitNorms() {
        return (flags & OMIT_NORMS_MASK) != 0;
    }

  /** @see #setOmitTermFreqAndPositions */
    public final boolean getOmitTermFreqAndPositions() {
        return (flags & OMIT_TERM_FREQ_AND_POSITIONS_MASK) != 0;
    }
  
  /** Expert:
   *
   * If set, omit term freq, positions and payloads from
   * postings for this field.
   *
   * <p><b>NOTE</b>: While this option reduces storage space
   * required in the index, it also means any query
   * requiring positional information, such as {@link
   * PhraseQuery} or {@link SpanQuery} subclasses will
   * silently fail to find results.
   */
    public final void setOmitTermFreqAndPositions(
        final boolean omitTermFreqAndPositions)
    {
        if (omitTermFreqAndPositions) {
            flags |= OMIT_TERM_FREQ_AND_POSITIONS_MASK;
        } else {
            flags &= ~OMIT_TERM_FREQ_AND_POSITIONS_MASK;
        }
    }

    public final void setIsLazy(final boolean lazy) {
        if (lazy) {
            flags |= LAZY_MASK;
        } else {
            flags &= ~LAZY_MASK;
        }
    }

    public final boolean isLazy() {
        return (flags & LAZY_MASK) != 0;
    }

  /** Prints a Field for human consumption. */
  @Override
  public final String toString() {
    StringBuilder result = new StringBuilder();
    if (isStored()) {
      result.append("stored");
    }
    if (isIndexed()) {
      if (result.length() > 0)
        result.append(",");
      result.append("indexed");
    }
    if (isTokenized()) {
      if (result.length() > 0)
        result.append(",");
      result.append("tokenized");
    }
    if (isTermVectorStored()) {
      if (result.length() > 0)
        result.append(",");
      result.append("termVector");
    }
    if (isStoreOffsetWithTermVector()) {
      if (result.length() > 0)
        result.append(",");
      result.append("termVectorOffsets");
    }
    if (isStorePositionWithTermVector()) {
      if (result.length() > 0)
        result.append(",");
      result.append("termVectorPosition");
    }
    if (isBinary()) {
      if (result.length() > 0)
        result.append(",");
      result.append("binary");
    }
    if (getOmitNorms()) {
      result.append(",omitNorms");
    }
    if (getOmitTermFreqAndPositions()) {
      result.append(",omitTermFreqAndPositions");
    }
    if (isLazy()){
      result.append(",lazy");
    }
    result.append('<');
    result.append(name);
    result.append(':');

    if (fieldsData != null && isLazy() == false) {
      result.append(fieldsData);
    }

    result.append('>');
    return result.toString();
  }
}
