package org.apache.lucene.document;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.Map;

/**
 * A {@link FieldSelector} based on a Map of field names to {@link FieldSelectorResult}s
 *
 */
public class MapFieldSelector implements FieldSelector {
    
    Map<String,FieldSelectorResult> fieldSelections;
    private final int maxAcceptedCount;
    
    /** Create a a MapFieldSelector
     * @param fieldSelections maps from field names (String) to {@link FieldSelectorResult}s
     */
    public MapFieldSelector(Map<String, FieldSelectorResult> fieldSelections) {
        if (fieldSelections.size() == 1) {
            Map.Entry<String, FieldSelectorResult> entry =
                fieldSelections.entrySet().iterator().next();
            FieldSelectorResult result = entry.getValue();
            this.fieldSelections =
                Collections.singletonMap(entry.getKey(), result);
            if (result == FieldSelectorResult.NO_LOAD) {
                maxAcceptedCount = 0;
            } else {
                maxAcceptedCount = 1;
            }
        } else {
            this.fieldSelections = fieldSelections;
            int toLoad = 0;
            for (Map.Entry<String, FieldSelectorResult> entry:
                    fieldSelections.entrySet())
            {
                FieldSelectorResult r = entry.getValue();
                if (r != FieldSelectorResult.NO_LOAD) {
                    toLoad++;
                }
            }
            maxAcceptedCount = toLoad;
        }
    }
    
    /** Create a a MapFieldSelector
     * @param fields fields to LOAD.  List of Strings.  All other fields are NO_LOAD.
     */
    public MapFieldSelector(Collection<String> fields) {
        if (fields.size() == 1) {
            fieldSelections = Collections.singletonMap(
                fields.iterator().next(),
                FieldSelectorResult.LOAD_AND_BREAK);
            maxAcceptedCount = 1;
        } else {
            fieldSelections = new IdentityHashMap<>(fields.size());
            for (String field: fields) {
                fieldSelections.put(field, FieldSelectorResult.LOAD);
            }
            maxAcceptedCount = fields.size();
        }
    }
    
    /** Create a a MapFieldSelector
     * @param fields fields to LOAD.  All other fields are NO_LOAD.
     */
    public MapFieldSelector(String... fields) {
      this(Arrays.asList(fields));
    }


    
    /** Load field according to its associated value in fieldSelections
     * @param field a field name
     * @return the fieldSelections value that field maps to or NO_LOAD if none.
     */
    public FieldSelectorResult accept(String field) {
        FieldSelectorResult selection = fieldSelections.get(field);
        return selection!=null ? selection : FieldSelectorResult.NO_LOAD;
    }

    public int maxFieldCount() {
        return maxAcceptedCount;
    }
}
