package org.apache.lucene.index;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.Reader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.document.AbstractField;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldMapping;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.FieldVisitor;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.codecs.MergeState;
import org.apache.lucene.index.SegmentMerger.DocIdStream;
import org.apache.lucene.index.SegmentMerger.FieldGroupWithReader;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.BufferedIndexInput;
import org.apache.lucene.store.Decompressor;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.store.PackedIndexedCompressedInputStream;
import org.apache.lucene.store.RawCachingInputStream;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CloseableThreadLocal;
import org.apache.lucene.util.SorterTemplate;
import org.apache.lucene.util.StringHelper;
//import org.apache.lucene.util.packed_ondisk.PackedInts;
import org.apache.lucene.util.packed.PackedInts;

import ru.yandex.msearch.collector.FieldToIndex;

import ru.yandex.msearch.util.JavaAllocator;
import ru.yandex.msearch.util.Compress;

public class DeflateFieldsReader implements FieldsReader {
    public static final String FAKE_NO_GROUP = "_GROUP_NONE_";
    private static final JavaAllocator mergeAllocator =
        JavaAllocator.get("DeflateFieldsReaderMergeIndex");
    private static final JavaAllocator allocator =
        JavaAllocator.get("DeflateFieldsReaderBlockCache");
    private static final int INDEX_CACHE_BLOCK_SIZE = 4096;
    private static final BytesRef GROUP_SEP = new BytesRef("#");
    private static final int FORMAT = 100;
    private final static int FORMAT_SIZE = 4;
    private static final boolean QUIET_STDERR = Compress.QUIET_STDERR;

    private final FieldInfos fieldInfos;

  // The main fieldStream, used only for cloning.
    private final IndexInput cloneableFieldsStream;

  // This is a clone of cloneableFieldsStream used for reading documents.
  // It should not be cloned outside of a synchronized context.
    private final PackedIndexedCompressedInputStream fieldsStream;

    private final IndexInput cloneableIndexStream;
    private final IndexInput indexStream;
    private final Directory dir;
//    private final PackedInts.Reader indexReader;
    private final FieldsIndex indexReader;
    private int numTotalDocs;
    private int size;
    private boolean closed;
    private String segment;

    private CloseableThreadLocal<IndexInput> fieldsStreamTL =
        new CloseableThreadLocal<IndexInput>();
    private boolean isOriginal = false;

  /** Returns a cloned FieldsReader that shares open
   *  IndexInputs with the original one.  It is the caller's
   *  job not to close the original FieldsReader until all
   *  clones are called (eg, currently SegmentReader manages
   *  this logic). */
    @Override
    public FieldsReader clone() {
        ensureOpen();
        DeflateFieldsReader clone = new DeflateFieldsReader(
            dir,
            fieldInfos,
            numTotalDocs,
            size,
            cloneableFieldsStream,
            cloneableIndexStream,
            indexReader);
        clone.segment = segment;
        return clone;
    }
/*
  public static void checkCodeVersion(Directory dir, String segment) throws IOException {
    final String indexStreamFN = IndexFileNames.segmentFileName(segment, "", IndexFileNames.FIELDS_INDEX_EXTENSION);
    IndexInput idxStream = dir.openInput(indexStreamFN, 1024);

    try {
      int format = idxStream.readInt();
      if (format < FieldsWriter.FORMAT_MINIMUM && format != 100 && format != 101)
        throw new IndexFormatTooOldException(indexStreamFN, format, FieldsWriter.FORMAT_MINIMUM, FieldsWriter.FORMAT_CURRENT);
      if (format > FieldsWriter.FORMAT_CURRENT && format != 100 && format != 101)
        throw new IndexFormatTooNewException(indexStreamFN, format, FieldsWriter.FORMAT_MINIMUM, FieldsWriter.FORMAT_CURRENT);
    } finally {
      idxStream.close();
    }

  }
*/

  // Used only by clone
    private DeflateFieldsReader(
        final Directory dir,
        final FieldInfos fieldInfos,
        final int numTotalDocs,
        final int size,
        final IndexInput cloneableFieldsStream,
        final IndexInput cloneableIndexStream,
        final FieldsIndex indexReader)
    {
        this.dir = dir;
        this.fieldInfos = fieldInfos;
        this.numTotalDocs = numTotalDocs;
        this.size = size;
        this.cloneableFieldsStream = cloneableFieldsStream;
        this.cloneableIndexStream = cloneableIndexStream;
        fieldsStream =
            (PackedIndexedCompressedInputStream) cloneableFieldsStream.clone();
        indexStream = (IndexInput) cloneableIndexStream.clone();
        this.indexReader = indexReader.clone();
    }

    public DeflateFieldsReader(
        final Decompressor decompressor,
        final Directory d,
        final String segment,
        final FieldInfos fn,
        final int readBufferSize,
        final boolean inMemoryIndex)
        throws IOException
    {
        this.dir = d;
        if (!QUIET_STDERR) {
            System.err.println("Directory: " + d + ", seg: " + segment + ", inmem: " + inMemoryIndex);
        }
        boolean success = false;
        isOriginal = true;
        this.segment = segment;
        try {
            fieldInfos = fn;

            final String indexStreamFN =
                IndexFileNames.segmentFileName(
                    segment,
                    "",
                    IndexFileNames.FIELDS_INDEX_EXTENSION);
            cloneableIndexStream = d.openInput(indexStreamFN, readBufferSize);
            int format = cloneableIndexStream.readInt();

            if (format < DeflateFieldsWriter.FORMAT) {
                throw new IndexFormatTooOldException(indexStreamFN, format, DeflateFieldsWriter.FORMAT, DeflateFieldsWriter.FORMAT);
            }
            if (format > DeflateFieldsWriter.FORMAT) {
                throw new IndexFormatTooNewException(indexStreamFN, format, DeflateFieldsWriter.FORMAT, DeflateFieldsWriter.FORMAT);
            }

            indexStream = (IndexInput) cloneableIndexStream.clone();

            final String dataFileName =
                IndexFileNames.segmentFileName(
                    segment,
                    "",
                    IndexFileNames.FIELDS_EXTENSION);
            IndexInput rawInput = d.openInput(dataFileName, readBufferSize);

/*
            if (format == 100) {
            decompressor = 
        } else if (format == 101) {
            decompressor = ZstdDecompressor.INSTANCE;
        }
*/
            cloneableFieldsStream =
                new PackedIndexedCompressedInputStream(
                    rawInput,
                    decompressor,
                    allocator,
                    "FieldsReader");
            indexStream.seek(FORMAT_SIZE);
            indexReader =
                FieldsIndex.getFieldsIndex(
                    segment,
                    indexStream,
                    d,
                    inMemoryIndex);

            fieldsStream =
                (PackedIndexedCompressedInputStream)
                    cloneableFieldsStream.clone();

            this.size = indexReader.size();
            numTotalDocs = size;
            success = true;
        } catch (IOException e) {
            throw new CorruptIndexException("DeflateFieldsReader error with segment: " + segment, e);
        } finally {
          //  With lock-less commits, it's entirely possible (and
          // fine) to hit a FileNotFound exception above. In
          // this case, we want to explicitly close any subset
          // of things that were opened so that we don't have to
          // wait for a GC to do so.
            if (!success) {
                close();
            }
        }
    }

    @Override
    public void directIO(final boolean direct) {
        fieldsStream.useCache(!direct);
    }

    /**
    * @throws AlreadyClosedException if this FieldsReader is closed
    */
    private void ensureOpen() throws AlreadyClosedException {
        if (closed) {
            throw new AlreadyClosedException("this FieldsReader is closed");
        }
    }

    /**
    * Closes the underlying {@link org.apache.lucene.store.IndexInput} streams, including any ones associated with a
    * lazy implementation of a Field.  This means that the Fields values will not be accessible.
    *
    * @throws IOException
    */
    public final void close() throws IOException {
        if (!closed) {
            if (fieldsStream != null) {
                fieldsStream.close();
            }
            if (isOriginal) {
                if (cloneableFieldsStream != null) {
                    cloneableFieldsStream.close();
                }
                if (cloneableIndexStream != null) {
                    cloneableIndexStream.close();
                    if (indexReader != null) {
                        indexReader.close();
                    }
                }
            }
            if (indexStream != null) {
                indexStream.close();
            }
            fieldsStreamTL.close();
            closed = true;
        }
    }

    public final int size() {
        return size;
    }

    private static class PosId implements Comparable<PosId> {
        public final int docId;
        public final int oldDocId;
        public final long position;
        public PosId(final int docId, final int oldDocId, final long position) {
            this.docId = docId;
            this.oldDocId = oldDocId;
            this.position = position;
        }

        public int compareTo(final PosId other) {
            return Long.compare(position, other.position);
        }
    }

    private static class DocIdPositionArray implements Closeable {
        private final PackedInts.Mutable offsets;
        private final PackedInts.Mutable docIds;
        private final PackedInts.Mutable oldDocIds;
        public int count;

        private static class Reference implements Closeable {
            private PackedInts.Mutable ref;

            Reference(final PackedInts.Mutable ref) {
                this.ref = ref;
            }

            public PackedInts.Mutable release() {
                final PackedInts.Mutable ret = ref;
                ref = null;
                return ret;
            }

            @Override
            public void close() {
                if (ref != null) {
                    ref.close();
                }
            }
        }

        private static final class DocIdPositionArraySorter
            extends SorterTemplate
        {
                private final PackedInts.Mutable offsets;
                private final PackedInts.Mutable docIds;
                private final PackedInts.Mutable oldDocIds;
                long pivot;

                DocIdPositionArraySorter(
                    final PackedInts.Mutable offsets,
                    final PackedInts.Mutable docIds,
                    final PackedInts.Mutable oldDocIds)
                {
                    this.offsets = offsets;
                    this.docIds = docIds;
                    this.oldDocIds = oldDocIds;
                }

                @Override
                protected void swap(final int i, final int j) {
                    final PackedInts.Mutable docIds = this.docIds;
                    final PackedInts.Mutable oldDocIds = this.oldDocIds;
                    final PackedInts.Mutable offsets = this.offsets;
                    long tmp = docIds.get(i);
                    docIds.set(i, docIds.get(j));
                    docIds.set(j, tmp);

                    tmp = oldDocIds.get(i);
                    oldDocIds.set(i, oldDocIds.get(j));
                    oldDocIds.set(j, tmp);

                    tmp = offsets.get(i);
                    offsets.set(i, offsets.get(j));
                    offsets.set(j, tmp);
                }

                @Override
                protected int compare(final int i, final int j) {
                    final long li = offsets.get(i);
                    final long lj = offsets.get(j);
                    return Long.compare(li, lj);
                }

                @Override
                protected void setPivot(final int i) {
                    pivot = offsets.get(i);
                }

                @Override
                protected int comparePivot(final int j) {
                    return Long.compare(
                        pivot,
                        offsets.get(j));
                }
        }

        DocIdPositionArray (final int maxDocId, final int offsetsBits) {
            try (
                Reference offsets = new Reference(
                    PackedInts.getMutable(
                        maxDocId,
                        offsetsBits,
                        mergeAllocator));
                Reference docIds = new Reference(
                    PackedInts.getMutable(
                        maxDocId,
                        PackedInts.bitsRequired(maxDocId),
                        mergeAllocator));
                Reference oldDocIds = new Reference(
                    PackedInts.getMutable(
                        maxDocId,
                        PackedInts.bitsRequired(maxDocId),
                        mergeAllocator)))
            {
                this.offsets = offsets.release();
                this.docIds = docIds.release();
                this.oldDocIds = oldDocIds.release();
            }
        }

        public void add(
            final int docId,
            final int oldDocId,
            final long position)
        {
            offsets.set(count, position);
            docIds.set(count, docId);
            oldDocIds.set(count, oldDocId);
            count++;
        }

        public void sort() {
            final DocIdPositionArraySorter sorter =
                new DocIdPositionArraySorter(offsets, docIds, oldDocIds);
            sorter.timSort(0, count - 1);
        }

        public int docId(int idx) {
            return (int) docIds.get(idx);
        }

        public int oldDocId(int idx) {
            return (int) oldDocIds.get(idx);
        }

        public long position(int idx) {
            return offsets.get(idx);
        }

        @Override
        public void close() throws IOException {
            offsets.close();
            docIds.close();
            oldDocIds.close();
        }
    }

    public long fillGroupDocFieldMap(
        final Set<String> groupFields,
        final SegmentMerger.FieldGroupWithReader reusableKey,
        final Map<FieldGroupWithReader, DocIdStream> groupDocFieldMap,
        final Bits deletedDocs,
        final Set<String> storedFields,
        final FieldInfos outFieldInfos,
        final MergeState.CheckAbort checkAbort)
        throws IOException
    {
        if (!QUIET_STDERR) {
            System.err.println("fillGroupDocFieldMap: " + groupFields);
        }
        final int[] groupFieldsNumbers = new int[groupFields.size()];
        int i = 0;
        for (final String field: groupFields) {
            final int fieldNum = fieldInfos.fieldNumber(field);
            if (!QUIET_STDERR) {
                System.err.println("groupField: " + field + ": " + fieldNum);
            }
            groupFieldsNumbers[i++] = fieldNum;
        }
        final Set<Integer> storedFieldsNumbers = new HashSet<Integer>();
        for (String field : storedFields) {
            int storedFieldNum = fieldInfos.fieldNumber(field);
            if (storedFieldNum != -1) {
                storedFieldsNumbers.add(storedFieldNum);
            }
        }
        boolean skipSort = false;
        if (groupFields.contains(FAKE_NO_GROUP)) {
            skipSort = true;
        }
        return fillGroupDocFieldMapNewFormat(
            groupFieldsNumbers,
            reusableKey,
            groupDocFieldMap,
            deletedDocs,
            storedFieldsNumbers,
            outFieldInfos,
            skipSort,
            checkAbort);
    }

    private long fillGroupDocFieldMapNewFormat(
        final int[] groupFieldsNumbers,
        final SegmentMerger.FieldGroupWithReader reusableKey,
        final Map<FieldGroupWithReader, DocIdStream> groupDocFieldMap,
        final Bits deletedDocs,
        final Set<Integer> storedFields,
        final FieldInfos outFieldInfos,
        final boolean skipSort,
        final MergeState.CheckAbort checkAbort)
        throws IOException
    {
        try (SortedFieldsIndexReader reverseIndexReader =
            new SortedFieldsIndexReader(segment, dir);
            PackedInts.Mutable newDocIds =
                PackedInts.getMutable(
                    this.size,
                    PackedInts.bitsRequired(this.size),
                    mergeAllocator))
        {
            for (int i = 0, docId = 0; i < this.size; i++) {
                if (deletedDocs != null && deletedDocs.get(i)) {
                    continue;
                }
                newDocIds.set(i, docId);
                docId++;
            }
            long totalSize = 0;
            final BytesRef groupRef = new BytesRef();
            int i = 0;
            final Map<Integer, BytesRef> groupFieldsValues =
                new HashMap<>();
            for (int fieldNum: groupFieldsNumbers) {
                groupFieldsValues.put(fieldNum, new BytesRef());
            }
            if (!QUIET_STDERR) {
                System.err.println("group fields mapping: " + groupFieldsValues);
            }
            for (i = 0; i < this.size; i++) {
                final int docSize;
                reverseIndexReader.next();
                final int oldDocId = reverseIndexReader.docId();
                if (deletedDocs != null
                    && deletedDocs.get(oldDocId))
                {
                    //skip deleted doc
                    continue;
                }
                final int docId = (int) newDocIds.get(oldDocId);
                final long position = reverseIndexReader.position();
                try {
                    docSize =
                        fillGroupCalcSize(
                            position,
                            groupFieldsValues,
                            storedFields,
                            outFieldInfos);
                } catch (IOException e) {
                    System.err.println("fillGroupDocFieldMapNewFormat: failed "
                        + "docId=" + docId + ", oldId="
                         + oldDocId + ", pos="
                         + position);
                    throw e;
                }
                groupRef.offset = groupRef.length = 0;
                for (int fieldNum: groupFieldsNumbers) {
                    final BytesRef value = groupFieldsValues.get(fieldNum);
                    groupRef.append(value);
                    groupRef.append(GROUP_SEP);
                    value.offset = value.length = 0;
                }
                totalSize += docSize;
                reusableKey.rehash(groupRef);
                SegmentMerger.DocIdStream docIdStream =
                    groupDocFieldMap.get(reusableKey);
                if (docIdStream == null) {
                    docIdStream = new DocIdStream();
                    groupDocFieldMap.put(reusableKey.clone(), docIdStream);
                }
                docIdStream.addDocId(docId);
                docIdStream.addDocId(oldDocId - docId);
                if (i % 1000 == 0) {
                    checkAbort.work(1000);
                }
            }
            if (!QUIET_STDERR) {
                System.err.println("ALL OK: " + i);
            }
            return totalSize;
        } catch (IOException e) {
            System.err.println("Use of SortedFieldsIndexReader failed "
                + "continuing with sorting mode");
            e.printStackTrace();
        }
        try (DocIdPositionArray reverseArray =
            new DocIdPositionArray(this.size, indexReader.getBitsPerValue()))
        {
            int docId = 0;
            for (int i = 0; i < this.size; i++) {
                if (deletedDocs != null && deletedDocs.get(i)) {
                    continue;
                }
                final long position = indexReader.get(i);
                reverseArray.add(docId, i, position);
                docId++;
            }
            checkAbort.work(10000);
            if (!QUIET_STDERR) {
                System.err.println("DocIdPositionArray size: " + reverseArray.count);
            }
            if (skipSort) {
                if (!QUIET_STDERR) {
                    System.err.println("DocIdPositionArray sorting skipped");
                }
            } else if (QUIET_STDERR) {
                reverseArray.sort();
            } else {
                long start = System.currentTimeMillis();
                reverseArray.sort();
                System.err.println("DocIdPositionArray sort time: "
                    + (System.currentTimeMillis() - start));
            }
            long totalSize = 0;
            final BytesRef groupRef = new BytesRef();
            int i = 0;
            final Map<Integer, BytesRef> groupFieldsValues =
                new HashMap<>();
            for (int fieldNum: groupFieldsNumbers) {
                groupFieldsValues.put(fieldNum, new BytesRef());
            }
            if (!QUIET_STDERR) {
                System.err.println("group fields mapping: " + groupFieldsValues);
            }
            for (i = 0; i < reverseArray.count; i++) {
                final int docSize;
                try {
                    docSize =
                        fillGroupCalcSize(
                            reverseArray.position(i),
                            groupFieldsValues,
                            storedFields,
                            outFieldInfos);
                } catch (IOException e) {
                    System.err.println("fillGroupDocFieldMapNewFormat: failed "
                        + "docId=" + reverseArray.docId(i) + ", oldId="
                         + reverseArray.oldDocId(i) + ", pos="
                         + reverseArray.position(i));
                    throw e;
                }
                groupRef.offset = groupRef.length = 0;
                for (int fieldNum: groupFieldsNumbers) {
                    final BytesRef value = groupFieldsValues.get(fieldNum);
                    groupRef.append(value);
                    groupRef.append(GROUP_SEP);
                    value.offset = value.length = 0;
                }
                totalSize += docSize;
                reusableKey.rehash(groupRef);
                SegmentMerger.DocIdStream docIdStream =
                    groupDocFieldMap.get(reusableKey);
                if (docIdStream == null) {
                    docIdStream = new DocIdStream();
                    groupDocFieldMap.put(reusableKey.clone(), docIdStream);
                }
                docIdStream.addDocId(reverseArray.docId(i));
                docIdStream.addDocId(
                    reverseArray.oldDocId(i) - reverseArray.docId(i));
                if (i % 1000 == 0) {
                    checkAbort.work(1000);
                }
            }
            if (!QUIET_STDERR) {
                System.err.println("ALL OK: " + i);
            }
            return totalSize;
        }
    }

    private int fillGroupCalcSize(
        final long position,
        final Map<Integer, BytesRef> groupFields,
        final Set<Integer> storedFields,
        final FieldInfos outFieldInfos)
        throws IOException
    {
        int size = 0;
        fieldsStream.seek(position);
        int numFields = fieldsStream.readVInt();
        int storedCount = 0;
        for (int i = 0; i < numFields; i++) {
            int fieldNumber = fieldsStream.readVInt();
            byte bits = fieldsStream.readByte();

            boolean binary = (bits & DeflateFieldsWriter.FIELD_IS_BINARY) != 0;

            if (!storedFields.contains(fieldNumber)) {
                skipField(fieldsStream);
            } else {
                storedCount++;
                final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
                final int outFieldNumber = outFieldInfos.fieldNumber(fi.name);
                size += vIntSize(outFieldNumber) + 1;
                final BytesRef groupFieldValue = groupFields.get(fieldNumber);
                if (groupFieldValue == null || binary) {
                    size += skipField(fieldsStream);
                } else {
                    size += getFieldValue(fieldsStream, groupFieldValue);
                }
            }
        }
        size += vIntSize(storedCount);
        return size;
    }

    private void dumpDoc(final long position) throws IOException {
        fieldsStream.seek(position);
        int numFields = fieldsStream.readVInt();
        System.err.println("DumpDoc: " + position);
        for (int i = 0; i < numFields; i++) {
            int fieldNumber = fieldsStream.readVInt();
            byte bits = fieldsStream.readByte();

            boolean binary = (bits & DeflateFieldsWriter.FIELD_IS_BINARY) != 0;
            final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
            final BytesRef value = new BytesRef();
            getFieldValue(fieldsStream, value);
            System.err.println("field: " + fi.name + ": "
                + value.utf8ToString());
        }
    }

    private static int getFieldValue(
        final IndexInput fldStream,
        final BytesRef value)
            throws IOException
    {
        final int length = fldStream.readVInt();
        if (value.bytes.length < length) {
            value.bytes = new byte[length * 2];
        }
        value.length = length;
        fldStream.readBytes(value.bytes, 0, length);
        return length + vIntSize(length);
    }

    private static int skipField(final IndexInput fieldsStream)
        throws IOException
    {
        final long pos = fieldsStream.getFilePointer();
        final int length = fieldsStream.readVInt();
        if (length < 0) {
            System.err.println("negative length: " + length);
            fieldsStream.seek(pos);
            for (int i = 0; i < 5; i++) {
                System.err.println("BD: "
                    + Integer.toHexString(fieldsStream.readByte()));
            }
        }
        skipField(length, fieldsStream);
        return length + vIntSize(length);
    }

    private static void skipField(
        final int toRead,
        final IndexInput fieldsStream)
        throws IOException
    {
        fieldsStream.seek(fieldsStream.getFilePointer() + toRead);
    }

    private static int vIntSize(int value) {
        int size = 1;
        while ((value & ~0x7F) != 0) {
            size++;
            value >>>= 7;
        }
//        return 5;
        return size;
    }

    @Override
    public Document doc(
        final int n,
        final FieldSelector fieldSelector)
        throws CorruptIndexException, IOException
    {
        boolean locked = fieldsStream.lockBuffer();
        try {
            return doGetDoc(n, fieldSelector);
        } finally {
            if (locked) {
                fieldsStream.releaseBuffer();
            }
        }
    }

    @Override
    public void readDocument(final int n, final FieldVisitor visitor)
        throws CorruptIndexException, IOException
    {
        boolean locked = fieldsStream.lockBuffer();
        try {
            doReadDoc(n, visitor);
        } finally {
            if (locked) {
                fieldsStream.releaseBuffer();
            }
        }
    }

    public final boolean lockBuffer() throws IOException {
        return fieldsStream.lockBuffer();
    }

    public final void releaseBuffer() throws IOException {
        fieldsStream.releaseBuffer();
    }

    private Document doGetDoc(
        final int n,
        final FieldSelector fieldSelector)
        throws CorruptIndexException, IOException
    {
        long position = indexReader.get(n);
        fieldsStream.seek(position);

        int numFields = fieldsStream.readVIntUnlockedSafe();
        final int maxToLoad =
            fieldSelector == null ? Integer.MAX_VALUE :
                fieldSelector.maxFieldCount();
        Document doc = new Document(Math.min(numFields, maxToLoad));
        int loaded = 0;
        for (int i = 0; i < numFields && loaded < maxToLoad; i++) {
            final int fieldNumber = fieldsStream.readVIntUnlockedSafe();
            final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
            final FieldSelectorResult acceptField =
                fieldSelector == null ? FieldSelectorResult.LOAD
                    : fieldSelector.accept(fi.name);

            final byte bits = fieldsStream.readByteUnlocked();
            assert bits <=
                DeflateFieldsWriter.FIELD_IS_TOKENIZED
                    + DeflateFieldsWriter.FIELD_IS_BINARY;

            final boolean tokenize =
                (bits & DeflateFieldsWriter.FIELD_IS_TOKENIZED) != 0;
            final boolean binary =
                (bits & DeflateFieldsWriter.FIELD_IS_BINARY) != 0;
          //TODO: Find an alternative approach here if this list continues to grow beyond the
          //list of 5 or 6 currently here.  See Lucene 762 for discussion
            if (acceptField.equals(FieldSelectorResult.LOAD)) {
                addField(doc, fi, binary, tokenize);
                loaded++;
            } else if (acceptField.equals(FieldSelectorResult.LOAD_AND_BREAK)) {
                addField(doc, fi, binary, tokenize);
                loaded++;
                break;//Get out of this loop
            } else if (acceptField.equals(FieldSelectorResult.LAZY_LOAD)) {
                addFieldLazy(doc, fi, binary, tokenize, true);
                loaded++;
            } else if (acceptField.equals(FieldSelectorResult.LATENT)) {
                addFieldLazy(doc, fi, binary, tokenize, false);
                loaded++;
            } else if (acceptField.equals(FieldSelectorResult.SIZE)) {
                skipField(addFieldSize(doc, fi, binary));
                loaded++;
            } else if (acceptField.equals(FieldSelectorResult.SIZE_AND_BREAK)) {
                addFieldSize(doc, fi, binary);
                loaded++;
                break;
            } else {
                skipField();
            }
        }
        return doc;
    }

    private void doReadDoc(
        final int n,
        final FieldVisitor visitor)
        throws CorruptIndexException, IOException
    {
        long position = indexReader.get(n);
        fieldsStream.seek(position);

        int numFields = fieldsStream.readVIntUnlockedSafe();
        FieldMapping mapping = visitor.fieldMapping(fieldInfos);
        FieldSelectorResult[] fieldSelectorResults =
            mapping.fieldSelectorResults();
        int[] fieldNumberToIndex = mapping.fieldNumberToIndex();
        int fieldsLeft = mapping.maxFieldCount();
        for (int i = 0; i < numFields; ++i) {
            final int fieldNumber = fieldsStream.readVIntUnlockedSafe();

            //ignored
            final byte bits = fieldsStream.readByteUnlocked();

            switch (fieldSelectorResults[fieldNumber]) {
                case LOAD:
                case LOAD_AND_BREAK:
                    visitor.storeFieldValue(
                        fieldNumberToIndex[fieldNumber],
                        readField());
                    if (--fieldsLeft == 0) {
                        return;
                    }
                    break;
                case SIZE:
                case SIZE_AND_BREAK:
                    visitor.storeFieldSize(
                        fieldNumberToIndex[fieldNumber],
                        skipField());
                    if (--fieldsLeft == 0) {
                        return;
                    }
                    break;
                default:
                    skipField();
                    break;
            }
        }
    }

    @Override
    public final void writeDocument(
        final int docId,
        final int newDocId,
        final FieldsWriter out,
        final Set<String> storedFields)
        throws IOException
    {
        writeDocument(docId, newDocId, out, storedFields, false);
    }

    @Override
    public final void writeDocument(
        final int docId,
        final int newDocId,
        final FieldsWriter out,
        final Set<String> storedFields,
        final boolean noIndex)
        throws IOException
    {
        long position = indexReader.get(docId);
        fieldsStream.seek(position);

        int numFields = fieldsStream.readVInt();
        position = fieldsStream.getFilePointer();
        int storedFieldsCount = 0;
        for (int i = 0; i < numFields; ++i) {
            final int fieldNumber = fieldsStream.readVInt();
            final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
            final byte bits = fieldsStream.readByte();
            final int length = fieldsStream.readVInt();
            if (storedFields.contains(fi.name)) {
                ++storedFieldsCount;
            }
            fieldsStream.seek(fieldsStream.getFilePointer() + length);
        }
        if (noIndex) {
            out.newDocumentNoIndex(newDocId, storedFieldsCount);
        } else {
            out.newDocument(newDocId, storedFieldsCount);
        }
        fieldsStream.seek(position);

        for (int i = 0; i < numFields; ++i) {
            final int fieldNumber = fieldsStream.readVInt();
            final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
            final byte bits = fieldsStream.readByte();
            final int length = fieldsStream.readVInt();
            if (storedFields.contains(fi.name)) {
                out.writeField(fi, bits, fieldsStream, length);
            } else {
                fieldsStream.seek(fieldsStream.getFilePointer() + length);
            }
        }
    }

    @Override
    public IndexInput rawDocs(
        final int[] lengths,
        final int startDocID,
        final int numDocs) throws IOException {
        if (numDocs != 1) {
            throw new IllegalStateException("Grouped compressed fieldsreader "
                + "supports only one doc in rawDocs mode but numDocs="
                + numDocs);
        }
        long offset = indexReader.get(startDocID);
        fieldsStream.seek(offset);
        try {
            int size = getDocSize(fieldsStream);
            lengths[0] = size;
            fieldsStream.seek(offset);
            return fieldsStream;
        } catch (IOException e) {
            e.printStackTrace();
            Document doc = doc(startDocID, null);
            System.err.println("Doc loaded successfuly");
            throw e;
        }
    }

    private int getDocSize(final IndexInput fldStream) throws IOException
    {
        long startOffset = fldStream.getFilePointer();
        int numFields = fldStream.readVIntUnlockedSafe();
        for (int i = 0; i < numFields; i++) {
            int fieldNumber = fldStream.readVIntUnlockedSafe();
            byte bits = fldStream.readByteUnlocked();
            int skip = fldStream.readVIntUnlockedSafe();
            if (fldStream.getFilePointer() + skip > fldStream.length()) {
                throw new IOException(
                    "Seeking behind end of fieldStream: curpos=" +
                        fldStream.getFilePointer() + ", skip=" + skip +
                        ", length=" + fldStream.length() +
                        ", numFields=" + numFields + ", i=" + i
                        + ", file=" + fldStream.getCacheKey());
            }
            fldStream.seek(fldStream.getFilePointer() + skip);
        }
        return (int)(fldStream.getFilePointer() - startOffset);
    }


    private int skipField() throws IOException {
        return skipField(fieldsStream.readVIntUnlockedSafe());
    }

    private int skipField(int toRead) throws IOException {
        fieldsStream.seek(fieldsStream.getFilePointer() + toRead);
        return toRead;
    }

    private void addFieldLazy(
        final Document doc,
        final FieldInfo fi,
        final boolean binary,
        final boolean tokenize,
        final boolean cacheResult)
        throws IOException
    {
        if (binary) {
            int toRead = fieldsStream.readVIntUnlockedSafe();
            long pointer = fieldsStream.getFilePointer();
            doc.add(new LazyField(fi.name, Field.Store.YES, toRead, pointer, binary, cacheResult));
            fieldsStream.seek(pointer + toRead);
        } else {
            Field.Store store = Field.Store.YES;
            Field.Index index = Field.Index.toIndex(fi.isIndexed, tokenize);
            Field.TermVector termVector = Field.TermVector.toTermVector(fi.storeTermVector, fi.storeOffsetWithTermVector, fi.storePositionWithTermVector);

            AbstractField f;
            int length = fieldsStream.readVIntUnlockedSafe();
            long pointer = fieldsStream.getFilePointer();

            //Skip ahead of where we are by the length of what is stored
            fieldsStream.seek(pointer+length);
            f = new LazyField(fi.name, store, index, termVector, length, pointer, binary, cacheResult);
            f.setOmitNorms(fi.omitNorms);
            f.setOmitTermFreqAndPositions(fi.omitTermFreqAndPositions);

            doc.add(f);
        }
    }

    private void addField(
        final Document doc,
        final FieldInfo fi,
        final boolean binary,
        final boolean tokenize)
        throws CorruptIndexException, IOException
    {
        if (binary) {
            int toRead = fieldsStream.readVIntUnlockedSafe();
            final byte[] b = new byte[toRead];
            fieldsStream.readBytesUnlockedSafe(b, 0, b.length);
            doc.add(new Field(fi.name, b));
        } else {
            Field.Store store = Field.Store.YES;
            Field.Index index = Field.Index.toIndex(fi.isIndexed, tokenize);
            Field.TermVector termVector =
                Field.TermVector.toTermVector(
                    fi.storeTermVector,
                    fi.storeOffsetWithTermVector,
                    fi.storePositionWithTermVector);

            AbstractField f;
            f = new Field(fi.name,     // name
                false,
                fieldsStream.readStringUnlockedSafe(), // read value
                store,
                index,
                termVector);
            f.setOmitTermFreqAndPositions(fi.omitTermFreqAndPositions);
            f.setOmitNorms(fi.omitNorms);
            doc.add(f);
        }
    }

    private byte[] readField() throws CorruptIndexException, IOException {
        final int toRead = fieldsStream.readVIntUnlockedSafe();
        final byte[] b = new byte[toRead];
        fieldsStream.readBytesUnlockedSafe(b, 0, b.length);
        return b;
    }

    // Add the size of field as a byte[] containing the 4 bytes of the integer byte size (high order byte first; char = 2 bytes)
    // Read just the size -- caller must skip the field content to continue reading fields
    // Return the size in bytes or chars, depending on field type
    private int addFieldSize(
        final Document doc,
        final FieldInfo fi,
        final boolean binary)
        throws IOException
    {
        int size = fieldsStream.readVIntUnlockedSafe();
        int bytesize = binary ? size : 2*size;
        byte[] sizebytes = new byte[4];
        sizebytes[0] = (byte) (bytesize>>>24);
        sizebytes[1] = (byte) (bytesize>>>16);
        sizebytes[2] = (byte) (bytesize>>> 8);
        sizebytes[3] = (byte)  bytesize      ;
        doc.add(new Field(fi.name, sizebytes));
        return size;
    }

  private class LazyField extends AbstractField implements Fieldable {
    private int toRead;
    private long pointer;
    private final boolean cacheResult;

    public LazyField(String name, Field.Store store, int toRead, long pointer, boolean isBinary, boolean cacheResult) {
      super(name, store, Field.Index.NO, Field.TermVector.NO);
      this.toRead = toRead;
      this.pointer = pointer;
      setIsBinary(isBinary);
      this.cacheResult = cacheResult;
      if (isBinary())
        binaryLength = toRead;
      setIsLazy(true);
    }

    public LazyField(String name, Field.Store store, Field.Index index, Field.TermVector termVector, int toRead, long pointer, boolean isBinary, boolean cacheResult) {
      super(name, store, index, termVector);
      this.toRead = toRead;
      this.pointer = pointer;
      setIsBinary(isBinary);
      this.cacheResult = cacheResult;
      if (isBinary())
        binaryLength = toRead;
      setIsLazy(true);
    }

    private IndexInput getFieldStream() {
      IndexInput localFieldsStream = fieldsStreamTL.get();
      if (localFieldsStream == null) {
        localFieldsStream = (IndexInput) cloneableFieldsStream.clone();
        fieldsStreamTL.set(localFieldsStream);
      }
      return localFieldsStream;
    }

    /** The value of the field as a Reader, or null.  If null, the String value,
     * binary value, or TokenStream value is used.  Exactly one of stringValue(), 
     * readerValue(), getBinaryValue(), and tokenStreamValue() must be set. */
    public Reader readerValue() {
      ensureOpen();
      return null;
    }

    /** The value of the field as a TokenStream, or null.  If null, the Reader value,
     * String value, or binary value is used. Exactly one of stringValue(), 
     * readerValue(), getBinaryValue(), and tokenStreamValue() must be set. */
    public TokenStream tokenStreamValue() {
      ensureOpen();
      return null;
    }

    /** The value of the field as a String, or null.  If null, the Reader value,
     * binary value, or TokenStream value is used.  Exactly one of stringValue(), 
     * readerValue(), getBinaryValue(), and tokenStreamValue() must be set. */
    public String stringValue() {
      ensureOpen();
      if (isBinary())
        return null;
      else {
        if (fieldsData == null) {
          String result = null;
          IndexInput localFieldsStream = getFieldStream();
          try {
            localFieldsStream.seek(pointer);
            byte[] bytes = new byte[toRead];
            localFieldsStream.readBytes(bytes, 0, toRead);
            result = new String(bytes, "UTF-8");
          } catch (IOException e) {
            throw new FieldReaderException(e);
          }
          if (cacheResult == true){
            fieldsData = result;
          }
          return result;
        } else {
          return (String) fieldsData;
        }
      }
    }

    @Override
    public byte[] getBinaryValue(byte[] result) {
      ensureOpen();

      if (isBinary()) {
        if (fieldsData == null) {
          // Allocate new buffer if result is null or too small
          final byte[] b;
          if (result == null || result.length < toRead)
            b = new byte[toRead];
          else
            b = result;
   
          IndexInput localFieldsStream = getFieldStream();

          // Throw this IOException since IndexReader.document does so anyway, so probably not that big of a change for people
          // since they are already handling this exception when getting the document
          try {
//            localFieldsStream.seek(pointer);
            localFieldsStream.seek(pointer);
            localFieldsStream.readBytes(b, 0, toRead);
          } catch (IOException e) {
            throw new FieldReaderException(e);
          }

          binaryOffset = 0;
          binaryLength = toRead;
          if (cacheResult == true){
            fieldsData = b;
          }
          return b;
        } else {
          return (byte[]) fieldsData;
        }
      } else
        return null;     
    }
  }

    private static abstract class FieldsIndex implements Closeable {
        public static FieldsIndex getFieldsIndex(
            final String segment,
            final IndexInput in,
            final Directory dir,
            final boolean inMemory)
            throws IOException
        {
            if (inMemory) {
                return new MemoryFieldsIndex(in);
            } else {
                return new OnDiskFieldsIndex(in);
//                return new TwoStageOnDiskFieldsIndex(segment, in, dir);
            }
        }

        public abstract long get(final int docId) throws IOException;
        public abstract int size();
        public abstract int getBitsPerValue();
        public abstract FieldsIndex clone();
    }

    private static final JavaAllocator indexAllocator =
        JavaAllocator.get("FieldsIndex");
    private static class MemoryFieldsIndex extends FieldsIndex {
        private org.apache.lucene.util.packed.PackedInts.Reader
            indexReader = null;
        public MemoryFieldsIndex(final IndexInput in) throws IOException {
            indexReader =
                org.apache.lucene.util.packed.PackedInts.getReader(
                    in,
                    indexAllocator);
        }

        private MemoryFieldsIndex(final MemoryFieldsIndex other) {
            this.indexReader = other.indexReader;
        }

        public long get(final int docId) {
            return indexReader.get(docId);
        }

        public int size() {
            return indexReader.size();
        }

        public void close() throws IOException {
            if (indexReader != null) {
                indexReader.close();
                indexReader = null;
            }
        }

        public int getBitsPerValue() {
            return indexReader.getBitsPerValue();
        }

        public FieldsIndex clone() {
            return new MemoryFieldsIndex(this);
        }
    }

    private static class OnDiskFieldsIndex extends FieldsIndex {
        private static final JavaAllocator indexAllocator =
            JavaAllocator.get("FieldsIndexCached");
        private org.apache.lucene.util.packed_ondisk.PackedInts.Reader
            indexReader = null;
        public OnDiskFieldsIndex(final IndexInput in) throws IOException {
            indexReader =
                org.apache.lucene.util.packed_ondisk.PackedInts.getReader(
                    new RawCachingInputStream(
                        in,
                        INDEX_CACHE_BLOCK_SIZE,
                        indexAllocator,
                        "FDX"));
        }

        private OnDiskFieldsIndex(final OnDiskFieldsIndex other) {
            this.indexReader = other.indexReader.clone();
        }

        public long get(final int docId) throws IOException {
            return indexReader.get(docId);
        }

        public int size() {
            return indexReader.size();
        }

        public void close() throws IOException {
            if (indexReader != null) {
                indexReader.close();
                indexReader = null;
            }
        }

        public int getBitsPerValue() {
            return indexReader.getBitsPerValue();
        }

        public FieldsIndex clone() {
            return new OnDiskFieldsIndex(this);
        }
    }

    //WARN: This is not finished yet
    private static class TwoStageOnDiskFieldsIndex extends FieldsIndex {
        private static final int DOC_INDEX_INTERVAL = 512;
        private static final JavaAllocator firstStageIndexAllocator =
            JavaAllocator.get("1STFieldsIndex");
        private static final JavaAllocator secondStageIndexAllocator =
            JavaAllocator.get("2STFieldsIndex");
        private org.apache.lucene.util.packed.PackedInts.Reader
            indexReader = null;
        private IndexInput rawSecondIndexStream;
        private IndexInput secondIndexStream;
        private IndexInput firstIndexStream;

        public TwoStageOnDiskFieldsIndex(
            final String segment,
            final IndexInput in,
            final Directory dir)
            throws IOException
        {
            final String secondIndexStreamName =
                IndexFileNames.segmentFileName(
                    segment,
                    "",
                    IndexFileNames.FIELDS_INDEX_EXTENSION_2ST);
            final String firstIndexStreamName =
                IndexFileNames.segmentFileName(
                    segment,
                    "",
                    IndexFileNames.FIELDS_INDEX_EXTENSION_1ST);
            boolean success = false;
            try {
                if (dir.fileExists(secondIndexStreamName)
                    && dir.fileExists(firstIndexStreamName))
                {
                    openStreams(
                        dir,
                        firstIndexStreamName,
                        secondIndexStreamName);
                    success = true;
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
            if (!success) {
                System.err.println("Can't load existing streams for segment: "
                    + segment + ", trying to recreate");
                createStreams(
                    dir,
                    in,
                    firstIndexStreamName,
                    secondIndexStreamName);
                openStreams(dir, firstIndexStreamName, secondIndexStreamName);
            }
        }

        private void openStreams(
            final Directory dir,
            final String firstIndexStreamName,
            final String secondIndexStreamName)
            throws IOException
        {
            System.err.println("OPENSTREAMS: " + dir + ", " + firstIndexStreamName + ", " + secondIndexStreamName);
            try {
                rawSecondIndexStream = dir.openInput(secondIndexStreamName);
                secondIndexStream =
                    new RawCachingInputStream(
                        rawSecondIndexStream,
                        INDEX_CACHE_BLOCK_SIZE,
                        secondStageIndexAllocator,
                        "FDOX");
                firstIndexStream = dir.openInput(firstIndexStreamName);
                indexReader =
                    org.apache.lucene.util.packed.PackedInts
                        .getReader(
                            firstIndexStream,
                            firstStageIndexAllocator);
                System.err.println("OPENSTREAMS success: " + rawSecondIndexStream + ", " + secondIndexStream);
            } catch (IOException e) {
                e.printStackTrace();
                if (rawSecondIndexStream != null) {
                    rawSecondIndexStream.close();
                    rawSecondIndexStream = null;
                }
                if (firstIndexStream != null) {
                    firstIndexStream.close();
                    firstIndexStream = null;
                }
                if (indexReader != null) {
                    indexReader.close();
                    indexReader = null;
                }
                throw e;
            }
        }

        private void createStreams(
            final Directory dir,
            final IndexInput in,
            final String firstIndexStreamName,
            final String secondIndexStreamName)
            throws IOException
        {
            try (
//                org.apache.lucene.util.packed_native.PackedInts.Reader
                org.apache.lucene.util.packed.PackedInts.Reader
                    tmpReader =
//                    org.apache.lucene.util.packed_native.PackedInts.getReader(
                    org.apache.lucene.util.packed.PackedInts.getReader(
                        in,
                        firstStageIndexAllocator);
                DocIdPositionArray reverseArray =
                    new DocIdPositionArray(
                        tmpReader.size(), tmpReader.getBitsPerValue());
                IndexOutput secondOut = dir.createOutput(secondIndexStreamName);
                IndexOutput firstOut = dir.createOutput(firstIndexStreamName);
                PackedInts.Mutable tempIndex =
                    PackedInts.getMutable(
                        tmpReader.size(),
                        tmpReader.getBitsPerValue(),
                        firstStageIndexAllocator))
            {
                int numDocs = tmpReader.size();
                long maxDataOffset = 0;
                for (int i = 0; i < numDocs; i++) {
                    final long position = tmpReader.get(i);
                    if (maxDataOffset < position) {
                        maxDataOffset = position;
                    }
                    reverseArray.add(i, i, position);
                }
                if (!QUIET_STDERR) {
                    System.err.println("DocIdPositionArray size: "
                        + reverseArray.count);
                }
                reverseArray.sort();
                int blockCount = 0;
                int block = 0;
                for (int i = 0; i < numDocs; i++) {
                    int docId = reverseArray.docId(i);
                    secondOut.writeVInt(docId);
                    tempIndex.set(docId, block);
                    blockCount++;
                    if (blockCount == DOC_INDEX_INTERVAL) {
                        for (int b = i - DOC_INDEX_INTERVAL + 1; b <= i; b++) {
                            long position = reverseArray.position(b);
                            secondOut.writeVLong(position);
                        }
                        blockCount = 0;
                        block++;
                    }
                }
                if (blockCount > 0) {
                    for (int b = numDocs - DOC_INDEX_INTERVAL; b < numDocs; b++) {
                        long position = reverseArray.position(b);
                        secondOut.writeVLong(position);
                    }
                    block++;
                }
                if (!QUIET_STDERR) {
                    long maxIndexOffset = secondOut.getFilePointer();
                    System.err.println("MAXES: mio: " + maxIndexOffset
                        + ", mdo: " + maxDataOffset
                        + ", bo: " + block
                        + ", miobits: " + PackedInts.bitsRequired(maxIndexOffset)
                        + ", mdobits: " + tmpReader.getBitsPerValue()
                        + ", bobits: " + PackedInts.bitsRequired(block));
                }
                PackedInts.Writer writer =
                    PackedInts.getWriter(
                        firstOut,
                        numDocs,
                        PackedInts.bitsRequired(block));
                for (int i = 0; i < numDocs; i++) {
                    final long off = tempIndex.get(i);
                    writer.add(off);
                }
                writer.finish();
            }
        }

        private TwoStageOnDiskFieldsIndex(
            final TwoStageOnDiskFieldsIndex other)
        {
            this.indexReader = other.indexReader;
            this.secondIndexStream =
                (IndexInput) other.secondIndexStream.clone();
        }

        public long get(final int docId) throws IOException {
            long offsetOffset = indexReader.get(docId);
            secondIndexStream.seek(offsetOffset);
            return secondIndexStream.readVLong();
        }

        public int size() {
            return indexReader.size();
        }

        public void close() throws IOException {
            if (indexReader != null) {
                indexReader.close();
                indexReader = null;
            }
            if (rawSecondIndexStream != null) {
                rawSecondIndexStream.close();
                rawSecondIndexStream = null;
            }
            if (firstIndexStream != null) {
                firstIndexStream.close();
                firstIndexStream = null;
            }
        }

        public int getBitsPerValue() {
            return indexReader.getBitsPerValue();
        }

        public FieldsIndex clone() {
            return new TwoStageOnDiskFieldsIndex(this);
        }
    }

    private static class SortedFieldsIndexReader implements Closeable {
        private static final int CURRENT_FORMAT =
            DeflateFieldsWriter.SORTED_INDEX_FORMAT;
        private final IndexInput in;
        private int docId = 0;
        private long position = 0;

        SortedFieldsIndexReader(final String segment, final Directory dir)
            throws IOException
        {
            in = dir.openInput(
                IndexFileNames.segmentFileName(
                    segment,
                    "",
                    IndexFileNames.FIELDS_INDEX_SORTED_EXTENSION));
            int format = in.readInt();
            if (format != CURRENT_FORMAT) {
                throw new IOException(
                    "SortedFieldsIndexReader: Unupported format version "
                        + format);
            }
        }

        public void next() throws IOException {
            docId += in.readSignedVInt();
            position += in.readVLong();
        }

        public int docId() {
            return docId;
        }

        public long position() {
            return position;
        }

        @Override
        public void close() throws IOException {
            in.close();
        }
    }
}
