package org.apache.lucene.index;

/**
 * Copyright 2004 The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.lucene.document.AbstractField;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.FieldVisitor;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IndexingCompressingOutputStream;
import org.apache.lucene.store.RAMOutputStream;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CloseableThreadLocal;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.IntsRef;
import org.apache.lucene.util.packed.*;
//import org.apache.lucene.util.packed_native.*;

import ru.yandex.msearch.util.JavaAllocator;

public class DeflateFieldsWriter implements FieldsWriter {
    private static final JavaAllocator ALLOCATOR =
        JavaAllocator.get("DeflateFieldsWriterTempIndex");
    public static final byte FIELD_IS_TOKENIZED = 0x1;
    public static final byte FIELD_IS_BINARY = 0x2;

    public static final int FORMAT_SIZE = 4;

    public static final int FORMAT = 100;
    public static final int SORTED_INDEX_FORMAT = 100;

    private Directory directory;
    private String segment;
    private FieldInfos fieldInfos;
    private IndexOutput fieldsStream;
    private IndexOutput indexStream;
    private IndexOutput sortedPositionsStream;
    private int docCount = 0;
    private int indexedDocs = 0;
    private long prevPosition = 0;
    private int prevDocId;

    public DeflateFieldsWriter(
        final Compressor compressor,
        final Directory directory,
        final String segment,
        final FieldInfos fn,
        final int bufferSize)
        throws IOException
    {
        this.directory = directory;
        this.segment = segment;
        fieldInfos = fn;

        boolean success = false;
        try {
            fieldsStream = directory.createOutput(
                IndexFileNames.segmentFileName(
                    segment,
                     "",
                     IndexFileNames.FIELDS_EXTENSION));
            indexStream = directory.createOutput(
                IndexFileNames.segmentFileName(
                    segment,
                    "",
                    IndexFileNames.FIELDS_INDEX_EXTENSION));
            sortedPositionsStream =
                directory.createOutput(
                    IndexFileNames.segmentFileName(
                        segment,
                        "",
                        IndexFileNames.FIELDS_INDEX_SORTED_EXTENSION));

            fieldsStream.writeInt(FORMAT);
            indexStream.writeInt(FORMAT);
            sortedPositionsStream.writeInt(SORTED_INDEX_FORMAT);
            fieldsStream =
                new IndexingCompressingOutputStream(
                    fieldsStream,
                    bufferSize,
                    true,
                    compressor);
            success = true;
        } finally {
            if (!success) {
                abort();
            }
        }
    }

    @Override
    public FieldInfos getFieldInfos() {
        return this.fieldInfos;
    }

    @Override
    public void close() throws IOException {
        if (directory != null) {
            if (indexedDocs > 0) {
                createIndex();
            }
            try {
                IOUtils.closeSafely(
                    fieldsStream,
                    indexStream,
                    sortedPositionsStream);
            } finally {
                fieldsStream = indexStream = sortedPositionsStream = null;
            }
        }
    }

    private void createIndex()
        throws IOException
    {
        indexStream.close();
        final IndexInput indexInput = directory.openInput(
            IndexFileNames.segmentFileName(
                segment,
                "",
                IndexFileNames.FIELDS_INDEX_EXTENSION));
        final long maxPosition = fieldsStream.getFilePointer();
        final PackedInts.Mutable tempIndex =
            PackedInts.getMutable(
                indexedDocs,
                PackedInts.bitsRequired(maxPosition),
                ALLOCATOR);
        try {
            try {
                indexInput.readInt(); //skip format
                for (int i = 0; i < indexedDocs; i++) {
                    final long pos = indexInput.readLong();
                    tempIndex.set(i, pos);
                }
            } finally {
                indexInput.close();
            }
            indexStream = directory.createOutput(
                IndexFileNames.segmentFileName(
                    segment,
                    "",
                    IndexFileNames.FIELDS_INDEX_EXTENSION));
            indexStream.writeInt(FORMAT);
            writeIndex(tempIndex);
        } finally {
            tempIndex.close();
        }
    }

    @Override
    public void abort() {
        if (directory != null) {
            try {
                close();
            } catch (IOException ignored) {
                ignored.printStackTrace();
            }
            try {
                directory.deleteFile(
                    IndexFileNames.segmentFileName(
                        segment,
                        "",
                        IndexFileNames.FIELDS_EXTENSION));
            } catch (IOException ignored) {
            }
            try {
                directory.deleteFile(
                    IndexFileNames.segmentFileName(
                        segment,
                        "",
                        IndexFileNames.FIELDS_INDEX_EXTENSION));
            } catch (IOException ignored) {
            }
        }
    }

    @Override
    public void writeField(
        final FieldInfo fi,
        final Fieldable field)
        throws IOException
    {
        fieldsStream.writeVInt(fi.number);
        byte bits = 0;
        if (field.isTokenized()) {
            bits |= FIELD_IS_TOKENIZED;
        }
        if (field.isBinary()) {
            bits |= FIELD_IS_BINARY;
        }
        fieldsStream.writeByte(bits);

        if (field.isBinary()) {
            final byte[] data;
            final int len;
            final int offset;
            data = field.getBinaryValue();
            len = field.getBinaryLength();
            offset =  field.getBinaryOffset();

            fieldsStream.writeVInt(len);
            fieldsStream.writeBytes(data, offset, len);
        } else {
            fieldsStream.writeString(field.stringValue());
        }
    }

    @Override
    public void writeField(
        final FieldInfo otherFi,
        final byte bits,
        final IndexInput data,
        final int length)
        throws IOException
    {
        final int fieldNumber = fieldInfos.fieldNumber(otherFi.name);
        fieldsStream.writeVInt(fieldNumber);
        fieldsStream.writeByte(bits);
        fieldsStream.writeVInt(length);
        fieldsStream.copyBytesDirect(data, length);
    }

    @Override
    public void newDocument(final int docId, final int fieldCount) throws IOException {
        indexPosition(docId, fieldsStream.getFilePointer(), false);
        fieldsStream.writeVInt(fieldCount);
        docCount++;
    }

    @Override
    public void newDocumentNoIndex(final int docId, final int fieldCount) throws IOException {
        indexPosition(docId, fieldsStream.getFilePointer(), true);
        fieldsStream.writeVInt(fieldCount);
        docCount++;
    }

    @Override
    public long getDataStreamPosition() {
        return fieldsStream.getFilePointer();
    }

    @Override
    public void writeIndex(final PackedInts.Mutable srcIndex)
        throws IOException
    {
        final int docs = srcIndex.size();
        PackedInts.Writer writer =
            PackedInts.getWriter(
                indexStream,
                docs,
                srcIndex.getBitsPerValue());
        for (int i = 0; i < docs; i++) {
            final long off = srcIndex.get(i);
            writer.add(off);
        }
        writer.finish();
    }

    private void indexPosition(final int docId, final long position, final boolean noIndex)
        throws IOException
    {
        sortedPositionsStream.writeSignedVInt(docId - prevDocId);
        prevDocId = docId;
        sortedPositionsStream.writeVLong(position - prevPosition);
        prevPosition = position;
        if (!noIndex) {
            indexStream.writeLong(position);
            indexedDocs++;
        }
    }

    @Override
    public void addRawDocuments(
        final IndexInput stream,
        final int[] lengths,
        final int numDocs)
        throws IOException
    {
        throw
            new IOException(
                new UnsupportedOperationException(
                    "DeflateFieldsWriter.addRawDocuments"));
    }

    @Override
    public void addDocument(final int docId, final Document doc) throws IOException {
        indexPosition(docId, fieldsStream.getFilePointer(), false);
        addDocumentNoIndex(docId, doc);
    }

    @Override
    public void addDocumentNoIndex(final int docId, final Document doc) throws IOException {
        indexPosition(docId, fieldsStream.getFilePointer(), true);
        int storedCount = 0;
        List<Fieldable> fields = doc.getFields();
        for (Fieldable field : fields) {
            if (field.isStored()) {
                storedCount++;
            }
        }
        fieldsStream.writeVInt(storedCount);

        for (Fieldable field : fields) {
            if (field.isStored()) {
                writeField(fieldInfos.fieldInfo(field.name()), field);
            }
        }
        docCount++;
    }

    public void flushDocument(
        final int docId,
        final int numStoredFields,
        final RAMOutputStream buffer)
        throws IOException
    {
        indexPosition(docId, fieldsStream.getFilePointer(), false);
        fieldsStream.writeVInt(numStoredFields);
        buffer.writeTo(fieldsStream);
        docCount++;
    }

    public void skipDocument(final int docId) throws IOException {
        indexPosition(docId, fieldsStream.getFilePointer(), false);
        fieldsStream.writeVInt(0);
        docCount++;
    }

    public Document doc(final int n, final FieldSelector fieldSelector) {
        throw new UnsupportedOperationException("Compressed fields writer"
            + "cannot be used in RT indexing");
    }

    @Override
    public void readDocument(final int n, final FieldVisitor visitor)
        throws CorruptIndexException, IOException
    {
        throw new UnsupportedOperationException("Compressed fields writer"
            + "cannot be used in RT indexing");
    }
}
