package org.apache.lucene.index;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.util.Bits;
import org.apache.lucene.util.ReaderUtil;
import java.io.IOException;

/**
 * Exposes flex API, merged from flex API of sub-segments.
 *
 * @lucene.experimental
 */

public final class MultiDocsEnum extends DocsEnum {
  private EnumsWithSlices subs;
  int numSubs;
  int upto;
  DocsEnum current;
  int currentBase;
  Bits skipDocs;
  int doc = -1;

  MultiDocsEnum reset(final EnumsWithSlices subs, final int numSubs) {
    this.numSubs = numSubs;

    if (this.subs == null
        || this.subs == subs
        || this.subs.docsEnums.length != subs.docsEnums.length) {
        this.subs = new EnumsWithSlices();
        this.subs.docsEnums = subs.docsEnums.clone();
        this.subs.slices = subs.slices.clone();
    } else {
        System.arraycopy(
            subs.docsEnums,
            0,
            this.subs.docsEnums,
            0,
            subs.docsEnums.length);
        System.arraycopy(
            subs.slices,
            0,
            this.subs.slices,
            0,
            subs.slices.length);
    }
    upto = -1;
    current = null;
    return this;
  }

  public int getNumSubs() {
    return numSubs;
  }

  public EnumsWithSlices getSubs() {
    return subs;
  }

  @Override
  public int freq() {
    return current.freq();
  }

  @Override
  public int docID() {
    return doc;
  }

  @Override
  public int advance(int target) throws IOException {
    while(true) {
      if (current != null) {
        final int doc = current.advance(target-currentBase);
        if (doc == NO_MORE_DOCS) {
          current = null;
        } else {
          return this.doc = doc + currentBase;
        }
      } else if (upto == numSubs-1) {
        return this.doc = NO_MORE_DOCS;
      } else {
        upto++;
        current = subs.docsEnums[upto];
        currentBase = subs.slices[upto].start;
      }
    }
  }

  @Override
  public int nextDoc() throws IOException {
    while(true) {
      if (current == null) {
        if (upto == numSubs-1) {
          return this.doc = NO_MORE_DOCS;
        } else {
          upto++;
          current = subs.docsEnums[upto];
          currentBase = subs.slices[upto].start;
        }
      }

      final int doc = current.nextDoc();
      if (doc != NO_MORE_DOCS) {
        return this.doc = currentBase + doc;
      } else {
        current = null;
      }
    }
  }

  // TODO: implement bulk read more efficiently than super
  public final static class EnumsWithSlices {
    public DocsEnum[] docsEnums;
    public ReaderUtil.Slice[] slices;
  }
}

