package org.apache.lucene.index;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldVisitor;
import org.apache.lucene.index.DocFieldProcessor.ArrayReference;
import org.apache.lucene.index.DocFieldProcessor.PerFieldAndThread;
import org.apache.lucene.index.FreqProxTermsWriterPerField.FreqProxPostingsArray;
import org.apache.lucene.index.TermsEnum.SeekStatus;
import org.apache.lucene.index.codecs.Codec;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.FieldCache; // javadocs
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.Similarity;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.Weight;
import org.apache.lucene.store.*;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.BitVector;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.ByteBlockPool;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CloseableThreadLocal;
import org.apache.lucene.util.IntsRef;
//import org.apache.lucene.util.NativeIntArray;
import org.apache.lucene.util.PriorityQueue;

import org.apache.lucene.util.ReaderUtil;         // for javadocs

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

public class PerFieldMergingIndexReader extends IndexReader {
    private static final boolean DEBUG = false;
    private static final ThreadLocal<Boolean> localDebug =
        new ThreadLocal<Boolean>();
//    private static final boolean DEBUG = false;
    private static final Document[] NOT_FOUND = new Document[0];
    private static final int FLUSH_THRESHOLD = 200;
    private static final Deletes EMPTY_DELETES = new Deletes();

    private boolean closed;
    private PerFieldMergingIndexReader[] clones =
        new PerFieldMergingIndexReader[0];
    private int[] deletes = new int[32];
    private volatile int deletesCount = 0;
    private final Deletes prevDeletes;
    private int[] docsInProgress;
    private final DocumentsWriter docWriter;
    private final Map<String,ArrayReference<PerFieldAndThread>> allPerFields;
    public final int lastDocId;
    private final FieldsWriter fieldsWriter;
    private final StoredFieldsWriter storedFieldsWriter;
    private final Directory directory;
    private MergingFields fields = null;
    private Object parent = null;
    public volatile boolean needFlush = false;
    private final AtomicInteger deletersRefs = new AtomicInteger(1);
    private Bits deletedDocuments = null;

    private static final boolean DEBUG() {
//        if (DEBUG) {
//            return true;
//        }
//        Boolean d = localDebug.get();
//        if (d == null) {
//            return false;
//        }
//        return d;
        return DEBUG;
    }

    private static final void setDebug() {
        localDebug.set(true);
    }

    private static final void resetDebug() {
        localDebug.set(null);
    }

    public PerFieldMergingIndexReader(
        final DocumentsWriter docWriter,
        final Map<String,ArrayReference<PerFieldAndThread>> allPerFields,
        final Deletes prevDeletes,
        final StoredFieldsWriter fieldsWriter,
        final Directory directory,
        final int lastDocId)
    {
        super();
        this.docWriter = docWriter;
        this.allPerFields = allPerFields;
        this.lastDocId = lastDocId;

        if (prevDeletes == null) {
            this.prevDeletes = EMPTY_DELETES;
        } else {
            this.prevDeletes = prevDeletes;
        }
        this.storedFieldsWriter = fieldsWriter;
        this.fieldsWriter = fieldsWriter.fieldsWriter;
        this.directory = directory;
        readerFinishedListeners = new FilterIndexReader.LazyInitMapBackedSet<ReaderFinishedListener>();
    }

    public void incDeletersRefs() {
        deletersRefs.incrementAndGet();
    }

    public synchronized int decDeletersRefs() {
        int refs = deletersRefs.decrementAndGet();
        if (refs == 0) {
            for (PerFieldMergingIndexReader clone : clones) {
                clone.decDeletersRefs();
            }
            clearClones();
        }
        return refs;
    }

    public void setParent(Object parent) {
        this.parent = parent;
    }

    public void clearClones() {
        clones = null;
    }

    public String toString() {
        if (parent == null) {
            return "PFMIR:" + Integer.toHexString(System.identityHashCode(this));
        } else {
            return parent.toString() + "," + System.identityHashCode(this);
        }
    }

    public synchronized PerFieldMergingIndexReader clone(int lastDocId) {
        int newLastDocId;
        int[] newDocsInProgress;
        synchronized(docWriter) {
            newLastDocId = docWriter.getLastDocId();
            newDocsInProgress = docWriter.getDocsInProgress();
        }
        int deletesCount = this.deletesCount;
        Deletes newPrevDeletes;
        if (deletesCount == 0) {
            newPrevDeletes = prevDeletes;
        } else {
            newPrevDeletes =
                new Deletes(
                    Arrays.copyOf(deletes, deletesCount),
                    prevDeletes.toArray());
        }
        PerFieldMergingIndexReader ret = new PerFieldMergingIndexReader(
            docWriter,
            allPerFields,
            newPrevDeletes,
            storedFieldsWriter,
            directory,
            newLastDocId);
        ret.setParent(parent);
        if (newDocsInProgress.length > 0) {
            ret.docsInProgress = newDocsInProgress;
        }
        clones = Arrays.copyOf(clones, clones.length + 1);
        clones[clones.length - 1] = ret;
        ret.incDeletersRefs();
        return ret;
    }

    public synchronized void deleteDocument(int docId) throws IOException {
        if (isDeleted(docId)) {
            return;
        }
        deletes[deletesCount++] = docId;
        if (deletesCount == deletes.length) {
            deletes = Arrays.copyOf(deletes, deletes.length << 1);
        }
        for (PerFieldMergingIndexReader clone : clones) {
            clone.deleteDocument(docId);
        }
    }

    public boolean isDeleted(int doc) {
        if (prevDeletes.get(doc)) {
            return true;
        }
        for (int i = 0; i < deletesCount; i++) {
            if (deletes[i] == doc) {
                return true;
            }
        }
        return false;
    }

    public Document[] getDocsForTerm(final Term term) throws IOException {
        final String field = term.field();
        final BytesRef needle = term.bytes();
        Terms terms = terms(field);
        if (terms == null) {
            return NOT_FOUND;
        }

        try (TermsEnum te = terms.getThreadTermsEnum(true)) {
            if (!te.seekExact(needle, true)) {
                return NOT_FOUND;
            }
            Bits deleted = getDeletedDocs();
            DocsEnum de = te.docs(deleted, null);
            int count = 0;
            int docId;
            while ((docId = de.nextDoc()) != Scorer.NO_MORE_DOCS) {
                count++;
            }
            Document[] docs = new Document[count];
            de = te.docs(deleted, de);
            count = 0;
            while ((docId = de.nextDoc()) != Scorer.NO_MORE_DOCS) {
                docs[count++] = document(docId);
            }
            return docs;
        }
    }

    public int deleteDocumentsFast(final Term term) throws IOException {
        final String field = term.field();
        final BytesRef needle = term.bytes();
        Terms terms = terms(field);
        int deletedCount = 0;
        if (terms == null) {
            return -1;
        }

        try (TermsEnum te = terms.getThreadTermsEnum(true)) {
            if (!te.seekExact(needle, true)) {
                return -2;
            }
            Bits deleted = getDeletedDocs();
            DocsEnum de = te.docs(deleted, null);
            int docId;
            while ((docId = de.nextDoc()) != Scorer.NO_MORE_DOCS) {
                deleteDocument(docId);
                deletedCount++;
            }
            if (deletedCount == 0) {
                try {
                    System.err.println("DeleteDocumentsFast.dump: term="
                        + term + ", lastDocId=" + lastDocId);
                    setDebug();
                    de = te.docs(null, null);
                    while ((docId = de.nextDoc()) != Scorer.NO_MORE_DOCS) {
                        System.err.println("DeleteDocumentsFast.dump: term=" + term
                            + ", docId=" + docId);
//                    deleteDocument(docId);
//                    deletedCount++;
                    }
                    System.err.println("DeleteDocumentsFast.dump: term="
                        + term + ", reseeking");
                    if (!te.seekExact(needle, true)) {
                        System.err.println("DeleteDocumentsFast.dump: term="
                        + term + " seek failed");
                        return -2;
                    }
                    de = te.docs(null, null);
                    while ((docId = de.nextDoc()) != Scorer.NO_MORE_DOCS) {
                        System.err.println("DeleteDocumentsFast.dump: term=" + term
                            + ", docId=" + docId);
//                    deleteDocument(docId);
//                    deletedCount++;
                    }
                } finally {
                    resetDebug();
                }
            }
        }
        return deletedCount;
    }

    public int deleteDocuments(final Query query) throws IOException {
        if (query instanceof TermQuery) {
            return deleteDocumentsFast(((TermQuery)query).getTerm());
//            return;
        }
        IndexSearcher searcher = new IndexSearcher(this);
        int deleted = 0;
        try {
            Weight weight = query.weight(searcher);
            Scorer scorer = weight.scorer(
                (AtomicReaderContext)searcher.getTopReaderContext(),
                Weight.ScorerContext.def());
            if (scorer != null) {
                while (true) {
                    int doc = scorer.nextDoc();
                    if (doc == Scorer.NO_MORE_DOCS) {
                        break;
                    }
                    deleted++;
                    deleteDocument(doc);
                }
            } else {
                return -1;
            }
        } finally {
            searcher.close();
        }
        return deleted;
    }

    public TermFreqVector[] getTermFreqVectors(int docNumber)
        throws IOException
    {
        throw new UnsupportedOperationException("This reader does not support this method.");
    }

    public TermFreqVector getTermFreqVector(int docNumber, String field)
        throws IOException
    {
        throw new UnsupportedOperationException("This reader does not support this method.");
    }

    public void getTermFreqVector(int docNumber, String field, TermVectorMapper mapper)
        throws IOException
    {
        throw new UnsupportedOperationException("This reader does not support this method.");
    }

    public void getTermFreqVector(int docNumber, TermVectorMapper mapper)
        throws IOException
    {
        throw new UnsupportedOperationException(
            "This reader does not support this method.");
    }

    public int numDocs() {
        int count = 0;
        Bits deleted = getDeletedDocs();
        for (int i = 0; i < maxDoc(); i++) {
            if (!deleted.get(i)) {
                count++;
            }
        }
        return count;
    }

    public int maxDoc() {
        return lastDocId + 1;
    }

    @Override
    public Directory directory() {
        return directory;
    }

    /** Returns the number of deleted documents. */
    public int numDeletedDocs() {
        if (prevDeletes == null) {
            return 0;
        }
        return prevDeletes.length();
    }

    public Document document(int n) throws CorruptIndexException, IOException {
        ensureOpen();
        return document(n, (FieldSelector) null);
    }

    public FieldsWriter getFieldsWriter() {
        return storedFieldsWriter.getFieldsWriter();
    }

    public Document document(int n, FieldSelector fieldSelector)
        throws CorruptIndexException, IOException
    {
        try {
//            return storedFieldsWriter.getFieldsWriter().doc(n, fieldSelector);
            return docWriter.doc(n, fieldSelector);
        } catch (java.lang.IndexOutOfBoundsException e) {
            System.err.println("PerFieldMergind.docsInProgress: "
                + Arrays.toString(docsInProgress)
                + ", lastDocId=" + lastDocId + ", n=" + n);
            throw e;
        } catch (IOException e) {
            throw new IOException("PerFieldMergind.docsInProgress: "
                + Arrays.toString(docsInProgress), e);
        }
    }

    @Override
    public void readDocument(final int n, final FieldVisitor visitor)
        throws CorruptIndexException, IOException
    {
        try {
            docWriter.readDocument(n, visitor);
        } catch (java.lang.IndexOutOfBoundsException e) {
            System.err.println("PerFieldMergind.docsInProgress: "
                + Arrays.toString(docsInProgress)
                + ", lastDocId=" + lastDocId + ", n=" + n);
            throw e;
        } catch (IOException e) {
            throw new IOException("PerFieldMergind.docsInProgress: "
                + Arrays.toString(docsInProgress), e);
        }
    }

    public boolean hasDeletions() {
        return (docsInProgress != null && docsInProgress.length > 0)
            || prevDeletes.length() > 0;
    }

    public byte[] norms(String field) throws IOException {
        return null;
    }

    protected void doSetNorm(int doc, String field, byte value)
          throws CorruptIndexException, IOException
    {
        throw new UnsupportedOperationException(
            "This reader does not support this method.");
    }

    public synchronized Fields fields() throws IOException {
        if (fields == null) {
            fields = new MergingFields(allPerFields, lastDocId, this);
        }
        return fields;
    }

    public int docFreq(Term term) throws IOException {
        return docFreq(term.field(), term.bytes());
    }

    public int docFreq(String field, BytesRef term) throws IOException {
        final Fields fields = fields();
        if (fields == null) {
            return 0;
        }
        final Terms terms = fields.terms(field);
        if (terms == null) {
            return 0;
        }
        return terms.docFreq(term);
    }

    public long totalTermFreq(String field, BytesRef term) throws IOException {
        final Fields fields = fields();
        if (fields == null) {
            return 0;
        }
        final Terms terms = fields.terms(field);
        if (terms == null) {
            return 0;
        }
        return terms.totalTermFreq(term);
    }

    public Terms terms(String field) throws IOException {
        final Fields fields = fields();
        if (fields == null) {
            return null;
        }
        return fields.terms(field);
    }

    public DocsEnum termDocsEnum(Bits skipDocs, String field, BytesRef term) throws IOException {
        final Fields fields = fields();
        if (fields == null) {
            return null;
        }
        final Terms terms = fields.terms(field);
        if (terms != null) {
            return terms.docs(skipDocs, term, null);
        } else {
            return null;
        }
    }

    public DocsAndPositionsEnum termPositionsEnum(Bits skipDocs, String field, BytesRef term) throws IOException {
        final Fields fields = fields();
        if (fields == null) {
            return null;
        }
        final Terms terms = fields.terms(field);
        if (terms != null) {
            return terms.docsAndPositions(skipDocs, term, null);
        } else {
            return null;
        }
    }

    public DocsEnum termDocsEnum(Bits skipDocs, String field, BytesRef term, TermState state) throws IOException {
        final Fields fields = fields();
        if (fields == null) {
            return null;
        }
        final Terms terms = fields.terms(field);
        if (terms != null) {
            return terms.docs(skipDocs, term, state, null);
        } else {
            return null;
        }
    }

    public DocsAndPositionsEnum termPositionsEnum(Bits skipDocs, String field, BytesRef term, TermState state) throws IOException {
        final Fields fields = fields();
        if (fields == null) {
            return null;
        }
        final Terms terms = fields.terms(field);
        if (terms != null) {
            return terms.docsAndPositions(skipDocs, term, state, null);
        } else {
            return null;
        }
    }

    protected void doDelete(int docNum)
        throws CorruptIndexException, IOException
    {
        throw new UnsupportedOperationException("This reader does not support this method.");
    }

    protected void doUndeleteAll() throws CorruptIndexException, IOException {
        throw new UnsupportedOperationException("This reader does not support this method.");
    }

    protected void doCommit(Map<String, String> commitUserData)
        throws IOException
    {
    }

    protected void doClose() throws IOException {
        if (fields != null) {
            fields.close();
        }
    }

    public Collection<String> getFieldNames(FieldOption fieldOption) {
        Set<String> fieldSet = new HashSet<String>();
        for (Map.Entry<String, ArrayReference<PerFieldAndThread>> entry :
            allPerFields.entrySet())
        {
            PerFieldAndThread[] pafArray = entry.getValue().get();
            for (DocFieldProcessor.PerFieldAndThread paf : pafArray) {
                FieldInfo fi = paf.perField.fieldInfo;
                if (fieldOption == IndexReader.FieldOption.ALL) {
                    fieldSet.add(fi.name);
                } else if (!fi.isIndexed && fieldOption == IndexReader.FieldOption.UNINDEXED) {
                    fieldSet.add(fi.name);
                } else if (fi.omitTermFreqAndPositions && fieldOption == IndexReader.FieldOption.OMIT_TERM_FREQ_AND_POSITIONS) {
                    fieldSet.add(fi.name);
                } else if (fi.storePayloads && fieldOption == IndexReader.FieldOption.STORES_PAYLOADS) {
                    fieldSet.add(fi.name);
                } else if (fi.isIndexed && fieldOption == IndexReader.FieldOption.INDEXED) {
                    fieldSet.add(fi.name);
                } else if (fi.isIndexed && fi.storeTermVector == false && fieldOption == IndexReader.FieldOption.INDEXED_NO_TERMVECTOR) {
                    fieldSet.add(fi.name);
                } else if (fi.storeTermVector == true &&
                    fi.storePositionWithTermVector == false &&
                    fi.storeOffsetWithTermVector == false &&
                    fieldOption == IndexReader.FieldOption.TERMVECTOR)
                {
                    fieldSet.add(fi.name);
                } else if (fi.isIndexed && fi.storeTermVector && fieldOption == IndexReader.FieldOption.INDEXED_WITH_TERMVECTOR) {
                    fieldSet.add(fi.name);
                } else if (fi.storePositionWithTermVector && fi.storeOffsetWithTermVector == false && fieldOption == IndexReader.FieldOption.TERMVECTOR_WITH_POSITION) {
                    fieldSet.add(fi.name);
                } else if (fi.storeOffsetWithTermVector && fi.storePositionWithTermVector == false && fieldOption == IndexReader.FieldOption.TERMVECTOR_WITH_OFFSET) {
                    fieldSet.add(fi.name);
                } else if ((fi.storeOffsetWithTermVector && fi.storePositionWithTermVector) &&
                    fieldOption == IndexReader.FieldOption.TERMVECTOR_WITH_POSITION_OFFSET)
                {
                    fieldSet.add(fi.name);
                }
            }
        }
        if (DEBUG()) {
            System.err.println("PFMIR.getFieldNames(" + fieldOption + ")=" + fieldSet);
        }
        return fieldSet;
    }

    public Bits getDeletedDocs() {
        if (deletedDocuments == null) {
            synchronized (this) {
                if (deletedDocuments == null) {
                    if (docsInProgress == null) {
                        deletedDocuments = prevDeletes;
                    } else {
                        deletedDocuments = createBits();
                    }
                }
            }
        }
        return deletedDocuments;
    }

    private Bits createBits() {
        return new Bits() {
            @Override
            public boolean get(int docId) {
                if (prevDeletes.get(docId)) {
                    return true;
                }
                for (int d: docsInProgress) {
                    if (d == docId) {
                        return true;
                    }
                }
                return false;
            }
            @Override
            public int length() {
                return prevDeletes.length();
            }
            @Override
            public int count() {
                int inp = 0;
                for (int d: docsInProgress) {
                    if (d != -1) {
                        inp++;
                    }
                }
                return prevDeletes.count() + inp;
            }
        };
    }

    public ReaderContext getTopReaderContext() {
        return new AtomicReaderContext(this);
    }

    public Object getCoreCacheKey() {
        return this;
    }

    private static class Deletes implements Bits {
        private final Bits[] prevDeletes;
        private final Bits newDeletes;

        public Deletes() {
            prevDeletes = new Deletes[0];
            newDeletes = new Bits.MatchNoBits(0);
        }

        public Deletes(final int[] newDeletes, final Bits[] prevDeletes) {
            if (complexity(newDeletes, prevDeletes) > 20) {
                this.prevDeletes = new Bits[1];
                this.prevDeletes[0] = collapseBits(newDeletes, prevDeletes);
                this.newDeletes = new Bits.MatchNoBits(0);
            } else {
                this.prevDeletes = prevDeletes;
                this.newDeletes = new IntArrayBits(newDeletes);
            }
        }

        public Bits[] toArray() {
            if (newDeletes.length() == 0) {
                return prevDeletes;
            } else {
                Bits[] array = new Bits[prevDeletes.length + 1];
                // Arrays.copyOf not used in order to avoid ArrayStoreException
                for (int i = 0; i < prevDeletes.length; ++i) {
                    array[i] = prevDeletes[i];
                }
                array[prevDeletes.length] = newDeletes;
                return array;
            }
        }

        //returns complexity in terms of number of compares
        private static int complexity(
            final int[] newDeletes,
            final Bits[] prevDeletes)
        {
            int compares = prevDeletes.length;
            for (Bits b : prevDeletes) {
                if (b instanceof IntArrayBits) {
                    compares += b.length();
                }
            }
            compares += newDeletes.length;
            return compares;
        }

        private static Bits collapseBits(
            final int[] newDeletes,
            final Bits[] prevDeletes)
        {
            int max = 0;
            for (Bits b : prevDeletes) {
                int size = maxBit(b);
                if (max < size) {
                    max = size;
                }
            }
            for (int d : newDeletes) {
                if (d > max) {
                    max = d;
                }
            }
            BitSet bits = new BitSet(max + 1);
            for (Bits b : prevDeletes) {
                extractBits(b, bits);
            }
            for (int d : newDeletes) {
                bits.set(d);
            }
            return new BitSetBits(bits);
        }

        private static int maxBit(Bits b) {
            int size;
            if (b instanceof BitSetBits) {
                size = ((BitSetBits)b).max();
            } else if (b instanceof IntArrayBits) {
                size = ((IntArrayBits)b).max();
            } else {
                size = b.length();
            }
            return size;
        }

        private static void extractBits(Bits b, BitSet bits) {
            if (b instanceof BitSetBits) {
                ((BitSetBits)b).fillBits(bits);
            } else if (b instanceof IntArrayBits) {
                ((IntArrayBits)b).fillBits(bits);
            } else {
                for (int i = 0; i < b.length(); i++) {
                    if (b.get(i)) {
                        bits.set(i);
                    }
                }
            }
        }

        @Override
        public boolean get(int bit) {
            for (Bits b : prevDeletes) {
                if (b.get(bit)) {
                    return true;
                }
            }
            return newDeletes.get(bit);
        }

        @Override
        public int length() {
            int size = 0;
            for (Bits b : prevDeletes) {
                size += b.length();
            }
            size += newDeletes.length();
            return size;
        }

        @Override
        public int count() {
            int count = 0;
            for (Bits b: prevDeletes) {
                count += b.count();
            }
            count += newDeletes.count();
            return count;
        }
    }

    private static class BitSetBits implements Bits {
        private final BitSet bits;
        public BitSetBits(final BitSet bits) {
            this.bits = bits;
        }

        public BitSetBits(final int[] bits) {
            int max = 0;
            for (int i : bits) {
                if (max < i) {
                    max = i;
                }
            }
            this.bits = new BitSet(max + 1);
            for (int i : bits) {
                this.bits.set(i);
            }
        }

        @Override
        public boolean get(int bit) {
            return bits.get(bit);
        }

        @Override
        public int length() {
            return bits.length();
        }

        public int max() {
            return bits.size() - 1;
        }

        public void fillBits(BitSet bits) {
            bits.or(this.bits);
        }

        @Override
        public int count() {
            return bits.cardinality();
        }
    }

    private static class IntArrayBits implements Bits {
        private final int[] array;
        public IntArrayBits(final int[] array) {
            this.array = array;
        }

        @Override
        public boolean get(int bit) {
            for (int b : array) {
                if (b == bit) {
                    return true;
                }
            }
            return false;
        }

        @Override
        public int length() {
            return array.length;
        }

        public int max() {
            int max = 0;
            for (int i : array) {
                if (max < i) {
                    max = i;
                }
            }
            return max;
        }

        public void fillBits(BitSet bits) {
            for (int b : array) {
                bits.set(b);
            }
        }

        @Override
        public int count() {
            return array.length;
        }
    }

    private static class MergingFields extends Fields {
        final Map<String,ArrayReference<PerFieldAndThread>> allPerFields;
        final int lastDocId;
        HashMap<String, MergingTerms> terms = null;
        final PerFieldMergingIndexReader parent;
        public MergingFields(
            final Map<String,ArrayReference<PerFieldAndThread>> allPerFields,
            final int lastDocId,
            final PerFieldMergingIndexReader parent)
        {
            this.allPerFields = allPerFields;
            this.lastDocId = lastDocId;
            this.parent = parent;
        }

        public synchronized Terms terms(String field) {
            if (terms == null) {
                terms = new HashMap<String, MergingTerms>();
            }
            if (DEBUG()) {
                System.err.println("PerFieldMerginIndexreader.MergingFields.terms(" + field + ")");
            }
            MergingTerms t = terms.get(field);
            if (t == null) {
                t = new MergingTerms(field, allPerFields, lastDocId, parent);
                terms.put(field, t);
            }
            return t;
        }

        public FieldsEnum iterator() {
            return new MergingFieldsEnum(this);
        }

        public void close() {
            for (MergingTerms mt : terms.values()) {
                mt.close();
            }
        }
    }

    private static class MergingFieldsEnum extends FieldsEnum {
        private final MergingFields fields;
        private String currentField = null;
        private final Iterator<String> iter;
        public MergingFieldsEnum(final MergingFields fields) {
            this.fields = fields;
            this.iter = fields.allPerFields.keySet().iterator();
        }

        public String next() {
            if (iter.hasNext()) {
                currentField = iter.next();
            } else {
                currentField = null;
            }
            if (DEBUG()) {
                System.err.println("PerFieldMerginIndexreader.MergingFieldsEnum.next()=" + currentField);
            }
            return currentField;
        }

        public TermsEnum terms(final boolean buffered) throws IOException {
            if (DEBUG()) {
                System.err.println("PerFieldMerginIndexreader.MergingFieldsEnum.terms()=" + currentField);
            }
            if (currentField == null) {
                return null;
            }
            return fields.terms(currentField).iterator(true);
        }

    }

    private static class MergingTerms extends Terms {
        private final Map<String,ArrayReference<PerFieldAndThread>> allPerFields;
        private final String field;
        private final FreqProxFieldMergeState[] states;
        private final int statesCount;
        private final int lastDocId;
        private final PerFieldMergingIndexReader parent;
        final ConcurrentLinkedQueue<MergingTermsIterator> enumCache =
            new ConcurrentLinkedQueue<MergingTermsIterator>();

        public MergingTerms(final String field, 
            final Map<String,ArrayReference<PerFieldAndThread>> allPerFields,
            final int lastDocId,
            final PerFieldMergingIndexReader parent)
        {
            this.field = field;
            this.allPerFields = allPerFields;
            this.lastDocId = lastDocId;
            this.parent = parent;
            ArrayReference<PerFieldAndThread> perFieldsRef = 
                allPerFields.get(field);
            if (perFieldsRef != null) {
                PerFieldAndThread[] perFields = perFieldsRef.get();
                statesCount = perFields.length;
                states = new FreqProxFieldMergeState[statesCount];
                int count = 0;
                for (DocFieldProcessor.PerFieldAndThread paf : perFields) {
                    DocInverterPerField di = (DocInverterPerField)paf.perField.consumer;
                    TermsHashPerField th = (TermsHashPerField)di.consumer;
                    FreqProxFieldMergeState state = new FreqProxFieldMergeState(
                        (FreqProxTermsWriterPerField)th.consumer,
                        BytesRef.getUTF8SortedAsUnicodeComparator(), lastDocId);
                    states[count++] = state;
                }
            } else {
                statesCount = 0;
                states = new FreqProxFieldMergeState[statesCount];
            }
        }

        public long getSumTotalTermFreq() {
            throw new UnsupportedOperationException("This reader does not support this method.");
        }

        public Comparator<BytesRef> getComparator() {
            return BytesRef.getUTF8SortedAsUnicodeComparator();
        }

        private MergingTermsIterator createIterator() {
            return new MergingTermsIterator(field, states, statesCount, getComparator(), lastDocId, parent, enumCache);
        }

        public TermsEnum getThreadTermsEnum() {
            MergingTermsIterator te = enumCache.poll();
            if (te == null) {
                return createIterator();
            }
            te.reset();
            return te;
        }

        public TermsEnum iterator(final boolean buffered) {
            return getThreadTermsEnum();
        }

        public void close() {
            enumCache.clear();
            for (int i = 0; i < statesCount; i++) {
                final FreqProxFieldMergeState state = states[i];
                state.close();
            }
        }
    }

    private static class MergingTermsIterator extends TermsEnum {
        private final FreqProxFieldMergeState[] states;
        private final FreqProxFieldMergeState[] mergedStates;
        private final int statesCount;
        private final String field;
        private final int lastDocId;
        private int docFreq;
        private BytesRef term = new BytesRef();
        private int merged;
        private final Comparator<BytesRef> termComp;
        private final PerFieldMergingIndexReader parent;
        private final ConcurrentLinkedQueue<MergingTermsIterator> enumCache;

        public MergingTermsIterator(final String field,
            final FreqProxFieldMergeState[] states,
            final int statesCount,
            final Comparator<BytesRef> termComp,
            final int lastDocId,
            final PerFieldMergingIndexReader parent,
            final ConcurrentLinkedQueue<MergingTermsIterator> enumCache)
        {
            this.field = field;
            this.parent = parent;
            this.states = new FreqProxFieldMergeState[statesCount];
            this.lastDocId = lastDocId;
            for (int i = 0; i < statesCount; i++) {
                this.states[i] = states[i].clone();
            }
            this.statesCount = statesCount;
            this.termComp = termComp;
            this.enumCache = enumCache;
            mergedStates = new FreqProxFieldMergeState[statesCount];
            merged = -1;
        }

        public void reset() {
            merged = -1;
            for (FreqProxFieldMergeState state : states) {
                state.reset();
            }
        }

        @Override
        public void close() {
            enumCache.add(this);
        }

        public String toString() {
            return parent.toString();
        }

        @Override
        public boolean seekExact(BytesRef text, boolean useCache) {
            boolean found = false;
            if (DEBUG()) {
                System.err.println("MergingTermsIterator<" + this + ">.seekExact: statesCount=" + statesCount);
            }
            merged = 0;
            for (int i = 0; i < statesCount; i++) {
                boolean success = states[i].seekExact(text);
                if (success) {
                    mergedStates[merged++] = states[i];
                }
            }
            docFreq = 0;
            if (merged > 0) {
                term.bytes = mergedStates[0].text.bytes;
                term.offset = mergedStates[0].text.offset;
                term.length = mergedStates[0].text.length;
                for (int i = 0; i < merged; i++) {
                    docFreq += mergedStates[i].postings.termFreqs.getInt(
                        mergedStates[i].currentTermID);
                }
                if (docFreq <= 0) {
                    docFreq = merged;
                }
                found = true;
            }
            if (DEBUG()) {
                System.err.println("MergingTermsIterator<" + this + ">.seekExact: text: " + text.utf8ToString() +", term: " + term.utf8ToString() + ", found: " + found);
            }
            return found;
        }

        @Override
        public SeekStatus seek(BytesRef text, boolean useCache) {
            if (DEBUG()) {
                System.err.println("MergingTermsIterator<" + this + ">.seek: statesCount=" + statesCount);
            }
            merged = 0;
            int start = -1;
            SeekStatus ss = null;
            for (int i = 0; i < statesCount; i++) {
                mergedStates[0] = states[i];
                ss = mergedStates[0].seek(text);
                if (ss != SeekStatus.END) {
                    start = i + 1;
                    merged = 1;
                    break;
                }
            }
            if (start == -1) {
                if (DEBUG()) {
                    System.err.println("MergingTermsIterator<" + this + ">: text: " + text.utf8ToString() + ", status: END");
                }
                return SeekStatus.END;
            }
            SeekStatus lastStatus = ss;

            // TODO: priority queue
            for(int i=start;i<statesCount;i++) {
                ss = states[i].seek(text);
                if (ss != SeekStatus.END) {
                    final int cmp = termComp.compare(states[i].text, mergedStates[0].text);
                    if (cmp < 0) {
                        mergedStates[0] = states[i];
                        merged = 1;
                        lastStatus = ss;
                    } else if (cmp == 0) {
                        mergedStates[merged++] = states[i];
                    }
                }
            }

            term.bytes = mergedStates[0].text.bytes;
            term.offset = mergedStates[0].text.offset;
            term.length = mergedStates[0].text.length;  
            docFreq = 0;
            for (int i = 0; i < merged; i++) {
                docFreq += mergedStates[i].postings.termFreqs.getInt(mergedStates[i].currentTermID);
            }
            if (docFreq <= 0) {
                docFreq = merged;
            }
            if (DEBUG()) {
                System.err.println("MergingTermsIterator<" + this + ">: text: " + text.utf8ToString() +", term: " + term.utf8ToString() + ", status: " + lastStatus);
            }
            return lastStatus;
        }

        public SeekStatus seek(long ord) {
            throw new UnsupportedOperationException("This reader does not support this method.");
        }

        public BytesRef next() {
            if (merged > 0) {
                for (int i = 0; i < merged; i++) {
                    mergedStates[i].nextTerm();
                }
                if (DEBUG()) {
                    System.err.println("MergingTermsIterator.next(): merged=" + merged);
                }
            } else if (merged == -1) {
                //first call to next
                if (DEBUG()) {
                    System.err.println("MergingTermsIterator.next(): first call");
                }
                for (int i = 0; i < statesCount; i++) {
                    states[i].nextTerm();
                }
                merged = 0;
            }

            int start = -1;
            for (int i = 0; i < statesCount; i++) {
                mergedStates[0] = states[i];
                if (!mergedStates[0].termsEof()) {
                    start = i + 1;
                    merged = 1;
                    break;
                }
            }

            if (DEBUG()) {
                System.err.println("MergingTermsIterator.next(): start=" + start);
            }

            if (start == -1) {
                return null;
            }

            // TODO: priority queue
            for(int i=start;i<statesCount;i++) {
                if (!states[i].termsEof()) {
                    final int cmp = termComp.compare(states[i].text, mergedStates[0].text);
                    if (cmp < 0) {
                        mergedStates[0] = states[i];
                        merged = 1;
                    } else if (cmp == 0) {
                        mergedStates[merged++] = states[i];
                    }
                }
            }


            term.bytes = mergedStates[0].text.bytes;
            term.offset = mergedStates[0].text.offset;
            term.length = mergedStates[0].text.length;
            if (DEBUG()) {
                System.err.println("MergingTermsIterator.next(): tem=" + term.utf8ToString());
            }
            docFreq = 0;
            for (int i = 0; i < merged; i++) {
                docFreq += mergedStates[i].postings.termFreqs.getInt(mergedStates[i].currentTermID);
            }
            if (docFreq <= 0) {
                docFreq = merged;
            }
            return term;
        }

        public BytesRef term() {
            return term;
        }

        public long ord() {
            throw new UnsupportedOperationException("This reader does not support this method.");
        }

        public int docFreq() {
            return docFreq;
        }

        public long totalTermFreq() throws IOException {
            throw new UnsupportedOperationException("This reader does not support this method.");
        }

        public DocsEnum docs(
            Bits skipDocs, DocsEnum reuse)
                throws IOException
        {
            if (reuse instanceof PerFieldMergingDocsEnum) {
                PerFieldMergingDocsEnum de = (PerFieldMergingDocsEnum)reuse;
                if (de.termsEnum == this) {
                    de.reset(skipDocs);
                    return de;
                }
                return new PerFieldMergingDocsEnum(this, skipDocs);
            } else {
                return new PerFieldMergingDocsEnum(this, skipDocs);
            }
        }

        public DocsAndPositionsEnum docsAndPositions(
            Bits skipDocs, DocsAndPositionsEnum reuse)
                throws IOException
        {
            if (reuse instanceof PerFieldMergingDocsAndPositionsEnum) {
                PerFieldMergingDocsAndPositionsEnum de = (PerFieldMergingDocsAndPositionsEnum)reuse;
                if (de.termsEnum == this) {
                    de.reset(skipDocs);
                    return de;
                }
                return new PerFieldMergingDocsAndPositionsEnum(this, skipDocs);
            } else {
                return new PerFieldMergingDocsAndPositionsEnum(this, skipDocs);
            }
        }

        public Comparator<BytesRef> getComparator() {
            return BytesRef.getUTF8SortedAsUnicodeComparator();
        }
    }

    private static final class BytesRefLocal extends ThreadLocal<BytesRef> {
        @Override
        protected BytesRef initialValue() {
            return new BytesRef();
        }
    }

    private static final class FreqProxFieldMergeState {
        final FreqProxTermsWriterPerField field;
        final int numPostings;
        final int lastDocId;
        private final ByteBlockPool bytePool;
//        final NativeIntArray[] termIDs;
        final int[][] termIDs;
        final int[] positionPerSlice;
        final FreqProxPostingsArray postings;
        int currentTermID;
        int lastPostingDocId = -1;
        int lastPostingTermFreq = -1;
        Bits skipDocs = null;
        boolean queueDirty = true;
        int skipProx = 0;

        private int pqSize;
        private final int maxPqSize;
        private int[] idxHeap;
        private static final BytesRefLocal cmpTermALocal = new BytesRefLocal();
        private static final BytesRefLocal cmpTermBLocal = new BytesRefLocal();

        final BytesRef text = new BytesRef();

        ByteSliceReader freq = null;//new ByteSliceReader();
        ByteSliceReader prox = null;//new ByteSliceReader();

        int docID;
        int termFreq;
        int position;
        boolean hitEof;
        boolean docsEof;
        boolean termsEof = false;
        boolean freqEof = false;
        int maxDocs = 0;
        int readDocs = 0;

        private FreqProxFieldMergeState(FreqProxFieldMergeState other) {
            this.lastDocId = other.lastDocId;
            this.field = other.field;
            this.numPostings = other.numPostings;
            this.bytePool = other.bytePool;
            this.termIDs = other.termIDs;
            this.postings = other.postings;
            this.currentTermID = other.currentTermID;
//            this.text = new BytesRef();
            this.docID = other.docID;
            this.termFreq = other.termFreq;
            maxPqSize = termIDs.length;
            this.positionPerSlice = new int[maxPqSize];
        }

        public FreqProxFieldMergeState(FreqProxTermsWriterPerField field, Comparator<BytesRef> termComp, final int lastDocId) {
            this.field = field;
            this.bytePool = field.perThread.termsHashPerThread.bytePool;
            this.termIDs = field.termsHashPerField.sortPostingsExternal(termComp);
            int totalTerms = 0;
            maxPqSize = termIDs.length;
            positionPerSlice = new int[maxPqSize];

            this.numPostings = totalTerms;
            this.postings = (FreqProxPostingsArray) field.termsHashPerField.postingsArray;
            this.lastDocId = lastDocId;
//            text = new BytesRef();
        }

        public void close() {
//            for (NativeIntArray array : termIDs) {
//                array.release();
//            }
        }

        public void reset() {
            queueDirty = true;
            lastPostingDocId = -1;
            lastPostingTermFreq = -1;
            skipDocs = null;
            termsEof = false;
            freqEof = false;
            freq = null;
            prox = null;
            skipProx = 0;
        }

        private final boolean sliceEof(final int idx) {
//            final NativeIntArray termIDsSlice = termIDs[idx];
            final int[] termIDsSlice = termIDs[idx];
            final int position = positionPerSlice[idx];
            return position >= termIDsSlice.length;
        }

        private final int sliceId(final int idx) {
//            final NativeIntArray termIDsSlice = termIDs[idx];
            final int[] termIDsSlice = termIDs[idx];
            final int position = positionPerSlice[idx];
            return termIDsSlice[position];
        }

        private final void setPosition(final int idx, final int position) {
            positionPerSlice[idx] = position;
        }

        private final void incrementPosition(final int idx) {
            positionPerSlice[idx]++;
        }

        private final boolean lessThan(int idxA, int  idxB) {
            final boolean aEof = sliceEof(idxA);
            final boolean bEof = sliceEof(idxB);
            if (aEof && bEof) {
                return false;
            }
            if (bEof) {
                return true;
            }
            if (aEof) {
                return false;
            }
            final int idA = sliceId(idxA);
            int textStart = postings.textStarts.getInt(idA);
            final BytesRef cmpTermA = cmpTermALocal.get();
            final BytesRef cmpTermB = cmpTermBLocal.get();
            bytePool.setBytesRef(cmpTermA, textStart);

            final int idB = sliceId(idxB);
            textStart = postings.textStarts.getInt(idB);
            bytePool.setBytesRef(cmpTermB, textStart);
            int cmp = cmpTermA.compareTo(cmpTermB);
            //Allow GC reclaim internal ByteBlockPool
            cmpTermA.bytes = null;
            cmpTermB.bytes = null;
            if (cmp < 0) {
                return true;
            }
            return false;
        }

        private void resetQueue() {
            pqSize = 0;
            if (idxHeap != null) {
                return;
            }
            int heapSize;
            if (maxPqSize == 0) {
                // We allocate 1 extra to avoid if statement in top()
                heapSize = 2;
            } else {
                heapSize = maxPqSize + 1;
            }
            idxHeap = new int[heapSize];
        }

        private final int addIdx(int idx) {
            pqSize++;
            idxHeap[pqSize] = idx;
            upHeap();
            return idxHeap[1];
        }

        private final int topIdx() {
            return idxHeap[1];
        }

        private final int updateTopIdx() {
            downHeap();
            return idxHeap[1];
        }

        private final void upHeap() {
            int i = pqSize;
            int node = idxHeap[i];			  // save bottom node
            int j = i >>> 1;
            while (j > 0 && lessThan(node, idxHeap[j])) {
                idxHeap[i] = idxHeap[j];			  // shift parents down
                i = j;
                j = j >>> 1;
            }
            idxHeap[i] = node;				  // install saved node
        }

        private final void downHeap() {
            int i = 1;
            int node = idxHeap[i];			  // save top node
            int j = i << 1;				  // find smaller child
            int k = j + 1;
            if (k <= pqSize && lessThan(idxHeap[k], idxHeap[j])) {
                j = k;
            }
            while (j <= pqSize && lessThan(idxHeap[j], node)) {
                idxHeap[i] = idxHeap[j];			  // shift up child
                i = j;
                j = i << 1;
                k = j + 1;
                if (k <= pqSize && lessThan(idxHeap[k], idxHeap[j])) {
                    j = k;
                }
            }
            idxHeap[i] = node;				  // install saved node
        }

        private void initQueue() {
            resetQueue();
            for (int idx = 0; idx < maxPqSize; idx++) {
                setPosition(idx, 0);
                addIdx(idx);
            }
            queueDirty = false;
        }

        public void initQueue(BytesRef term) {
            resetQueue();
            for (int idx = 0; idx < maxPqSize; idx++) {
                seekSlice(term, idx);
                addIdx(idx);
            }
            queueDirty = false;
        }

        private void seekSlice(BytesRef term, final int idx) {
            int min = 0;
//            final NativeIntArray ids = termIDs[idx];
            final int[] ids = termIDs[idx];
            int max = ids.length - 1;
            int needle = -1;
            if (DEBUG()) {
                System.err.println("PFMIR.seek.ref.start: " + text.utf8ToString() + ", min=" + min + ", max=" + max + ", needle=" + needle);// + ", scans: " + scans);
            }

            int mid = 0;
            int id;
            while (max >= min) {
                mid = (min + max) >>> 1;
                id = ids[mid];
                int textStart = postings.textStarts.getInt(id);
                bytePool.setBytesRef(text, textStart);
                if (DEBUG()) {
                    System.err.println("PFMIR.seek.ref.iter: " + text.utf8ToString() + ",  min=" + min + ", max=" + max + ", needle=" + needle + ", mid=" + mid + ", tstart=" + textStart + ", curTID=" + id);
                }
                int delta = term.compareTo(text);
                if (delta < 0) {
                    max = mid - 1;
                } else if (delta > 0) {
                    min = mid + 1;
                } else {
                    needle = mid;
                    break;
                }
            }
            if (DEBUG()) {
                System.err.println("PFMIR.seek.ref.end: " + text.utf8ToString() + ", min=" + min + ", max=" + max + ", needle=" + needle + ", mid=" + mid);// + ", scans: " + scans);
            }
            if (needle != -1) {
                setPosition(idx, needle);
            } else if (min > ids.length - 1) {
                setPosition(idx, ids.length);
            } else {
                setPosition(idx, max + 1);
                if (DEBUG()) {
                    id = ids[max + 1];
                    final int textStart = postings.textStarts.getInt(id);
                    bytePool.setBytesRef(text, textStart);
                    System.err.println("PFMIR.seek.ref.end: NOT_FOUND " + text.utf8ToString());// + ", scans: " + scans);
                }
            }
        }


        boolean termsEof() {
            if (DEBUG()) {
                System.err.println("FreqProxFieldMergeState.termsEof: " + termsEof);
            }
            return termsEof;
        }

        boolean docsEof() {
            return docsEof;
        }

        boolean nextTerm() { //throws IOException {
            if (queueDirty) {
                if (DEBUG()) {
                    System.err.println("FreqProxFieldMergeState.nextTerm.initQueue");
                }
                initQueue();
            }

            int topIdx = topIdx();
            if (termIDs.length == 0) {
                termsEof = true;
                return false;
            }
            if (sliceEof(topIdx)) {
                if (DEBUG()) {
                    System.err.println("FreqProxFieldMergeState: nextTerm: false");
                }
                termsEof = true;
                return false;
            }
            currentTermID = sliceId(topIdx);
            incrementPosition(topIdx);
            updateTopIdx();
            docID = -1;

            // Get BytesRef
            final int textStart = postings.textStarts.getInt(currentTermID);
            bytePool.setBytesRef(text, textStart);
            if (DEBUG()) {
                System.err.println("FreqProxFieldMergeState: nextTerm: term=" + text.utf8ToString());
            }
            return true;
        }

        public void initDocReader() {
            if (DEBUG()) {
                System.err.println("PFMIR.initDocReader");
            }
            hitEof = false;
            docsEof = false;
            if (freq == null) {
                freq = new ByteSliceReader();
            }
            while (true) {
                if (postings.lastDocCodes.getInt(currentTermID) != -1) {
                    lastPostingDocId = postings.lastDocIDs.getInt(currentTermID);
                    lastPostingTermFreq = postings.docFreqs.getInt(currentTermID);
                    maxDocs = postings.termFreqs.getInt(currentTermID);
                    if (lastPostingDocId == postings.lastDocIDs.getInt(currentTermID)) {
                        field.termsHashPerField.initReader(freq, currentTermID, 0);
                        break;
                    }
                } else {
                    lastPostingDocId = -1;
                    maxDocs = 0;
                    break;
                }
            }
            freqEof = false;
            docID = -1;
            readDocs = 0;
        }

        public void initDocProxReader() {
            if (DEBUG()) {
                System.err.println("PFMIR.initDocProxReader: termID=" + currentTermID);
            }
            hitEof = false;
            docsEof = false;
            if (freq == null) {
                freq = new ByteSliceReader();
            }
            while(true) {
                if (postings.lastDocCodes.getInt(currentTermID) != -1) {
                    lastPostingDocId = postings.lastDocIDs.getInt(currentTermID);
                    maxDocs = postings.termFreqs.getInt(currentTermID);
                    field.termsHashPerField.initReader(freq, currentTermID, 0);
                    if (!field.fieldInfo.omitTermFreqAndPositions) {
                        if (prox == null) {
                            prox = new ByteSliceReader();
                        }
                        field.termsHashPerField.initReader(prox, currentTermID, 1);
                        lastPostingTermFreq = postings.docFreqs.getInt(currentTermID);
                        if (lastPostingDocId == postings.lastDocIDs.getInt(currentTermID)) {
                            if (DEBUG()) {
                                System.err.println("PFMIR.initDocProxReader.break");
                            }
                            break;
                        } else {
                            if (DEBUG()) {
                                System.err.println("PFMIR.initDocProxReader.retry");
                            }
                            //lastPostingDocId changed while initializing proxReader
                            //retrying
                        }
                    } else {
                        if (DEBUG()) {
                            System.err.println("PFMIR.initDocProxReader.omitTermFreqAndPositions");
                        }
                        break;
                    }
                } else {
                    lastPostingDocId = -1;
                    maxDocs = 0;
                    break;
                }
            }
            freqEof = false;
            readDocs = 0;
            docID = -1;
        }

        public boolean seekExact(BytesRef term) {
            queueDirty = true;
            int termId = field.termsHashPerField.findTerm(term, text);
            final int postingsLen = postings.textStarts.size();
            if (termId >= postingsLen) {
                System.err.print(this + ": term=" + term + " postings.textStarts.length="
                    + postingsLen + ", termId=" + termId);
            }
            if (termId != -1 && termId < postingsLen) {
                currentTermID = termId;
                final int textStart = postings.textStarts.getInt(currentTermID);
                bytePool.setBytesRef(text, textStart);
                return true;
            } else {
                return false;
            }
        }

        public SeekStatus seek(BytesRef term) {
            initQueue(term);
            int topIdx = topIdx();
            if (sliceEof(topIdx)) {
                termsEof = true;
                return SeekStatus.END;
            }
            currentTermID = sliceId(topIdx);
            final int textStart = postings.textStarts.getInt(currentTermID);
            bytePool.setBytesRef(text, textStart);

            incrementPosition(topIdx);
            updateTopIdx();

            if (term.compareTo(text) == 0) {
                return SeekStatus.FOUND;
            }
            return SeekStatus.NOT_FOUND;
        }

        public boolean advance(int target) throws IOException {
            if (docsEof) return false;
            if (lastPostingDocId != -1) {
                if (target > lastPostingDocId) {
                    docsEof = true;
                    return false;
                } else if (target == lastPostingDocId
                     && lastPostingDocId > docID
                     && !hitEof
                     && lastPostingDocId <= lastDocId)
                {
                    hitEof = true;
                    docID = lastPostingDocId;
                    if (!field.omitTermFreqAndPositions) {
                        termFreq = lastPostingTermFreq;
                    }
                    freqEof = true;
                    return true;
                }
            }
            while (target > docID) {
                nextDoc();
            }
            if (docsEof) {
                return false;
            }
            return true;
        }

        public final boolean nextDoc() throws IOException {
            int prevDoc = -2;
            while (nextDocInt()) {
                if (docID == prevDoc) {
                    throw new RuntimeException("Duplicating DocId: " + docID + ", prevDoc=" + prevDoc);
                }
                prevDoc = docID;
                if (skipDocs != null && skipDocs.get(docID)) {
                    continue;
                }
                break;
            }
            return !docsEof;
        }

        public final boolean nextDocInt() throws IOException {
            if (DEBUG()) {
                System.err.println("FreqProxFieldMergeState: nextDoc(): freqEof=" + freqEof + ", docsEof=" + docsEof + ", hitEof=" + hitEof + ", maxDocs=" + maxDocs);
            }
            if (lastPostingDocId == -1) {
                docsEof = true;
                return false;
            }
            if (readDocs == maxDocs - 1) {
                freqEof = true;
            }

            if (prox != null) {
                while (skipProx > 0) {
                    skipPosition();
                }
                position = 0;
            }

            if (freqEof) {
                if (DEBUG()) {
                    System.err.println("FreqProxFieldMergeState: nextDoc().freq.oef()");
                }
                if (lastPostingDocId != -1 && lastPostingDocId > docID && !hitEof) {
                    if (DEBUG()) {
                        System.err.println("FreqProxFieldMergeState.nextDoc(): lastDocCodes[currentTermID]=" + postings.lastDocCodes.getInt(currentTermID) + ", lastDocIDs[currentTermID]=" + postings.lastDocIDs.getInt(currentTermID) + ", docID=" + docID + ", lastPostingDocId=" + lastPostingDocId);
                    }
                    hitEof = true;
                    docID = lastPostingDocId;
                    if (!field.omitTermFreqAndPositions) {
                        skipProx = termFreq = lastPostingTermFreq;
                        if (DEBUG()) {
                            System.err.println("FreqProxFieldMergeState.nextDoc(): lastTermFreq[currentTermID]=" + postings.docFreqs.getInt(currentTermID) + ", lastPostingTermFreq=" + lastPostingTermFreq);
                        }
                    }
                    if (docID > lastDocId) {
                        docsEof = true;
                        if (DEBUG()) {
                            System.err.println("FreqProxFieldMergeState: nextDoc().freq.oef(): docId > lastDocId: " + docID + "/" + lastDocId);
                        }
                        return false;
                    }
                    if (DEBUG()) {
                        System.err.println("FreqProxFieldMergeState: nextDoc().freq.oef(): docId: " + docID + " this=" + this);
                    }
                    return true;
                } else {
                // EOF
                    if (DEBUG()) {
                        System.err.println("FreqProxFieldMergeState: nextDoc().freq.oef() : return false");
                    }
                    docsEof = true;
                    return false;
                }
            }
            readDocs++;
            if (docID == -1) docID++;
            final int code = freq.readVInt();
            if (field.omitTermFreqAndPositions) {
                docID += code;
            } else {
                docID += code >>> 1;
                if ((code & 1) != 0) {
                    termFreq = 1;
                } else {
                    termFreq = freq.readVInt();
                }
                skipProx = termFreq;
            }
            if (docID > lastDocId) {
                if (DEBUG()) {
                    System.err.println("FreqProxFieldMergeState: nextDoc(): docId > lastDocId: " + docID + "/" + lastDocId);
                }
                docsEof = true;
                return false;
            }
            if (DEBUG()) {
                System.err.println("FreqProxFieldMergeState: nextDoc(): docId: " + docID + " this=" + this);
            }

            return true;
        }

        public int nextPosition(
            final PerFieldMergingDocsAndPositionsEnum docsEnum)
            throws IOException
        {
            skipProx--;
            final int code = prox.readVInt();
            position += code >> 1;
            if (DEBUG()) {
                System.err.println("FreqProxFieldMergeState: nextPosition: " + code + ", position=" + position);
            }

            if ((code & 1) != 0) {
              // This position has a payload
              docsEnum.payloadLength = prox.readVInt();  
              if (docsEnum.payload == null) {
                docsEnum.payload = new BytesRef();
                docsEnum.payload.bytes = new byte[docsEnum.payloadLength];
              } else if (docsEnum.payload.bytes.length < docsEnum.payloadLength) {
                docsEnum.payload.grow(docsEnum.payloadLength);
              }

              prox.readBytes(docsEnum.payload.bytes, 0, docsEnum.payloadLength);
              docsEnum.payload.length = docsEnum.payloadLength;
            } else {
              docsEnum.payloadLength = 0;
            }
            return position;
        }

        private void skipPosition() throws IOException {
            skipProx--;
            final int code = prox.readVInt();
            position += code >> 1;
            if (DEBUG()) {
                System.err.println("FreqProxFieldMergeState: skipPosition: " + code + ", position=" + position);
            }

            if ((code & 1) != 0) {
              // This position has a payload
              int payloadLength = prox.readVInt();
              while (payloadLength-- > 0) {
                prox.readByte();
              }
            }
        }

        public FreqProxFieldMergeState clone() {
            return new FreqProxFieldMergeState(this);
        }

    }

    private static class PerFieldMergingDocsEnum extends DocsEnum {
        public final MergingTermsIterator termsEnum;
        private int freq;
        private int docId;
        private int merged;
        private FreqProxFieldMergeState[] mergedStates;
        private Bits skipDocs;
        private FreqProxFieldMergeState minState;
        private int minStateNum;

        public PerFieldMergingDocsEnum(final MergingTermsIterator termsEnum, Bits skipDocs) throws IOException {
            this.termsEnum = termsEnum;
            reset(skipDocs);
        }

        public void reset(Bits skipDocs) throws IOException {
            this.skipDocs = skipDocs;
            this.mergedStates = termsEnum.mergedStates;
            this.merged = termsEnum.merged;
            for (int i = 0; i < merged; i++) {
                FreqProxFieldMergeState state = mergedStates[i];
                state.initDocReader();
                state.skipDocs = skipDocs;
                state.nextDoc();
                minState = null;
            }
            if (DEBUG()) {
                System.err.println("PerFieldMergingDocsEnum<" + termsEnum.parent + ">.reset(): merged=" + merged);
            }
        }

        public int docID() {
            return docId;
        }

        public int nextDoc() throws IOException {
            while(true) {
                if (minState != null) {
                    minState.nextDoc();
                    minState = null;
                }
                for(int i=0;i<merged;i++) {
                    if (!mergedStates[i].docsEof()) {
                        if (minState == null) {
                            minState = mergedStates[i];
                            minStateNum = i;
                        } else {
                            if (mergedStates[i].docID < minState.docID) {
                                minState = mergedStates[i];
                                minStateNum = i;
                            }
                        }
                    }
                }
                if (minState == null) {
                    return DocIdSetIterator.NO_MORE_DOCS;
                }
                freq = minState.termFreq;
                docId = minState.docID;
                if (DEBUG()) {
                    System.err.println("PerFieldMergingDocsEnum<" + termsEnum.parent + ">: nextDoc: " + docId);
                }
                return docId;
            }
        }

        public int freq() {
            return freq;
        }

        public int advance(int target) throws IOException {
            if (target == DocIdSetIterator.NO_MORE_DOCS) {
                return target;
            }
            minState = null;
            for (int i = 0; i < merged; i++) {
                if (mergedStates[i].advance(target)) {
                    if (minState == null) {
                        minState = mergedStates[i];
                        minStateNum = i;
                    } else {
                        if (mergedStates[i].docID < minState.docID) {
                            minState = mergedStates[i];
                            minStateNum = i;
                        }
                    }
                }
            }
            if (minState == null) {
                return DocIdSetIterator.NO_MORE_DOCS;
            }
            freq = minState.termFreq;
            docId = minState.docID;
            if (skipDocs != null && skipDocs.get(docId)) {
                int doc;
                int iters = 0;
                while ((doc = nextDoc()) < target) {
                    iters++;
                }
                if (iters > FLUSH_THRESHOLD) {
                    termsEnum.parent.needFlush = true;
                }
                return doc;
            }
            return docId;
        }

    }

    private static class PerFieldMergingDocsAndPositionsEnum extends DocsAndPositionsEnum {
        public final MergingTermsIterator termsEnum;
        private int freq;
        private int docId;
        private int merged;
        private FreqProxFieldMergeState[] mergedStates;
        private int payloadLength = 0;
        private BytesRef payload = null;
        private FreqProxFieldMergeState minState = null;
        private int minStateNum;
        private Bits skipDocs;

        public PerFieldMergingDocsAndPositionsEnum(final MergingTermsIterator termsEnum, Bits skipDocs) throws IOException {
            this.termsEnum = termsEnum;
            reset(skipDocs);
        }

        public void reset(Bits skipDocs) throws IOException {
            this.skipDocs = skipDocs;
            this.mergedStates = termsEnum.mergedStates;
            this.merged = termsEnum.merged;
            for (int i = 0; i < merged; i++) {
                FreqProxFieldMergeState state = mergedStates[i];
                state.initDocProxReader();
                state.skipDocs = skipDocs;
                state.nextDoc();
                minState = null;
            }
            if (DEBUG()) {
                System.err.println("PerFieldMergingDocsAndPositionsEnum<" + System.identityHashCode(termsEnum) + ">.reset(): merged=" + merged);
            }
        }

        public int docID() {
            return docId;
        }

        public int nextDoc() throws IOException {
            while(true) {
                if (minState != null) {
                    minState.nextDoc();
                    minState = null;
                }
                for(int i=0;i<merged;i++) {
                    if (!mergedStates[i].docsEof()) {
                        if (minState == null) {
                            minState = mergedStates[i];
                                minStateNum = i;
                        } else {
                            if (mergedStates[i].docID < minState.docID) {
                                minState = mergedStates[i];
                                minStateNum = i;
                            }
                        }
                    }
                }
                if (minState == null) return DocIdSetIterator.NO_MORE_DOCS;
                freq = minState.termFreq;
                docId = minState.docID;
                if (DEBUG()) {
                    System.err.println("PerFieldMergingDocsAndPositionsEnum<" + System.identityHashCode(termsEnum) + ">: nextDoc: " + docId);
                }
                return docId;
            }
        }

        public int nextPosition() throws IOException {
            return minState.nextPosition(this);
        }

        public BytesRef getPayload() {
            return payload;
        }

        public boolean hasPayload() {
            return payloadLength != 0;
        }

        public int freq() {
            return freq;
        }

        public int advance(int target) throws IOException {
            int doc;
            int iters = 0;
            while ((doc = nextDoc()) < target) {
                iters++;
            }
            if (iters > FLUSH_THRESHOLD) {
                termsEnum.parent.needFlush = true;
            }
            return doc;
        }
    }
}

