package org.apache.lucene.index;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import java.io.IOException;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.FutureTask;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.lucene.store.Directory;
import org.apache.lucene.util.ThreadInterruptedException;
import org.apache.lucene.util.CollectionUtil;

import ru.yandex.concurrent.NamedThreadFactory;

import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOScheduler;

public class SharedThreadPoolMergeScheduler extends MergeScheduler {
    private final PriorityBlockingQueue<Runnable> writersQueue;
    private final ThreadPoolExecutor executor;
    private final AtomicInteger mergesInProgress = new AtomicInteger(0);
    private final ConcurrentHashMap<IndexWriter, IndexWriter> allWriters =
        new ConcurrentHashMap<>();

    private int threadPriority = Thread.NORM_PRIORITY;
    private int ioclass = Compress.IOPRIO_CLASS_IDLE;
    private int ioprio = 0;

    public SharedThreadPoolMergeScheduler(int threads) {
        writersQueue = new PriorityBlockingQueue<>();
        executor = new ThreadPoolExecutor(
            threads,
            threads,
            1,
            TimeUnit.MINUTES,
            writersQueue,
            new NamedThreadFactory("Merger"));
    }

    public void setMergeThreadPriority(
        final int pri,
        final int ioclass,
        final int ioprio)
    {
        if (pri > Thread.MAX_PRIORITY || pri < Thread.MIN_PRIORITY) {
            throw new IllegalArgumentException("priority must be in range "
                + Thread.MIN_PRIORITY + " .. "
                + Thread.MAX_PRIORITY + " inclusive");
        }
        threadPriority = pri;
        this.ioclass = ioclass;
        this.ioprio = ioprio;
    }

    public int mergesInProgress() {
        return mergesInProgress.get();
    }

    public int queuedMerges() {
        int sum = 0;
        for (IndexWriter writer : allWriters.values()) {
            sum += writer.pendingMerges();
        }
        return sum;
    }

    @Override
    public void close() {
    }

    @Override
    public void merge(IndexWriter writer) throws IOException {
        allWriters.put(writer, writer);
        MergePolicy.OneMerge merge = writer.getNextMerge();
        if (merge != null) {
            executor.execute(
                new PriorityMergeTask(
                    new MergeTask(writer, merge)));
        }
    }

    private static class PriorityMergeTask extends FutureTask<Void>
        implements Comparable<PriorityMergeTask>
    {
        private final MergeTask task;

        PriorityMergeTask(final MergeTask task) {
            super(task, null);
            this.task = task;
        }

        @Override
        public int compareTo(final PriorityMergeTask other) {
            return task.compareTo(other.task);
        }
    }

    private class MergeTask implements Comparable <MergeTask>, Runnable {
        private final IndexWriter writer;
        private final MergePolicy.OneMerge merge;

        MergeTask(final IndexWriter writer,  MergePolicy.OneMerge merge)
            throws IOException
        {
            this.writer = writer;
            this.merge = merge;
            if (merge != null) {
                writer.mergeInit(merge);
            }
        }

        @Override
        public int compareTo(final MergeTask other) {
            if (other == null || other.merge == null) {
                return 1;
            }
            return Integer.compare(
                merge.segments.totalDocCount(),
                other.merge.segments.totalDocCount());
        }

        @Override
        public void run() {
            if (merge == null) {
                return;
            }
            Compress.setThreadPriority(
                threadPriority,
                true);
            Compress.setThreadIOPriority(
                ioclass,
                ioprio);
            IOScheduler.setThreadReadPrio(IOScheduler.IOPRIO_MERGE);
            IOScheduler.setThreadWritePrio(IOScheduler.IOPRIO_MERGE);
            boolean success = false;
            Exception mergeException = null;
            mergesInProgress.incrementAndGet();
            try {
                writer.merge(merge);
                success = true;
            } catch (Exception e) {
                e.printStackTrace();
                mergeException = e;
            } finally {
                mergesInProgress.decrementAndGet();
                if (!success) {
                    try {
                        writer.mergeFinish(merge);
                    } catch (Exception e) {
                        if (mergeException != null) {
                            mergeException.addSuppressed(e);
                        } else {
                            mergeException = e;
                        }
                    }
                }
            }
            if (mergeException != null) {
                if (merge.getException() == null) {
                    merge.setException(mergeException);
                } else {
                    merge.getException().addSuppressed(mergeException);
                }
            }
            if (writer.pendingMerges() > 0) {
                try {
                    merge(writer);
                } catch (IOException ignore) {
                    ignore.printStackTrace();
                }
            }
        }
    }
}
