package org.apache.lucene.index;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.Reader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.document.AbstractField;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldMapping;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.FieldVisitor;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.codecs.MergeState;
import org.apache.lucene.index.SegmentMerger.DocIdStream;
import org.apache.lucene.index.SegmentMerger.FieldGroupWithReader;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.BufferedIndexInput;
import org.apache.lucene.store.Decompressor;
import org.apache.lucene.store.DeflateDecompressor;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.store.PackedIndexedCompressedInputStream;
import org.apache.lucene.store.ZstdDecompressor;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CloseableThreadLocal;
import org.apache.lucene.util.packed_ondisk.PackedInts;

import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;

/**
 * Class responsible for access to stored document fields.
 * <p/>
 * It uses &lt;segment&gt;.fdt and &lt;segment&gt;.fdx; files.
 * 
 * @lucene.internal
 */
public class StandardFieldsReader implements FieldsReader {
    public static final int FORMAT = StandardFieldsWriter.FORMAT_CURRENT;

    private static final BytesRef GROUP_SEP = new BytesRef("#");

  private static final NativeMemoryAllocator allocator =
    NativeMemoryAllocator.get("FieldsReaderBlockCache");
  private final static int FORMAT_SIZE = 4;

  private final FieldInfos fieldInfos;

  // The main fieldStream, used only for cloning.
  private final IndexInput cloneableFieldsStream;
//  private final IndexInput cloneableFieldsStreamBackend;

  // This is a clone of cloneableFieldsStream used for reading documents.
  // It should not be cloned outside of a synchronized context.
  private final IndexInput fieldsStream;

  private final IndexInput cloneableIndexStream;
  private final IndexInput indexStream;
  private final PackedInts.Reader indexReader;
  private int numTotalDocs;
  private int size;
  private boolean closed;
  private final int format;
  private String segment;

  // The docID offset where our docs begin in the index
  // file.  This will be 0 if we have our own private file.
  private int docStoreOffset;

  private CloseableThreadLocal<IndexInput> fieldsStreamTL = new CloseableThreadLocal<IndexInput>();
  private boolean isOriginal = false;

  /** Returns a cloned FieldsReader that shares open
   *  IndexInputs with the original one.  It is the caller's
   *  job not to close the original FieldsReader until all
   *  clones are called (eg, currently SegmentReader manages
   *  this logic). */
  @Override
  public FieldsReader clone() {
    ensureOpen();
    StandardFieldsReader clone = new StandardFieldsReader(
        fieldInfos,
        numTotalDocs,
        size,
        format,
        docStoreOffset,
        cloneableFieldsStream,
        cloneableIndexStream,
        indexReader);
    clone.segment = segment;
    return clone;
  }

  /** Verifies that the code version which wrote the segment is supported. */
  public static void checkCodeVersion(Directory dir, String segment) throws IOException {
    final String indexStreamFN = IndexFileNames.segmentFileName(segment, "", IndexFileNames.FIELDS_INDEX_EXTENSION);
    IndexInput idxStream = dir.openInput(indexStreamFN, 1024);

    try {
      int format = idxStream.readInt();
      if (format < StandardFieldsWriter.FORMAT_MINIMUM && format != 100 && format != 101)
        throw new IndexFormatTooOldException(indexStreamFN, format, StandardFieldsWriter.FORMAT_MINIMUM, StandardFieldsWriter.FORMAT_CURRENT);
      if (format > StandardFieldsWriter.FORMAT_CURRENT && format != 100 && format != 101)
        throw new IndexFormatTooNewException(indexStreamFN, format, StandardFieldsWriter.FORMAT_MINIMUM, StandardFieldsWriter.FORMAT_CURRENT);
    } finally {
      idxStream.close();
    }

  }

  // Used only by clone
  private StandardFieldsReader(FieldInfos fieldInfos, int numTotalDocs, int size, int format, int docStoreOffset,
                       IndexInput cloneableFieldsStream, IndexInput cloneableIndexStream, PackedInts.Reader indexReader) {
    this.fieldInfos = fieldInfos;
    this.numTotalDocs = numTotalDocs;
    this.size = size;
    this.format = format;
    this.docStoreOffset = docStoreOffset;
    this.cloneableFieldsStream = cloneableFieldsStream;
    this.cloneableIndexStream = cloneableIndexStream;
    fieldsStream = (IndexInput) cloneableFieldsStream.clone();
    indexStream = (IndexInput) cloneableIndexStream.clone();
    if (indexReader != null) {
        this.indexReader = indexReader.clone();
    } else {
        this.indexReader = null;
    }
//    this.indexReader = indexReader;
/*
    if( format == 100 )
    {
    }
    else
    {
	this.cloneableFieldsStreamInflater = null;
	fieldsStreamInflater = null;
        indexReader = null;
        indexOffsetReader = null;
    }
*/
  }
  
  public StandardFieldsReader(Directory d, String segment, FieldInfos fn) throws IOException {
    this(d, segment, fn, BufferedIndexInput.BUFFER_SIZE, -1, 0);
  }

  public StandardFieldsReader(Directory d, String segment, FieldInfos fn, int readBufferSize, int docStoreOffset, int size) throws IOException {
    boolean success = false;
    isOriginal = true;
    this.segment = segment;
    try {
      fieldInfos = fn;

//      cloneableFieldsStreamBackend = d.openInput(IndexFileNames.segmentFileName(segment, "", IndexFileNames.FIELDS_EXTENSION), readBufferSize);
      final String indexStreamFN = IndexFileNames.segmentFileName(segment, "", IndexFileNames.FIELDS_INDEX_EXTENSION);
      cloneableIndexStream = d.openInput(indexStreamFN, readBufferSize);
      
      format = cloneableIndexStream.readInt();

      if (format < StandardFieldsWriter.FORMAT_MINIMUM && format != 100 && format != 101)
        throw new IndexFormatTooOldException(indexStreamFN, format, StandardFieldsWriter.FORMAT_MINIMUM, StandardFieldsWriter.FORMAT_CURRENT);
      if (format > StandardFieldsWriter.FORMAT_CURRENT && format != 100 && format != 101)
        throw new IndexFormatTooNewException(indexStreamFN, format, StandardFieldsWriter.FORMAT_MINIMUM, StandardFieldsWriter.FORMAT_CURRENT);


      indexStream = (IndexInput) cloneableIndexStream.clone();

      cloneableFieldsStream = d.openInput(IndexFileNames.segmentFileName(segment, "", IndexFileNames.FIELDS_EXTENSION), readBufferSize);
      indexReader = null;

      fieldsStream = (IndexInput) cloneableFieldsStream.clone();


      final long indexSize = cloneableIndexStream.length() - FORMAT_SIZE;
      
      if (docStoreOffset != -1) {
        if( format == 100 || format == 101 ) throw new IOException( "Shared docstore is not supported in compressed format" );
        // We read only a slice out of this shared fields file
        this.docStoreOffset = docStoreOffset;
        this.size = size;

        // Verify the file is long enough to hold all of our
        // docs
        assert ((int) (indexSize / 8)) >= size + this.docStoreOffset: "indexSize=" + indexSize + " size=" + size + " docStoreOffset=" + docStoreOffset;
      } else {
        this.docStoreOffset = 0;
        if( format == 100 || format == 101 )
        {
    	    this.size = indexReader.size();
        }
        else
        {
    	    this.size = (int) (indexSize >> 3);
    	}
        
      }

      numTotalDocs = (int) (indexSize >> 3);
      success = true;
    } catch (IOException e) {
        throw new IOException("segment fields reader error: " + segment, e);
    } finally {
      // With lock-less commits, it's entirely possible (and
      // fine) to hit a FileNotFound exception above. In
      // this case, we want to explicitly close any subset
      // of things that were opened so that we don't have to
      // wait for a GC to do so.
      if (!success) {
        close();
      }
    }
  }

  public int format() {
    return format;
  }

    public void directIO(final boolean direct) {
        if (fieldsStream instanceof PackedIndexedCompressedInputStream) {
            ((PackedIndexedCompressedInputStream)fieldsStream).useCache(!direct);
        }
    }

  /**
   * @throws AlreadyClosedException if this FieldsReader is closed
   */
  private void ensureOpen() throws AlreadyClosedException {
    if (closed) {
      throw new AlreadyClosedException("this FieldsReader is closed");
    }
  }

  /**
   * Closes the underlying {@link org.apache.lucene.store.IndexInput} streams, including any ones associated with a
   * lazy implementation of a Field.  This means that the Fields values will not be accessible.
   *
   * @throws IOException
   */
  public final void close() throws IOException {
    if (!closed) {
      if (fieldsStream != null) {
        fieldsStream.close();
      }
      if (isOriginal) {
        if (cloneableFieldsStream != null) {
          cloneableFieldsStream.close();
        }
        if (cloneableIndexStream != null) {
          cloneableIndexStream.close();
          if( indexReader != null ) indexReader.close();
        }
      }
      if (indexStream != null) {
        indexStream.close();
      }
      fieldsStreamTL.close();
      closed = true;
    }
  }

  public final int size() {
    return size;
  }

  private void seekIndex(int docID) throws IOException {
    indexStream.seek(FORMAT_SIZE + (docID + docStoreOffset) * 8L);
  }

    public static void main(String[] args) throws IOException {
        final String filePath = args[0];
        final String dirPath = filePath.substring(0, filePath.lastIndexOf('/'));
        final String name = filePath.substring(dirPath.length() + 1);
        final Directory dir = NIOFSDirectory.get(new File(dirPath));
        final IndexInput ip = dir.openInput(name);
        int format = ip.readInt();
        System.out.println("Format: " + format);
        PackedInts.Reader indexReader = PackedInts.getReader(ip);
        for (int i = 0; i < indexReader.size(); i++) {
            System.out.println(i + " pos=" + indexReader.get(i));
        }
//    FSDirectory dir = FSDirectory.open( new File("/u0/failindex.2/") );
//    FieldInfos fis = new FieldInfos( dir, "_6bu.fnm" );
//    FieldsReader fr = new FieldsReader( dir, "_6bu", fis, 4 * 4096, -1, -1 );
//    fr.dumpDocs();
  }

    private static class PosId implements Comparable<PosId> {
        public final int docId;
        public final int oldDocId;
        public final long position;
        public PosId(final int docId, final int oldDocId, final long position) {
            this.docId = docId;
            this.oldDocId = oldDocId;
            this.position = position;
        }

        public int compareTo(final PosId other) {
            return Long.compare(position, other.position);
        }
    }

    public long fillGroupDocFieldMap(
        final Set<String> groupFields,
        final SegmentMerger.FieldGroupWithReader reusableKey,
        final Map<FieldGroupWithReader, DocIdStream> groupDocFieldMap,
        final Bits deletedDocs,
        final Set<String> storedFields,
        final FieldInfos outFieldInfos,
        final MergeState.CheckAbort checkAbort)
        throws IOException
    {
        final int[] groupFieldsNumbers = new int[groupFields.size()];
        int i = 0;
        for (final String field: groupFields) {
            final int fieldNum = fieldInfos.fieldNumber(field);
            groupFieldsNumbers[i++] = fieldNum;
        }
        final Set<Integer> storedFieldsNumbers = new HashSet<Integer>();
        for (String field : storedFields) {
            int storedFieldNum = fieldInfos.fieldNumber(field);
            if (storedFieldNum != -1) {
                storedFieldsNumbers.add(storedFieldNum);
            }
        }
//        if (false) {
        if (format == 100 || format == 101) {
//            try {
//                directIO(true);
                return fillGroupDocFieldMapNewFormat(
                    groupFieldsNumbers,
                    reusableKey,
                    groupDocFieldMap,
                    deletedDocs,
                    storedFieldsNumbers,
                    outFieldInfos);
//            } finally {
//                directIO(false);
//            }
        } else {
            return fillGroupDocFieldMapOldFormat(
                groupFieldsNumbers,
                reusableKey,
                groupDocFieldMap,
                deletedDocs,
                storedFieldsNumbers,
                outFieldInfos);
        }
    }

    private long fillGroupDocFieldMapNewFormat(
        final int[] groupFieldsNumbers,
        final SegmentMerger.FieldGroupWithReader reusableKey,
        final Map<FieldGroupWithReader, DocIdStream> groupDocFieldMap,
        final Bits deletedDocs,
        final Set<Integer> storedFields,
        final FieldInfos outFieldInfos)
        throws IOException
    {
        //first reverse index mapping to make reading of fields data sequential
        ArrayList<PosId> reverseList = new ArrayList<>(this.size);
        int docId = 0;
        for (int i = 0; i < this.size; i++) {
            if (deletedDocs != null && deletedDocs.get(i)) {
                continue;
            }
            final long position = indexReader.get(i);
//            System.err.println("BS<" + segment +">: pos=" + position + ", i=" + i);
            reverseList.add(new PosId(docId, i, position));
            docId++;
        }
        Collections.sort(reverseList);
        long totalSize = 0;
        final BytesRef groupRef = new BytesRef();
        final Map<Integer, BytesRef> groupFieldsValues =
            new HashMap<>();
        for (int fieldNum: groupFieldsNumbers) {
            groupFieldsValues.put(fieldNum, new BytesRef());
        }
        for (PosId posId : reverseList) {
            final int docSize;
            try {
                docSize =
                    fillGroupCalcSize(
                        posId.position,
                        groupFieldsValues,
                        storedFields,
                        outFieldInfos);
            } catch (IOException e) {
                System.err.println("fillGroupDocFieldMapNewFormat: failed "
                    + "docId=" + posId.docId + ", oldId="
                     + posId.oldDocId + ", pos=" + posId.position);
                throw e;
            }
            groupRef.offset = groupRef.length = 0;
            for (int fieldNum: groupFieldsNumbers) {
                groupRef.append(groupFieldsValues.get(fieldNum));
            }
            totalSize += docSize;
            reusableKey.rehash(groupRef);
            SegmentMerger.DocIdStream docIdStream =
                groupDocFieldMap.get(reusableKey);
            if (docIdStream == null) {
                docIdStream = new DocIdStream();
                groupDocFieldMap.put(reusableKey.clone(), docIdStream);
            }
            docIdStream.addDocId(posId.docId);
            docIdStream.addDocId(posId.oldDocId - posId.docId);
        }
        return totalSize;
    }

    private long fillGroupDocFieldMapOldFormat(
        final int[] groupFieldsNumbers,
        final SegmentMerger.FieldGroupWithReader reusableKey,
        final Map<FieldGroupWithReader, DocIdStream> groupDocFieldMap,
        final Bits deletedDocs,
        final Set<Integer> storedFields,
        final FieldInfos outFieldInfos)
        throws IOException
    {
        long totalSize = 0;
        final BytesRef groupRef = new BytesRef();
        final Map<Integer, BytesRef> groupFieldsValues =
            new HashMap<>();
        for (int fieldNum: groupFieldsNumbers) {
            groupFieldsValues.put(fieldNum, new BytesRef());
        }
        int docId = 0;
        for (int i = 0; i < size; i++) {
            if (deletedDocs != null && deletedDocs.get(i)) {
                continue;
            }
            seekIndex(i);
            final long position = indexStream.readLong();
            final int size =
                fillGroupCalcSize(
                    position,
                    groupFieldsValues,
                    storedFields,
                    outFieldInfos);
            groupRef.offset = groupRef.length = 0;
            for (int fieldNum: groupFieldsNumbers) {
                groupRef.append(GROUP_SEP);
                groupRef.append(groupFieldsValues.get(fieldNum));
            }
            totalSize += size;
            reusableKey.rehash(groupRef);
            SegmentMerger.DocIdStream docIdStream =
                groupDocFieldMap.get(reusableKey);
            if (docIdStream == null) {
                docIdStream = new DocIdStream();
                groupDocFieldMap.put(reusableKey.clone(), docIdStream);
            }
            docIdStream.addDocId(docId);
            docIdStream.addDocId(i - docId);
            docId++;
        }
        return totalSize;
    }

    private int fillGroupCalcSize(
        final long position,
        final Map<Integer, BytesRef> groupFields,
        final Set<Integer> storedFields,
        final FieldInfos outFieldInfos)
        throws IOException
    {
        int size = 0;
        fieldsStream.seek(position);
//        System.err.println("fgcs<" + segment + ">: " + position);
        int numFields = fieldsStream.readVInt();
        int storedCount = 0;
        for (int i = 0; i < numFields; i++) {
            int fieldNumber = fieldsStream.readVInt();
            byte bits = fieldsStream.readByte();

            boolean binary = (bits & StandardFieldsWriter.FIELD_IS_BINARY) != 0;

            if (!storedFields.contains(fieldNumber)) {
                skipField(fieldsStream);
//                System.err.println("Skipping field: " + fieldNumber
//                    + ", name=" + fieldInfos.fieldInfo(fieldNumber).name);
            } else {
                storedCount++;
                final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
                final int outFieldNumber = outFieldInfos.fieldNumber(fi.name);
                size += vIntSize(outFieldNumber) + 1;
                final BytesRef groupFieldValue = groupFields.get(fieldNumber);
                if (groupFieldValue == null || binary) {
                    size += skipField(fieldsStream);
                } else {
                    size += getFieldValue(fieldsStream, groupFieldValue);
                }
            }
        }
        size += vIntSize(storedCount);
//        System.err.println("fillGroupCalcSize: " + size);
        return size;
    }

    private static int getFieldValue(
        final IndexInput fldStream,
        final BytesRef value)
            throws IOException
    {
        final int length = fldStream.readVInt();
        if (value.bytes.length < length) {
            value.bytes = new byte[length * 2];
        }
        value.length = length;
        fldStream.readBytes(value.bytes, 0, length);
        return length + vIntSize(length);
    }

    private static int skipField(final IndexInput fieldsStream)
        throws IOException
    {
        final long pos = fieldsStream.getFilePointer();
        final int length = fieldsStream.readVInt();
        if (length < 0) {
            System.err.println("negative length: " + length);
            fieldsStream.seek(pos);
            for (int i = 0; i < 5; i++) {
                System.err.println("BD: "
                    + Integer.toHexString(fieldsStream.readByte()));
            }
        }
        skipField(length, fieldsStream);
        return length + vIntSize(length);
    }

    private static void skipField(
        final int toRead,
        final IndexInput fieldsStream)
        throws IOException
    {
//        System.err.println("skipField.filePointer="
//            + fieldsStream.getFilePointer()
//            + ", toRead=" + toRead);
        fieldsStream.seek(fieldsStream.getFilePointer() + toRead);
    }

    private static int vIntSize(int value) {
        int size = 1;
        while ((value & ~0x7F) != 0) {
            size++;
            value >>>= 7;
        }
//        return 5;
        return size;
    }

  public final Document doc(int n, FieldSelector fieldSelector) throws CorruptIndexException, IOException {
    boolean locked = false;
    if (format == 100 || format == 101) {
        locked = ((PackedIndexedCompressedInputStream)fieldsStream).lockBuffer();
    }
    try {
        return doGetDoc(n, fieldSelector);
    } finally {
        if (locked) {
            ((PackedIndexedCompressedInputStream)fieldsStream).releaseBuffer();
        }
    }
  }

    public void readDocument(final int n, final FieldVisitor visitor)
        throws CorruptIndexException, IOException
    {
        boolean locked = false;
        if (format == 100 || format == 101) {
            locked =
                ((PackedIndexedCompressedInputStream)fieldsStream).lockBuffer();
        }
        try {
            doReadDoc(n, visitor);
        } finally {
            if (locked) {
                ((PackedIndexedCompressedInputStream)fieldsStream)
                    .releaseBuffer();
            }
        }
    }

  public final boolean lockBuffer() throws IOException {
    boolean locked = false;
    if (format == 100 || format == 101) {
        locked = ((PackedIndexedCompressedInputStream)fieldsStream).lockBuffer();
    }
    return locked;
  }

  public final void releaseBuffer() throws IOException {
    if (format == 100 || format == 101) {
        ((PackedIndexedCompressedInputStream)fieldsStream).releaseBuffer();
    }
  }

  public final Document doGetDoc(int n, FieldSelector fieldSelector) throws CorruptIndexException, IOException {
    long position;

    boolean locked = false;
    if( format == 100 || format == 101 )
    {
	position = indexReader.get(n);
    }
    else
    {
	seekIndex(n);
	position = indexStream.readLong();
    }
    fieldsStream.seek(position);


    int numFields = fieldsStream.readVIntUnlockedSafe();
    final int maxToLoad =
        fieldSelector == null ? Integer.MAX_VALUE :
            fieldSelector.maxFieldCount();
    Document doc = new Document(Math.min(numFields, maxToLoad));
    int loaded = 0;
    for (int i = 0; i < numFields && loaded < maxToLoad; i++) {
      int fieldNumber = fieldsStream.readVIntUnlockedSafe();
      FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
      FieldSelectorResult acceptField = fieldSelector == null ? FieldSelectorResult.LOAD : fieldSelector.accept(fi.name);

      byte bits = fieldsStream.readByteUnlocked();
      assert bits <= StandardFieldsWriter.FIELD_IS_TOKENIZED + StandardFieldsWriter.FIELD_IS_BINARY;

      boolean tokenize = (bits & StandardFieldsWriter.FIELD_IS_TOKENIZED) != 0;
      boolean binary = (bits & StandardFieldsWriter.FIELD_IS_BINARY) != 0;
      //TODO: Find an alternative approach here if this list continues to grow beyond the
      //list of 5 or 6 currently here.  See Lucene 762 for discussion
      if (acceptField.equals(FieldSelectorResult.LOAD)) {
        addField(doc, fi, binary, tokenize);
        loaded++;
      }
      else if (acceptField.equals(FieldSelectorResult.LOAD_AND_BREAK)){
        addField(doc, fi, binary, tokenize);
        loaded++;
        break;//Get out of this loop
      }
      else if (acceptField.equals(FieldSelectorResult.LAZY_LOAD)) {
        addFieldLazy(doc, fi, binary, tokenize, true);
        loaded++;
//        System.err.println( "LAZY FIELD" );
      }
      else if (acceptField.equals(FieldSelectorResult.LATENT)) {
        addFieldLazy(doc, fi, binary, tokenize, false);
        loaded++;
//        System.err.println( "LAZY FIELD" );
      }
      else if (acceptField.equals(FieldSelectorResult.SIZE)){
        skipField(addFieldSize(doc, fi, binary));
        loaded++;
      }
      else if (acceptField.equals(FieldSelectorResult.SIZE_AND_BREAK)){
        addFieldSize(doc, fi, binary);
        loaded++;
//        System.err.println( "SIZE FIELD" );
        break;
      }
      else {
        skipField();
      }
    }

    return doc;
  }

    public final void doReadDoc(int n, FieldVisitor visitor)
        throws CorruptIndexException, IOException
    {
        long position;
        if (format == 100 || format == 101) {
            position = indexReader.get(n);
        } else {
            seekIndex(n);
            position = indexStream.readLong();
        }
        fieldsStream.seek(position);

        // TODO: Remove code duplication with DeflateFieldsReader
        int numFields = fieldsStream.readVIntUnlockedSafe();
        FieldMapping mapping = visitor.fieldMapping(fieldInfos);
        FieldSelectorResult[] fieldSelectorResults =
            mapping.fieldSelectorResults();
        int[] fieldNumberToIndex = mapping.fieldNumberToIndex();
        int fieldsLeft = mapping.maxFieldCount();
        for (int i = 0; i < numFields; ++i) {
            final int fieldNumber = fieldsStream.readVIntUnlockedSafe();

            //ignored
            final byte bits = fieldsStream.readByteUnlocked();

            switch (fieldSelectorResults[fieldNumber]) {
                case LOAD:
                case LOAD_AND_BREAK:
                    visitor.storeFieldValue(
                        fieldNumberToIndex[fieldNumber],
                        readField());
                    if (--fieldsLeft == 0) {
                        return;
                    }
                    break;
                case SIZE:
                case SIZE_AND_BREAK:
                    visitor.storeFieldSize(
                        fieldNumberToIndex[fieldNumber],
                        skipField());
                    if (--fieldsLeft == 0) {
                        return;
                    }
                    break;
                default:
                    skipField();
                    break;
            }
        }
    }

    public final void writeDocument(
        final int docId,
        final int newDocId,
        final FieldsWriter out,
        final Set<String> storedFields)
        throws IOException
    {
        writeDocument(docId, newDocId, out, storedFields, false);
    }

    public final void writeDocument(
        final int docId,
        final int newDocId,
        final FieldsWriter out,
        final Set<String> storedFields,
        final boolean noIndex)
        throws IOException
    {
        long position;
        if (format == 100 || format == 101) {
            position = indexReader.get(docId);
        } else {
            seekIndex(docId);
            position = indexStream.readLong();
        }
        fieldsStream.seek(position);

        int numFields = fieldsStream.readVInt();
        position = fieldsStream.getFilePointer();
        int storedFieldsCount = 0;
        for (int i = 0; i < numFields; ++i) {
            final int fieldNumber = fieldsStream.readVInt();
            final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
            final byte bits = fieldsStream.readByte();
            final int length = fieldsStream.readVInt();
            if (storedFields.contains(fi.name)) {
                ++storedFieldsCount;
            } else {
//                System.err.println("Skipping field: " + fi.name);
            }
            fieldsStream.seek(fieldsStream.getFilePointer() + length);
        }
        if (noIndex) {
            out.newDocumentNoIndex(newDocId, storedFieldsCount);
        } else {
            out.newDocument(newDocId, storedFieldsCount);
        }
        fieldsStream.seek(position);

        for (int i = 0; i < numFields; ++i) {
            final int fieldNumber = fieldsStream.readVInt();
            final FieldInfo fi = fieldInfos.fieldInfo(fieldNumber);
            final byte bits = fieldsStream.readByte();
            final int length = fieldsStream.readVInt();
            if (storedFields.contains(fi.name)) {
//                java.util.zip.CRC32 crc = new java.util.zip.CRC32();
                out.writeField(fi, bits, fieldsStream, length);
//                System.err.println("FR: writeField: name=" + fi.name + ", length=" + length);
            } else {
//                System.err.println("Skipping field: " + fi.name);
                fieldsStream.seek(fieldsStream.getFilePointer() + length);
            }
        }
    }


  /** Returns the length in bytes of each raw document in a
   *  contiguous range of length numDocs starting with
   *  startDocID.  Returns the IndexInput (the fieldStream),
   *  already seeked to the starting point for startDocID.*/
  public final IndexInput rawDocs(int[] lengths, int startDocID, int numDocs) throws IOException {
    if (format == 100 || format == 101) {
        if (numDocs != 1) {
            throw new IllegalStateException("Grouped compressed fieldsreader supports only one doc in rawDocs mode but numDocs=" + numDocs);
        }
        long offset = indexReader.get(startDocID);
        fieldsStream.seek(offset);
        try {
        int size = getDocSize(fieldsStream);
	lengths[0] = size;
	fieldsStream.seek(offset);
	return fieldsStream;
	} catch (IOException e) {
	    e.printStackTrace();
	    Document doc = doc(startDocID, null);
	    System.err.println("Doc loaded successfuly");
	    throw e;
	}
    } else {
        seekIndex(startDocID);
        long startOffset = indexStream.readLong();
        long lastOffset = startOffset;
        int count = 0;
        while (count < numDocs) {
            final long offset;
            final int docID = docStoreOffset + startDocID + count + 1;
            assert docID <= numTotalDocs;
            if (docID < numTotalDocs) 
                offset = indexStream.readLong();
            else
                offset = fieldsStream.length();
            lengths[count++] = (int) (offset-lastOffset);
            lastOffset = offset;
        }

        fieldsStream.seek(startOffset);

        return fieldsStream;
    }
  }

    private int getDocSize(final IndexInput fldStream) throws IOException
    {
        long startOffset = fldStream.getFilePointer();
        int numFields = fldStream.readVIntUnlockedSafe();
        for (int i = 0; i < numFields; i++) {
            int fieldNumber = fldStream.readVIntUnlockedSafe();
            byte bits = fldStream.readByteUnlocked();
            int skip = fldStream.readVIntUnlockedSafe();
            if (fldStream.getFilePointer() + skip > fldStream.length()) {
                throw new IOException("Seeking behind end of fieldStream: curpos=" +
                    fldStream.getFilePointer() + ", skip=" + skip +
                        ", length=" + fldStream.length() +
                            ", numFields=" + numFields + ", i=" + i + ", file=" + fldStream.getCacheKey());
            }
            fldStream.seek(fldStream.getFilePointer() + skip);
        }
        return (int)(fldStream.getFilePointer() - startOffset);
    }


  /**
   * Skip the field.  We still have to read some of the information about the field, but can skip past the actual content.
   * This will have the most payoff on large fields.
   */
  private int skipField() throws IOException {
    return skipField(fieldsStream.readVIntUnlockedSafe());
  }
  
  private int skipField(int toRead) throws IOException {
    fieldsStream.seek(fieldsStream.getFilePointer() + toRead);
    return toRead;
  }

  private void addFieldLazy(Document doc, FieldInfo fi, boolean binary, boolean tokenize, boolean cacheResult) throws IOException {
    if (binary) {
      int toRead = fieldsStream.readVIntUnlockedSafe();
      long pointer = fieldsStream.getFilePointer();
      //was: doc.add(new Fieldable(fi.name, b, Fieldable.Store.YES));
      doc.add(new LazyField(fi.name, Field.Store.YES, toRead, pointer, binary, cacheResult));
      //Need to move the pointer ahead by toRead positions
      fieldsStream.seek(pointer + toRead);
    } else {
      Field.Store store = Field.Store.YES;
      Field.Index index = Field.Index.toIndex(fi.isIndexed, tokenize);
      Field.TermVector termVector = Field.TermVector.toTermVector(fi.storeTermVector, fi.storeOffsetWithTermVector, fi.storePositionWithTermVector);

      AbstractField f;
      int length = fieldsStream.readVIntUnlockedSafe();
//      long pointer = fieldsStream.getFilePointer();
      long pointer = fieldsStream.getFilePointer();

      //Skip ahead of where we are by the length of what is stored
      fieldsStream.seek(pointer+length);
      f = new LazyField(fi.name, store, index, termVector, length, pointer, binary, cacheResult);
      f.setOmitNorms(fi.omitNorms);
      f.setOmitTermFreqAndPositions(fi.omitTermFreqAndPositions);

      doc.add(f);
    }

  }

  private void addField(Document doc, FieldInfo fi, boolean binary, boolean tokenize) throws CorruptIndexException, IOException {

    if (binary) {
      int toRead = fieldsStream.readVIntUnlockedSafe();
      final byte[] b = new byte[toRead];
      fieldsStream.readBytesUnlockedSafe(b, 0, b.length);
      doc.add(new Field(fi.name, b));
    } else {
      Field.Store store = Field.Store.YES;
      Field.Index index = Field.Index.toIndex(fi.isIndexed, tokenize);
      Field.TermVector termVector = Field.TermVector.toTermVector(fi.storeTermVector, fi.storeOffsetWithTermVector, fi.storePositionWithTermVector);

      AbstractField f;
      f = new Field(fi.name,     // name
       false,
              fieldsStream.readStringUnlockedSafe(), // read value
              store,
              index,
              termVector);
      f.setOmitTermFreqAndPositions(fi.omitTermFreqAndPositions);
      f.setOmitNorms(fi.omitNorms);

      doc.add(f);
    }
  }

    private byte[] readField() throws CorruptIndexException, IOException {
        final int toRead = fieldsStream.readVIntUnlockedSafe();
        final byte[] b = new byte[toRead];
        fieldsStream.readBytesUnlockedSafe(b, 0, b.length);
        return b;
    }

  // Add the size of field as a byte[] containing the 4 bytes of the integer byte size (high order byte first; char = 2 bytes)
  // Read just the size -- caller must skip the field content to continue reading fields
  // Return the size in bytes or chars, depending on field type
  private int addFieldSize(Document doc, FieldInfo fi, boolean binary) throws IOException {
    int size = fieldsStream.readVIntUnlockedSafe(), bytesize = binary ? size : 2*size;
    byte[] sizebytes = new byte[4];
    sizebytes[0] = (byte) (bytesize>>>24);
    sizebytes[1] = (byte) (bytesize>>>16);
    sizebytes[2] = (byte) (bytesize>>> 8);
    sizebytes[3] = (byte)  bytesize      ;
    doc.add(new Field(fi.name, sizebytes));
    return size;
  }

  /**
   * A Lazy implementation of Fieldable that differs loading of fields until asked for, instead of when the Document is
   * loaded.
   */
  private class LazyField extends AbstractField implements Fieldable {
    private int toRead;
    private long pointer;
    private final boolean cacheResult;

    public LazyField(String name, Field.Store store, int toRead, long pointer, boolean isBinary, boolean cacheResult) {
      super(name, store, Field.Index.NO, Field.TermVector.NO);
      this.toRead = toRead;
      this.pointer = pointer;
      setIsBinary(isBinary);
      this.cacheResult = cacheResult;
      if (isBinary())
        binaryLength = toRead;
      setIsLazy(true);
    }

    public LazyField(String name, Field.Store store, Field.Index index, Field.TermVector termVector, int toRead, long pointer, boolean isBinary, boolean cacheResult) {
      super(name, store, index, termVector);
      this.toRead = toRead;
      this.pointer = pointer;
      setIsBinary(isBinary);
      this.cacheResult = cacheResult;
      if (isBinary())
        binaryLength = toRead;
      setIsLazy(true);
    }

    private IndexInput getFieldStream() {
      IndexInput localFieldsStream = fieldsStreamTL.get();
      if (localFieldsStream == null) {
        localFieldsStream = (IndexInput) cloneableFieldsStream.clone();
        fieldsStreamTL.set(localFieldsStream);
      }
      return localFieldsStream;
    }

    /** The value of the field as a Reader, or null.  If null, the String value,
     * binary value, or TokenStream value is used.  Exactly one of stringValue(), 
     * readerValue(), getBinaryValue(), and tokenStreamValue() must be set. */
    public Reader readerValue() {
      ensureOpen();
      return null;
    }

    /** The value of the field as a TokenStream, or null.  If null, the Reader value,
     * String value, or binary value is used. Exactly one of stringValue(), 
     * readerValue(), getBinaryValue(), and tokenStreamValue() must be set. */
    public TokenStream tokenStreamValue() {
      ensureOpen();
      return null;
    }

    /** The value of the field as a String, or null.  If null, the Reader value,
     * binary value, or TokenStream value is used.  Exactly one of stringValue(), 
     * readerValue(), getBinaryValue(), and tokenStreamValue() must be set. */
    public String stringValue() {
      ensureOpen();
      if (isBinary())
        return null;
      else {
        if (fieldsData == null) {
          String result = null;
          IndexInput localFieldsStream = getFieldStream();
          try {
            localFieldsStream.seek(pointer);
            byte[] bytes = new byte[toRead];
            localFieldsStream.readBytes(bytes, 0, toRead);
            result = new String(bytes, "UTF-8");
          } catch (IOException e) {
            throw new FieldReaderException(e);
          }
          if (cacheResult == true){
            fieldsData = result;
          }
          return result;
        } else {
          return (String) fieldsData;
        }
      }
    }

    @Override
    public byte[] getBinaryValue(byte[] result) {
      ensureOpen();

      if (isBinary()) {
        if (fieldsData == null) {
          // Allocate new buffer if result is null or too small
          final byte[] b;
          if (result == null || result.length < toRead)
            b = new byte[toRead];
          else
            b = result;
   
          IndexInput localFieldsStream = getFieldStream();

          // Throw this IOException since IndexReader.document does so anyway, so probably not that big of a change for people
          // since they are already handling this exception when getting the document
          try {
//            localFieldsStream.seek(pointer);
            localFieldsStream.seek(pointer);
            localFieldsStream.readBytes(b, 0, toRead);
          } catch (IOException e) {
            throw new FieldReaderException(e);
          }

          binaryOffset = 0;
          binaryLength = toRead;
          if (cacheResult == true){
            fieldsData = b;
          }
          return b;
        } else {
          return (byte[]) fieldsData;
        }
      } else
        return null;     
    }
  }
}
