package org.apache.lucene.index;

/**
 * Copyright 2004 The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.lucene.document.AbstractField;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldMapping;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.FieldVisitor;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.RAMInputStream;
import org.apache.lucene.store.RAMOutputStream;
import org.apache.lucene.store.DeflateOutputStream;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CloseableThreadLocal;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.IntsRef;
import org.apache.lucene.util.packed.*;

public class StandardFieldsWriter implements FieldsWriter {
    static final byte FIELD_IS_TOKENIZED = 0x1;
    static final byte FIELD_IS_BINARY = 0x2;

    private final static int FORMAT_SIZE = 4;

    // Lucene 3.0: Removal of compressed fields
    static final int FORMAT_LUCENE_3_0_NO_COMPRESSED_FIELDS = 2;

    // NOTE: if you introduce a new format, make it 1 higher
    // than the current one, and always change this if you
    // switch to a new format!
    public static final int FORMAT_CURRENT = FORMAT_LUCENE_3_0_NO_COMPRESSED_FIELDS;
    // when removing support for old versions, leave the last supported version here
    static final int FORMAT_MINIMUM = FORMAT_LUCENE_3_0_NO_COMPRESSED_FIELDS;

    // If null - we were supplied with streams, if notnull - we manage them ourselves
    private Directory directory;
    private String segment;
    private FieldInfos fieldInfos;
    private IndexOutput fieldsStream;
    private IndexOutput indexStream;
    private IndexInput fieldsStreamInput = null;
    private IndexInput indexStreamInput = null;
    private IndexInput rawDocsFieldsStreamInput = null;
    private IndexInput rawDocsIndexStreamInput = null;

    private String groupField = null;
    private int docCount = 0;
    private AtomicInteger refs = new AtomicInteger(0);
    private volatile int flushedDocCount = -1;
    private AtomicInteger indexedDocCount = new AtomicInteger(0);
    private CloseableThreadLocal<FieldsReaderLocal> threadLocalReader =
        new CloseableThreadLocal<FieldsReaderLocal>();

    private static class FieldsReaderLocal {
        private final IndexInput fieldsStream;
        private final IndexInput indexStream;
        private final StandardFieldsWriter writer;
        public FieldsReaderLocal(
            final IndexInput fieldsStream,
            final IndexInput indexStream,
            final StandardFieldsWriter writer)
        {
            this.fieldsStream = fieldsStream;
            this.indexStream = indexStream;
            this.writer = writer;
        }

        public final Document doc(final int n, final FieldSelector fieldSelector)
            throws CorruptIndexException, IOException
        {
            return writer.getDoc(n, fieldSelector, indexStream, fieldsStream);
        }

        public void readDocument(final int n, final FieldVisitor visitor)
            throws CorruptIndexException, IOException
        {
            writer.readDocument(n, visitor, indexStream, fieldsStream);
        }
    }

    public StandardFieldsWriter(
        final Directory directory,
        final String segment,
        final FieldInfos fn)
        throws IOException
    {
        this.directory = directory;
        this.segment = segment;
        fieldInfos = fn;

        boolean success = false;
        try {
            fieldsStream =
                directory.createOutput(
                    IndexFileNames.segmentFileName(
                        segment,
                        "",
                        IndexFileNames.FIELDS_EXTENSION));
             indexStream =
                directory.createOutput(
                    IndexFileNames.segmentFileName(
                        segment,
                        "",
                        IndexFileNames.FIELDS_INDEX_EXTENSION));

            fieldsStream.writeInt(FORMAT_CURRENT);
            indexStream.writeInt(FORMAT_CURRENT);
            success = true;
        } finally {
            if (!success) {
                abort();
            }
        }
    }

    StandardFieldsWriter(
        final IndexOutput fdx,
        final IndexOutput fdt,
        final FieldInfos fn)
    {
        directory = null;
        segment = null;
        fieldInfos = fn;
        fieldsStream = fdt;
        indexStream = fdx;
    }

    public static FieldsWriter getFieldsWriter(
        final Directory directory,
        final String segment,
        final FieldInfos fn)
        throws IOException
    {
        return new StandardFieldsWriter(
            directory,
            segment,
            fn);
    }

    @Override
    public FieldInfos getFieldInfos() {
        return this.fieldInfos;
    }

  void groupByField( String field )
  {
    groupField = field;
//    if( groupField != null )
//    {
//	groupFieldCache = new HashMap<BytesRef, IntsRef>();
//	System.err.println( "GROUP BY FIELD: " + field );
//    }
  }

  void setFieldsStream(IndexOutput stream) {
    this.fieldsStream = stream;
  }

  // Writes the contents of buffer into the fields stream
  // and adds a new entry for this document into the index
  // stream.  This assumes the buffer was already written
  // in the correct fields format.
  public void flushDocument(final int docId, int numStoredFields, RAMOutputStream buffer) throws IOException {
    if( groupField != null ) throw new IOException( "FieldsWriter.flushDocument: writing raw documents is not supported in grouping mode" );
    indexStream.writeLong(fieldsStream.getFilePointer());
    fieldsStream.writeVInt(numStoredFields);
    buffer.writeTo(fieldsStream);
    if (indexStream instanceof RAMOutputStream) {
        indexStream.flush();
    }
    if (fieldsStream instanceof RAMOutputStream) {
        fieldsStream.flush();
    }
    indexedDocCount.incrementAndGet();
//    synchronized(this) {
//        notifyAll();
///    }
  }

  public void skipDocument(final int docId) throws IOException {
    indexStream.writeLong(fieldsStream.getFilePointer());
    fieldsStream.writeVInt(0);
    if (indexStream instanceof RAMOutputStream) {
        indexStream.flush();
    }
    if (fieldsStream instanceof RAMOutputStream) {
        fieldsStream.flush();
    }
    indexedDocCount.incrementAndGet();
    synchronized(this) {
        notifyAll();
    }
  }

    @Override
    public void close() throws IOException {
        if (directory != null) {
            try {
                IOUtils.closeSafely(fieldsStream, indexStream);
            } finally {
                fieldsStream = indexStream = null;
            }
            threadLocalReader.close();
            if (indexStreamInput != null) {
                System.err.println("FieldsWriter.close() : Closing readers");
                IOUtils.closeSafely(fieldsStreamInput, indexStreamInput);
            }
            indexStreamInput = fieldsStreamInput = null;
            if (rawDocsIndexStreamInput != null) {
                IOUtils.closeSafely(rawDocsFieldsStreamInput,
                    rawDocsIndexStreamInput);
            }
            rawDocsIndexStreamInput = rawDocsFieldsStreamInput = null;
        }
    }

    @Override
    public void abort() {
        if (directory != null) {
            try {
                close();
            } catch (IOException ignored) {
                ignored.printStackTrace();
            }
            try {
                directory.deleteFile(
                    IndexFileNames.segmentFileName(
                        segment,
                        "",
                        IndexFileNames.FIELDS_EXTENSION));
            } catch (IOException ignored) {
            }
            try {
                directory.deleteFile(
                    IndexFileNames.segmentFileName(
                        segment,
                        "",
                        IndexFileNames.FIELDS_INDEX_EXTENSION));
            } catch (IOException ignored) {
            }
        }
    }

    @Override
    public void writeField(
        final FieldInfo fi,
        final Fieldable field)
        throws IOException
    {
        fieldsStream.writeVInt(fi.number);
        byte bits = 0;
        if (field.isTokenized()) {
          bits |= FIELD_IS_TOKENIZED;
        }
        if (field.isBinary()) {
            bits |= FIELD_IS_BINARY;
        }

        fieldsStream.writeByte(bits);

        if (field.isBinary()) {
            final byte[] data;
            final int len;
            final int offset;
            data = field.getBinaryValue();
            len = field.getBinaryLength();
            offset =  field.getBinaryOffset();

            fieldsStream.writeVInt(len);
            fieldsStream.writeBytes(data, offset, len);
        } else {
            fieldsStream.writeString(field.stringValue());
        }
    }

    @Override
    public void writeField(
        final FieldInfo otherFi,
        final byte bits,
        final IndexInput data,
        final int length)
        throws IOException
    {
        final int fieldNumber = fieldInfos.fieldNumber(otherFi.name);
        fieldsStream.writeVInt(fieldNumber);
        fieldsStream.writeByte(bits);
        fieldsStream.writeVInt(length);
        fieldsStream.copyBytesDirect(data, length);
    }

    @Override
    public void newDocument(final int docId, final int fieldCount) throws IOException {
        indexStream.writeLong(fieldsStream.getFilePointer());
        fieldsStream.writeVInt(fieldCount);
        docCount++;
    }

    @Override
    public void newDocumentNoIndex(final int docId, final int fieldCount) throws IOException {
        fieldsStream.writeVInt(fieldCount);
        docCount++;
    }

    @Override
    public long getDataStreamPosition() {
        return fieldsStream.getFilePointer();
    }

    public void writeIndex(final PackedInts.Mutable srcIndex)
        throws IOException
    {
        final int docs = srcIndex.size();
        PackedInts.Writer writer =
            PackedInts.getWriter(
                indexStream,
                docs,
                srcIndex.getBitsPerValue());
        for (int i = 0; i < docs; i++) {
            final long off = srcIndex.get(i);
            writer.add(off);
        }
        writer.finish();
    }

    /** Bulk write a contiguous series of documents.  The
     *  lengths array is the length (in bytes) of each raw
     *  document.  The stream IndexInput is the
     *  fieldsStream from which we should bulk-copy all
     *  bytes. */
    @Override
    public void addRawDocuments(
        final IndexInput stream,
        final int[] lengths,
        final int numDocs)
        throws IOException
    {
        long position = fieldsStream.getFilePointer();
        long start = position;
        for(int i=0;i<numDocs;i++) {
            indexStream.writeLong(position);
            position += lengths[i];
        }
        fieldsStream.copyBytesDirect(stream, position-start);
        assert fieldsStream.getFilePointer() == position;
        docCount += numDocs;
    }

    @Override
    public void addDocument(final int docId, final Document doc) throws IOException {
        indexStream.writeLong(fieldsStream.getFilePointer());
        addDocumentNoIndex(docId, doc);
    }

    @Override
    public void addDocumentNoIndex(final int docId, final Document doc) throws IOException {
        int storedCount = 0;
        List<Fieldable> fields = doc.getFields();
        for (Fieldable field : fields) {
            if (field.isStored()) {
                storedCount++;
            }
        }
        fieldsStream.writeVInt(storedCount);

        for (Fieldable field : fields) {
            if (field.isStored()) {
                writeField(fieldInfos.fieldInfo(field.name()), field);
            }
        }
        docCount++;
    }

    private void seekIndex(final int docID, final IndexInput indexStream)
        throws IOException
    {
        indexStream.seek(FORMAT_SIZE + docID * 8L);
    }

  public void incRef() {
    refs.incrementAndGet();
  }

  public void decRef() throws IOException {
    int r = refs.decrementAndGet();
    if (r == 0) {
        synchronized(this) {
            if (indexStreamInput != null) {
                IOUtils.closeSafely(fieldsStreamInput, indexStreamInput);
            }
            indexStreamInput = fieldsStreamInput = null;
        }
    }
  }

  public synchronized final IndexInput rawDocs(int[] lengths, int startDocID, int numDocs) throws IOException {
    synchronized(this) {
        if (rawDocsIndexStreamInput == null) {
            indexStream.flush();
            fieldsStream.flush();
            rawDocsFieldsStreamInput = directory.openInput(IndexFileNames.segmentFileName(segment, "", IndexFileNames.FIELDS_EXTENSION));
            rawDocsIndexStreamInput = directory.openInput(IndexFileNames.segmentFileName(segment, "", IndexFileNames.FIELDS_INDEX_EXTENSION));
        }
    }
    rawDocsIndexStreamInput.seek(FORMAT_SIZE + startDocID * 8L);
    long startOffset = rawDocsIndexStreamInput.readLong();
    long lastOffset = startOffset;
    int count = 0;
    final int numTotalDocs = indexedDocCount.get();
    while (count < numDocs) {
        final long offset;
        final int docID = startDocID + count + 1;
        assert docID <= numTotalDocs;
        if (docID < numTotalDocs) 
            offset = rawDocsIndexStreamInput.readLong();
        else
            offset = rawDocsFieldsStreamInput.length();
        lengths[count++] = (int) (offset-lastOffset);
        lastOffset = offset;
    }
    rawDocsFieldsStreamInput.seek(startOffset);
    return rawDocsFieldsStreamInput;
  }

  private synchronized boolean flushOutputStreams(int n) throws IOException {
    while (n >= indexedDocCount.get()) {
        try {
            wait(10);
        } catch (InterruptedException e) {
            throw new IOException("Document wait interrupted", e);
        }
    }
    if (!(indexStream instanceof RAMOutputStream)) {
        if (n > flushedDocCount) {
            indexStream.flush();
            fieldsStream.flush();
            flushedDocCount = n;
            return true;
        }
    }
    return false;
  }

    private void checkInputStreams() throws IOException {
        if (fieldsStreamInput == null) {
            synchronized(this) {
                if (fieldsStreamInput == null) {
                    fieldsStreamInput = directory.openInput(
                        IndexFileNames.segmentFileName(
                            segment, "", IndexFileNames.FIELDS_EXTENSION));
                }
            }
        }
        if (indexStreamInput == null) {
            synchronized(this) {
                if (indexStreamInput == null) {
                    indexStreamInput = directory.openInput(
                        IndexFileNames.segmentFileName(
                            segment, "", IndexFileNames.FIELDS_INDEX_EXTENSION));
                }
            }
        }
    }

    private final FieldsReaderLocal getFieldsReaderLocal()
        throws IOException
    {
        FieldsReaderLocal reader = threadLocalReader.get();
        if (reader == null) {
            checkInputStreams();
            reader = new FieldsReaderLocal(
                (IndexInput)fieldsStreamInput.clone(),
                (IndexInput)indexStreamInput.clone(),
                this);
            threadLocalReader.set(reader);
        }
        return reader;
    }

  public final Document doc(int n, FieldSelector fieldSelector) throws CorruptIndexException, IOException {
    try {
        FieldsReaderLocal localReader = getFieldsReaderLocal();
        return localReader.doc(n, fieldSelector);
    } catch (IOException e) {
        throw new IOException("FieldsWriter error. DocCount=" + indexedDocCount.get()
            + ", n=" + n
            + ", flushedDocCount=" + flushedDocCount
            + ", indexWritePos=" + fieldsStream.getFilePointer()
            + ", length=" + fieldsStream.length()
            + ", fileLength=" + ((RAMInputStream)fieldsStreamInput).file().getLength(), e);
//            + ", fileLength=" + ((NativeRAMInputStream)fieldsStreamInput).file().getLength(), e);
    }
  }

    @Override
    public void readDocument(final int n, final FieldVisitor visitor)
        throws CorruptIndexException, IOException
    {
        try {
            FieldsReaderLocal localReader = getFieldsReaderLocal();
            localReader.readDocument(n, visitor);
        } catch (IOException e) {
            throw new IOException(
                "FieldsWriter error. DocCount="
                + indexedDocCount.get()
                + ", n=" + n
                + ", flushedDocCount=" + flushedDocCount
                + ", indexWritePos=" + fieldsStream.getFilePointer()
                + ", length=" + fieldsStream.length()
//                + ", fileLength=" + ((NativeRAMInputStream)fieldsStreamInput)
                + ", fileLength=" + ((RAMInputStream)fieldsStreamInput)
                    .file().getLength(),
                e);
        }
    }

  public final Document getDoc(
    final int n,
    final FieldSelector fieldSelector,
    final IndexInput indexStream,
    final IndexInput fieldsStream)
    throws CorruptIndexException, IOException
  {
    boolean flushWas = flushOutputStreams(n);
    try {
        seekIndex(n, indexStream);
    } catch (IOException e) {
        throw new IOException("FieldsWriter seekIndexError. DocCount=" + indexedDocCount.get()
            + ", n=" + n
            + ", flushedDocCount=" + flushedDocCount
            + ", indexWritePos=" + indexStream.getFilePointer()
            + ", flushWas=" + flushWas
            + ", length=" + indexStream.length()
            + ", fileLength=" + ((RAMInputStream)indexStreamInput).file().getLength(), e);
//            + ", fileLength=" + ((NativeRAMInputStream)indexStreamInput).file().getLength(), e);
    }
    long position = indexStream.readLong();
    fieldsStream.seek(position);

    Document doc = new Document();
    int numFields = fieldsStream.readVInt();
//    System.err.println( this + "Doc: " + n + " Pos : " + position + " : numfields: " + numFields );
    for (int i = 0; i < numFields; i++) {
      int fieldNumber = fieldsStream.readVInt();
      FieldInfo fi = fieldInfos.fieldInfoSynchronized(fieldNumber);
      FieldSelectorResult acceptField = fieldSelector == null ? FieldSelectorResult.LOAD : fieldSelector.accept(fi.name);
      
      byte bits = fieldsStream.readByte();
      assert bits <= FIELD_IS_TOKENIZED + FIELD_IS_BINARY;

      boolean tokenize = (bits & FIELD_IS_TOKENIZED) != 0;
      boolean binary = (bits & FIELD_IS_BINARY) != 0;
      //TODO: Find an alternative approach here if this list continues to grow beyond the
      //list of 5 or 6 currently here.  See Lucene 762 for discussion
      if (acceptField.equals(FieldSelectorResult.LOAD)) {
        addField(doc, fi, binary, tokenize, fieldsStream);
      }
      else if (acceptField.equals(FieldSelectorResult.LOAD_AND_BREAK)){
        addField(doc, fi, binary, tokenize, fieldsStream);
        break;//Get out of this loop
      }
      else if (acceptField.equals(FieldSelectorResult.LAZY_LOAD)) {
        addField(doc, fi, binary, tokenize, fieldsStream);
      }
      else if (acceptField.equals(FieldSelectorResult.LATENT)) {
        addField(doc, fi, binary, tokenize, fieldsStream);
      }
      else if (acceptField.equals(FieldSelectorResult.SIZE)){
        skipField(addFieldSize(doc, fi, binary, fieldsStream), fieldsStream);
      }
      else if (acceptField.equals(FieldSelectorResult.SIZE_AND_BREAK)){
        addFieldSize(doc, fi, binary, fieldsStream);
        break;
      }
      else {
        skipField(fieldsStream);
      }
    }

    return doc;
  }

    public void readDocument(
        final int n,
        final FieldVisitor visitor,
        final IndexInput indexStream,
        final IndexInput fieldsStream)
        throws CorruptIndexException, IOException
    {
        boolean flushWas = flushOutputStreams(n);
        try {
            seekIndex(n, indexStream);
        } catch (IOException e) {
            throw new IOException(
                "FieldsWriter seekIndexError. DocCount=" + indexedDocCount.get()
                    + ", n=" + n
                    + ", flushedDocCount=" + flushedDocCount
                    + ", indexWritePos=" + indexStream.getFilePointer()
                    + ", flushWas=" + flushWas
                    + ", length=" + indexStream.length()
                    + ", fileLength="
                    + ((RAMInputStream) indexStreamInput)
//                    + ((NativeRAMInputStream)indexStreamInput)
                        .file().getLength(),
                e);
        }
        final long position = indexStream.readLong();
        fieldsStream.seek(position);

        final int numFields = fieldsStream.readVInt();
        FieldMapping mapping = visitor.fieldMapping(fieldInfos);
        FieldSelectorResult[] fieldSelectorResults =
            mapping.fieldSelectorResults();
        int[] fieldNumberToIndex = mapping.fieldNumberToIndex();
        int fieldsLeft = mapping.maxFieldCount();
        for (int i = 0; i < numFields; ++i) {
            final int fieldNumber = fieldsStream.readVInt();
            //bits are ignore here
            byte bits = fieldsStream.readByte();

            switch (fieldSelectorResults[fieldNumber]) {
                case LOAD:
                case LOAD_AND_BREAK:
                    visitor.storeFieldValue(
                        fieldNumberToIndex[fieldNumber],
                        readField(fieldsStream));
                    if (--fieldsLeft == 0) {
                        return;
                    }
                    break;
                case SIZE:
                case SIZE_AND_BREAK:
                    visitor.storeFieldSize(
                        fieldNumberToIndex[fieldNumber],
                        skipField(fieldsStream));
                    if (--fieldsLeft == 0) {
                        return;
                    }
                    break;
                default:
                    skipField(fieldsStream);
                    break;
            }
        }
    }

  private int skipField(final IndexInput fieldsStream) throws IOException {
    return skipField(fieldsStream.readVInt(), fieldsStream);
  }

  private int skipField(final int toRead, final IndexInput fieldsStream)
    throws IOException
  {
    fieldsStream.seek(fieldsStream.getFilePointer() + toRead);
    return toRead;
  }

  private int addFieldSize(
    final Document doc,
    final FieldInfo fi,
    final boolean binary,
    final IndexInput fieldsStream) throws IOException {
    int size = fieldsStream.readVInt(), bytesize = binary ? size : 2*size;
    byte[] sizebytes = new byte[4];
    sizebytes[0] = (byte) (bytesize>>>24);
    sizebytes[1] = (byte) (bytesize>>>16);
    sizebytes[2] = (byte) (bytesize>>> 8);
    sizebytes[3] = (byte)  bytesize      ;
    doc.add(new Field(fi.name, sizebytes));
    return size;
  }

  private void addField(
    final Document doc,
    final FieldInfo fi,
    final boolean binary,
    final boolean tokenize,
    final IndexInput fieldsStream) throws CorruptIndexException, IOException {

    if (binary) {
      int toRead = fieldsStream.readVInt();
      final byte[] b = new byte[toRead];
      fieldsStream.readBytes(b, 0, b.length);
      doc.add(new Field(fi.name, b));
    } else {
      Field.Store store = Field.Store.YES;
      Field.Index index = Field.Index.toIndex(fi.isIndexed, tokenize);
      Field.TermVector termVector = Field.TermVector.toTermVector(
        fi.storeTermVector,
        fi.storeOffsetWithTermVector,
        fi.storePositionWithTermVector);

      AbstractField f;
      f = new Field(fi.name,     // name
       false,
              fieldsStream.readString(), // read value
              store,
              index,
              termVector);
      f.setOmitTermFreqAndPositions(fi.omitTermFreqAndPositions);
      f.setOmitNorms(fi.omitNorms);

      doc.add(f);
    }
  }

    private byte[] readField(final IndexInput fieldsStream)
        throws CorruptIndexException, IOException
    {
        int toRead = fieldsStream.readVInt();
        final byte[] b = new byte[toRead];
        fieldsStream.readBytes(b, 0, b.length);
        return b;
    }

    private boolean scanFields(
        final String groupField,
        final HashMap<BytesRef, IntsRef> groupFieldCache,
        final IndexInput idxStream,
        final IndexInput fldStream, final int docCount)
            throws IOException
    {
        if (groupField == null) {
            System.err.println("FieldsWriter.scanFields: groupField=null");
            return false;
        }
        int field = fieldInfos.fieldNumber(groupField);
        if (field == -1) {
            System.err.println("FieldsWriter.scanFields: groupFieldNumber=-1");
            return false;
        }
        BytesRef groupValue = new BytesRef(20);
        for (int i = 0; i < docCount; i++) {
            getDocGroupValue(groupValue, i, idxStream, fldStream, field);

	    IntsRef docs = groupFieldCache.get(groupValue);
            if (docs == null) {
    	        docs = new IntsRef(10);
    	        groupFieldCache.put((BytesRef)groupValue.clone(), docs);
            }
            if (docs.length == docs.ints.length) {
                docs.grow(docs.length * 2);
            }
            docs.ints[docs.length++] = i;
//            docs.add(i);
        }
        return true;
    }

    private void getDocGroupValue(final BytesRef value, final int docId,
        final IndexInput idxStream, final IndexInput fldStream, final int field)
            throws IOException
    {
        idxStream.seek(FORMAT_SIZE + docId * 8L);
	long position = idxStream.readLong();
	fldStream.seek(position);
//        System.err.println("getDocGroupValue: docId=" + docId + ", position=" + position + ", fdt.size=" + fldStream.length());
        int numFields = fldStream.readVInt();
        for (int i = 0; i < numFields; i++) {
            int fieldNumber = fldStream.readVInt();
            byte bits = fldStream.readByte();

            boolean binary = (bits & FIELD_IS_BINARY) != 0;

            if (fieldNumber != field || binary) {
                skipField(fldStream);
            } else {
                getFieldValue(fldStream, value);
                break;
            }
        }
//        return fieldValue;
    }

    private void getFieldValue(final IndexInput fldStream,
        final BytesRef value)
            throws IOException
    {
        int length = fldStream.readVInt();
        if (value.bytes.length < length) {
            value.bytes = new byte[length * 2];
        }
        value.length = length;
        fldStream.readBytes(value.bytes, 0, length);
//        return fldStream.readString();
    }
}
