package org.apache.lucene.index;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import java.util.Comparator;
import java.util.concurrent.atomic.LongAdder;

import org.apache.lucene.analysis.tokenattributes.TermToBytesRefAttribute;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.util.ByteBlockPool;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefHash;
import org.apache.lucene.util.BytesRefHash.BytesStartArray;
import org.apache.lucene.util.BytesRefHash.MaxBytesLengthExceededException;

import ru.yandex.collection.ChunkedIntList;

final class TermsHashPerField extends InvertedDocConsumerPerField {
  private static final int HASH_INIT_SIZE = 32;
  private static final boolean DEBUG = false;

  final TermsHashConsumerPerField consumer;

  final TermsHashPerField nextPerField;
  final TermsHashPerThread perThread;
  final DocumentsWriter.DocState docState;
  final FieldInvertState fieldState;
  TermToBytesRefAttribute termAtt;

  // Copied from our perThread
  final IntBlockPool intPool;
  final ByteBlockPool bytePool;
  final ByteBlockPool termBytePool;

  final int streamCount;
  final int numPostingInt;

  final FieldInfo fieldInfo;

  final BytesRefHash bytesHash;

  ParallelPostingsArray postingsArray;
  private final BytesRef termBytesRef;

  public TermsHashPerField(DocInverterPerField docInverterPerField, final TermsHashPerThread perThread, final TermsHashPerThread nextPerThread, final FieldInfo fieldInfo) {
    this.perThread = perThread;
    intPool = perThread.intPool;
    bytePool = perThread.bytePool;
    termBytePool = perThread.termBytePool;
    docState = perThread.docState;

    fieldState = docInverterPerField.fieldState;
    this.consumer = perThread.consumer.addField(this, fieldInfo);
    BytesStartArray byteStarts;
    if (perThread.termsHash.trackAllocations) {
        byteStarts = new TrackingPostingsBytesStartArray(this, perThread.termsHash.docWriter.bytesUsed);
    } else {
        byteStarts = new PostingsBytesStartArray(this);
    }
    bytesHash = new BytesRefHash(termBytePool, HASH_INIT_SIZE, byteStarts);
    streamCount = consumer.getStreamCount();
    numPostingInt = streamCount << 1;
    termBytesRef = perThread.termBytesRef;
    this.fieldInfo = fieldInfo;
    if (nextPerThread != null)
      nextPerField = (TermsHashPerField) nextPerThread.addField(docInverterPerField, fieldInfo);
    else
      nextPerField = null;
  }

  void shrinkHash(int targetSize) {
    // Fully free the bytesHash on each flush but keep the pool untouched
    // bytesHash.clear will clear the ByteStartArray and in turn the ParallelPostingsArray too
    bytesHash.clear(false);
  }

  public void reset() {
    bytesHash.clear(false);
    if (nextPerField != null)
      nextPerField.reset();
  }

  @Override
  synchronized public void abort() {
    reset();
    if (nextPerField != null)
      nextPerField.abort();
  }

  @Override
  public void close() {
    bytesHash.close();
    if (nextPerField != null) {
      nextPerField.close();
    }
  }

  public void initReader(ByteSliceReader reader, int termID, int stream) {
    assert stream < streamCount;
    int intStart = postingsArray.intStarts.getInt(termID);
    final int[] ints = intPool.buffers[intStart >> DocumentsWriter.INT_BLOCK_SHIFT];
    final int upto = intStart & DocumentsWriter.INT_BLOCK_MASK;
    reader.init(bytePool,
                postingsArray.byteStarts.getInt(termID)+stream*ByteBlockPool.FIRST_LEVEL_SIZE,
                ints[upto+stream]);
  }


  /** Collapse the hash table & sort in-place. */
  public int[] sortPostings(Comparator<BytesRef> termComp) {
   return bytesHash.sort(termComp);
  }

  public int[][] sortPostingsExternal(Comparator<BytesRef> termComp) {
    return bytesHash.getSorted(termComp);
  }

  public int findTerm(final BytesRef term, final BytesRef scratch) {
    return bytesHash.find(term, scratch);
  }

  private boolean doCall;
  private boolean doNextCall;

  @Override
  void start(Fieldable f) {
    termAtt = fieldState.attributeSource.getAttribute(TermToBytesRefAttribute.class);
    consumer.start(f);
    if (nextPerField != null) {
      nextPerField.start(f);
    }
  }
  
  @Override
  boolean start(Fieldable[] fields, int count) throws IOException {
    doCall = consumer.start(fields, count);
    bytesHash.reinit();
    if (nextPerField != null)
      doNextCall = nextPerField.start(fields, count);
    return doCall || doNextCall;
  }

  // Secondary entry point (for 2nd & subsequent TermsHash),
  // because token text has already been "interned" into
  // textStart, so we hash by textStart
  public void add(int textStart) throws IOException {
    int termID = bytesHash.addByPoolOffset(textStart);
    if (termID >= 0) {      // New posting
      // First time we are seeing this token since we last
      // flushed the hash.
      // Init stream slices
      if (numPostingInt + intPool.intUpto > DocumentsWriter.INT_BLOCK_SIZE)
        intPool.nextBuffer();

      if (ByteBlockPool.BYTE_BLOCK_SIZE - bytePool.byteUpto < numPostingInt*ByteBlockPool.FIRST_LEVEL_SIZE)
        bytePool.nextBuffer();

      intUptos = intPool.buffer;
      intUptoStart = intPool.intUpto;
      intPool.intUpto += streamCount;

      postingsArray.intStarts.setInt(termID, intUptoStart + intPool.intOffset);

      for(int i=0;i<streamCount;i++) {
        final int upto = bytePool.newSlice(ByteBlockPool.FIRST_LEVEL_SIZE);
        intUptos[intUptoStart+i] = upto + bytePool.byteOffset;
      }
      postingsArray.byteStarts.setInt(termID, intUptos[intUptoStart]);

      consumer.newTerm(termID);

    } else {
      termID = (-termID)-1;
      int intStart = postingsArray.intStarts.getInt(termID);
      intUptos = intPool.buffers[intStart >> DocumentsWriter.INT_BLOCK_SHIFT];
      intUptoStart = intStart & DocumentsWriter.INT_BLOCK_MASK;
      consumer.addTerm(termID);
    }
  }

  // Primary entry point (for first TermsHash)
  @Override
  void add() throws IOException {

    // We are first in the chain so we must "intern" the
    // term text into textStart address
    // Get the text & hash of this term.
    int termID;
    try{
       termID = bytesHash.add(termBytesRef, termAtt.toBytesRef(termBytesRef));
    }catch (MaxBytesLengthExceededException e) {
      // Not enough room in current block
      // Just skip this term, to remain as robust as
      // possible during indexing.  A TokenFilter
      // can be inserted into the analyzer chain if
      // other behavior is wanted (pruning the term
      // to a prefix, throwing an exception, etc).
      if (docState.maxTermPrefix == null) {
        final int saved = termBytesRef.length;
        try {
          termBytesRef.length = Math.min(30, DocumentsWriter.MAX_TERM_LENGTH_UTF8);
          docState.maxTermPrefix = termBytesRef.toString();
        } finally {
          termBytesRef.length = saved;
        }
      }
      consumer.skippingLongTerm();
      return;
    }
    if (termID >= 0) {// New posting
      bytesHash.byteStart(termID);
      // Init stream slices
      if (numPostingInt + intPool.intUpto > DocumentsWriter.INT_BLOCK_SIZE) {
        intPool.nextBuffer();
      }

      if (ByteBlockPool.BYTE_BLOCK_SIZE - bytePool.byteUpto < numPostingInt*ByteBlockPool.FIRST_LEVEL_SIZE) {
        bytePool.nextBuffer();
      }

      intUptos = intPool.buffer;
      intUptoStart = intPool.intUpto;
      intPool.intUpto += streamCount;

      postingsArray.intStarts.setInt(termID, intUptoStart + intPool.intOffset);

      for(int i=0;i<streamCount;i++) {
        final int upto = bytePool.newSlice(ByteBlockPool.FIRST_LEVEL_SIZE);
        intUptos[intUptoStart+i] = upto + bytePool.byteOffset;
      }
      postingsArray.byteStarts.setInt(termID, intUptos[intUptoStart]);

      consumer.newTerm(termID);

    } else {
      termID = (-termID)-1;
      final int intStart = postingsArray.intStarts.getInt(termID);
      intUptos = intPool.buffers[intStart >> DocumentsWriter.INT_BLOCK_SHIFT];
      intUptoStart = intStart & DocumentsWriter.INT_BLOCK_MASK;
      consumer.addTerm(termID);
    }

    if (doNextCall)
      nextPerField.add(postingsArray.textStarts.getInt(termID));
  }

  int[] intUptos;
  int intUptoStart;

  void writeByte(int stream, byte b) {
    int upto = intUptos[intUptoStart+stream];
    byte[] bytes = bytePool.buffers[upto >> ByteBlockPool.BYTE_BLOCK_SHIFT];
    assert bytes != null;
    int offset = upto & ByteBlockPool.BYTE_BLOCK_MASK;
    if (bytes[offset] != 0) {
      if (DEBUG) System.err.println("thpf.writeByte(" + stream + "," + b +"): slice[upto]=" + bytes[offset] + ", slice=" + bytes);
      // End of slice; allocate a new one
      offset = bytePool.allocSlice(bytes, offset + 3);
      bytes = bytePool.buffer;
      intUptos[intUptoStart+stream] = offset + bytePool.byteOffset;
      if (DEBUG) System.err.println("thpf.writeByte(" + stream + "," + b +"): new slice upto=" + offset + ", slice=" + bytes);
    }
    bytes[offset] = b;
    if (DEBUG) System.err.println("BytesSliceWriter.writeByte(" + stream + "," + b +"): upto=" + offset + ", slice=" + bytes);
    (intUptos[intUptoStart+stream])++;
  }

  public void writeBytes(int stream, byte[] b, int offset, int len) {
    // TODO: optimize
    final int end = offset + len;
    for(int i=offset;i<end;i++)
      writeByte(stream, b[i]);
  }

  void writeVInt(int stream, int i) {
    assert stream < streamCount;
    while ((i & ~0x7F) != 0) {
      writeByte(stream, (byte)((i & 0x7f) | 0x80));
      i >>>= 7;
    }
    writeByte(stream, (byte) i);
  }

  @Override
  void finish() throws IOException {
    bytesHash.finishDoc();
    consumer.finish();
    if (nextPerField != null) {
        bytesHash.finishDoc();
        nextPerField.finish();
    }
  }

  private static final class PostingsBytesStartArray extends BytesStartArray {
      private final TermsHashPerField perField;

      private PostingsBytesStartArray(final TermsHashPerField perField) {
          this.perField = perField;
      }

      @Override
      public ChunkedIntList init() {
          if (perField.postingsArray == null) {
              perField.postingsArray = perField.consumer.createPostingsArray(2);
          }
          return perField.postingsArray.textStarts;
      }

      @Override
      public ChunkedIntList grow() {
          perField.postingsArray.grow();
          return perField.postingsArray.textStarts;
      }

      @Override
      public ChunkedIntList clear() {
          perField.postingsArray = null;
          return null;
      }

      @Override
      public LongAdder bytesUsed() {
          return BytesRefHash.FAKE_LONG_ADDER;
      }
  }

  private static final class TrackingPostingsBytesStartArray extends BytesStartArray {
      private final TermsHashPerField perField;
      private final LongAdder bytesUsed;

      TrackingPostingsBytesStartArray(
          final TermsHashPerField perField,
          final LongAdder bytesUsed)
      {
          this.perField = perField;
          this.bytesUsed = bytesUsed;
      }

      @Override
      public ChunkedIntList init() {
          if (perField.postingsArray == null) {
              perField.postingsArray = perField.consumer.createPostingsArray(2);
              int size = perField.postingsArray.size;
              int postingSize = perField.postingsArray.bytesPerPosting();
              bytesUsed.add(size * postingSize);
          }
          return perField.postingsArray.textStarts;
      }

      @Override
      public ChunkedIntList grow() {
          ParallelPostingsArray postingsArray = perField.postingsArray;
          int oldSize = postingsArray.size;
          postingsArray.grow();
          int size = postingsArray.size;
          int postingSize = postingsArray.bytesPerPosting();
          bytesUsed.add((size - oldSize) * postingSize);
          return postingsArray.textStarts;
      }

      @Override
      public ChunkedIntList clear() {
          if (perField.postingsArray != null) {
              int size = perField.postingsArray.size;
              int postingSize = perField.postingsArray.bytesPerPosting();
              bytesUsed.add(-size * postingSize);
              perField.postingsArray = null;
          }
          return null;
      }

      @Override
      public LongAdder bytesUsed() {
          return bytesUsed;
      }
  }
}
