package org.apache.lucene.index.codecs;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.util.BytesRef;

/**
 * @lucene.experimental
 */

public abstract class PostingsConsumer {

  /** Adds a new doc in this term.  If this field omits term
   *  freqs & positions then termDocFreq should be ignored,
   *  and, finishDoc will not be called. */
  public abstract void startDoc(int docID, int termDocFreq) throws IOException;

  public static class PostingsMergeState {
    DocsEnum docsEnum;
    int[] docMap;
    int docBase;
  }

  /** Add a new position & payload.  A null payload means no
   *  payload; a non-null payload with zero length also
   *  means no payload.  Caller may reuse the {@link
   *  BytesRef} for the payload between calls (method must
   *  fully consume the payload). */
  public abstract void addPosition(int position, BytesRef payload) throws IOException;

  /** Called when we are done adding positions & payloads
   *  for each doc.  Not called  when the field omits term
   *  freq and positions. */
  public abstract void finishDoc() throws IOException;

  /** Default merge impl: append documents, mapping around
   *  deletes */
  public void merge(final MergeState mergeState, final DocsEnum postings, final TermStats stats) throws IOException {

    int df = 0;
    long totTF = 0;

    if (mergeState.fieldInfo.omitTermFreqAndPositions) {
      while(true) {
        final int doc = postings.nextDoc();
        if (doc == DocIdSetIterator.NO_MORE_DOCS) {
          break;
        }
        this.startDoc(doc, postings.freq());
        this.finishDoc();
        df++;
        totTF++;
      }
    } else {
      final DocsAndPositionsEnum postingsEnum = (DocsAndPositionsEnum) postings;
      while(true) {
        final int doc = postingsEnum.nextDoc();
        if (doc == DocIdSetIterator.NO_MORE_DOCS) {
          break;
        }
        final int freq = postingsEnum.freq();
        this.startDoc(doc, freq);
        totTF += freq;
        for(int i=0;i<freq;i++) {
          final int position = postingsEnum.nextPosition();
          final BytesRef payload;
          if (postingsEnum.hasPayload()) {
            payload = postingsEnum.getPayload();
          } else {
            payload = null;
          }
          this.addPosition(position, payload);
        }
        this.finishDoc();
        df++;
      }
    }
    stats.docFreq = df;
    stats.totalTermFreq = totTF;
//    return new TermStats(df, totTF);
  }
}
