package org.apache.lucene.index.codecs;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.SegmentInfo;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CodecUtil;
import org.apache.lucene.util.automaton.fst.Builder;
import org.apache.lucene.util.automaton.fst.BytesRefFSTEnum;
import org.apache.lucene.util.automaton.fst.FST;
import org.apache.lucene.util.automaton.fst.PositiveIntOutputs;
//import org.apache.lucene.util.automaton.fst_native.Builder;
//import org.apache.lucene.util.automaton.fst_native.BytesRefFSTEnum;
//import org.apache.lucene.util.automaton.fst_native.FST;
//import org.apache.lucene.util.automaton.fst_native.PositiveIntOutputs;

/** See {@link VariableGapTermsIndexWriter}
 * 
 * @lucene.experimental */
public class VariableGapTermsIndexReader extends TermsIndexReaderBase {

  private final PositiveIntOutputs fstOutputs = PositiveIntOutputs.getSingleton(true);
  private final String indexFileName;
  private final Directory dir;
  private int indexDivisor;
  final Map<String, Integer> indexDivisors;
  final Set<String> indexedFields;

  // Closed if indexLoaded is true:
  private IndexInput in;
  private volatile boolean indexLoaded;

  final Map<FieldInfo, FieldIndexData> fields;
  
  // start of the field info data
  protected long dirOffset;

  public VariableGapTermsIndexReader(Directory dir, FieldInfos fieldInfos, String segment, int indexDivisor, String codecId)
    throws IOException {
    this(dir, fieldInfos, segment, indexDivisor, codecId, null, Collections.<String, Integer>emptyMap());
  }

  public VariableGapTermsIndexReader(Directory dir, FieldInfos fieldInfos, String segment, int indexDivisor, String codecId, final Set<String> indexedFields, final Map<String, Integer> indexDivisors)
    throws IOException {

    this.dir = dir;

    indexFileName =
        IndexFileNames.segmentFileName(
            segment,
            codecId,
            "");
    final String indexFileExtension = VariableGapTermsIndexWriter.TERMS_INDEX_EXTENSION;
    in = dir.openInput(indexFileName + "." + indexFileExtension);
    this.indexedFields = indexedFields;
    this.indexDivisors = indexDivisors;
    boolean success = false;

    try {
      
      readHeader(in);
      this.indexDivisor = indexDivisor;

      seekDir(in, dirOffset);

      // Read directory
      final int numFields = in.readVInt();
      fields = new IdentityHashMap<>(numFields << 1);

      for(int i=0;i<numFields;i++) {
        final int field = in.readVInt();
        final long indexStart = in.readVLong();
        final FieldInfo fieldInfo = fieldInfos.fieldInfo(field);
        if (indexedFields == null || indexedFields.contains(fieldInfo.name)) {
            fields.put(fieldInfo, new FieldIndexData(fieldInfo, indexStart));
        } else {
            System.err.println("VariableGapTermsIndexReader: skipping field: "
                + fieldInfo.name);
        }
      }
      success = true;
    } finally {
      if (indexDivisor > 0) {
        in.close();
        in = null;
        if (success) {
          indexLoaded = true;
        }
      }
    }
  }

  @Override
  public int getDivisor() {
    return indexDivisor;
  }

  public int getDivisor(final FieldInfo info) {
    int indexDivisor = indexDivisor(info.name);
    return indexDivisor;
  }

  protected void readHeader(IndexInput input) throws IOException {
    CodecUtil.checkHeader(input, VariableGapTermsIndexWriter.CODEC_NAME,
      VariableGapTermsIndexWriter.VERSION_START, VariableGapTermsIndexWriter.VERSION_START);
    dirOffset = input.readLong();
  }

  private static class IndexEnum extends FieldIndexEnum {
    private final BytesRefFSTEnum<Long> fstEnum;
    private BytesRefFSTEnum.InputOutput<Long> current;

    public IndexEnum(FST<Long> fst) {
      fstEnum = new BytesRefFSTEnum<Long>(fst);
    }

    @Override
    public BytesRef term() {
      if (current == null) {
        return null;
      } else {
        return current.input;
      }
    }

    @Override
    public long seek(BytesRef target) throws IOException {
      //System.out.println("VGR: seek field=" + fieldInfo.name + " target=" + target);
      current = fstEnum.seekFloor(target);
      //System.out.println("  got input=" + current.input + " output=" + current.output);
      return current.output;
    }

    public long seekCeil(final BytesRef target) throws IOException {
        current = fstEnum.seekCeil(target);
        return current.output;
    }

    @Override
    public long prev() throws IOException {
        current = fstEnum.prev();
        if (current == null) {
            return -1;
        } else {
            return current.output;
        }
    }

    @Override
    public void seekEnd() {
        fstEnum.seekEnd();
    }

    @Override
    public long next() throws IOException {
      //System.out.println("VGR: next field=" + fieldInfo.name);
      current = fstEnum.next();
      if (current == null) {
        //System.out.println("  eof");
        return -1;
      } else {
        return current.output;
      }
    }

    @Override
    public long ord() {
      throw new UnsupportedOperationException();
    }

    @Override
    public long seek(long ord) {
      throw new UnsupportedOperationException();
    }
  }

  @Override
  public boolean supportsOrd() {
    return false;
  }

  public int indexDivisor(final String fieldName) {
    final Integer perFieldDivisor = indexDivisors.get(fieldName);
    if (perFieldDivisor == null) {
        return indexDivisor;
    } else {
        return perFieldDivisor;
    }
  }

  private final class FieldIndexData {

    private final FieldInfo fieldInfo;
    private final long indexStart;

    // Set only if terms index is loaded:
    private volatile FST<Long> fst;

    public FieldIndexData(FieldInfo fieldInfo, long indexStart) throws IOException {

      this.fieldInfo = fieldInfo;
      this.indexStart = indexStart;

      if (indexDivisor > 0) {
        loadTermsIndex();
      }
    }

    private void loadSubSampledIndex(
        final String fileName,
        final int indexDivisor)
    {
        IndexInput subIdxInput = null;
        try {
            if (dir.fileExists(fileName)) {
                subIdxInput = dir.openInput(fileName);
                final int divisor = subIdxInput.readInt();
                if (divisor != indexDivisor) {
                    subIdxInput.close();
                    subIdxInput = null;
                    dir.deleteFile(fileName);
                    System.err.println(
                        "loadSubSampledIndex: "
                            + fileName + ": indexDivisor not matches"
                            + ": expected " + indexDivisor + ", found "
                            + divisor + ". Deleting index.");
                } else {
                    fst = new FST<Long>(subIdxInput, fstOutputs);
                }
            }
        } catch (IOException ign) {
            ign.printStackTrace();
        } finally {
            if (subIdxInput != null) {
                try {
                    subIdxInput.close();
                } catch (IOException ign) {
                    ign.printStackTrace();
                }
            }
        }
    }

    private void saveSubSampledIndex(
        final String fileName,
        final FST<Long> fst,
        final int divisor)
    {
        try (final IndexOutput out = dir.createOutput(fileName)) {
            out.writeInt(divisor);
            fst.save(out);
        } catch (IOException ign) {
            ign.printStackTrace();
        }
    }

    private void loadOriginalIndex() throws IOException {
        IndexInput clone = (IndexInput) in.clone();
        clone.seek(indexStart);
        fst = new FST<Long>(clone, fstOutputs);
        clone.close();
        System.err.println("FST: fi= " + fieldInfo.name
            + " , size= " + fst.sizeInBytes());
    }

    public void loadTermsIndex() throws IOException {
        if (fst == null) {
            final int indexDivisor = indexDivisor(fieldInfo.name);

            final String subsampledIndexName = indexFileName + '_'
                + fieldInfo.name + '.'
                + VariableGapTermsIndexWriter.TERMS_INDEX_EXTENSION;
            if (indexDivisor > 1) {
                loadSubSampledIndex(subsampledIndexName, indexDivisor);
                if (fst == null) {
                // subsample
                    loadOriginalIndex();
                    final PositiveIntOutputs outputs =
                        PositiveIntOutputs.getSingleton(true);
                    final Builder<Long> builder =
                        new Builder<Long>(
                            FST.INPUT_TYPE.BYTE1,
                            0,
                            0,
                            true,
                            outputs);
                    final BytesRefFSTEnum<Long> fstEnum =
                        new BytesRefFSTEnum<Long>(fst);
                    BytesRefFSTEnum.InputOutput<Long> result;
                    int count = indexDivisor;
                    while((result = fstEnum.next()) != null) {
                        if (count == indexDivisor) {
                            builder.add(result.input, result.output);
                            count = 0;
                        }
                        count++;
                    }
                    fst.close();
                    fst = builder.finish();
                    saveSubSampledIndex(subsampledIndexName, fst, indexDivisor);
                } else {
                    System.err.println(
                        "loadTermsIndex: reused subsampled index: "
                            + subsampledIndexName);
                }
            } else {
                loadOriginalIndex();
                if (dir.fileExists(subsampledIndexName)) {
                    System.err.println(
                        "no subSampledIndex is required: "
                            + ", deleting: " + subsampledIndexName);
                    try {
                        dir.deleteFile(subsampledIndexName);
                    } catch (IOException ign) {
                        ign.printStackTrace();
                    }
                }
            }
        }
    }

    public void close() {
        if (fst != null) fst.close();
    }

  }

  // Externally synced in IndexWriter
  @Override
  public void loadTermsIndex(int indexDivisor) throws IOException {
    if (!indexLoaded) {

      if (indexDivisor < 0) {
        this.indexDivisor = -indexDivisor;
      } else {
        this.indexDivisor = indexDivisor;
      }

      Iterator<FieldIndexData> it = fields.values().iterator();
      while(it.hasNext()) {
        it.next().loadTermsIndex();
      }

      indexLoaded = true;
      in.close();
    }
  }

  @Override
  public FieldIndexEnum getFieldEnum(FieldInfo fieldInfo) {
    final FieldIndexData fieldData = fields.get(fieldInfo);
    if (fieldData.fst == null) {
      return null;
    } else {
      return new IndexEnum(fieldData.fst);
    }
  }

  public static void files(Directory dir, SegmentInfo info, String id, Collection<String> files) throws IOException {
    files.add(IndexFileNames.segmentFileName(info.name, id, VariableGapTermsIndexWriter.TERMS_INDEX_EXTENSION));
    final String indexFileName =
        IndexFileNames.segmentFileName(
        info.name,
        id,
        "") + '_';
    for (final String file : dir.listAll(indexFileName)) {
        if (file.startsWith(indexFileName)
            && file.endsWith(VariableGapTermsIndexWriter.TERMS_INDEX_EXTENSION))
        {
            files.add(file);
        }
    }
  }

  public static void getIndexExtensions(Collection<String> extensions) {
    extensions.add(VariableGapTermsIndexWriter.TERMS_INDEX_EXTENSION);
  }

  @Override
  public void getExtensions(Collection<String> extensions) {
    getIndexExtensions(extensions);
  }

  @Override
  public void close() throws IOException {
    Iterator<FieldIndexData> it = fields.values().iterator();
    while(it.hasNext()) {
      it.next().close();
    }
    if (in != null && !indexLoaded) {
      in.close();
    }
  }

  protected void seekDir(IndexInput input, long dirOffset) throws IOException {
    input.seek(dirOffset);
  }
}
