package org.apache.lucene.index.codecs;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.Closeable;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.index.FieldsEnum;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.SegmentInfo;
import org.apache.lucene.index.TermState;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
//import org.apache.lucene.index.codecs.standard.StandardPostingsReader; // javadocs
import org.apache.lucene.index.codecs.bloom.FuzzySet; // javadocs
import org.apache.lucene.index.codecs.bloom.FuzzySet.ContainsResult; // javadocs
//import org.apache.lucene.index.codecs.bloom.NativeFuzzySet; // javadocs
//import org.apache.lucene.index.codecs.bloom.NativeFuzzySet.ContainsResult; // javadocs
import org.apache.lucene.index.codecs.yandex.YandexPostingsReader; // javadocs
import org.apache.lucene.index.codecs.yandex.YandexPostingsReader.YandexTermState; // javadocs
import org.apache.lucene.index.codecs.yandex.YandexPostingsWriter; // javadocs
import org.apache.lucene.store.ByteArrayDataInput;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.DeflateDecompressor;
import org.apache.lucene.store.Decompressor;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.ChainedBlockCompressedInputStream;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CodecUtil;
import org.apache.lucene.util.CloseableThreadLocal;
import org.apache.lucene.util.DoubleBarrelLRUCache;

import ru.yandex.msearch.util.JavaAllocator;

/** Handles a terms dict, but decouples all details of
 *  doc/freqs/positions reading to an instance of {@link
 *  PostingsReaderBase}.  This class is reusable for
 *  codecs that use a different format for
 *  docs/freqs/positions (though codecs are also free to
 *  make their own terms dict impl).
 *
 * <p>This class also interacts with an instance of {@link
 * TermsIndexReaderBase}, to abstract away the specific
 * implementation of the terms dict index. 
 * @lucene.experimental */

public class YandexTermsReader extends FieldsProducer {
  private static final JavaAllocator allocator =
    JavaAllocator.get("DeflateYandexTermsReaderBlockCache");
  private static final String TERMS_TAG = "terms";
//  private static final Decompressor decompressor = DeflateDecompressor.INSTANCE;
  private static final AtomicLong bloomSuccess = new AtomicLong(0);
  private static final AtomicLong bloomFail = new AtomicLong(0);
  // Open input to the main terms dict file (_X.tis)
  private final IndexInput in;
  private final Compressor compressor;
  private final Decompressor decompressor;

  // Reads the terms dict entries, to gather state to
  // produce DocsEnum on demand
  private final PostingsReaderBase postingsReader;

  private final TreeMap<String,FieldReader> fields = new TreeMap<String,FieldReader>();
  private final HashMap<String,FuzzySet> bloomsByFieldName;

  // Caches the most recently looked-up field + terms:
  private final DoubleBarrelLRUCache<FieldAndTerm,YandexTermState> termsCache;
  private static final AtomicLong cacheSuccess = new AtomicLong(0);
  private static final AtomicLong cacheFail = new AtomicLong(0);

  // Reads the terms index
  private TermsIndexReaderBase indexReader;

  // keeps the dirStart offset
  protected long dirOffset;

  static {
    Thread bloomStats = new Thread("BloomStats") {
        public double round(final double value) {
            double round = value * 100;
            round = (double)((int) round);
            return round / 100;
        }

        @Override
        public void run() {
            long prevHits = 0;
            long prevMisses = 0;
            Averager avg10sec = new Averager(2);
            Averager avg60sec = new Averager(12);
            Averager avg300sec = new Averager(60);
            String prevStats = "";
            while (true) {
                final long currentHits = bloomSuccess.get();
                final long currentMisses = bloomFail.get();
                final long hits = currentHits - prevHits;
                final long misses = currentMisses - prevMisses;
                if (hits != 0 || misses != 0) {
                    prevHits = currentHits;
                    prevMisses = currentMisses;
                    double diffRatio =
                        (hits * 100.0) /
                        (double)(hits + misses);
                    avg10sec.push(diffRatio);
                    avg60sec.push(diffRatio);
                    avg300sec.push(diffRatio);
                }
                double totalRatio =
                    (currentHits * 100.0) /
                    (double)(currentHits + currentMisses);
                String stats = "BloomFilter: "
                        + "lifetimeRatio=" + round(totalRatio)
                        + ", 10secRatio=" + round(avg10sec.value())
                        + ", 60secRatio=" + round(avg60sec.value())
                        + ", 300secRatio=" + round(avg300sec.value());
                if (!stats.equals(prevStats)) {
                    System.err.println(stats);
                    prevStats = stats;
                }
                try {
                    Thread.sleep(5000);
                } catch (InterruptedException ign) {
                }
            }
        }
    };
    bloomStats.setDaemon(true);
    bloomStats.start();
  }

  static {
    Thread bloomStats = new Thread("TermsCacheStats") {
        public double round(final double value) {
            double round = value * 100;
            round = (double)((int) round);
            return round / 100;
        }

        @Override
        public void run() {
            long prevHits = 0;
            long prevMisses = 0;
            Averager avg10sec = new Averager(2);
            Averager avg60sec = new Averager(12);
            Averager avg300sec = new Averager(60);
            String prevStats = "";
            while (true) {
                final long currentHits = cacheSuccess.get();
                final long currentMisses = cacheFail.get();
                final long hits = currentHits - prevHits;
                final long misses = currentMisses - prevMisses;
                if (hits != 0 || misses != 0) {
                    prevHits = currentHits;
                    prevMisses = currentMisses;
                    double diffRatio =
                        (hits * 100.0) /
                        (double)(hits + misses);
                    avg10sec.push(diffRatio);
                    avg60sec.push(diffRatio);
                    avg300sec.push(diffRatio);
                }
                double totalRatio =
                    (currentHits * 100.0) /
                    (double)(currentHits + currentMisses);
                String stats = "TermsCache: "
                        + "lifetimeRatio=" + round(totalRatio)
                        + ", 10secRatio=" + round(avg10sec.value())
                        + ", 60secRatio=" + round(avg60sec.value())
                        + ", 300secRatio=" + round(avg300sec.value());
                if (!stats.equals(prevStats)) {
                    System.err.println(stats);
                    prevStats = stats;
                }
                try {
                    Thread.sleep(5000);
                } catch (InterruptedException ign) {
                }
            }
        }
    };
    bloomStats.setDaemon(true);
    bloomStats.start();
  }

    private static class Averager {
        public double sum = 0;
        public double[] values;
        public int idx = 0;
        public int count = 0;

        public Averager(int depth) {
            this.values = new double[depth];
        }

        public void push(double value) {
            sum += value;
            sum -= values[idx];
            values[idx++] = value;
            if (idx >= values.length) {
                idx = 0;
            }
            if (count < values.length) {
                count++;
            }
        }

        public double value() {
            return sum / count;
        }
    }


  // Used as key for the terms cache
  private static class FieldAndTerm extends DoubleBarrelLRUCache.CloneableKey {
    String field;
    BytesRef term;

    public FieldAndTerm() {
    }

    public FieldAndTerm(FieldAndTerm other) {
      field = other.field;
      term = new BytesRef(other.term);
    }

    @Override
    public boolean equals(Object _other) {
      FieldAndTerm other = (FieldAndTerm) _other;
      return other.field == field && term.bytesEquals(other.term);
    }

    @Override
    public Object clone() {
      return new FieldAndTerm(this);
    }

    @Override
    public int hashCode() {
      return field.hashCode() * 31 + term.hashCode();
    }
  }
  
  //private String segment;
  
  public YandexTermsReader(
    final Compressor compressor,
    TermsIndexReaderBase indexReader, Directory dir, FieldInfos fieldInfos, String segment, PostingsReaderBase postingsReader, int readBufferSize,
                          int termsCacheSize, String codecId, Set<String> bloomSet)
    throws IOException
  {
    this.compressor = compressor;
    this.decompressor = compressor.decompressor();
    this.postingsReader = postingsReader;

    //this.segment = segment;
    in = dir.openInput(IndexFileNames.segmentFileName(segment, codecId, YandexTermsWriter.TERMS_EXTENSION),
                       readBufferSize);

    boolean success = false;
    int totalNumTerms = 0;
    try {

      bloomsByFieldName = new HashMap<>();
      String bloomFileName = IndexFileNames.segmentFileName(
          segment, codecId, YandexTermsWriter.BLOOM_EXTENSION);
      if (dir.fileExists(bloomFileName)) {
        try (IndexInput bloomIn = dir.openInput(bloomFileName, readBufferSize)) {
            CodecUtil.checkHeader(
                bloomIn,
                YandexTermsWriter.BLOOM_CODEC_NAME,
                YandexTermsWriter.BLOOM_CODEC_VERSION,
                YandexTermsWriter.BLOOM_CODEC_VERSION);
            // Load the delegate postings format
            int numBlooms = bloomIn.readInt();
            for (int i = 0; i < numBlooms; i++) {
                int fieldNum = bloomIn.readInt();
                final FieldInfo fieldInfo = fieldInfos.fieldInfo(fieldNum);
                FuzzySet bloom = FuzzySet.deserialize(bloomIn, fieldInfo.name);
                if (bloomSet.contains(fieldInfo.name)) {
                    bloomsByFieldName.put(fieldInfo.name, bloom);
                } else {
                    bloom.close();
                }
            }
        }
      }

      readHeader(in);

      // Have PostingsReader init itself
      postingsReader.init(in);

      // Read per-field details
      seekDir(in, dirOffset);

      final int numFields = in.readVInt();

      this.indexReader = indexReader;

      for(int i=0;i<numFields;i++) {
        final int field = in.readVInt();
        final long numTerms = in.readVLong();
        totalNumTerms += numTerms;
        assert numTerms >= 0;
        final long termsStartPointer = in.readVLong();
        final FieldInfo fieldInfo = fieldInfos.fieldInfo(field);
        final long sumTotalTermFreq = fieldInfo.omitTermFreqAndPositions ? -1 : in.readVLong();
        assert !fields.containsKey(fieldInfo.name);
        fields.put(fieldInfo.name, new FieldReader(fieldInfo, numTerms, termsStartPointer, sumTotalTermFreq, bloomsByFieldName.get(fieldInfo.name)));
      }


      success = true;
    } finally {
      if (!success) {
        in.close();
      }
    }

    if( termsCacheSize == -1 )
    {
	termsCacheSize = totalNumTerms * 1024 / 10000000;
	if( termsCacheSize < 50 ) termsCacheSize = 50;
	if( termsCacheSize > 1024 ) termsCacheSize = 1024;
//	System.err.println( "totalNumTerms=" + totalNumTerms + " / termsCacheSize=" + termsCacheSize );
    }

    termsCache = new DoubleBarrelLRUCache<FieldAndTerm,YandexTermState>(termsCacheSize);
  }

  private String codecName() {
    if (compressor.id() != null) {
        return YandexTermsWriter.CODEC_NAME + "_" + compressor.id();
    } else {
        return YandexTermsWriter.CODEC_NAME;
    }
  }

  protected void readHeader(IndexInput input) throws IOException {
    CodecUtil.checkHeader(in, codecName(),
                          YandexTermsWriter.VERSION_START,
                          YandexTermsWriter.VERSION_CURRENT);
    dirOffset = in.readLong();    
  }
  
  protected void seekDir(IndexInput input, long dirOffset)
      throws IOException {
    input.seek(dirOffset);
  }
  
  @Override
  public void loadTermsIndex(int indexDivisor) throws IOException {
    indexReader.loadTermsIndex(indexDivisor);
  }

  @Override
  public void close() throws IOException {
    try {
      for (FuzzySet bloom : bloomsByFieldName.values()) {
            bloom.close();
      }
      bloomsByFieldName.clear();
      try {
        if (indexReader != null) {
          indexReader.close();
        }
      } finally {
        // null so if an app hangs on to us (ie, we are not
        // GCable, despite being closed) we still free most
        // ram
        indexReader = null;
        if (in != null) {
          in.close();
        }
      }
    } finally {
      try {
        if (postingsReader != null) {
          postingsReader.close();
        }
      } finally {
        for(FieldReader field : fields.values()) {
          field.close();
        }
      }
    }
  }

  public static void files(Directory dir, SegmentInfo segmentInfo, String id, Collection<String> files) throws IOException {
    files.add(IndexFileNames.segmentFileName(segmentInfo.name, id, YandexTermsWriter.TERMS_EXTENSION));
    if (dir.fileExists(IndexFileNames.segmentFileName(segmentInfo.name, id, YandexTermsWriter.BLOOM_EXTENSION))) {
        files.add(IndexFileNames.segmentFileName(segmentInfo.name, id, YandexTermsWriter.BLOOM_EXTENSION));
    }
  }

  public static void getExtensions(Collection<String> extensions) {
    extensions.add(YandexTermsWriter.TERMS_EXTENSION);
    extensions.add(YandexTermsWriter.BLOOM_EXTENSION);
  }

  @Override
  public FieldsEnum iterator() {
    return new TermFieldsEnum();
  }

  @Override
  public Terms terms(String field) throws IOException {
    return fields.get(field);
  }

  // Iterates through all fields
  private class TermFieldsEnum extends FieldsEnum {
    final Iterator<FieldReader> it;
    FieldReader current;

    TermFieldsEnum() {
      it = fields.values().iterator();
    }

    @Override
    public String next() {
      if (it.hasNext()) {
        current = it.next();
        return current.fieldInfo.name;
      } else {
        current = null;
        return null;
      }
    }
    
    @Override
    public TermsEnum terms(final boolean buffered) throws IOException {
      return current.iterator(buffered);
    }
  }

  private class FieldReader extends Terms implements Closeable {
    final long numTerms;
    final FieldInfo fieldInfo;
    final long termsStartPointer;
    final long sumTotalTermFreq;
    final FuzzySet bloomFilter;
//    private final CloseableThreadLocal<TermsEnum> termsEnumTreadLocal =
//        new CloseableThreadLocal<TermsEnum>();
//    private final ThreadLocal<WeakReference<TermsEnum>> termsEnumTreadLocal =
//        new ThreadLocal<WeakReference<TermsEnum>>();
    final ConcurrentLinkedQueue<TermsEnum> cachedEnums =
        new ConcurrentLinkedQueue<TermsEnum>();
    final int indexDivisor;

    FieldReader(FieldInfo fieldInfo, long numTerms, long termsStartPointer, long sumTotalTermFreq, FuzzySet bloomFilter) {
      assert numTerms > 0;
      this.fieldInfo = fieldInfo;
      this.numTerms = numTerms;
      this.termsStartPointer = termsStartPointer;
      this.sumTotalTermFreq = sumTotalTermFreq;
      this.bloomFilter = bloomFilter;
      this.indexDivisor =
        ((VariableGapTermsIndexReader) indexReader).getDivisor(fieldInfo);
    }

    @Override
    public Comparator<BytesRef> getComparator() {
      return BytesRef.getUTF8SortedAsUnicodeComparator();
    }

    @Override
    public void close() {
//      termsEnumTreadLocal.close();
      cachedEnums.clear();
      super.close();
    }
    
    @Override
    public TermsEnum iterator(final boolean buffered) throws IOException {
      return new SegmentTermsEnum(buffered);
    }

    @Override
    public TermsEnum reverseIterator() throws IOException {
        return new ReverseSegmentTermsEnum();
    }

    @Override
    public long getUniqueTermCount() {
      return numTerms;
    }
    
    @Override
    public TermsEnum getThreadTermsEnum(final boolean buffered)
        throws IOException
    {
/*        WeakReference<TermsEnum> teRef = termsEnumTreadLocal.get();
        TermsEnum te;
        if (teRef == null) {
            te = iterator();
            teRef = new WeakReference<TermsEnum>(te);
            termsEnumTreadLocal.set(teRef);
            return te;
        }
        te = teRef.get();
        if (te == null) {
            te = iterator();
            teRef = new WeakReference<TermsEnum>(te);
            return te;
        }
	return te;
*/
        TermsEnum te = cachedEnums.poll();
        if (te != null) {
            ((SegmentTermsEnum)te).reset(buffered);
        } else {
            te = iterator(buffered);
        }
        return te;
    }

    @Override
    public long getSumTotalTermFreq() {
      return sumTotalTermFreq;
    }

    // Iterates through terms in this field
    private final class SegmentTermsEnum extends TermsEnum {
      private final IndexInput in;
      private final YandexTermState state;
      private final boolean doOrd;
      private final FieldAndTerm fieldTerm = new FieldAndTerm();
      private final TermsIndexReaderBase.FieldIndexEnum indexEnum;
      private final BytesRef term = new BytesRef();
      private long currentBlockPointer = -1;
      private long currentTermPos = -1;
      private long proxPointer = -1;
      private long freqPointer = -1;
      private long skipPointer = -1;
      private long lastBytesReaderPos = 0;
      private long lastBytesReaderBlockPos = 0;
      private boolean useCache;
      private boolean bufferLocked = false;
      private int[] bytesReaderPos = null;
      private boolean reused = false;

      /* True after seek(TermState), do defer seeking.  If the app then
         calls next() (which is not "typical"), then we'll do the real seek */
      private boolean seekPending;

      ChainedBlockCompressedInputStream bytesReader = null; //new ChainedBlockCompressedInputStream();

      public SegmentTermsEnum(final boolean buffered) throws IOException {
        in = (IndexInput) YandexTermsReader.this.in.clone();
        in.seek(termsStartPointer);
	lastBytesReaderBlockPos = in.getFilePointer();
        indexEnum = indexReader.getFieldEnum(fieldInfo);
        doOrd = indexReader.supportsOrd();
        fieldTerm.field = fieldInfo.name;
        state = (YandexTermState)postingsReader.newTermState();
        state.totalTermFreq = -1;
        state.ord = -1;

        useCache = buffered;

        //System.out.println("BTR.enum init this=" + this + " postingsReader=" + postingsReader);
      }

      @Override
      public void close() {
        if (bytesReader != null) {
            try {
                bytesReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            bytesReader = null;
        }
      }

      public void reset(final boolean buffered) throws IOException {
        this.useCache = buffered;
        if (bytesReader != null) {
            bytesReader.useCache(buffered);
        }
        in.seek(termsStartPointer);
	lastBytesReaderBlockPos = in.getFilePointer();
        state.totalTermFreq = -1;
        state.ord = -1;
      }

      @Override
      public Comparator<BytesRef> getComparator() {
        return BytesRef.getUTF8SortedAsUnicodeComparator();
      }
      
      
      private void checkBytesReader() throws IOException
      {
        if( bytesReader != null )
        {
            return;
        }
        bytesReader = new ChainedBlockCompressedInputStream(in, decompressor, allocator, TERMS_TAG);
        bytesReaderPos = bytesReader.getPositionPtr();
        bytesReader.useCache(useCache);
        bytesReader.seek( lastBytesReaderBlockPos, lastBytesReaderPos );
        bytesReader.clearNextBlock();
        postingsReader.readTermsBlock( bytesReader, fieldInfo, state );

      }

      public boolean seekExact(BytesRef text, boolean useCache) throws IOException {
        if (bloomFilter != null && bloomFilter.contains(text) == ContainsResult.NO) {
//          System.err.println("bloomFilter reject: " + text.utf8ToString());
          bloomSuccess.incrementAndGet();
          return false;
        }
        final boolean found = seek(text, useCache) == SeekStatus.FOUND;
        if (!found) {
            bloomFail.incrementAndGet();
        }
        return found;
      }

      // TODO: we may want an alternate mode here which is
      // "if you are about to return NOT_FOUND I won't use
      // the terms data from that"; eg FuzzyTermsEnum will
      // (usually) just immediately call seek again if we
      // return NOT_FOUND so it's a waste for us to fill in
      // the term that was actually NOT_FOUND
      @Override
      public SeekStatus seek(final BytesRef target, final boolean useCache1) throws IOException {

        if (indexEnum == null) {
          throw new IllegalStateException("terms index was not loaded");
        }
        
//        boolean useCache = false;
        boolean useCache = useCache1;
        
//        if( true ) return SeekStatus.NOT_FOUND;
//	useCache = false;
        
        if (useCache) {
//	if( false ) {
          fieldTerm.term = target;
          // TODO: should we differentiate "frozen"
          // TermState (ie one that was cloned and
          // cached/returned by termState()) from the
          // malleable (primary) one?
          final YandexTermState cachedState = termsCache.get(fieldTerm);
          if (cachedState != null) {
            cacheSuccess.incrementAndGet();
            seekPending = true;
//            System.out.println("  cached!");
            seek(target, cachedState);
//            if( cachedState.seekStatus != SeekStatus.FOUND )
//            {
        	term.copy(cachedState.term);
//            }
//            System.err.println( this + "   cached!  term=" + fieldInfo.name + ":" + term.utf8ToString() + " df=" + state.docFreq + " freqP=" + state.freqBlockStart + " count=" + state.termCount + ", termPos=" + state.termPointer);
            return state.seekStatus;
          } else {
            cacheFail.incrementAndGet();
          }
//          else System.err.println( "    not cached: " + fieldInfo.name + ":" + target.utf8ToString() );
        }
        
        seekPending = false;
        
        
        long start = System.currentTimeMillis();
        long data = 0;
        long nextData;
//        for( int i = 0; i < 10; i++ )

	data = indexEnum.seek(target);
        long time = System.currentTimeMillis() - start;
        term.copy(indexEnum.term());
//        nextData = indexEnum.next();
//	long dictPos = (data & 0xFFFFFFFFFFFF0000L) >> 16;
	long dictPos = data;
	if( dictPos == 0 ) 
	{
//    	    state.blockFilePointer = currentBlockPointer;
//    	    state.termPointer = bytesReader.getFilePointer();
//    	    YandexTermState clonedState = (YandexTermState)state.clone();
//    	    clonedState.term = new BytesRef(term);
//	    termsCache.put(new FieldAndTerm(fieldTerm), clonedState);
	    return SeekStatus.END;
	}
//	long termPos = data & 0xFFFF;
//	long nextDictPos = (data & 0xFFFFFFFFFFFF0000L) >> 16;
//	long nextDictPos = data;
//	long nextTermPos = data & 0xFFFF;
        if( BytesRef.getUTF8SortedAsUnicodeComparator().compare(target, term) < 0 )
    	    return SeekStatus.NOT_FOUND;
//	if( true )    
	    
//        System.err.println( "Index: target=" + target.utf8ToString() + ", time=" + time + " termPrefix: " + term.utf8ToString() + " / " + dictPos + " / " + data );
//        while( indexEnum.next() != -1 )
//	    System.out.println( "nextIndex: termPrefix: " + indexEnum.term().utf8ToString() );
//        System.out.println( "Index: " + time + " termPrefix: " + term.utf8ToString() + " / " + dictPos + " / " + termPos + " / " + data );
        
//        if( currentBlockPointer != dictPos )
//        {
//    	    in.seek( dictPos );
//    	}
//        loadBlock();
        boolean locked = false;
	try
	{
//	if( bytesReader == null ) bytesReader = new ChainedBlockCompressedInputStream( in );
	lastBytesReaderBlockPos = dictPos;
	lastBytesReaderPos = 0;
	checkBytesReader();
	bytesReader.seek( dictPos, 0 );
        bytesReader.clearNextBlock();
        locked = lockBuffer();
	state.termCount = 0;
        
//        bytesReader.seek( termPos );
//        term.copy( termPrefix );
	int scans = 0;
        while( !bytesReader.eof() )
        {
    	    loadTermUnlocked();
//    	    System.err.println( "LOADED TERM: " + term.utf8ToString() );
    	    int cmp = BytesRef.getUTF8SortedAsUnicodeComparator().compare(target, term);
    	    if( cmp < 0 )
    	    {
//    		bytesReader.seek(currentTermPos);
//    		state.termCount--;
//		System.err.println( "NOT_FOUND: " + target.utf8ToString() + ", scans: " + scans );
//    		state.copyFrom(tmpState);
    		if( useCache )
    		{
    		    state.blockFilePointer = bytesReader.getCurrentBlockFilePointer();
    		    state.termPointer = bytesReader.getFilePointer();
    		    state.seekStatus = SeekStatus.NOT_FOUND;
    		    YandexTermState clonedState = (YandexTermState)state.clone();
    		    clonedState.term = new BytesRef(term);
//    		    if( state.termCount == 1 ) clonedState.termCount = 0;
//		    clonedState.termCount--;
////    		    state.termSuffix = 
        	    termsCache.put(new FieldAndTerm(fieldTerm), clonedState);
//        	    System.err.println( this + "Caching not found: term=" + term.utf8ToString() + ", termCount=" + state.termCount  + " termPos=" + state.termPointer + ", freqFP=" + state.freqBlockStart + ", freqFPOffset" + state.freqBlockOffset );
//        	    System.err.println( this + "Caching not found: term=" + term.utf8ToString() + ", termCount=" + clonedState.termCount  + " termPos=" + clonedState.termPointer + ", freqFP=" + clonedState.freqBlockStart );
        	}
    		return SeekStatus.NOT_FOUND;
    	    }
    	    if( cmp == 0 )
    	    {
//    		bytesReader.seek(currentTermPos);
//    		state.termCount--;
//    		state.copyFrom(tmpState);
//		System.err.println( "FOUND: " + target.utf8ToString() + ", scans: " + scans );
    		if( useCache )
    		{
    		    state.blockFilePointer = bytesReader.getCurrentBlockFilePointer();
//    		    state.blockFilePointer = currentBlockPointer;
    		    state.termPointer = bytesReader.getFilePointer();
    		    state.seekStatus = SeekStatus.FOUND;
    		    YandexTermState clonedState = (YandexTermState)state.clone();
    		    clonedState.term = new BytesRef(term);
//		    clonedState.termCount--;
//    		    state.termSuffix = 
        	    termsCache.put(new FieldAndTerm(fieldTerm), clonedState);
//        	    System.err.println( "Caching: term=" + term.utf8ToString() + ", termCount=" + state.termCount  + " termPos=" + state.termPointer + ", freqFP=" + state.freqBlockStart );
        	}
//    		return SeekStatus.FOUND;
    		return SeekStatus.FOUND;
//    		return SeekStatus.NOT_FOUND;
    	    }
    	    
    	    scans++;
    	}
	state.seekStatus = SeekStatus.END;
//	System.err.println( "SeekStatus.END: " + target.utf8ToString() + ", scans=" + scans );
        if( useCache )
	{
    	    state.blockFilePointer = currentBlockPointer;
    	    state.termPointer = bytesReader.getFilePointer();
    	    YandexTermState clonedState = (YandexTermState)state.clone();
    	    clonedState.term = new BytesRef(term);
	    termsCache.put(new FieldAndTerm(fieldTerm), clonedState);
	}
	return state.seekStatus; //we are now at the end of this block
	
	
	} finally {
	    if (locked) {
	        releaseBuffer();
	    }
	}
      }

        private final boolean lockBuffer() throws IOException {
            if (bytesReader != null && !bufferLocked) {
                bytesReader.lockBuffer();
                bufferLocked = true;
                return true;
            } else {
                return false;
            }
        }

        private final void releaseBuffer() throws IOException {
            if (bufferLocked) {
                bufferLocked = false;
                bytesReader.releaseBuffer();
            }
        }

        private final boolean lockBufferLite() throws IOException {
            if (bufferLocked) {
                return false;
            }
            bytesReader.lockBuffer();
            bufferLocked = true;
            return true;
        }

      private void loadTerm() throws IOException
      {
        final boolean locked = lockBufferLite();
        try {
            currentTermPos = bytesReader.getFilePointer();
	    int prefix = bytesReader.readVInt();
    	    int suffix = bytesReader.readVInt();
//    	System.out.println( this + " loadterm: " + prefix + " / " + suffix + ", termCount=" + state.termCount + ", currentTermPos=" + currentTermPos );
    	    term.length = prefix + suffix;
    	    if (term.bytes.length < term.length) {
    	        term.grow(term.length);
    	    }
            bytesReader.readBytes(term.bytes, prefix, suffix);
    	    
//        postingsReader.readTermsBlock(bytesReader, fieldInfo, state);
            state.docFreq = bytesReader.readVInt();
//    	    System.err.println( "DF: " + df );
            if( !fieldInfo.omitTermFreqAndPositions ) {
	        state.totalTermFreq = state.docFreq + bytesReader.readVLong();
	    }
	

            if( bytesReader.isNextBlock() )
            {
//	    System.err.println( "bytesReader.isNextBlock" );
    	        bytesReader.clearNextBlock();
	        state.termCount = 0;
	        currentBlockPointer = bytesReader.getCurrentBlockFilePointer();
	    }

	    postingsReader.nextTerm( fieldInfo, state );
	} finally {
	    if (locked) {
	        releaseBuffer();
	    }
	}

        state.termCount++;
      }

        private void loadTermUnlocked() throws IOException {
            //currentTermPos = bytesReader.getFilePointer();
            currentTermPos = bytesReaderPos[0];
            int prefix = bytesReader.readVIntUnlocked();
            int suffix = bytesReader.readVIntUnlocked();
            term.length = prefix + suffix;
            if (term.bytes.length < term.length) {
                term.grow(term.length);
            }
            bytesReader.readBytesUnlocked(term.bytes, prefix, suffix);

            state.docFreq = bytesReader.readVIntUnlocked();
            if( !fieldInfo.omitTermFreqAndPositions ) {
                state.totalTermFreq = state.docFreq + bytesReader.readVLongUnlocked();
            }

            if (bytesReader.isNextBlock()) {
                bytesReader.clearNextBlock();
                state.termCount = 0;
                currentBlockPointer = bytesReader.getCurrentBlockFilePointer();
            }

            postingsReader.nextTerm(fieldInfo, state);

            state.termCount++;
      }

      @Override
      public BytesRef next() throws IOException {
        //System.out.println("BTR.next() seekPending=" + seekPending + " pendingSeekCount=" + state.termCount);

//	if( bytesReader == null ) bytesReader = new ChainedBlockCompressedInputStream( in );
	checkBytesReader();

	if( seekPending )
	{
	    bytesReader.seek( state.blockFilePointer, state.termPointer );
	    seekPending = false;
	    bytesReader.clearNextBlock();
	}
	if( !bytesReader.eof() )
	{
	    loadTerm();
        // NOTE: meaningless in the non-ord case
    	    state.ord++;

        //System.out.println("  return term=" + fieldInfo.name + ":" + term.utf8ToString() + " " + term);
    	    return term;
	}
	return null;
      }

      /* Decodes only the term bytes of the next term.  If caller then asks for
         metadata, ie docFreq, totalTermFreq or pulls a D/&PEnum, we then (lazily)
         decode all metadata up to the current term. */

      @Override
      public BytesRef term() {
        return term;
      }

      @Override
      public int docFreq() throws IOException {
        //System.out.println("BTR.docFreq");
        //System.out.println("  return " + state.docFreq);
        return state.docFreq;
      }

      @Override
      public long totalTermFreq() throws IOException {
        return state.totalTermFreq;
      }

      @Override
      public DocsEnum docs(Bits skipDocs, DocsEnum reuse) throws IOException {
        //System.out.println("BTR.docs this=" + this);
        //System.out.println("  state.docFreq=" + state.docFreq);
        final DocsEnum docsEnum = postingsReader.docs(fieldInfo, state, skipDocs, reuse, useCache);
        assert docsEnum != null;
        return docsEnum;
      }

      @Override
      public DocsAndPositionsEnum docsAndPositions(Bits skipDocs, DocsAndPositionsEnum reuse) throws IOException {
        //System.out.println("BTR.d&p this=" + this);
        if (fieldInfo.omitTermFreqAndPositions) {
          return null;
        } else {
          DocsAndPositionsEnum dpe = postingsReader.docsAndPositions(fieldInfo, state, skipDocs, reuse, useCache);
          //System.out.println("  return d&pe=" + dpe);
          return dpe;
        }
      }

      @Override
      public void seek(BytesRef target, TermState otherState) throws IOException {
        //System.out.println("BTR.seek termState target=" + target.utf8ToString() + " " + target + " this=" + this);
        assert otherState != null && otherState instanceof YandexTermState;
        assert !doOrd || ((YandexTermState) otherState).ord < numTerms;
        state.copyFrom(otherState);
        seekPending = true;
        term.copy(((YandexTermState) otherState).term);
      }

      @Override
      public TermState termState() throws IOException {
        //System.out.println("BTR.termState this=" + this);
        YandexTermState ts = state.clone();
        ts.term = new BytesRef(term);
        //System.out.println("  return ts=" + ts);
        return ts;
      }

      @Override
      public SeekStatus seek(long ord) throws IOException {
        //System.out.println("BTR.seek by ord ord=" + ord);
        if (indexEnum == null) {
          throw new IllegalStateException("terms index was not loaded");
        }

        if (ord >= numTerms) {
          state.ord = numTerms-1;
          return SeekStatus.END;
        }

        // TODO: if ord is in same terms block and
        // after current ord, we should avoid this seek just
        // like we do in the seek(BytesRef) case
        bytesReader.seek( indexEnum.seek(ord), 0 );

        // Block must exist since ord < numTerms:

        seekPending = false;

        state.ord = indexEnum.ord()-1;
        assert state.ord >= -1: "ord=" + state.ord;
        term.copy(indexEnum.term());
        loadTerm();

        // Now, scan:
        int left = (int) (ord - state.ord);
        while(left > 0) {
          final BytesRef term = next();
          assert term != null;
          left--;
        }

        // always found
        return SeekStatus.FOUND;
      }

      @Override
      public long ord() {
        if (!doOrd) {
          throw new UnsupportedOperationException();
        }
        return state.ord;
      }

      private void doPendingSeek() {
      }

    }

        private final class ReverseSegmentTermsEnum extends TermsEnum {
            private final IndexInput in;
            private YandexTermState state = null;
            private final TermsIndexReaderBase.FieldIndexEnum indexEnum;
            private BytesRef term = null;
            private final boolean useCache = true;
            private boolean bufferLocked = false;
            private boolean reused = false;
            private List<YandexTermState> blockTerms = null;
            private boolean end = false;

            ChainedBlockCompressedInputStream bytesReader = null;

            public ReverseSegmentTermsEnum() throws IOException {
                in = (IndexInput) YandexTermsReader.this.in.clone();
                in.seek(termsStartPointer);
                indexEnum = indexReader.getFieldEnum(fieldInfo);
                state = (YandexTermState)postingsReader.newTermState();
                state.totalTermFreq = -1;
                state.ord = -1;
            }

            @Override
            public void close() {
                if (bytesReader != null) {
                    try {
                        bytesReader.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                    bytesReader = null;
                }
            }

            public void reset() throws IOException {
                blockTerms.clear();
            }

            @Override
            public Comparator<BytesRef> getComparator() {
                return BytesRef.getUTF8SortedAsUnicodeComparator();
            }

            private void checkBytesReader() throws IOException {
                if (bytesReader != null) {
                    return;
                }
                bytesReader =
                    new ChainedBlockCompressedInputStream(
                        in,
                        decompressor,
                        allocator,
                        TERMS_TAG);
            }

            public boolean seekExact(
                final BytesRef text,
                final boolean useCache)
                throws IOException
            {
                if (bloomFilter != null
                    && bloomFilter.contains(text) == ContainsResult.NO)
                {
                    bloomSuccess.incrementAndGet();
                    return false;
                }
                final boolean found = seek(text, useCache) == SeekStatus.FOUND;
                if (!found) {
                    bloomFail.incrementAndGet();
                }
                return found;
            }

            // TODO: we may want an alternate mode here which is
            // "if you are about to return NOT_FOUND I won't use
            // the terms data from that"; eg FuzzyTermsEnum will
            // (usually) just immediately call seek again if we
            // return NOT_FOUND so it's a waste for us to fill in
            // the term that was actually NOT_FOUND
            @Override
            public SeekStatus seek(
                final BytesRef target,
                final boolean useCache1)
                throws IOException
            {
                if (indexEnum == null) {
                    throw new IllegalStateException("terms index was not loaded");
                }
                end = false;
                long blockOffset = indexEnum.seek(target); //CEIL
                boolean skipPrev = true;
                if (blockOffset == 0) {
                    //scan to the latest block ever
                    if ((blockOffset = indexEnum.prev()) == 0) {
                        while ((blockOffset = indexEnum.prev()) == 0) {
                        }
                        skipPrev = blockOffset == termsStartPointer;
                    }
                }
                term = new BytesRef(indexEnum.term());
                final Comparator<BytesRef> termComp =
                    BytesRef.getUTF8SortedAsUnicodeComparator();

                if (skipPrev) {
                    //first prev() after seek() will point to the current block
                    // so skip it if iteration will be required later
                    final long skipOffset = indexEnum.prev();
                    if (skipOffset == termsStartPointer && indexDivisor <= 1) {
                        //skip one more index position
                        //skip first empty entry;
                         indexEnum.prev();
                    }
                }

                checkBytesReader();
                bytesReader.seek(blockOffset, 0);
                state = new YandexTermState();
                state.bytesReader = bytesReader;
                state.term = term;
                state.termCount = 0;
                if (blockTerms == null) {
                    blockTerms = new ArrayList<>();
                } else {
                    blockTerms.clear();
                }
                boolean locked = lockBuffer();
                try {
                    while (!bytesReader.eof()) {
                        final int prefix = bytesReader.readVIntUnlocked();
                        final int suffix = bytesReader.readVIntUnlocked();
                        term.length = prefix + suffix;
                        if (term.bytes.length < term.length) {
                            term.grow(term.length);
                        }
                        bytesReader.readBytesUnlocked(term.bytes, prefix, suffix);

                        state.docFreq = bytesReader.readVIntUnlocked();
                        if (!fieldInfo.omitTermFreqAndPositions) {
                            state.totalTermFreq =
                                state.docFreq + bytesReader.readVLongUnlocked();
                        }
                        postingsReader.nextTerm(fieldInfo, state);

                        int cmp = termComp.compare(target, term);
                        if (cmp == 0) {
                            return SeekStatus.FOUND;
                        } else if (cmp < 0) {
                            if (blockTerms.size() > 0) {
                                state = blockTerms.remove(blockTerms.size() - 1);
                                term = state.term;
                                return SeekStatus.NOT_FOUND;
                            } else {
                                end = true;
                                return SeekStatus.END;
                            }
                        }

                        final YandexTermState clonedState = (YandexTermState) state.clone();
                        clonedState.term = new BytesRef(term);
                        blockTerms.add(clonedState);
                        state.termCount++;
                    }
                    if (blockTerms.size() > 0) {
                        state = blockTerms.remove(blockTerms.size() - 1);
                        term = state.term;
                        return SeekStatus.NOT_FOUND;
                    } else {
                        end = true;
                        return SeekStatus.END;
                    }
                } finally {
                    if (locked) {
                        releaseBuffer();
                    }
                }
            }

        private final boolean lockBuffer() throws IOException {
            if (bytesReader != null && !bufferLocked) {
                bytesReader.lockBuffer();
                bufferLocked = true;
                return true;
            } else {
                return false;
            }
        }

        private final void releaseBuffer() throws IOException {
            if (bufferLocked) {
                bufferLocked = false;
                bytesReader.releaseBuffer();
            }
        }

        private void loadPrevBlock() throws IOException {
            if (blockTerms == null) {
                indexEnum.seekEnd();
                //skip the latest element because it pointing to the 0, not the last block start
                final long offset = indexEnum.prev();
                if (offset != 0) {
                    indexEnum.seekEnd();
                }
                blockTerms = new ArrayList<>();
            }
            final long blockOffset = indexEnum.prev();
            if (blockOffset == -1) {
                end = true;
                return;
            }
            final BytesRef tempTerm = new BytesRef(indexEnum.term());
            if (blockOffset == termsStartPointer && indexDivisor <= 1) {
                //skip first empty entry;
                indexEnum.prev();
            }
            checkBytesReader();
            bytesReader.seek(blockOffset, 0);
            boolean locked = lockBuffer();
            try {
                final int blocks;
                if (blockOffset == termsStartPointer && indexDivisor > 1) {
                    //there are to index pointers at the start which are poining
                    //to the first block
                    //so after index division we should read one block less
                    //if this is a first index pointer
                    blocks = indexDivisor - 1;
                } else {
                    blocks = indexDivisor;
                }
                for (int b = 0; b < blocks; b++) {
                    final YandexTermState tempState = new YandexTermState();
                    tempState.bytesReader = bytesReader;
                    tempState.termCount = 0;
                    while (!bytesReader.blockEof()) {
                        final int prefix = bytesReader.readVIntUnlocked();
                        final int suffix = bytesReader.readVIntUnlocked();
                        tempTerm.length = prefix + suffix;
                        if (tempTerm.bytes.length < tempTerm.length) {
                            tempTerm.grow(tempTerm.length);
                        }
                        bytesReader.readBytesUnlocked(tempTerm.bytes, prefix, suffix);
                        tempState.docFreq = bytesReader.readVIntUnlocked();
                        if (!fieldInfo.omitTermFreqAndPositions) {
                            tempState.totalTermFreq =
                                tempState.docFreq + bytesReader.readVLongUnlocked();
                        }
                        postingsReader.nextTerm(fieldInfo, tempState);
                        final YandexTermState clonedState = (YandexTermState) tempState.clone();
                        clonedState.term = new BytesRef(tempTerm);
                        blockTerms.add(clonedState);
                        tempState.termCount++;
                    }
                    if (blocks > 1 && bytesReader.eof()) {
                        break;
                    }
                    if (bytesReader.isNextBlock()) {
                        bytesReader.clearNextBlock();
                    }
                }
            } finally {
                if (locked) {
                    releaseBuffer();
                }
            }
        }

        @Override
        public BytesRef next() throws IOException {
            if (end) {
                return null;
            }

            if (blockTerms == null || blockTerms.size() == 0) {
                loadPrevBlock();
                return next();
            }

            state = blockTerms.remove(blockTerms.size() - 1);
            term = state.term;
            return term;
        }

      /* Decodes only the term bytes of the next term.  If caller then asks for
         metadata, ie docFreq, totalTermFreq or pulls a D/&PEnum, we then (lazily)
         decode all metadata up to the current term. */

      @Override
      public BytesRef term() {
        return term;
      }

      @Override
      public int docFreq() throws IOException {
        //System.out.println("BTR.docFreq");
        //System.out.println("  return " + state.docFreq);
        return state.docFreq;
      }

      @Override
      public long totalTermFreq() throws IOException {
        return state.totalTermFreq;
      }

      @Override
      public DocsEnum docs(Bits skipDocs, DocsEnum reuse) throws IOException {
        //System.out.println("BTR.docs this=" + this);
        //System.out.println("  state.docFreq=" + state.docFreq);
        final DocsEnum docsEnum = postingsReader.docs(fieldInfo, state, skipDocs, reuse, useCache);
        assert docsEnum != null;
        return docsEnum;
      }

      @Override
      public DocsAndPositionsEnum docsAndPositions(Bits skipDocs, DocsAndPositionsEnum reuse) throws IOException {
        //System.out.println("BTR.d&p this=" + this);
        if (fieldInfo.omitTermFreqAndPositions) {
          return null;
        } else {
          DocsAndPositionsEnum dpe = postingsReader.docsAndPositions(fieldInfo, state, skipDocs, reuse, useCache);
          //System.out.println("  return d&pe=" + dpe);
          return dpe;
        }
      }

      @Override
      public void seek(BytesRef target, TermState otherState) throws IOException {
        //System.out.println("BTR.seek termState target=" + target.utf8ToString() + " " + target + " this=" + this);
        assert otherState != null && otherState instanceof YandexTermState;
        seek(target);
      }

      @Override
      public TermState termState() throws IOException {
        //System.out.println("BTR.termState this=" + this);
        TermState ts = (TermState) state.clone();
        //System.out.println("  return ts=" + ts);
        return ts;
      }

      @Override
      public SeekStatus seek(long ord) throws IOException {
        throw new UnsupportedOperationException();
      }

      @Override
      public long ord() {
          throw new UnsupportedOperationException();
      }

      private void doPendingSeek() {
      }

    }

  }
}
