package org.apache.lucene.index.codecs.fast_commit;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.Set;

import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.index.FieldsReader;
import org.apache.lucene.index.FieldsWriter;
import org.apache.lucene.index.SegmentInfo;
import org.apache.lucene.index.SegmentWriteState;
import org.apache.lucene.index.SegmentReadState;
import org.apache.lucene.index.StandardFieldsReader;
import org.apache.lucene.index.StandardFieldsWriter;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.codecs.Codec;
import org.apache.lucene.index.codecs.FieldsConsumer;
import org.apache.lucene.index.codecs.FieldsProducer;
import org.apache.lucene.index.codecs.fast_commit.FastCommitFieldsReader.FastCommitTermState;
import org.apache.lucene.store.Directory;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.DoubleBarrelLRUCache;

public class FastCommitCodec extends Codec {
  public final DoubleBarrelLRUCache<FieldAndTermAndReader,FastCommitTermState> termsCache = new DoubleBarrelLRUCache<FieldAndTermAndReader,FastCommitTermState>(1000);
  private final Set<String> bloomSet;

  public FastCommitCodec(Set<String> bloomSet) {
    name = "FastCommit";
    this.bloomSet = bloomSet;
  }

  @Override
  public FieldsReader fieldsReader(
    Directory d,
    String segment,
    FieldInfos fn,
    int readBufferSize)
    throws IOException
  {
    return new StandardFieldsReader(d, segment, fn, readBufferSize, -1, 0);
  }

  @Override
  public FieldsWriter fieldsWriter(
    final boolean merge,
    final Directory dir,
    final String segment,
    final FieldInfos fi)
    throws IOException
  {
    return new StandardFieldsWriter(dir, segment, fi);
  }

  @Override
  public FieldsConsumer fieldsConsumer(SegmentWriteState state) throws IOException {
    return new FastCommitFieldsWriter(state, bloomSet);
  }

  @Override
  public FieldsProducer fieldsProducer(SegmentReadState state) throws IOException {
    return new FastCommitFieldsReader(state, termsCache, bloomSet);
  }

  /** Extension of freq postings file */
  static final String POSTINGS_EXTENSION = "fcp";
  static final String POSTINGS_OFFSETS_EXTENSION = "fcpoff";

  static String getPostingsFileName(String segment, String id) {
    return IndexFileNames.segmentFileName(segment, id, POSTINGS_EXTENSION);
  }

  static String getPostingsOffsetsFileName(String segment, String id) {
    return IndexFileNames.segmentFileName(segment, id, POSTINGS_OFFSETS_EXTENSION);
  }

  @Override
  public void files(Directory dir, SegmentInfo segmentInfo, String id, Set<String> files) throws IOException {
    files.add(getPostingsFileName(segmentInfo.name, id));
    files.add(getPostingsOffsetsFileName(segmentInfo.name, id));
    if (dir.fileExists(IndexFileNames.segmentFileName(segmentInfo.name, id, FastCommitFieldsWriter.BLOOM_EXTENSION))) {
        files.add(IndexFileNames.segmentFileName(segmentInfo.name, id, FastCommitFieldsWriter.BLOOM_EXTENSION));
    }
  }

  @Override
  public void getExtensions(Set<String> extensions) {
    extensions.add(POSTINGS_EXTENSION);
    extensions.add(POSTINGS_OFFSETS_EXTENSION);
  }

  // Used as key for the terms cache
  public static class FieldAndTermAndReader extends DoubleBarrelLRUCache.CloneableKey {
    public FastCommitFieldsReader reader;
    public String field;
    public BytesRef term;

    public FieldAndTermAndReader() {
    }

    public FieldAndTermAndReader(FieldAndTermAndReader other) {
      field = other.field;
      term = new BytesRef(other.term);
      reader = other.reader;
    }

    @Override
    public boolean equals(Object _other) {
      FieldAndTermAndReader other = (FieldAndTermAndReader) _other;
      return other.reader == reader && other.field == field && term.bytesEquals(other.term);
    }

    @Override
    public Object clone() {
      return new FieldAndTermAndReader(this);
    }

    @Override
    public int hashCode() {
      return field.hashCode() * 31 + term.hashCode() + reader.hashCode();
    }
  }

}
