package org.apache.lucene.index.codecs.fast_commit;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.index.codecs.FieldsProducer;
import org.apache.lucene.index.codecs.bloom.FuzzySet; // javadocs
import org.apache.lucene.index.codecs.bloom.FuzzySet.ContainsResult; // javadocs
import org.apache.lucene.index.codecs.fast_commit.FastCommitCodec.FieldAndTermAndReader;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.SegmentReadState;
import org.apache.lucene.index.FieldsEnum;
import org.apache.lucene.index.TermState;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.index.TermsEnum.SeekStatus;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CodecUtil;
import org.apache.lucene.util.DoubleBarrelLRUCache;
import org.apache.lucene.util.StringHelper;
import org.apache.lucene.util.UnicodeUtil;
import org.apache.lucene.util.automaton.fst.Builder;
import org.apache.lucene.util.automaton.fst.BytesRefFSTEnum;
import org.apache.lucene.util.automaton.fst.FST;
import org.apache.lucene.util.automaton.fst.PositiveIntOutputs;
import org.apache.lucene.util.automaton.fst.PairOutputs;
import org.apache.lucene.util.packed.PackedInts;
//import org.apache.lucene.util.packed_native.PackedInts;
import java.io.Closeable;
import java.io.IOException;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;

import ru.yandex.msearch.util.JavaAllocator;

class FastCommitFieldsReader extends FieldsProducer {
    private static final JavaAllocator ALLOCATOR =
        JavaAllocator.get("FastCommitFieldsReader");
  private final IndexInput postingsIn;
  private final IndexInput offsetsIn;
  private final FieldInfos fieldInfos;
  private final HashMap<String,FastCommitFieldReader> fields = new HashMap<String,FastCommitFieldReader>();
  private final String segmentName;
  private boolean closed = false;
  private final DoubleBarrelLRUCache<FieldAndTermAndReader,FastCommitTermState> termsCache;
  private final HashMap<String,FuzzySet> bloomsByFieldName;

  public FastCommitFieldsReader(SegmentReadState state, DoubleBarrelLRUCache<FieldAndTermAndReader,FastCommitTermState> termsCache, Set<String> bloomSet) throws IOException {
    this.segmentName = state.segmentInfo.name;
    this.termsCache = termsCache;
    postingsIn = state.dir.openInput(FastCommitCodec.getPostingsFileName(state.segmentInfo.name, ""+state.codecId));
    offsetsIn = state.dir.openInput(FastCommitCodec.getPostingsOffsetsFileName(state.segmentInfo.name, ""+state.codecId));

    fieldInfos = state.fieldInfos;

      bloomsByFieldName = new HashMap<>();
      String bloomFileName = IndexFileNames.segmentFileName(
          state.segmentInfo.name, state.codecId, FastCommitFieldsWriter.BLOOM_EXTENSION);
      if (state.dir.fileExists(bloomFileName)) {
        try (IndexInput bloomIn = state.dir.openInput(bloomFileName)) {
            CodecUtil.checkHeader(bloomIn, FastCommitFieldsWriter.BLOOM_CODEC_NAME, FastCommitFieldsWriter.BLOOM_CODEC_VERSION,
            FastCommitFieldsWriter.BLOOM_CODEC_VERSION);
            // // Load the hash function used in the BloomFilter
            // hashFunction = HashFunction.forName(bloomIn.readString());
            // Load the delegate postings format
            int numBlooms = bloomIn.readInt();
            for (int i = 0; i < numBlooms; i++) {
                int fieldNum = bloomIn.readInt();
                final FieldInfo fieldInfo = fieldInfos.fieldInfo(fieldNum);
                FuzzySet bloom = FuzzySet.deserialize(bloomIn, fieldInfo.name);
                if (bloomSet.contains(fieldInfo.name)) {
                    bloomsByFieldName.put(fieldInfo.name, bloom);
                } else {
                    bloom.close();
                }
            }
        }
      }

    initFields();

  }

  private void initFields() throws IOException
  {
    int fieldsCount = offsetsIn.readVInt();
    for( int i = 0; i < fieldsCount; i++ )
    {
        String fieldName = StringHelper.intern(offsetsIn.readString());
        FieldInfo fi = fieldInfos.fieldInfo(fieldName);
        fields.put( fieldName, new FastCommitFieldReader(fi.name, fi.omitTermFreqAndPositions, fi.storePayloads, offsetsIn.readVLong(), bloomsByFieldName.get(fieldName)) );
    }
  }

  @Override
  public void loadTermsIndex( int indexDivisor )
  {
  }

  @Override
  public Terms terms( String field )
  {
    return fields.get(field);
  }

  @Override
  public FieldsEnum iterator()
  {
    return new FastCommitFieldsEnum();
  }

  @Override
  public final void close() throws IOException
  {
    if( closed ) return;
    closed = true;
    bloomsByFieldName.clear();
    for( FastCommitFieldReader reader : fields.values() )
    {
        reader.close();
    }
    fields.clear();
    offsetsIn.close();
    postingsIn.close();
    invalidateCache();
  }

  private final void invalidateCache() {
    if (termsCache == null) {
        return;
    }
    Iterator<FieldAndTermAndReader> keysIter = termsCache.keysIterator();
    while (keysIter.hasNext()) {
        FieldAndTermAndReader ftr = keysIter.next();
        if (ftr.reader == this) {
            keysIter.remove();
        }
    }
  }

  public static class FastCommitTermState extends TermState
  {
    public int currentIndex;
    public SeekStatus status;
    public void copyFrom(TermState o)
    {
        FastCommitTermState other = (FastCommitTermState)o;
        currentIndex = other.currentIndex;
        status = other.status;
    }
  }

  private class FastCommitFieldReader extends Terms implements Closeable
  {
    protected String fieldName;
    protected boolean omitTf;
    protected boolean hasPayloads;
    private long offset;
    private PackedInts.Reader index = null;
    private PackedInts.Reader docsFreqs = null;
    private PackedInts.Reader totalTermFreqs = null;
    private IndexInput postingsInClone;
    private long sumTotalTermFreq;
    private boolean initialized = false;
    final FuzzySet bloomFilter;

    public FastCommitFieldReader( String fieldName, boolean omitTf, boolean hasPayloads, long offset, FuzzySet bloomFilter ) throws IOException
    {
        this.fieldName = fieldName;
        this.omitTf = omitTf;
        this.hasPayloads = hasPayloads;
        this.offset = offset;
        this.bloomFilter = bloomFilter;
        this.postingsInClone = (IndexInput) FastCommitFieldsReader.this.postingsIn.clone();
        this.postingsInClone.seek(offset);
        sumTotalTermFreq = this.postingsInClone.readVLong();
//        init();
    }

    private synchronized final void init() throws IOException {
        if (initialized) return;
        boolean success = false;
        try {
            if (CodecUtil.hasHeader(this.postingsInClone)) {
                index = PackedInts.getReader(this.postingsInClone, ALLOCATOR);
                docsFreqs = PackedInts.getReader(this.postingsInClone, ALLOCATOR);
                if (!omitTf) {
                    totalTermFreqs = PackedInts.getReader(this.postingsInClone, ALLOCATOR);
                }
            } else {
                index = PackedInts.getReaderNoCodecHeader(this.postingsInClone, ALLOCATOR);
                docsFreqs =
                    PackedInts.getReaderNoCodecHeader(this.postingsInClone, ALLOCATOR);
                if (!omitTf) {
                    totalTermFreqs =
                        PackedInts.getReaderNoCodecHeader(this.postingsInClone, ALLOCATOR);
                }
            }
            success = true;
        } finally {
            if (!success) {
                freePacked();
            }
        }
        initialized = true;
    }

    @Override
    public Comparator<BytesRef> getComparator() {
      return BytesRef.getUTF8SortedAsUnicodeComparator();
    }

    @Override
    public long getSumTotalTermFreq()
    {
        return sumTotalTermFreq;
    }

    @Override
    public TermsEnum iterator(final boolean buffered) throws IOException
    {
        if( !initialized ) init();
        return new FastCommitTermsEnum( fieldName, index, docsFreqs, totalTermFreqs, omitTf, hasPayloads, termsCache, bloomFilter );
    }

    private final void freePacked() throws IOException {
        if (index != null) {
            index.close();
            index = null;
        }
        if (docsFreqs != null) {
            docsFreqs.close();
            docsFreqs = null;
        }
        if (totalTermFreqs != null) {
            totalTermFreqs.close();
            totalTermFreqs = null;
        }
    }

    @Override
    public synchronized void close() {
        try {
            //This should not cause IOException
            freePacked();
            postingsInClone.close();
        }
        catch( IOException ignore )
        {
        }
    }

  }

  private class FastCommitFieldsEnum extends FieldsEnum
  {
    private Iterator<String> iter;
    private String current;

    public FastCommitFieldsEnum()
    {
        iter = fields.keySet().iterator();
    }

    @Override
    public String next() throws IOException
    {
        if( iter.hasNext() ) current = iter.next();
        else current = null;
        return current;
    }

    @Override
    public TermsEnum terms(final boolean buffered) throws IOException
    {
      return fields.get(current).iterator(buffered);
    }
  }

  private class FastCommitTermsEnum extends TermsEnum
  {
    private final IndexInput in;
    private int docFreq;
    private long totalTermFreq;
    private final boolean omitTf;
    private final boolean hasPayloads;
    private final PackedInts.Reader index;
    private final PackedInts.Reader docsFreqs;
    private final PackedInts.Reader totalTermFreqs;
    private int currentIndex = 0;
    private long docsStart;
    private BytesRef currentTerm = null;
    private BytesRef loadTerm = new BytesRef();
    private final FieldAndTermAndReader fieldTerm = new FieldAndTermAndReader();
    private final String fieldName;
    private final DoubleBarrelLRUCache<FieldAndTermAndReader,FastCommitTermState> termsCache;
    private final FuzzySet bloomFilter;


    public FastCommitTermsEnum( String fieldName, PackedInts.Reader index, PackedInts.Reader docsFreqs,
        PackedInts.Reader totalTermFreqs, boolean omitTf, boolean hasPayloads,
        DoubleBarrelLRUCache<FieldAndTermAndReader,FastCommitTermState> termsCache, FuzzySet bloomFilter ) throws IOException
    {
      this.in = (IndexInput) FastCommitFieldsReader.this.postingsIn.clone();
      this.fieldName = fieldName;
      this.index = index;
      this.docsFreqs = docsFreqs;
      this.totalTermFreqs = totalTermFreqs;
      this.omitTf = omitTf;
      this.hasPayloads = hasPayloads;
      this.bloomFilter = bloomFilter;
      fieldTerm.field = fieldName;
      fieldTerm.reader = FastCommitFieldsReader.this;
      this.termsCache = termsCache;
    }

    @Override
    public void close() {
    }

    @Override
    public boolean seekExact(BytesRef text, boolean useCache) throws IOException {
        if (bloomFilter != null && bloomFilter.contains(text) == ContainsResult.NO) {
//          System.err.println("FastCommitCodec.bloomFilter reject: " + text.utf8ToString());
          return false;
        }
//          System.err.println("FastCommitCodec.bloomFilter accept: " + text.utf8ToString());
        return seek(text, useCache) == SeekStatus.FOUND;
    }

    @Override
    public SeekStatus seek(BytesRef text, boolean useCache /* ignored */) throws IOException
    {

      if( useCache )
      {
        fieldTerm.term = text;
        FastCommitTermState state = termsCache.get(fieldTerm);
        if( state != null )
        {
            currentIndex = state.currentIndex;
            if( state.status != SeekStatus.END )
            {
                next();
            }
            return state.status;
        }
      }

      int min = 0;
      int max = index.size() - 1;
      int needle = -1;
      int scans = 0;
//      System.err.println( "FastCommitTermsEnum.seek.start: min=" + min + ", max=" + max );
      while( max >= min )
      {
        scans++;
        int mid = (min + max) >>> 1;
        BytesRef pivot = loadTerm(mid);
        int delta = text.compareTo(pivot);
        if( delta < 0 )
        {
          max = mid - 1;
        }
        else if (delta > 0)
        {
          min = mid + 1;
        }
        else
        {
          needle = mid;
          break;
        }
      }
//      System.err.println( "FastCommitTermsEnum.seek.end: " + text.utf8ToString() + ", min=" + min + ", max=" + max + ", needle=" + needle + ", scans: " + scans);
      if( needle != -1 )
      {
        currentIndex = needle + 1;
        currentTerm = new BytesRef(loadTerm);
        if( useCache )
        {
            FastCommitTermState state = new FastCommitTermState();
            state.currentIndex = needle;
            state.status = SeekStatus.FOUND;
            termsCache.put((FieldAndTermAndReader)fieldTerm.clone(), state);
        }
        return SeekStatus.FOUND;
      }
      if( min > index.size() - 1 )
      {
        currentIndex = index.size();
        if( useCache )
        {
            FastCommitTermState state = new FastCommitTermState();
            state.currentIndex = index.size();
            state.status = SeekStatus.END;
            termsCache.put((FieldAndTermAndReader)fieldTerm.clone(), state);
        }
        return SeekStatus.END;
      }

      if( useCache )
      {
         FastCommitTermState state = new FastCommitTermState();
         state.currentIndex = max + 1;
         state.status = SeekStatus.NOT_FOUND;
         termsCache.put((FieldAndTermAndReader)fieldTerm.clone(), state);
      }

      currentIndex = max + 1;
      next();
      return SeekStatus.NOT_FOUND;
    }

    private BytesRef loadTerm( int indexPos ) throws IOException
    {
      in.seek( index.get(indexPos) );
      loadTerm.length = in.readVInt();
      if( loadTerm.bytes.length < loadTerm.length )
      {
        loadTerm.bytes = new byte[loadTerm.length * 2];
      }
      in.readBytes( loadTerm.bytes, 0, loadTerm.length );

      docFreq = (int)docsFreqs.get(indexPos);
      if( !omitTf ) totalTermFreq = totalTermFreqs.get(indexPos);
      docsStart = in.getFilePointer();

      return loadTerm;
    }

    @Override
    public BytesRef next() throws IOException {
      if( currentIndex >= index.size() )
      {
        currentTerm = null;
      }
      else
      {
        currentTerm = (BytesRef)loadTerm(currentIndex++);
      }
      return currentTerm;
    }

    @Override
    public BytesRef term() {
      return currentTerm;
    }

    @Override
    public long ord() throws IOException {
      throw new UnsupportedOperationException();
    }

    @Override
    public SeekStatus seek(long ord) {
      throw new UnsupportedOperationException();
    }

    @Override
    public int docFreq() {
      return docFreq;
    }

    @Override
    public long totalTermFreq() {
      return totalTermFreq;
    }
 
    @Override
    public DocsEnum docs(Bits skipDocs, DocsEnum reuse) throws IOException {
      FastCommitDocsEnum docsEnum;
      if (
        reuse != null
        && reuse instanceof FastCommitDocsEnum
        && ((FastCommitDocsEnum) reuse).canReuse(FastCommitFieldsReader.this.postingsIn)) {
        docsEnum = (FastCommitDocsEnum) reuse;
      } else {
        docsEnum = new FastCommitDocsEnum();
      }
      return docsEnum.reset(docsStart, skipDocs, omitTf, hasPayloads, docFreq);
    }

    @Override
    public DocsAndPositionsEnum docsAndPositions(Bits skipDocs, DocsAndPositionsEnum reuse) throws IOException {
      if (omitTf) {
        return null;
      }

      FastCommitDocsAndPositionsEnum docsAndPositionsEnum;
      if (
        reuse != null
        && reuse instanceof FastCommitDocsAndPositionsEnum
        && ((FastCommitDocsAndPositionsEnum) reuse).canReuse(FastCommitFieldsReader.this.postingsIn))
      {
        docsAndPositionsEnum = (FastCommitDocsAndPositionsEnum) reuse;
      } else {
        docsAndPositionsEnum = new FastCommitDocsAndPositionsEnum();
      }
      return docsAndPositionsEnum.reset(docsStart, skipDocs, hasPayloads, docFreq);
    }
    
    @Override
    public Comparator<BytesRef> getComparator() {
      return BytesRef.getUTF8SortedAsUnicodeComparator();
    }
  }

  private class FastCommitDocsEnum extends DocsEnum
  {
    private final IndexInput in;
    private boolean omitTF;
    private boolean hasPayloads;
    private int docID;
    private int tf;
    private int docFreq;
    private Bits skipDocs;
    private int currentPos = 0;
    
    public FastCommitDocsEnum()
    {
      this.in = (IndexInput) FastCommitFieldsReader.this.postingsIn.clone();
    }

    public boolean canReuse(IndexInput in)
    {
      return in == FastCommitFieldsReader.this.postingsIn;
    }

    public FastCommitDocsEnum reset(long fp, Bits skipDocs, boolean omitTF, boolean hasPayloads, int docFreq) throws IOException
    {
      this.skipDocs = skipDocs;
      in.seek(fp);
      this.omitTF = omitTF;
      this.docFreq = docFreq;
      this.hasPayloads = hasPayloads;
      this.docID = 0;
      if (omitTF) {
        tf = 1;
      }
      currentPos = 0;
      return this;
    }

    @Override
    public int docID()
    {
      return docID;
    }

    @Override
    public int freq()
    {
      return tf;
    }

    @Override
    public int nextDoc() throws IOException
    {
      while( currentPos < docFreq )
      {
        docID += in.readVInt();
        if( !omitTF )
        {
            tf = in.readVInt();
            //skip term's positions
            for( int i = 0; i < tf; i++ )
            {
            int code;
                code = in.readVInt();
                if( hasPayloads )
                {
                    if( (code & 1) != 0 )
                    {
                        int payLoadLength = in.readVInt();
                        in.seek(in.getFilePointer() + payLoadLength);
                    }
                }
            }
        }
        currentPos++;
        if( skipDocs == null || (skipDocs != null && !skipDocs.get(docID)) ) return docID;
      }
      return NO_MORE_DOCS;
    }

    @Override
    public int advance(int target) throws IOException {
      // Naive -- better to index skip data
      int id;
      do
      {
        id = nextDoc();
        if( id == NO_MORE_DOCS ) return NO_MORE_DOCS;
      }
      while( id < target );
      return docID;
    }
  }

  private class FastCommitDocsAndPositionsEnum extends DocsAndPositionsEnum {
    private final IndexInput in;
    private boolean hasPayloads;
    private int docID;
    private int tf;
    private Bits skipDocs;
    private BytesRef payload;
    private int docFreq;
    private int docPos = 0;
    private int freqPos = 0;
    private int pos = 0;
    private boolean posHasPayload;

    public FastCommitDocsAndPositionsEnum()
    {
      this.in = (IndexInput) FastCommitFieldsReader.this.postingsIn.clone();
    }

    public boolean canReuse(IndexInput in)
    {
      return in == FastCommitFieldsReader.this.postingsIn;
    }

    public FastCommitDocsAndPositionsEnum reset(long fp, Bits skipDocs, boolean hasPayloads, int docFreq) throws IOException
    {
      this.skipDocs = skipDocs;
      this.docFreq = docFreq;
      this.hasPayloads = hasPayloads;
      this.docID = 0;
      tf = 0;
      docPos = 0;
      freqPos = 0;
      in.seek(fp);
      return this;
    }

    @Override
    public int docID()
    {
      return docID;
    }

    @Override
    public int freq()
    {
      return tf;
    }

    @Override
    public int nextDoc() throws IOException
    {
      while( docPos < docFreq )
      {
        if( tf > freqPos ) //skip previous freqs
        {
            for( int i = freqPos; i < tf; i++ )
            {
                int code = in.readVInt();
                if( hasPayloads )
                {
                    if( (code & 1) != 0 )
                    {
                        int payloadLength = in.readVInt();
                        in.seek(in.getFilePointer() + payloadLength);
                    }
                }
            }
        }

        docID += in.readVInt();
        tf = in.readVInt();
        pos = 0;
        freqPos = 0;
        docPos++;
        if( skipDocs == null || (skipDocs != null && !skipDocs.get(docID)) ) return docID;
      }
      return NO_MORE_DOCS;
    }

    @Override
    public int advance(int target) throws IOException {
      // Naive -- better to index skip data
      int id;
      do
      {
        id = nextDoc();
        if( id == NO_MORE_DOCS ) return NO_MORE_DOCS;
      }
      while( id < target );
      return docID;
    }

    @Override
    public int nextPosition() throws IOException {
      if( freqPos >= tf ) return NO_MORE_DOCS;
      int code = in.readVInt();
      posHasPayload = false;
      if( hasPayloads )
      {
        pos += code >>> 1;
        if( (code & 1) != 0 )
        {
            posHasPayload = true;
            payload.length = in.readVInt();
            if( payload.length > payload.bytes.length )
            {
                payload.bytes = new byte[payload.length];
            }
            in.readBytes( payload.bytes, 0, payload.length );
        }
      }
      else
      {
        pos += code;
      }
      freqPos++;
      return pos;
    }

    @Override
    public BytesRef getPayload() {
      // Some tests rely on only being able to retrieve the
      // payload once
      return new BytesRef(payload);
    }

    @Override
    public boolean hasPayload() {
      return posHasPayload;
    }
  }

  static class TermData {
    public long docsStart;
    public int docFreq;

    public TermData(long docsStart, int docFreq) {
      this.docsStart = docsStart;
      this.docFreq = docFreq;
    }
  }

}
