package org.apache.lucene.index.codecs.fast_commit;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.CodecUtil;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.UnicodeUtil;
import org.apache.lucene.util.packed.PackedInts;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.codecs.FieldsConsumer;
import org.apache.lucene.index.codecs.TermsConsumer;
import org.apache.lucene.index.codecs.PostingsConsumer;
import org.apache.lucene.index.codecs.TermStats;
import org.apache.lucene.index.codecs.bloom.FuzzySet;
import org.apache.lucene.index.codecs.bloom.BloomFilterFactory;
import org.apache.lucene.index.codecs.bloom.DefaultBloomFilterFactory;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.SegmentWriteState;
import org.apache.lucene.store.IndexOutput;

class FastCommitFieldsWriter extends FieldsConsumer {
  public static final String BLOOM_CODEC_NAME = "BloomFilter";
  public static final int BLOOM_CODEC_VERSION = 1;
  
  /** Extension of Bloom Filters file */
  static final String BLOOM_EXTENSION = "blm";

  private final IndexOutput postingsOut;
  private final IndexOutput offsetsOut;
  protected final int OMIT_TF = 1;
  protected final int HAS_PAYLOADS = 2;
  private LinkedList<FieldAndOffset> fields;
  private final Set<String> bloomSet;
  private final SegmentWriteState state;
  private Map<FieldInfo,FuzzySet> bloomFilters = new HashMap<FieldInfo,FuzzySet>();
  BloomFilterFactory bloomFilterFactory = new DefaultBloomFilterFactory();

  private static class FieldAndOffset
  {
    public String fieldName;
    public long offset;
  }

  public FastCommitFieldsWriter(SegmentWriteState state, Set<String> bloomSet) throws IOException {
    this.state = state;
    this.bloomSet = bloomSet;
    final String postingsFileName = FastCommitCodec.getPostingsFileName(state.segmentName, state.codecId);
    final String offsetsFileName = FastCommitCodec.getPostingsOffsetsFileName(state.segmentName, state.codecId);
    postingsOut = state.directory.createOutput(postingsFileName);
    offsetsOut = state.directory.createOutput(offsetsFileName);
    fields = new LinkedList<FieldAndOffset>();
  }

  @Override
  public TermsConsumer addField(FieldInfo field) throws IOException {
    FuzzySet bloomFilter = null;
    if (bloomSet.contains(field.name)) {
      bloomFilter = bloomFilterFactory.getSetForField(state,field);
      if (bloomFilter != null) {
        bloomFilters.put(field, bloomFilter);
      }
    }
    return new FastCommitTermsWriter(field.name, field.omitTermFreqAndPositions, field.storePayloads, bloomFilter);
  }

  protected void addFieldAndOffset( String field, long offset )
  {
    FieldAndOffset fao = new FieldAndOffset();
    fao.fieldName = field;
    fao.offset = offset;
    fields.add( fao );
  }

  private interface DocFreqIncrementor
  {
    public void inc();
    public void reset();
  }

  private class FastCommitTermsWriter extends TermsConsumer implements DocFreqIncrementor {
    private final FastCommitPostingsWriter postingsWriter;
    private long[] offsets = new long[100];
    private long maxOffset = 0;
    private int[] docsFreqs = new int[100];
    private int maxDocFreq = 0;
    private long[] totalTermFreqs;
    private long maxTotalTermFreq = 0;
    private int termsCount = 0;
    protected boolean omitTf;
    protected boolean hasPayloads;
    protected int docFreq;
    private String fieldName;
    private final FuzzySet bloomFilter;

    public FastCommitTermsWriter( String fieldName, boolean omitTf, boolean hasPayloads, FuzzySet bloomFilter )
    {
        this.fieldName = fieldName;
        this.omitTf = omitTf;
        this.hasPayloads = hasPayloads;
        this.bloomFilter = bloomFilter;
        if( !omitTf ) totalTermFreqs = new long[100];
        postingsWriter = new FastCommitPostingsWriter( omitTf, hasPayloads, this );
    }

    @Override
    public void inc()
    {
      docFreq++;
    }

    @Override
    public void reset()
    {
      docFreq = 0;
    }

    @Override
    public PostingsConsumer startTerm(BytesRef term) throws IOException {
      if( termsCount >= offsets.length )
      {
        offsets = ArrayUtil.grow( offsets );
        docsFreqs = ArrayUtil.grow( docsFreqs );
        if( !omitTf )
        {
          totalTermFreqs = ArrayUtil.grow( totalTermFreqs );
        }
      }
      offsets[termsCount] = postingsOut.getFilePointer();
      if( maxOffset < offsets[termsCount] ) maxOffset = offsets[termsCount];
      postingsOut.writeVInt( term.length );
      postingsOut.writeBytes( term.bytes, term.offset, term.length );
//      System.err.println("FastCommitFieldsWriter.starTerm: " + term.utf8ToString());
      return postingsWriter.reset(term);
    }

    @Override
    public void finishTerm(BytesRef term, TermStats stats) throws IOException {
      if (stats.docFreq > 0 && bloomFilter != null) {
        bloomFilter.addValue(term);
      }

      docsFreqs[termsCount] = docFreq;
      if( maxDocFreq < docFreq ) maxDocFreq = docFreq;
      if( !omitTf )
      {
        totalTermFreqs[termsCount] = stats.totalTermFreq;
        if( maxTotalTermFreq < stats.totalTermFreq ) maxTotalTermFreq = stats.totalTermFreq;
      }
      termsCount++;
    }

    @Override
    public void finish(long sumTotalTermFreq) throws IOException {
//      offsetsOut.writeVLong(postingsOut.getFilePointer());
      addFieldAndOffset(fieldName, postingsOut.getFilePointer());
      postingsOut.writeVLong(sumTotalTermFreq);
      writePacked( postingsOut, offsets, termsCount, maxOffset );
      writePacked( postingsOut, docsFreqs, termsCount, maxDocFreq );
      if( !omitTf )
      {
        writePacked( postingsOut, totalTermFreqs, termsCount, maxTotalTermFreq );
      }
    }

    private void writePacked( IndexOutput out, int[] array, int count, int maxValue ) throws IOException
    {
      PackedInts.Writer writer = PackedInts.getWriterNoCodecHeader( out, count, PackedInts.bitsRequired(maxValue) );
      for( int i = 0; i < count; i++ )
      {
        writer.add( array[i] );
      }
      writer.finish();
    }

    private void writePacked( IndexOutput out, long[] array, int count, long maxValue ) throws IOException
    {
      PackedInts.Writer writer = PackedInts.getWriterNoCodecHeader( out, count, PackedInts.bitsRequired(maxValue) );
      for( int i = 0; i < count; i++ )
      {
        writer.add( array[i] );
      }
      writer.finish();
    }

    @Override
    public Comparator<BytesRef> getComparator() {
      return BytesRef.getUTF8SortedAsUnicodeComparator();
    }
  }

  private class FastCommitPostingsWriter extends PostingsConsumer {
    private BytesRef term;
    private boolean wroteTerm;
    private int prevDocId;
    private int prevPosition;
    private final boolean omitTf;
    private final boolean hasPayloads;
    private final DocFreqIncrementor docFreq;

    public FastCommitPostingsWriter( boolean omitTf, boolean hasPayloads, DocFreqIncrementor docFreq )
    {
        this.omitTf = omitTf;
        this.hasPayloads = hasPayloads;
        this.docFreq = docFreq;
    }

    @Override
    public void startDoc(int docId, int termDocFreq) throws IOException {
        postingsOut.writeVInt( docId - prevDocId );
        prevDocId = docId;
        if( !omitTf )
        {
            postingsOut.writeVInt(termDocFreq);
        }
        prevPosition = 0;
        docFreq.inc();
    }

    public PostingsConsumer reset(BytesRef term) {
      prevDocId = 0;
      docFreq.reset();
      return this;
    }

    @Override
    public void addPosition(int position, BytesRef payload) throws IOException {
      int code = position - prevPosition;
      prevPosition = position;
      if( hasPayloads )
      {
        if( payload != null ) code = (code << 1) | 1;
        postingsOut.writeVInt( code );
        if( payload != null )
        {
            postingsOut.writeVInt(payload.length);
            postingsOut.writeBytes(payload.bytes, payload.offset, payload.length);
        }
      }
      else
      {
        postingsOut.writeVInt( code );
      }
    }

    @Override
    public void finishDoc() {
    }
  }

  @Override
  public void close() throws IOException {
      List<Entry<FieldInfo,FuzzySet>> nonSaturatedBlooms = new ArrayList<Map.Entry<FieldInfo,FuzzySet>>();
      
      for (Entry<FieldInfo,FuzzySet> entry : bloomFilters.entrySet()) {
        FuzzySet bloomFilter = entry.getValue();
        if(!bloomFilterFactory.isSaturated(bloomFilter,entry.getKey())){          
          nonSaturatedBlooms.add(entry);
        }
      }
      String bloomFileName = IndexFileNames.segmentFileName(
          state.segmentName, state.codecId, BLOOM_EXTENSION);
      try (IndexOutput bloomOutput =
            state.directory.createOutput(bloomFileName))
      {
        CodecUtil.writeHeader(bloomOutput, BLOOM_CODEC_NAME,
            BLOOM_CODEC_VERSION);

        // First field in the output file is the number of fields+blooms saved
        bloomOutput.writeInt(nonSaturatedBlooms.size());
        for (Entry<FieldInfo,FuzzySet> entry : nonSaturatedBlooms) {
          FieldInfo fieldInfo = entry.getKey();
          FuzzySet bloomFilter = entry.getValue();
          bloomOutput.writeInt(fieldInfo.number);
          saveAppropriatelySizedBloomFilter(bloomOutput, bloomFilter, fieldInfo);
        }
      }
      //We are done with large bitsets so no need to keep them hanging around
      bloomFilters.clear(); 

    offsetsOut.writeVInt( fields.size() );
    for( FieldAndOffset fao : fields )
    {
      offsetsOut.writeString(fao.fieldName);
      offsetsOut.writeVLong(fao.offset);
    }
    postingsOut.close();
    offsetsOut.close();
  }

  private void saveAppropriatelySizedBloomFilter(IndexOutput bloomOutput,
    FuzzySet bloomFilter, FieldInfo fieldInfo) throws IOException {
      
    FuzzySet rightSizedSet = bloomFilterFactory.downsize(fieldInfo,
      bloomFilter);
    if (rightSizedSet == null) {
       rightSizedSet = bloomFilter;
    }
    rightSizedSet.serialize(bloomOutput);
  }
}
