package org.apache.lucene.index.codecs.yandex;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.DeflateFieldsReader;
import org.apache.lucene.index.DeflateFieldsWriter;
import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.index.FieldsReader;
import org.apache.lucene.index.FieldsWriter;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.apache.lucene.index.SegmentInfo;
import org.apache.lucene.index.SegmentWriteState;
import org.apache.lucene.index.SegmentReadState;
import org.apache.lucene.index.StandardFieldsReader;
import org.apache.lucene.index.StandardFieldsWriter;
import org.apache.lucene.index.codecs.Codec;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.index.codecs.FieldsConsumer;
import org.apache.lucene.index.codecs.FieldsProducer;
import org.apache.lucene.index.codecs.PostingsWriterBase;
import org.apache.lucene.index.codecs.PostingsReaderBase;
import org.apache.lucene.index.codecs.TermsIndexWriterBase;
import org.apache.lucene.index.codecs.TermsIndexReaderBase;
import org.apache.lucene.index.codecs.VariableGapTermsIndexWriter;
import org.apache.lucene.index.codecs.VariableGapTermsIndexWriter.IndexTermSelector;
import org.apache.lucene.index.codecs.VariableGapTermsIndexReader;
import org.apache.lucene.index.codecs.YandexTermsWriter;
import org.apache.lucene.index.codecs.YandexTermsReader;
import org.apache.lucene.index.codecs.TermStats;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.DeflateCompressor;
import org.apache.lucene.store.Directory;
import ru.yandex.msearch.Config;

/** Default codec. 
 *  @lucene.experimental */
public class YandexCodec extends Codec {
  public static final int DEFAULT_FIELD_INDEX_READ_BUFFER_SIZE = 0;
  public static final boolean DEFAULT_IN_MEMORY_FIELDS_INDEX = false;
  private static final Compressor DEFAULT_COMPRESSOR =
    new DeflateCompressor();
  private static final int DEFAULT_FIELDS_WRITER_BLOCK_SIZE = 6144;
  private static final int DEFAULT_TERMS_WRITER_BLOCK_SIZE = 4096;
  private static final Set<String> DEFAULT_GROUP_FIELDS =
    Collections.singleton(Config.PREFIX_FIELD_KEY);
  private static final int DEFAULT_POSTINGS_BLOCK_SIZE =
    YandexPostingsWriter.DEFAULT_BLOCK_SIZE;
  private static final boolean DEFAULT_USE_STANDARD_FIELDS_WRITER = true;

  private static final String CONFIG_FIELDS_BLOCKS_SIZE = "fbs";
  private static final String CONFIG_TERMS_BLOCK_SIZE = "tbs";
  private static final String CONFIG_POSTINGS_BLOCK_SIZE = "pbs";
  private static final String CONFIG_GROUP_FIELDS = "gf";
  private static final String CONFIG_BLOOM_FIELDS = "bf";

  private final int fieldsWriterBlockSize;
  private final int termsWriterBlockSize;
  private final int postingsBlockSize;
  private final Set<String> groupFields;
  private final Set<String> bloomSet;
  private final Map<String, Integer> indexDivisors;
  private final Compressor compressor;

  private boolean inMemoryIndex = DEFAULT_IN_MEMORY_FIELDS_INDEX;
  private int fieldIndexReadBufferSize = DEFAULT_FIELD_INDEX_READ_BUFFER_SIZE;
  private boolean useStandardFieldsWriter = DEFAULT_USE_STANDARD_FIELDS_WRITER;

  private static final int DEFLATE_VERSION = 0;
  private static final int LZ4_VERSION = 1;

  public YandexCodec(
    final Compressor compressor,
    final int fieldsWriterBlockSize,
    final int termsWriterBlockSize,
    final int postingsBlockSize,
    final Set<String> groupFields,
    final Set<String> bloomSet,
    final Map<String, Integer> indexDivisors)
  {
    this.compressor = compressor;
    name = "Yandex";
    if (compressor.id() != null) { //back compat
        name += "_" + compressor.id();
    }
    this.fieldsWriterBlockSize = fieldsWriterBlockSize;
    this.termsWriterBlockSize = termsWriterBlockSize;
    this.postingsBlockSize = postingsBlockSize;
    this.groupFields = internFields(groupFields);
    this.bloomSet = bloomSet;
    this.indexDivisors = indexDivisors;
    name += CodecProvider.SETTINGS_SEPARATOR
        + CONFIG_FIELDS_BLOCKS_SIZE + ':' + fieldsWriterBlockSize + ';'
        + CONFIG_TERMS_BLOCK_SIZE + ':' + termsWriterBlockSize + ';'
        + CONFIG_POSTINGS_BLOCK_SIZE + ':' + postingsBlockSize + ';'
        + CONFIG_GROUP_FIELDS + ':' + groupFields + ';'
        + CONFIG_BLOOM_FIELDS + bloomSet;
  }

  public YandexCodec(
    final Compressor compressor,
    final int fieldsWriterBlockSize,
    final int termsWriterBlockSize,
    final int postingsBlockSize,
    final Set<String> groupFields,
    final Set<String> bloomSet)
  {
    this(
        compressor,
        fieldsWriterBlockSize,
        termsWriterBlockSize,
        postingsBlockSize,
        groupFields,
        bloomSet,
        Collections.<String, Integer>emptyMap());
  }

  public YandexCodec(final Compressor compressor) {
    this(
        compressor,
        DEFAULT_FIELDS_WRITER_BLOCK_SIZE,
        DEFAULT_TERMS_WRITER_BLOCK_SIZE,
        DEFAULT_POSTINGS_BLOCK_SIZE,
        DEFAULT_GROUP_FIELDS,
        new HashSet(),
        Collections.<String, Integer>emptyMap());
  }

  /* TODO: maybe also introduce separate constructors for the values, or a builder class*/
  public YandexCodec() {
    this(
        DEFAULT_COMPRESSOR,
        DEFAULT_FIELDS_WRITER_BLOCK_SIZE,
        DEFAULT_TERMS_WRITER_BLOCK_SIZE,
        DEFAULT_POSTINGS_BLOCK_SIZE,
        DEFAULT_GROUP_FIELDS,
        new HashSet(),
        Collections.<String, Integer>emptyMap());
  }

  public Codec cloneByName(final String name) {
    //TODO: parse params
    YandexCodec clone = new YandexCodec(
        compressor,
        fieldsWriterBlockSize,
        termsWriterBlockSize,
        postingsBlockSize,
        groupFields,
        bloomSet,
        indexDivisors);
    clone.name = name;
    return clone;
  }

  public static final class BlockStartSelector extends IndexTermSelector {

    public BlockStartSelector() {
    }

    @Override
    public boolean isIndexTerm(BytesRef term, TermStats stats) {
	return false;
    }

    @Override
    public void newField(FieldInfo fieldInfo) {
    }
  }

  public YandexCodec setFieldIndexReadBufferSize(final int size) {
    this.fieldIndexReadBufferSize = size;
    return this;
  }

  public YandexCodec useInMemoryFieldsIndex(final boolean use) {
    this.inMemoryIndex = use;
    return this;
  }

  public YandexCodec useStandardFieldsWriter(final boolean use) {
    this.useStandardFieldsWriter = use;
    return this;
  }

  @Override
  public FieldsReader fieldsReader(
    Directory d,
    String segment,
    FieldInfos fn,
    int readBufferSize)
    throws IOException
  {
    final int fdxReadBufferSize;
    if (fieldIndexReadBufferSize == 0) {
        fdxReadBufferSize = readBufferSize;
    } else {
        fdxReadBufferSize = fieldIndexReadBufferSize;
    }
    try {
        return new DeflateFieldsReader(
            compressor.decompressor(),
            d,
            segment,
            fn,
            fdxReadBufferSize,
            inMemoryIndex);
    } catch (CorruptIndexException e) {
        try {
            return new StandardFieldsReader(
                d,
                segment,
                fn,
                fdxReadBufferSize,
                -1,
                0);
        } catch (Exception ex) {
            e.addSuppressed(ex);
            throw e;
        }
    }
  }

  @Override
  public FieldsWriter fieldsWriter(
    final boolean merge,
    final Directory dir,
    final String segment,
    final FieldInfos fi)
    throws IOException
  {
    if (!merge && useStandardFieldsWriter) {
        return new StandardFieldsWriter(
            dir,
            segment,
            fi);
    } else {
        return new DeflateFieldsWriter(
            compressor,
            dir,
            segment,
            fi,
            fieldsWriterBlockSize);
    }
  }


  @Override
  public FieldsConsumer fieldsConsumer(SegmentWriteState state) throws IOException {
    PostingsWriterBase docs = new YandexPostingsWriter(compressor, postingsBlockSize, state);

    // TODO: should we make the terms index more easily
    // pluggable?  Ie so that this codec would record which
    // index impl was used, and switch on loading?
    // Or... you must make a new Codec for this?
    TermsIndexWriterBase indexWriter;
    boolean success = false;
    try {
      indexWriter = new VariableGapTermsIndexWriter(state, new BlockStartSelector());
//      indexWriter = new VariableGapTermsIndexWriter(state, new VariableGapTermsIndexWriter.EveryNTermSelector(state.termIndexInterval));
      success = true;
    } finally {
      if (!success) {
        docs.close();
      }
    }

    success = false;
    try {
      FieldsConsumer ret = new YandexTermsWriter(compressor, indexWriter, state, docs, termsWriterBlockSize, bloomSet);
      success = true;
      return ret;
    } finally {
      if (!success) {
        try {
          docs.close();
        } finally {
          indexWriter.close();
        }
      }
    }
  }

  public final static int TERMS_CACHE_SIZE = -1; //auto

  @Override
  public FieldsProducer fieldsProducer(SegmentReadState state) throws IOException {
    PostingsReaderBase postings = new YandexPostingsReader(compressor, state.dir, state.segmentInfo, state.readBufferSize, state.codecId);
    TermsIndexReaderBase indexReader;

    boolean success = false;
    try {
      indexReader = new VariableGapTermsIndexReader(state.dir,
                                                    state.fieldInfos,
                                                    state.segmentInfo.name,
                                                    state.termsIndexDivisor,
                                                    state.codecId,
                                                    null,
                                                    indexDivisors);
      success = true;
    } finally {
      if (!success) {
        postings.close();
      }
    }

    success = false;
    try {
      FieldsProducer ret = new YandexTermsReader(
                                                compressor,
                                                indexReader,
                                                state.dir,
                                                state.fieldInfos,
                                                state.segmentInfo.name,
                                                postings,
                                                state.readBufferSize,
                                                TERMS_CACHE_SIZE,
                                                state.codecId,
                                                bloomSet);
      success = true;
      return ret;
    } finally {
      if (!success) {
        try {
          postings.close();
        } finally {
          indexReader.close();
        }
      }
    }
  }

  /** Extension of freq postings file */
  static final String FREQ_EXTENSION = "frqy";

  /** Extension of prox postings file */
  static final String PROX_EXTENSION = "prxy";

  static final String INDEX_OFFSET_EXTENSION = "fdxo";

  @Override
  public void files(Directory dir, SegmentInfo segmentInfo, String id, Set<String> files) throws IOException {
    YandexPostingsReader.files(dir, segmentInfo, id, files);
    YandexTermsReader.files(dir, segmentInfo, id, files);
    VariableGapTermsIndexReader.files(dir, segmentInfo, id, files);
  }

  @Override
  public void getExtensions(Set<String> extensions) {
    getStandardExtensions(extensions);
  }

  public static void getStandardExtensions(Set<String> extensions) {
    extensions.add(FREQ_EXTENSION);
    extensions.add(PROX_EXTENSION);
    extensions.add(INDEX_OFFSET_EXTENSION);
    extensions.add(IndexFileNames.FIELDS_INDEX_EXTENSION_1ST);
    extensions.add(IndexFileNames.FIELDS_INDEX_EXTENSION_2ST);
    YandexTermsReader.getExtensions(extensions);
    VariableGapTermsIndexReader.getIndexExtensions(extensions);
  }
  
  @Override
  public boolean groupFieldsWriter() {
    return true;
  }
  
  @Override
  public Set<String> getGroupFields() {
    return groupFields;
  }

    @Override
    public String toString() {
        return name + "_group(" + groupFields + ")_bloom(" + bloomSet + ')';
    }
  
}
