package org.apache.lucene.index.codecs.yandex;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.Collection;

import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.SegmentInfo;
import org.apache.lucene.index.TermState;
import org.apache.lucene.index.codecs.BlockTermState;
import org.apache.lucene.index.codecs.PostingsReaderBase;
import org.apache.lucene.index.TermsEnum.SeekStatus;
import org.apache.lucene.store.ByteArrayDataInput;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.ChainedBlockCompressedInputStream;
import org.apache.lucene.store.DeflateDecompressor;
import org.apache.lucene.store.Decompressor;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.Lockable;
import org.apache.lucene.store.MutableBlockCompressedInputStream;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CodecUtil;

import ru.yandex.msearch.util.JavaAllocator;

/** Concrete class that reads the current doc/freq/skip
 *  postings format. 
 *  @lucene.experimental */

public class YandexPostingsReader extends PostingsReaderBase {
    private static final JavaAllocator ALLOCATOR =
        JavaAllocator.get("DeflateYandexPostingsReaderBlockCache");
//    private static final Decompressor decompressor = DeflateDecompressor.INSTANCE;
    private static final String FREQ_TAG = "doc-freqs";
    private static final String PROX_TAG = "doc-positions";
    private static final String SKIPS_TAG = "skip-table";

  private final IndexInput freqIn;
  private final IndexInput proxIn;
  private final Compressor compressor;
  private final Decompressor decompressor;

  private int blockSize;

  int skipInterval;
  int maxSkipLevels;
  int skipMinimum;

  //private String segment;

  public YandexPostingsReader(
    final Compressor compressor,
    Directory dir,
    SegmentInfo segmentInfo,
    int readBufferSize,
    String codecId)
    throws IOException
  {
    this.compressor = compressor;
    this.decompressor = compressor.decompressor();
    freqIn = dir.openInput(IndexFileNames.segmentFileName(segmentInfo.name, codecId, YandexCodec.FREQ_EXTENSION),
                           readBufferSize);
    //this.segment = segmentInfo.name;
    if (segmentInfo.getHasProx()) {
      boolean success = false;
      try {
        proxIn = dir.openInput(IndexFileNames.segmentFileName(segmentInfo.name, codecId, YandexCodec.PROX_EXTENSION),
                               readBufferSize);
        success = true;
      } finally {
        if (!success) {
          freqIn.close();
        }
      }
    } else {
      proxIn = null;
    }
  }

  public static void files(Directory dir, SegmentInfo segmentInfo, String id, Collection<String> files) throws IOException {
    files.add(IndexFileNames.segmentFileName(segmentInfo.name, id, YandexCodec.FREQ_EXTENSION));
    if (segmentInfo.getHasProx()) {
      files.add(IndexFileNames.segmentFileName(segmentInfo.name, id, YandexCodec.PROX_EXTENSION));
    }
  }

  private String codecName() {
    if (compressor.id() != null) {
        return YandexPostingsWriter.CODEC + "_" + compressor.id();
    } else {
        return YandexPostingsWriter.CODEC;
    }
  }

  @Override
  public void init(IndexInput termsIn) throws IOException {

    // Make sure we are talking to the matching past writer
    int version = CodecUtil.checkHeader(termsIn, codecName(),
      YandexPostingsWriter.VERSION_START, Integer.MAX_VALUE);
    if (version > 0) {
        blockSize = version;
    } else {
        blockSize = YandexPostingsWriter.DEFAULT_BLOCK_SIZE;
    }

    skipInterval = termsIn.readInt();
    maxSkipLevels = termsIn.readInt();
    skipMinimum = termsIn.readInt();
  }

  // Must keep final because we do non-standard clone
  public final static class YandexTermState extends BlockTermState {
    public long freqBlockStart;
    public long freqBlockOffset;
    long proxBlockStart;
    long proxBlockOffset;
    long skipBlockStart;
    long skipBlockOffset;
    public boolean fullTerm;

    public ChainedBlockCompressedInputStream bytesReader;
    public long termPointer;
    public BytesRef term;
    public BytesRef blockFirstTerm;

    public SeekStatus seekStatus;


    @Override
    public YandexTermState clone() {
      YandexTermState other = new YandexTermState();
      other.copyFrom(this);
      return other;
    }

    @Override
    public void copyFrom(TermState _other) {
      super.copyFrom(_other);
      YandexTermState other = (YandexTermState) _other;
      freqBlockStart = other.freqBlockStart;
      freqBlockOffset = other.freqBlockOffset;
      proxBlockStart = other.proxBlockStart;
      proxBlockOffset = other.proxBlockOffset;
      skipBlockStart = other.skipBlockStart;
      skipBlockOffset = other.skipBlockOffset;
//      bytesReader = other.bytesReader
      term = other.term;
      seekStatus = other.seekStatus;
      termPointer = other.termPointer;
      fullTerm = other.fullTerm;
      blockFirstTerm = other.blockFirstTerm;
      blockTermCount = other.blockTermCount;

      // Do not copy bytes, bytesReader (else TermState is
      // very heavy, ie drags around the entire block's
      // byte[]).  On seek back, if next() is in fact used
      // (rare!), they will be re-read from disk.
    }

    @Override
    public String toString() {
      return super.toString()
        + " freqFP=" + freqBlockStart
        + " proxFP=" + proxBlockStart
        + " skipOffset=" + skipBlockStart
        + " term=" + (term != null ? term.utf8ToString() : "null")
        + " blockFirstTerm=" + blockFirstTerm.utf8ToString()
        + " blockTermCount=" + blockTermCount
        + " termPointer=" + termPointer
        + " seekStatus=" + seekStatus
        + " fullTerm=" + fullTerm
        + " blockFirstTerm= " + blockFirstTerm.utf8ToString()
        + " blockTermCount= " + blockTermCount;
    }
  }

  @Override
  public BlockTermState newTermState() {
    return new YandexTermState();
  }

  @Override
  public void close() throws IOException {
    try {
      if (freqIn != null) {
        freqIn.close();
      }
    } finally {
      if (proxIn != null) {
        proxIn.close();
      }
    }
  }

  /* Reads but does not decode the byte[] blob holding
     metadata for the current terms block */
  @Override
  public void readTermsBlock(IndexInput termsIn, FieldInfo fieldInfo, BlockTermState _termState) throws IOException {
    final YandexTermState termState = (YandexTermState) _termState;
/*
    final int len = termsIn.readVInt();
    //System.out.println("SPR.readTermsBlock termsIn.fp=" + termsIn.getFilePointer());
    if (termState.bytes == null) {
      termState.bytes = new byte[ArrayUtil.oversize(len, 1)];
      termState.bytesReader = new ByteArrayDataInput(null);
    } else if (termState.bytes.length < len) {
      termState.bytes = new byte[ArrayUtil.oversize(len, 1)];
    }

    termsIn.readBytes(termState.bytes, 0, len);
    termState.bytesReader.reset(termState.bytes, 0, len);
*/
    termState.bytesReader = (ChainedBlockCompressedInputStream)termsIn;
  }

  @Override
  public void nextTerm(FieldInfo fieldInfo, BlockTermState _termState)
    throws IOException {
    final YandexTermState termState = (YandexTermState) _termState;
    //System.out.println("StandardR.nextTerm seg=" + segment);
    final boolean isFirstTerm = termState.termCount == 0 || termState.fullTerm;
//    final boolean isFirstTerm = termState.freqBlockStart == 0
//        || termState.termCount == 0;

//    System.err.println( "YandexPostingsReader.nextTerm: isFirstTerm=" + isFirstTerm );
    if (isFirstTerm) {
      termState.freqBlockStart = termState.bytesReader.readVLongUnlocked();
      termState.freqBlockOffset = termState.bytesReader.readVLongUnlocked();
    } else {
      long newBlockStart = termState.freqBlockStart + termState.bytesReader.readVLongUnlocked();
      if( termState.freqBlockStart == newBlockStart )
      {
        termState.freqBlockOffset += termState.bytesReader.readVLongUnlocked();
      }
      else
      {
        termState.freqBlockStart = newBlockStart;
        termState.freqBlockOffset = termState.bytesReader.readVLongUnlocked();
      }
    }
//    System.err.println("  dF=" + termState.docFreq);
//    System.err.println("  freqFP=" + termState.freqBlockStart + " / " + termState.freqBlockOffset + " (" + freqIn.length() + ")" );
    assert termState.freqBlockStart < freqIn.length();

    if (termState.docFreq >= skipMinimum) {
      termState.skipBlockStart = termState.freqBlockStart + termState.bytesReader.readVIntUnlocked();
      if( termState.skipBlockStart == termState.freqBlockStart )
      {
        termState.skipBlockOffset = (termState.freqBlockOffset + (termState.bytesReader.readVIntUnlocked() >> 1)) << 1;
      }
      else
      {
        termState.skipBlockOffset = termState.bytesReader.readVIntUnlocked();
      }
//      System.out.println("  skipOffset=" + termState.skipOffset + " vs freqIn.length=" + freqIn.length());
      assert termState.freqBlockStart < freqIn.length();
    } else {
      // undefined
    }

    if (!fieldInfo.omitTermFreqAndPositions) {
	if (isFirstTerm) {
    	    termState.proxBlockStart = termState.bytesReader.readVLongUnlocked();
    	    termState.proxBlockOffset = termState.bytesReader.readVLongUnlocked();
	} else {
    	    long newBlockStart = termState.proxBlockStart + termState.bytesReader.readVLongUnlocked();
    	    if( termState.proxBlockStart == newBlockStart )
    	    {
    	       termState.proxBlockOffset += termState.bytesReader.readVLongUnlocked();
    	    }
    	    else
    	    {
        	termState.proxBlockStart = newBlockStart;
        	termState.proxBlockOffset = termState.bytesReader.readVLongUnlocked();
    	    }
	}
//      System.out.println("  proxFP=" + termState.proxBlockStart + " / " + termState.proxBlockOffset);
    }
  }
    
  @Override
  public DocsEnum docs(FieldInfo fieldInfo, BlockTermState termState, Bits skipDocs, DocsEnum reuse, final boolean useCache) throws IOException {
    SegmentDocsEnum docsEnum;
    if (reuse == null || !(reuse instanceof SegmentDocsEnum)) {
      docsEnum = new SegmentDocsEnum(freqIn);
    } else {
      docsEnum = (SegmentDocsEnum) reuse;
      if (docsEnum.startFreqIn != freqIn) {
        // If you are using ParellelReader, and pass in a
        // reused DocsEnum, it could have come from another
        // reader also using standard codec
        System.err.println( "CANT RESET NEW ENUM" );
        docsEnum = new SegmentDocsEnum(freqIn);
      }
    }
    return docsEnum.reset(fieldInfo, (YandexTermState) termState, skipDocs, useCache);
  }

  @Override
  public DocsAndPositionsEnum docsAndPositions(FieldInfo fieldInfo, BlockTermState termState, Bits skipDocs, DocsAndPositionsEnum reuse, final boolean useCache) throws IOException {
    if (fieldInfo.omitTermFreqAndPositions) {
      return null;
    }
    
    // TODO: refactor
    if (fieldInfo.storePayloads) {
      SegmentDocsAndPositionsAndPayloadsEnum docsEnum;
      if (reuse == null || !(reuse instanceof SegmentDocsAndPositionsAndPayloadsEnum)) {
        docsEnum = new SegmentDocsAndPositionsAndPayloadsEnum(freqIn, proxIn);
      } else {
        docsEnum = (SegmentDocsAndPositionsAndPayloadsEnum) reuse;
        if (docsEnum.startFreqIn != freqIn) {
          // If you are using ParellelReader, and pass in a
          // reused DocsEnum, it could have come from another
          // reader also using standard codec
//        System.err.println( "CANT RESET NEW ENUM" );
          docsEnum = new SegmentDocsAndPositionsAndPayloadsEnum(freqIn, proxIn);
        }
      }
      return docsEnum.reset(fieldInfo, (YandexTermState) termState, skipDocs);
    } else {
      SegmentDocsAndPositionsEnum docsEnum;
      if (reuse == null || !(reuse instanceof SegmentDocsAndPositionsEnum)) {
        docsEnum = new SegmentDocsAndPositionsEnum(freqIn, proxIn);
//        System.err.println( "CANT RESET NEW ENUM: reuse=" + reuse  );
//        throw new IOException( "cant reset" );
      } else {
        docsEnum = (SegmentDocsAndPositionsEnum) reuse;
        if (docsEnum.startFreqIn != freqIn) {
          // If you are using ParellelReader, and pass in a
          // reused DocsEnum, it could have come from another
          // reader also using standard codec
        System.err.println( "CANT RESET NEW ENUM" );
          docsEnum = new SegmentDocsAndPositionsEnum(freqIn, proxIn);
        }
      }
      return docsEnum.reset(fieldInfo, (YandexTermState) termState, skipDocs, useCache);
    }
  }

  // Decodes only docs
  private class SegmentDocsEnum extends DocsEnum implements Lockable {
    final IndexInput freqIn;
    final IndexInput startFreqIn;

    boolean omitTF;                               // does current field omit term freq?
    boolean storePayloads;                        // does current field store payloads?

    int limit;                                    // number of docs in this posting
    int ord;                                      // how many docs we've read
    int doc;                                      // doc we last read
    int freq;                                     // freq we last read

    Bits skipDocs;

    long freqBlockStart;
    long freqBlockOffset;
    long skipBlockStart;
    long skipBlockOffset;
    
    long currentFreqBlockStart = -1;

    boolean skipped;
    boolean bufferLocked;
    YandexSkipListReader skipper;
    MutableBlockCompressedInputStream skipReader = null;


    MutableBlockCompressedInputStream bytesReader = null; 

    public SegmentDocsEnum(IndexInput freqIn) throws IOException {
        startFreqIn = freqIn;
        this.freqIn = (IndexInput) freqIn.clone();
        this.bytesReader = new MutableBlockCompressedInputStream(
            this.freqIn,
            compressor.decompressor(),
            ALLOCATOR,
            FREQ_TAG);
    }

    public void free() {
        if (skipReader != null) {
            try {
                skipReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            skipReader = null;
        }
        if (bytesReader != null) {
            try {
                bytesReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            bytesReader = null;
        }
//        System.err.println("Freed");
    }

    public SegmentDocsEnum reset(FieldInfo fieldInfo, YandexTermState termState, Bits skipDocs, final boolean useCache) throws IOException {
      bytesReader.useCache(useCache);
      omitTF = fieldInfo.omitTermFreqAndPositions;
      if (omitTF) {
        freq = 1;
      }
      storePayloads = fieldInfo.storePayloads;
      this.skipDocs = skipDocs;
      freqBlockStart = termState.freqBlockStart;
      freqBlockOffset = termState.freqBlockOffset;
      skipBlockStart = termState.skipBlockStart;
      skipBlockOffset = termState.skipBlockOffset;
//      System.err.println("Termstate: " + termState);

//      skip

      // TODO: for full enum case (eg segment merging) this
      // seek is unnecessary; maybe we can avoid in such
      // cases
//      System.err.println( this + " SegmentDocsEnum.reset: freqBlockStart=" + freqBlockStart + ", freqBlockOffset=" + freqBlockOffset );
      
//      freqIn.seek(termState.freqBlockStart);
//      loadBlock();
//      bytesReader.inflateFrom( freqIn );
//      checkFreqBytesReader();
//      bytesReader.seek( freqBlockStart, 0 );
//      bytesReader.readVInt();
//      bytesReader = new MutableBlockCompressedInputStream( this.freqIn );
      if (freqBlockStart < 0) {
        System.err.println("YandexPostingsReader: freqBlockStart=" + freqBlockStart + ", freqBlockOffset=" + freqBlockOffset);
      }
//      System.err.println("br.seek: " + freqBlockStart + ", " + freqBlockOffset);
      bytesReader.seek( freqBlockStart, freqBlockOffset );
      limit = termState.docFreq;
      assert limit > 0;
      ord = 0;
      doc = 0;
      //System.out.println("  sde limit=" + limit + " freqFP=" + freqOffset);

      skipped = false;

      return this;
    }

    @Override
    public final boolean lockBuffer() throws IOException {
        if (bytesReader != null && !bufferLocked) {
            bytesReader.lockBuffer();
            bufferLocked = true;
            return true;
        } else {
            return false;
        }
    }

    @Override
    public final void releaseBuffer() throws IOException {
        if (bufferLocked) {
            bufferLocked = false;
            bytesReader.releaseBuffer();
        }
    }

/*    
    private final void loadBlock() throws IOException
    {
	try
	{
	    throw new IOException("fake");
	} catch( IOException e )
	{
	    System.err.println( "LOAD BLOCK: " + freqIn.getFilePointer() );
	    e.printStackTrace();
	}
	    if( freqIn.getFilePointer() != currentFreqBlockStart )
	    {
		currentFreqBlockStart = freqIn.getFilePointer();
		bytesReader.inflateFrom( freqIn );
	    } 
    }
    
    private final void checkFreqBytesReader() throws IOException
    {
	if( bytesReader.getFilePointer() == bytesReader.length() )
	{
	    loadBlock();
	}
    }
*/
    @Override
    public int nextDoc() throws IOException {
      while(true) {
        if (ord == limit) {
          return doc = NO_MORE_DOCS;
        }

        ord++;

        // Decode next doc/freq pair
        final int code = bytesReader.readVInt();
        if (omitTF) {
//    	    System.err.println( "OMITTF code=" + code );
          doc += code;
        } else {
          doc += code >>> 1;              // shift off low bit
          if ((code & 1) != 0) {          // if low bit is set
            freq = 1;                     // freq is one
          } else {
            freq = bytesReader.readVInt();     // else read freq
          }
        }

//	checkFreqBytesReader();

        if (skipDocs == null || !skipDocs.get(doc)) {
          break;
        }
      }

      return doc;
    }

    private int nextDocUnlocked() throws IOException {
      while(true) {
        if (ord == limit) {
          return doc = NO_MORE_DOCS;
        }

        ord++;

        // Decode next doc/freq pair
        final int code = bytesReader.readVIntUnlockedSafe();
        if (omitTF) {
//    	    System.err.println( "OMITTF code=" + code );
          doc += code;
        } else {
          doc += code >>> 1;              // shift off low bit
          if ((code & 1) != 0) {          // if low bit is set
            freq = 1;                     // freq is one
          } else {
            freq = bytesReader.readVIntUnlockedSafe();     // else read freq
          }
        }

//	checkFreqBytesReader();

        if (skipDocs == null || !skipDocs.get(doc)) {
          break;
        }
      }

      return doc;
    }

    @Override
    public int read() throws IOException {
        boolean locked = bytesReader.lockBuffer();
        try {
            if (omitTF) {
                return readOmitTF();
            } else {
                return readWithTF();
            }
        } finally {
            if (locked) {
                bytesReader.releaseBuffer();
            }
        }
    }

    private final int readOmitTF() throws IOException {
        final int[] docs = bulkResult.docs.ints;
        final int[] freqs = bulkResult.freqs.ints;
        int i = 0;
        final int length = docs.length;
        if (skipDocs == null) {
            while (i < length && ord < limit) {
                ord++;
                // manually inlined call to next() for speed
                final int code = bytesReader.readVIntUnlockedSafe();
                doc += code;
                docs[i] = doc;
                freqs[i] = 1;
                ++i;
            }
        } else {
            while (i < length && ord < limit) {
                ord++;
                // manually inlined call to next() for speed
                final int code = bytesReader.readVIntUnlockedSafe();
                doc += code;
                if (!skipDocs.get(doc)) {
                    docs[i] = doc;
                    freqs[i] = 1;
                    ++i;
                }
            }
        }
        return i;
    }

    private final int readWithTF() throws IOException {
        final int[] docs = bulkResult.docs.ints;
        final int[] freqs = bulkResult.freqs.ints;
        int i = 0;
        final int length = docs.length;
        int skipped = 0;
        while (i < length && ord < limit) {
            ord++;
            // manually inlined call to next() for speed
            final int code = bytesReader.readVIntUnlockedSafe();
            doc += code >>> 1;              // shift off low bit
            if ((code & 1) != 0) {          // if low bit is set
                freq = 1;                     // freq is one
            } else {
                freq = bytesReader.readVIntUnlockedSafe();     // else read freq
            }
            if (skipDocs == null || !skipDocs.get(doc)) {
                docs[i] = doc;
                freqs[i] = freq;
                ++i;
            } else {
                skipped++;
            }
        }
        return i;
    }

    @Override
    public int docID() {
      return doc;
    }

    @Override
    public int freq() {
      return freq;
    }

    @Override
    public int advance(int target) throws IOException {

      if ((target - skipInterval) >= doc && limit >= skipMinimum) {

        // There are enough docs in the posting to have
        // skip data, and it isn't too close.

        if (skipper == null) {
          // This is the first time this enum has ever been used for skipping -- do lazy init
	  skipReader = new MutableBlockCompressedInputStream( (IndexInput)freqIn.clone(), blockSize, false, decompressor, ALLOCATOR, SKIPS_TAG);
	  skipReader.useCache(bytesReader.useCache());
          skipper = new YandexSkipListReader((IndexInput) skipReader, bytesReader, maxSkipLevels, skipInterval);
        }

        if (!skipped) {

          // This is the first time this posting has
          // skipped since reset() was called, so now we
          // load the skip data for this posting
          if( skipBlockStart == freqBlockStart )
          {
//            System.err.println( "CopyMode" );
            skipReader.copyFrom( bytesReader );
          }
          else
          {
            if( skipBlockOffset == 1 ) // indexed mode
            {
//            System.err.println( "IndexedMode" );
        	skipReader.setIndexedMode( true );
    	    }
    	    else
    	    {
//            System.err.println( "NOT IndexedMode" );
    		skipReader.setIndexedMode( false );
    	    }
    	    try {
        	    skipReader.seek( skipBlockStart, 0 );
            } catch (IOException e) {
                new IOException(
                    e.getMessage() + "; skipBlockStart=" + skipBlockStart
                    + ", freqBlockStart=" + freqBlockStart
                    + ", skipBlockOffset=" + skipBlockOffset,
                    e).printStackTrace();
                return advanceSlow(target);
            }
          }


	  if( skipBlockOffset == 1 ) skipBlockOffset = 0;
	  else skipBlockOffset = skipBlockOffset >> 1;
          skipper.init(skipBlockOffset, 
                       freqBlockStart, freqBlockOffset,
                       0, 0,
                       limit, storePayloads);

          skipped = true;
        }

        final int newOrd = skipper.skipTo(target); 

        if (newOrd > ord) {
          // Skipper moved

          //save in case of Exception
          int oldOrd = ord;
          int oldDoc = doc;
          long oldFreqBlockStart = freqBlockStart;
          long oldFreqBlockOffset = freqBlockOffset;

          ord = newOrd;
          doc = skipper.getDoc();
          long newFreqBlockStart = skipper.getFreqBlockPointer();
          long newFreqBlockOffset = skipper.getFreqBlockOffset();
          try {
            bytesReader.seek( newFreqBlockStart, newFreqBlockOffset );
          } catch (IOException e) {
            new IOException(
                e.getMessage() + "; newFreqBlockStart=" + newFreqBlockStart
                + ", freqBlockStart=" + freqBlockStart
                + ", newFreqBlockOffset=" + newFreqBlockOffset,
                e)
                .printStackTrace();
            ord = oldOrd;
            doc = oldDoc;
            freqBlockStart = oldFreqBlockStart;
            freqBlockOffset = oldFreqBlockOffset;
            bytesReader.seek(freqBlockStart, freqBlockOffset);
            return advanceSlow(target);
          }
        }
      }
      return advanceSlow(target);
    }

    private int advanceSlow(final int target) throws IOException {
      // scan for the rest:
      boolean locked = bytesReader.lockBuffer();
      try {
        do {
            nextDocUnlocked();
        } while (target > doc);
      } finally {
        if (locked) {
            bytesReader.releaseBuffer();
        }
      }

      return doc;
    }
  }

  // Decodes docs & positions. payloads are not present.
  private class SegmentDocsAndPositionsEnum extends DocsAndPositionsEnum {
    final IndexInput startFreqIn;
    private final IndexInput freqIn;
    private final IndexInput proxIn;

    int limit;                                    // number of docs in this posting
    int ord;                                      // how many docs we've read
    int doc;                                      // doc we last read
    int freq;                                     // freq we last read
    int position;

    boolean storePayloads;                        // does current field store payloads?

    Bits skipDocs;

//    long freqOffset;
//    int skipOffset;
//    long proxOffset;

    long freqBlockStart;
    long freqBlockOffset;
    long proxBlockStart;
    long proxBlockOffset;

    long skipBlockStart;
    long skipBlockOffset;
    
    long currentFreqBlockStart = -1;
    long currentProxBlockStart = -1;

    int posPendingCount;

    boolean skipped;
    YandexSkipListReader skipper;
    private long lazyProxBlockPointer;
    private long lazyProxBlockOffset;
    MutableBlockCompressedInputStream bytesReader = null; //new MutableBlockCompressedInputStream();
    MutableBlockCompressedInputStream proxReader = null;
    MutableBlockCompressedInputStream skipReader = null;

    public SegmentDocsAndPositionsEnum(IndexInput freqIn, IndexInput proxIn) throws IOException {
      startFreqIn = freqIn;
      this.freqIn = (IndexInput) freqIn.clone();
      this.proxIn = (IndexInput) proxIn.clone();
      this.bytesReader = new MutableBlockCompressedInputStream( this.freqIn, decompressor, ALLOCATOR, FREQ_TAG );
//      System.err.println( "New SegmentDocsAndPositionsEnum" );
    }

    public void free() {
        if (skipReader != null) {
            try {
                skipReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            skipReader = null;
        }
        if (bytesReader != null) {
            try {
                bytesReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            bytesReader = null;
        }
        if (proxReader != null) {
            try {
                proxReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            proxReader = null;
        }
    }

    public SegmentDocsAndPositionsEnum reset(FieldInfo fieldInfo, YandexTermState termState, Bits skipDocs, final boolean useCache) throws IOException {
      assert !fieldInfo.omitTermFreqAndPositions;
      assert !fieldInfo.storePayloads;

      bytesReader.useCache(useCache);
      this.skipDocs = skipDocs;

      // TODO: for full enum case (eg segment merging) this
      // seek is unnecessary; maybe we can avoid in such
      // cases
      freqBlockStart = termState.freqBlockStart;
      freqBlockOffset = termState.freqBlockOffset;
      skipBlockStart = termState.skipBlockStart;
      skipBlockOffset = termState.skipBlockOffset;

      storePayloads = fieldInfo.storePayloads;
      // TODO: for full enum case (eg segment merging) this
      // seek is unnecessary; maybe we can avoid in such
      // cases
//      freqIn.seek(termState.freqBlockStart);
//      bytesReader.inflateFrom( freqIn );
//      loadBlock();
//      checkFreqBytesReader();
      bytesReader.seek( freqBlockStart, freqBlockOffset );

      proxBlockStart = lazyProxBlockPointer = termState.proxBlockStart;
      proxBlockOffset = lazyProxBlockOffset = termState.proxBlockOffset;
//      System.err.println( "RESET: freqBlockStart=" + freqBlockStart + ", freqBlockOffset=" + freqBlockOffset + ", lazyProxBlockPointer=" + lazyProxBlockPointer + ", lazyProxBlockOffset=" + lazyProxBlockOffset );
      currentProxBlockStart = -1;

      limit = termState.docFreq;
      assert limit > 0;

      ord = 0;
      doc = 0;
      position = 0;

      skipped = false;
      posPendingCount = 0;

//      freqOffset = termState.freqOffset;
//      proxOffset = termState.proxOffset;
//      skipOffset = termState.skipOffset;
      //System.out.println("StandardR.D&PE reset seg=" + segment + " limit=" + limit + " freqFP=" + freqOffset + " proxFP=" + proxOffset);

      return this;
    }

/*
    private final void loadBlock() throws IOException
    {
	    if( freqIn.getFilePointer() != currentFreqBlockStart )
	    {
		currentFreqBlockStart = freqIn.getFilePointer();
		bytesReader.inflateFrom( freqIn );
	    }
    }

    private final void loadProxBlock() throws IOException
    {
	if( proxReader == null ) proxReader = new MutableBlockCompressedInputStream();
        currentProxBlockStart = proxIn.getFilePointer();
//        System.err.println("loadProxBlock: currentProxBlockStart=" + currentProxBlockStart);
        proxReader.inflateFrom( proxIn );
    }
    
    private final void checkFreqBytesReader() throws IOException
    {
	if( bytesReader.getFilePointer() == bytesReader.length() )
	{
	    loadBlock();
	}
    }

    private final void checkProxBytesReader() throws IOException
    {
	if( proxReader.getFilePointer() == proxReader.length() )
	{
	    loadProxBlock();
	}
    }
*/

    @Override
    public int nextDoc() throws IOException {
      while(true) {
        if (ord == limit) {
          //System.out.println("StandardR.D&PE seg=" + segment + " nextDoc return doc=END");
          return doc = NO_MORE_DOCS;
        }

        ord++;

        // Decode next doc/freq pair
        final int code = bytesReader.readVInt();

        doc += code >>> 1;              // shift off low bit
        if ((code & 1) != 0) {          // if low bit is set
          freq = 1;                     // freq is one
        } else {
          freq = bytesReader.readVInt();     // else read freq
        }
        posPendingCount += freq;
        
//        checkFreqBytesReader();

        if (skipDocs == null || !skipDocs.get(doc)) {
          break;
        }
      }

      position = 0;

      //System.out.println("StandardR.D&PE nextDoc seg=" + segment + " return doc=" + doc);
      return doc;
    }

    @Override
    public int docID() {
      return doc;
    }

    @Override
    public int freq() {
      return freq;
    }

    @Override
    public int advance(int target) throws IOException {

      //System.out.println("StandardR.D&PE advance target=" + target);

      if ((target - skipInterval) >= doc && limit >= skipMinimum) {
//    if( false ) {

        // There are enough docs in the posting to have
        // skip data, and it isn't too close

        if (skipper == null) {
          // This is the first time this enum has ever been used for skipping -- do lazy init
//	  skipReader = new MutableBlockCompressedInputStream();
//	  skipReader = new MutableBlockCompressedInputStream( (IndexInput)freqIn.clone() );
	  skipReader = new MutableBlockCompressedInputStream( (IndexInput)freqIn.clone(), blockSize, false, decompressor, ALLOCATOR, SKIPS_TAG );
	  skipReader.useCache(bytesReader.useCache());

/*	  if( skipBlockStart == freqBlockStart )
	  {
		System.err.println( "\n\tNOT chained stream clone" );
	    skipReader = (MutableBlockCompressedInputStream)bytesReader.clone();
	  }
	  else
	  {
	    if( skipBlockOffset == 1 )
	    {
		System.err.println( "\n\tchained stream: skipBlockStart=" + skipBlockStart );
		IndexInput freqInClone = (IndexInput)freqIn.clone();
		freqInClone.seek( skipBlockStart );
		skipReader = new MutableBlockCompressedInputStream( freqInClone, YandexPostingsWriter.BLOCK_SIZE );
	    }
	    else
	    {
		System.err.println( "\n\tNOT chained stream" );
		skipReader = new MutableBlockCompressedInputStream( (IndexInput)freqIn.clone() );
//		skipReader.inflateFrom( (IndexInput)freqIn.clone() );
	    }
	  }
*/
          skipper = new YandexSkipListReader((IndexInput) skipReader, bytesReader, maxSkipLevels, skipInterval);
        }

        if (!skipped) {

          // This is the first time this posting has
          // skipped since reset() was called, so now we
          // load the skip data for this posting
          if( skipBlockStart == freqBlockStart )
          {
            skipReader.copyFrom( bytesReader );
          }
          else
          {
            if( skipBlockOffset == 1 ) // indexed mode
            {
        	skipReader.setIndexedMode( true );
    	    }
    	    else
    	    {
    		skipReader.setIndexedMode( false );
    	    }
    	    skipReader.seek( skipBlockStart, 0 );
          }


//	  System.err.println( "\n\tSkipListInit: skipBlockOffset=" + skipBlockOffset );
	  if( skipBlockOffset == 1 ) skipBlockOffset = 0;
	  else skipBlockOffset = skipBlockOffset >> 1;
//	  System.err.println( "\n\tSkipListInit: skipBlockOffset=" + skipBlockOffset );
          skipper.init(skipBlockOffset, 
                       freqBlockStart, freqBlockOffset,
                       proxBlockStart, proxBlockOffset,
                       limit, storePayloads);

          skipped = true;
        }

        final int newOrd = skipper.skipTo(target); 

        if (newOrd > ord) {
          // Skipper moved
          ord = newOrd;
          doc = skipper.getDoc();
          
          long newFreqBlockStart = skipper.getFreqBlockPointer();
          long newFreqBlockOffset = skipper.getFreqBlockOffset();
//          if( newFreqBlockStart != currentFreqBlockStart )
//          {
//            if( newFreqBlockStart != freqIn.getFilePointer() )
//          	freqIn.seek( newFreqBlockStart );
//            loadBlock();
//          }
          lazyProxBlockPointer = skipper.getProxBlockPointer();
          lazyProxBlockOffset = skipper.getProxBlockOffset();

//          System.err.println( "ADVANCE TO["+target+"]: newFreqBlockStart=" + newFreqBlockStart + ", newFreqBlockOffset=" + newFreqBlockOffset + ", lazyProxBlockPointer=" + lazyProxBlockPointer + ", lazyProxBlockOffset=" + lazyProxBlockOffset );


          bytesReader.seek( newFreqBlockStart, newFreqBlockOffset );
          
          
          
          posPendingCount = 0;
          position = 0;
        }
      }
        
      // Now, linear scan for the rest:
      do {
        nextDoc();
      } while (target > doc);

      return doc;
    }

    @Override
    public int nextPosition() throws IOException {

      if (lazyProxBlockPointer != -1) {
//        if( currentProxBlockStart != lazyProxBlockPointer )
//        {
//    	    System.err.println( "lazyProxBlockPointer=" + lazyProxBlockPointer + ", lazyProxBlockOffset=" + lazyProxBlockOffset );
//	    proxIn.seek(lazyProxBlockPointer);
//	    loadProxBlock();
//	}
	if( proxReader == null ) {
	    proxReader = new MutableBlockCompressedInputStream( this.proxIn, decompressor, ALLOCATOR, PROX_TAG );
	    proxReader.useCache(bytesReader.useCache());
	}
	proxReader.seek( lazyProxBlockPointer, lazyProxBlockOffset );
        lazyProxBlockPointer = -1;
      }

      // scan over any docs that were iterated without their positions
      if (posPendingCount > freq) {
        position = 0;
        while(posPendingCount != freq) {
          if ((proxReader.readByte() & 0x80) == 0) {
            posPendingCount--;
          }
//          checkProxBytesReader();
        }
      }

      position += proxReader.readVInt();
//      checkProxBytesReader();

      posPendingCount--;

      assert posPendingCount >= 0: "nextPosition() was called too many times (more than freq() times) posPendingCount=" + posPendingCount;

      return position;
    }

    /** Returns the payload at this position, or null if no
     *  payload was indexed. */
    @Override
    public BytesRef getPayload() throws IOException {
      throw new IOException("No payloads exist for this field!");
    }

    @Override
    public boolean hasPayload() {
      return false;
    }
  }
  
  // Decodes docs & positions & payloads
  private class SegmentDocsAndPositionsAndPayloadsEnum extends DocsAndPositionsEnum {
    final IndexInput startFreqIn;
    private final IndexInput freqIn;
    private final IndexInput proxIn;

    int limit;                                    // number of docs in this posting
    int ord;                                      // how many docs we've read
    int doc;                                      // doc we last read
    int freq;                                     // freq we last read
    int position;
    boolean storePayloads;

    Bits skipDocs;

//    long freqOffset;
//    int skipOffset;
//    long proxOffset;

    long freqBlockStart;
    long freqBlockOffset;
    long proxBlockStart;
    long proxBlockOffset;
    long skipBlockStart;
    long skipBlockOffset;
    
    long currentFreqBlockStart = -1;
    long currentProxBlockStart = -1;


    int posPendingCount;
    int payloadLength;
    boolean payloadPending;

    boolean skipped;
    YandexSkipListReader skipper;
    private BytesRef payload;
    private long lazyProxBlockPointer;
    private long lazyProxBlockOffset;
    MutableBlockCompressedInputStream bytesReader = null; //new MutableBlockCompressedInputStream();
    MutableBlockCompressedInputStream proxReader = null;
    MutableBlockCompressedInputStream skipReader = null;

    public SegmentDocsAndPositionsAndPayloadsEnum(IndexInput freqIn, IndexInput proxIn) throws IOException {
      startFreqIn = freqIn;
      this.freqIn = (IndexInput) freqIn.clone();
      this.proxIn = (IndexInput) proxIn.clone();
      bytesReader = new MutableBlockCompressedInputStream( this.freqIn, decompressor, ALLOCATOR, FREQ_TAG );
    }

    public void free() {
        if (skipReader != null) {
            try {
                skipReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            skipReader = null;
        }
        if (bytesReader != null) {
            try {
                bytesReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            bytesReader = null;
        }
        if (proxReader != null) {
            try {
                proxReader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            proxReader = null;
        }
    }

    public SegmentDocsAndPositionsAndPayloadsEnum reset(FieldInfo fieldInfo, YandexTermState termState, Bits skipDocs) throws IOException {
      assert !fieldInfo.omitTermFreqAndPositions;
      assert fieldInfo.storePayloads;
      if (payload == null) {
        payload = new BytesRef();
        payload.bytes = new byte[1];
      }

      this.skipDocs = skipDocs;

      storePayloads = fieldInfo.storePayloads;
      // TODO: for full enum case (eg segment merging) this
      // seek is unnecessary; maybe we can avoid in such
      // cases
//      freqIn.seek(termState.freqBlockStart);
//      bytesReader.inflateFrom( freqIn );
//      loadBlock();
//      checkFreqBytesReader();
      freqBlockStart = termState.freqBlockStart;
      freqBlockOffset = termState.freqBlockOffset;
      skipBlockStart = termState.skipBlockStart;
      skipBlockOffset = termState.skipBlockOffset;


      bytesReader.seek( termState.freqBlockStart, termState.freqBlockOffset );
      

      proxBlockStart = lazyProxBlockPointer = termState.proxBlockStart;
      proxBlockOffset = lazyProxBlockOffset = termState.proxBlockOffset;
      currentProxBlockStart = -1;

      limit = termState.docFreq;
      assert limit > 0;

      ord = 0;
      doc = 0;
      position = 0;

      skipped = false;
      posPendingCount = 0;

      return this;
    }

    @Override
    public int nextDoc() throws IOException {
      while(true) {
        if (ord == limit) {
          //System.out.println("StandardR.D&PE seg=" + segment + " nextDoc return doc=END");
          return doc = NO_MORE_DOCS;
        }

        ord++;

        // Decode next doc/freq pair
        final int code = bytesReader.readVInt();

        doc += code >>> 1;              // shift off low bit
        if ((code & 1) != 0) {          // if low bit is set
          freq = 1;                     // freq is one
        } else {
          freq = bytesReader.readVInt();     // else read freq
        }
        posPendingCount += freq;

//        checkFreqBytesReader();

        if (skipDocs == null || !skipDocs.get(doc)) {
          break;
        }
      }

      position = 0;

      //System.out.println("StandardR.D&PE nextDoc seg=" + segment + " return doc=" + doc);
      return doc;
    }
/*
    private final void loadBlock() throws IOException
    {
	if( freqIn.getFilePointer() != currentFreqBlockStart )
	{
    	    currentFreqBlockStart = freqIn.getFilePointer();
	    bytesReader.inflateFrom( freqIn );
	}
    }

    private final void loadProxBlock() throws IOException
    {
	if( proxReader == null ) proxReader = new MutableBlockCompressedInputStream();
	    currentProxBlockStart = proxIn.getFilePointer();
	    proxReader.inflateFrom( proxIn );
    }
    
    private final void checkFreqBytesReader() throws IOException
    {
	if( bytesReader.getFilePointer() == bytesReader.length() )
	{
	    loadBlock();
	}
    }

    private final void checkProxBytesReader() throws IOException
    {
	if( proxReader.getFilePointer() == proxReader.length() )
	{
	    loadProxBlock();
	}
    }
*/


    @Override
    public int docID() {
      return doc;
    }

    @Override
    public int freq() {
      return freq;
    }

    @Override
    public int advance(int target) throws IOException {

      //System.out.println("StandardR.D&PE advance seg=" + segment + " target=" + target + " this=" + this);

      if ((target - skipInterval) >= doc && limit >= skipMinimum) {

        // There are enough docs in the posting to have
        // skip data, and it isn't too close


        if (skipper == null) {
          // This is the first time this enum has ever been used for skipping -- do lazy init
//	  skipReader = new MutableBlockCompressedInputStream( (IndexInput)freqIn.clone() );
	  skipReader = new MutableBlockCompressedInputStream( (IndexInput)freqIn.clone(), blockSize, false, decompressor, ALLOCATOR, SKIPS_TAG );
	  skipReader.useCache(bytesReader.useCache());
/*	  if( skipBlockStart == freqBlockStart )
	  {
		System.err.println( "\n\tNOT chained stream clone" );
	    skipReader = (MutableBlockCompressedInputStream)bytesReader.clone();
	  }
	  else
	  {
	    if( skipBlockOffset == 1 )
	    {
		System.err.println( "\n\tchained stream: skipBlockStart=" + skipBlockStart );
		IndexInput freqInClone = (IndexInput)freqIn.clone();
		freqInClone.seek( skipBlockStart );
		skipReader = new MutableBlockCompressedInputStream( freqInClone, YandexPostingsWriter.BLOCK_SIZE );
	    }
	    else
	    {
		System.err.println( "\n\tNOT chained stream" );
		skipReader = new MutableBlockCompressedInputStream( (IndexInput)freqIn.clone() );
//		skipReader.inflateFrom( (IndexInput)freqIn.clone() );
	    }
	  }
*/
          skipper = new YandexSkipListReader((IndexInput) skipReader, bytesReader, maxSkipLevels, skipInterval);
        }

        if (!skipped) {

          // This is the first time this posting has
          // skipped since reset() was called, so now we
          // load the skip data for this posting
          if( skipBlockStart == freqBlockStart )
          {
            skipReader.copyFrom( bytesReader );
          }
          else
          {
            if( skipBlockOffset == 1 ) // indexed mode
            {
        	skipReader.setIndexedMode( true );
    	    }
    	    else
    	    {
    		skipReader.setIndexedMode( false );
    	    }
    	    skipReader.seek( skipBlockStart, 0 );
          }

          // This is the first time this posting has
          // skipped, since reset() was called, so now we
          // load the skip data for this posting
          //System.out.println("  init skipper freqOffset=" + freqOffset + " skipOffset=" + skipOffset + " vs len=" + freqIn.length());
	  if( skipBlockOffset == 1 ) skipBlockOffset = 0;
	  else skipBlockOffset = skipBlockOffset >> 1;
          skipper.init(skipBlockOffset, 
                       freqBlockStart, freqBlockOffset,
                       proxBlockStart, proxBlockOffset,
                       limit, storePayloads);

          skipped = true;
        }

        final int newOrd = skipper.skipTo(target); 

        if (newOrd > ord) {
          // Skipper moved
          ord = newOrd;
          doc = skipper.getDoc();
//          freqIn.seek(skipper.getFreqPointer());
//          lazyProxPointer = skipper.getProxPointer();
          long newFreqBlockStart = skipper.getFreqBlockPointer();
          long newFreqBlockOffset = skipper.getFreqBlockOffset();
//          if( newFreqBlockStart != currentFreqBlockStart )
//          {
//            if( newFreqBlockStart != freqIn.getFilePointer() )
//          	freqIn.seek( newFreqBlockStart );
//            loadBlock();
//          }
          bytesReader.seek( newFreqBlockStart, newFreqBlockOffset );

          
          lazyProxBlockPointer = skipper.getProxBlockPointer();
          lazyProxBlockOffset = skipper.getProxBlockOffset();



          posPendingCount = 0;
          position = 0;
          payloadPending = false;
          payloadLength = skipper.getPayloadLength();
        }
      }
        
      // Now, linear scan for the rest:
      do {
        nextDoc();
      } while (target > doc);

      return doc;
    }

    @Override
    public int nextPosition() throws IOException {

      if (lazyProxBlockPointer != -1) {
//        if( currentProxBlockStart != lazyProxBlockPointer )
//        {
//	    proxIn.seek(lazyProxBlockPointer);
//	    loadProxBlock();
//	}
	proxReader.seek( lazyProxBlockPointer, lazyProxBlockOffset );
        lazyProxBlockPointer = -1;
      }
      
      if (payloadPending && payloadLength > 0) {
        // payload of last position as never retrieved -- skip it
//        proxIn.seek(proxIn.getFilePointer() + payloadLength);
	proxReader.seek(proxReader.getFilePointer() + payloadLength);
        payloadPending = false;
      }

      // scan over any docs that were iterated without their positions
      while(posPendingCount > freq) {

        final int code = proxReader.readVInt();

        if ((code & 1) != 0) {
          // new payload length
          payloadLength = proxReader.readVInt();
          assert payloadLength >= 0;
        }
        
        assert payloadLength != -1;
        proxReader.seek(proxReader.getFilePointer() + payloadLength);
//	checkProxBytesReader();

        posPendingCount--;
        position = 0;
        payloadPending = false;
        //System.out.println("StandardR.D&PE skipPos");
      }

      // read next position
      if (payloadPending && payloadLength > 0) {
        // payload wasn't retrieved for last position
        proxReader.seek(proxReader.getFilePointer()+payloadLength);
      }

      final int code = proxReader.readVInt();
      if ((code & 1) != 0) {
        // new payload length
        payloadLength = proxReader.readVInt();
        assert payloadLength >= 0;
      }
      assert payloadLength != -1;
//      checkProxBytesReader();
          
          
      payloadPending = true;
      position += code >>> 1;

      posPendingCount--;

      assert posPendingCount >= 0: "nextPosition() was called too many times (more than freq() times) posPendingCount=" + posPendingCount;

      //System.out.println("StandardR.D&PE nextPos   return pos=" + position);
      return position;
    }

    /** Returns the payload at this position, or null if no
     *  payload was indexed. */
    @Override
    public BytesRef getPayload() throws IOException {
      assert lazyProxBlockPointer == -1;
      assert posPendingCount < freq;
      if (!payloadPending) {
        throw new IOException("Either no payload exists at this term position or an attempt was made to load it more than once.");
      }
      if (payloadLength > payload.bytes.length) {
        payload.grow(payloadLength);
      }

      proxReader.readBytes(payload.bytes, 0, payloadLength);
      payload.length = payloadLength;
      payloadPending = false;
//      checkProxBytesReader();

      return payload;
    }

    @Override
    public boolean hasPayload() {
      return payloadPending && payloadLength > 0;
    }
  }
}
