package org.apache.lucene.index.codecs.yandex;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** Consumes doc & freq, writing them using the current
 *  index file format */

import java.io.IOException;

import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.SegmentWriteState;
import org.apache.lucene.index.codecs.PostingsWriterBase;
import org.apache.lucene.index.codecs.TermStats;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.RAMOutputStream;
import org.apache.lucene.store.DeflateOutputStream;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CodecUtil;

/** @lucene.experimental */
public final class YandexPostingsWriter extends PostingsWriterBase {
  public static final int DEFAULT_BLOCK_SIZE = 4096;

  final static String CODEC = "YandexPostingsWriterImpl";

  // Increment version to change it:
  final static int VERSION_START = 0;
  final static int VERSION_CURRENT = VERSION_START;

//  static final int BLOCK_SIZE = 4096;
  final int blockSize;

  final IndexOutput freqOut;
  final IndexOutput proxOut;
  final YandexSkipListWriter skipListWriter;
  /** Expert: The fraction of TermDocs entries stored in skip tables,
   * used to accelerate {@link DocsEnum#advance(int)}.  Larger values result in
   * smaller indexes, greater acceleration, but fewer accelerable cases, while
   * smaller values result in bigger indexes, less acceleration and more
   * accelerable cases. More detailed experiments would be useful here. */
  final int skipInterval = 16;
  
  /**
   * Expert: minimum docFreq to write any skip data at all
   */
  final int skipMinimum = skipInterval;

  /** Expert: The maximum number of skip levels. Smaller values result in 
   * slightly smaller indexes, but slower skipping in big posting lists.
   */
  final int maxSkipLevels = 10;
  final int totalNumDocs;
  IndexOutput termsOut;

  boolean omitTermFreqAndPositions;
  boolean storePayloads;
  // Starts a new term
  long lastFreqBlockStart;
  long lastFreqBlockOffset;
  long freqBlockStart;
  long freqBlockOffset;
  long lastProxBlockStart;
  long lastProxBlockOffset;
  long proxBlockStart;
  long proxBlockOffset;
  FieldInfo fieldInfo;
  int lastPayloadLength;
  int lastPosition;

  private int pendingCount;

  //private String segment;

//  private RAMOutputStream bytesWriter = new RAMOutputStream();
  private IndexOutput bytesWriter = null;
  private final DeflateOutputStream freqBytesWriter;// = new DeflateOutputStream( BLOCK_SIZE );
  private final DeflateOutputStream proxBytesWriter;// = new DeflateOutputStream( BLOCK_SIZE );
  private RAMOutputStream tmpWriter = new RAMOutputStream();
  private final Compressor compressor;

  public YandexPostingsWriter(final Compressor compressor, final int blockSize, SegmentWriteState state) throws IOException {
    super();
    this.compressor = compressor;
    this.blockSize = blockSize;
    freqBytesWriter = new DeflateOutputStream(blockSize, compressor);
    proxBytesWriter = new DeflateOutputStream(blockSize, compressor);
    //this.segment = state.segmentName;
    String fileName = IndexFileNames.segmentFileName(state.segmentName, state.codecId, YandexCodec.FREQ_EXTENSION);
    freqOut = state.directory.createOutput(fileName);

    if (state.fieldInfos.hasProx()) {
      // At least one field does not omit TF, so create the
      // prox file
      fileName = IndexFileNames.segmentFileName(state.segmentName, state.codecId, YandexCodec.PROX_EXTENSION);
      proxOut = state.directory.createOutput(fileName);
    } else {
      // Every field omits TF so we will write no prox file
      proxOut = null;
    }

    totalNumDocs = state.numDocs;

    skipListWriter = new YandexSkipListWriter(skipInterval,
                                               maxSkipLevels,
                                               state.numDocs, state.fieldInfos.hasProx());
//    skipListWriter = new DefaultSkipListWriter(skipInterval,
//                                               maxSkipLevels,
//                                               state.numDocs,
//                                               freqOut,
//                                               proxOut);
  }

  private String codecName() {
    if (compressor.id() != null) {
        return CODEC + "_" + compressor.id();
    } else {
        return CODEC;
    }
  }

  @Override
  public void start(IndexOutput termsOut) throws IOException {
    this.termsOut = termsOut;
    final int version;
    if (blockSize == DEFAULT_BLOCK_SIZE) {
        version = VERSION_CURRENT;
    } else {
        version = blockSize;
    }
    CodecUtil.writeHeader(termsOut, codecName(), blockSize);
    termsOut.writeInt(skipInterval);                // write skipInterval
    termsOut.writeInt(maxSkipLevels);               // write maxSkipLevels
    termsOut.writeInt(skipMinimum);                 // write skipMinimum
  }

  @Override
  public void startTerm() {
    freqBlockStart = freqOut.getFilePointer();
    freqBlockOffset = freqBytesWriter.getFilePointer();
    if (proxOut != null) {
      proxBlockStart = proxOut.getFilePointer();
      proxBlockOffset = proxBytesWriter.getFilePointer();
      // force first payload to write its length
      lastPayloadLength = -1;
    }
//    System.err.println("StandardW: startTerm pendingCount=" + pendingCount + " freqBlockStart=" + freqBlockStart + ", freqBlockOffset=" + freqBlockOffset);
    skipListWriter.resetSkip( freqBlockStart, freqBlockOffset, proxBlockStart, proxBlockOffset );
  }

  // Currently, this instance is re-used across fields, so
  // our parent calls setField whenever the field changes
  @Override
  public void setField(FieldInfo fieldInfo) {
    //System.out.println("SPW: setField");
    this.fieldInfo = fieldInfo;
    omitTermFreqAndPositions = fieldInfo.omitTermFreqAndPositions;
    storePayloads = fieldInfo.storePayloads;
    //System.out.println("  set init blockFreqStart=" + freqStart);
    //System.out.println("  set init blockProxStart=" + proxStart);
  }

  int lastDocID;
  int df;

  private void flushFreqBlock() throws IOException
  {
    tmpWriter.reset();
    int deflated = freqBytesWriter.deflateTo( tmpWriter );
    freqOut.writeVInt( deflated );
    freqOut.writeVInt( (int)(freqBytesWriter.length() & 0xFFFFFFFF) ); 
    tmpWriter.writeTo( freqOut ); 
    tmpWriter.reset();
    freqBytesWriter.reset();
//    freqBlockStart = freqOut.getFilePointer();
//    freqBlockOffset = freqBytesWriter.getFilePointer();
  }

  private void flushChainedFreqBlock() throws IOException
  {
    freqBytesWriter.writeChainedTo( freqOut );
    freqBytesWriter.reset();
  }

  private void flushProxBlock() throws IOException
  {
    try {
        tmpWriter.reset();
        int deflated = proxBytesWriter.deflateTo( tmpWriter );
        proxOut.writeVInt( deflated );
        proxOut.writeVInt( (int)(proxBytesWriter.length() & 0xFFFFFFFF) ); 
        tmpWriter.writeTo( proxOut ); 
        tmpWriter.reset();
        proxBytesWriter.reset();
    } catch (IOException e) {
        e.printStackTrace();
        throw e;
    }
//    freqBlockStart = freqOut.getFilePointer();
//    freqBlockOffset = freqBytesWriter.getFilePointer();
  }
  
  /** Adds a new doc in this term.  If this returns null
   *  then we just skip consuming positions/payloads. */
  @Override
  public void startDoc(int docID, int termDocFreq) throws IOException {
//    System.out.println("StandardW: " + this + "  startDoc docID=" + docID + " tf=" + termDocFreq);

    final int delta = docID - lastDocID;
    
    if (docID < 0 || (df > 0 && delta <= 0)) {
      throw new CorruptIndexException("docs out of order (" + docID + " <= " + lastDocID + " )");
    }

    if ((++df % skipInterval) == 0) {
//      skipListWriter.setSkipData(lastDocID, storePayloads, lastPayloadLength, freqBlockStart, freqBlockOffset, proxBlockStart, proxBlockOffset);
	long curProxBlockStart = 0;
	long curProxBlockOffset = 0;
	if (proxOut != null) 
	{
    	    curProxBlockStart = proxOut.getFilePointer();
    	    curProxBlockOffset = proxBytesWriter.getFilePointer();
    	}
    
//      System.err.println( "setSkipData: freqBlockStart=" + freqOut.getFilePointer() + ", freqBlockOffset=" + freqBytesWriter.getFilePointer() );
      skipListWriter.setSkipData(lastDocID, storePayloads, lastPayloadLength, freqOut.getFilePointer(), freqBytesWriter.getFilePointer(), curProxBlockStart, curProxBlockOffset);
      skipListWriter.bufferSkip(df);
    }

    assert docID < totalNumDocs: "docID=" + docID + " totalNumDocs=" + totalNumDocs;
    if( docID > totalNumDocs ) throw new RuntimeException( "docID=" + docID + " totalNumDocs=" + totalNumDocs );

    lastDocID = docID;
    if (omitTermFreqAndPositions) {
//      freqOut.writeVInt(delta);
	freqBytesWriter.writeVInt( delta );
//      freqOut.writeInt(delta);
    } else if (1 == termDocFreq) {
//      freqOut.writeVInt((delta<<1) | 1);
      freqBytesWriter.writeVInt((delta<<1) | 1);
//	freqBytesWriter
//      freqOut.writeInt((delta<<1) | 1);
    } else {
//      freqOut.writeVInt(delta<<1);
//      freqOut.writeVInt(termDocFreq);
      freqBytesWriter.writeVInt(delta<<1);
      freqBytesWriter.writeVInt(termDocFreq);
//      freqOut.writeInt(delta<<1);
//      freqOut.writeInt(termDocFreq);
    }
    
    if( freqBytesWriter.getFilePointer() > blockSize )
    {
	flushFreqBlock();
    }

    lastPosition = 0;
  }

  /** Add a new position & payload */
  @Override
  public void addPosition(int position, BytesRef payload) throws IOException {
    //System.out.println("StandardW:     addPos pos=" + position + " payload=" + (payload == null ? "null" : (payload.length + " bytes")) + " proxFP=" + proxOut.getFilePointer());
    assert !omitTermFreqAndPositions: "omitTermFreqAndPositions is true";
    assert proxOut != null;

    final int delta = position - lastPosition;
    
    assert delta > 0 || position == 0: "position=" + position + " lastPosition=" + lastPosition;            // not quite right (if pos=0 is repeated twice we don't catch it)

    lastPosition = position;

    if (storePayloads) {
      final int payloadLength = payload == null ? 0 : payload.length;

      if (payloadLength != lastPayloadLength) {
        lastPayloadLength = payloadLength;
        proxBytesWriter.writeVInt((delta<<1)|1);
        proxBytesWriter.writeVInt(payloadLength);
//        proxOut.writeVInt((delta<<1)|1);
//        proxOut.writeVInt(payloadLength);
      } else {
//        proxOut.writeVInt(delta << 1);
        proxBytesWriter.writeVInt(delta << 1);
      }

      if (payloadLength > 0) {
//        proxOut.writeBytes(payload.bytes, payload.offset, payloadLength);
        proxBytesWriter.writeBytes(payload.bytes, payload.offset, payloadLength);
      }
    } else {
      proxBytesWriter.writeVInt(delta);
//      proxOut.writeVInt(delta);
    }
    
    if( proxBytesWriter.getFilePointer() > blockSize )
    {
	flushProxBlock();
    }
    
  }

  @Override
  public void finishDoc() {
  }

  public void setWriter(IndexOutput out)
  {
    bytesWriter = out;
  }

  /** Called when we are done adding docs to this term */
  @Override
  public void finishTerm(TermStats stats) throws IOException {
    final boolean isFirstTerm = pendingCount == 0;
    //System.out.println("  isFirstTerm=" + isFirstTerm);

    //System.out.println("  freqFP=" + freqStart);
    if (isFirstTerm) {
      bytesWriter.writeVLong(freqBlockStart);
      bytesWriter.writeVLong(freqBlockOffset);
    } else {
      bytesWriter.writeVLong(freqBlockStart-lastFreqBlockStart);
      long offset = freqBlockOffset;
      if( freqBlockStart == lastFreqBlockStart ) offset = freqBlockOffset-lastFreqBlockOffset;
      bytesWriter.writeVLong(offset);
    }
    lastFreqBlockStart = freqBlockStart;
    lastFreqBlockOffset = freqBlockOffset;

    if (df >= skipMinimum) {
        long offset;
        long start;

        skipListWriter.writeSkip(tmpWriter);
        if (tmpWriter.getFilePointer()
            > blockSize - freqBytesWriter.getFilePointer())
        {
            flushFreqBlock();
            start = (int)(freqOut.getFilePointer() - freqBlockStart);
            offset= 1;
            freqBytesWriter.setChainedMode(true);
            skipListWriter.writeSkip(freqBytesWriter);
            flushChainedFreqBlock();
        } else {
            start = (int)(freqOut.getFilePointer() - freqBlockStart);
            offset = freqBytesWriter.getFilePointer();
            if (freqBlockStart == freqOut.getFilePointer()) {
                offset -= freqBlockOffset;
            }
            offset = offset << 1;
            skipListWriter.writeSkip(freqBytesWriter);
            if (freqBytesWriter.getFilePointer() > blockSize) {
                flushFreqBlock();
            }
        }
        tmpWriter.reset();
        bytesWriter.writeVInt((int) start);
        bytesWriter.writeVInt((int) offset);
    }

    if (!omitTermFreqAndPositions) {
      //System.out.println("  proxFP=" + proxStart);
      if (isFirstTerm) {
        bytesWriter.writeVLong(proxBlockStart);
        bytesWriter.writeVLong(proxBlockOffset);
      } else {
        bytesWriter.writeVLong(proxBlockStart - lastProxBlockStart);
        long offset = proxBlockOffset;
        if( proxBlockStart == lastProxBlockStart ) offset = proxBlockOffset-lastProxBlockOffset;
        bytesWriter.writeVLong(offset);
      }
      lastProxBlockStart = proxBlockStart;
      lastProxBlockOffset = proxBlockOffset;
    }
     
     
    lastDocID = 0;
    df = 0;
    pendingCount++;
  }

  @Override
  public void flushTermsBlock() throws IOException {
    //System.out.println("SPW.flushBlock pendingCount=" + pendingCount);
//    termsOut.writeVInt((int) bytesWriter.getFilePointer());
//    bytesWriter.writeTo(termsOut);
//    bytesWriter.reset();
    pendingCount = 0;
  }

  @Override
  public void close() throws IOException {
    try {
      if( freqBytesWriter.getFilePointer() > 0 ) flushFreqBlock();
      freqBytesWriter.close();
      freqOut.close();
    } finally {
      if (proxOut != null) {
        if( proxBytesWriter.getFilePointer() > 0 ) flushProxBlock();
        proxBytesWriter.close();
        proxOut.close();
      }
    }
  }
}
