package org.apache.lucene.index.codecs.yandex;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.Arrays;

import org.apache.lucene.index.codecs.MultiLevelSkipListReader;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.MutableBlockCompressedInputStream;

/**
 * Implements the skip list reader for the default posting list format
 * that stores positions and payloads.
 * @lucene.experimental
 */
public class YandexSkipListReader extends MultiLevelSkipListReader {
  private boolean currentFieldStoresPayloads;
  private long freqBlockPointer[];
  private long proxBlockPointer[];
  private long freqBlockOffset[];
  private long proxBlockOffset[];
  private int payloadLength[];
  
  private long lastFreqBlockPointer;
  private long lastProxBlockPointer;
  private long lastFreqBlockOffset;
  private long lastProxBlockOffset;
  private int lastPayloadLength;
                           

  public YandexSkipListReader(IndexInput skipStream, MutableBlockCompressedInputStream bytesReader, int maxSkipLevels, int skipInterval) {
    super(skipStream, maxSkipLevels, skipInterval);
    
    freqBlockPointer = new long[maxSkipLevels];
    proxBlockPointer = new long[maxSkipLevels];
    freqBlockOffset = new long[maxSkipLevels];
    proxBlockOffset = new long[maxSkipLevels];
    payloadLength = new int[maxSkipLevels];
  }

  public void init(long skipPointer, long freqBaseBlockPointer, long freqBaseBlockOffset, long proxBaseBlockPointer, long proxBaseBlockOffset, int df, boolean storesPayloads) {
    super.init(skipPointer, df);
    this.currentFieldStoresPayloads = storesPayloads;
    lastFreqBlockPointer = freqBaseBlockPointer;
    lastFreqBlockOffset = freqBaseBlockOffset;
    lastProxBlockPointer = proxBaseBlockPointer;
    lastProxBlockOffset = proxBaseBlockOffset;

    Arrays.fill(freqBlockPointer, freqBaseBlockPointer);
    Arrays.fill(freqBlockOffset, freqBaseBlockOffset);
    Arrays.fill(proxBlockPointer, proxBaseBlockPointer);
    Arrays.fill(proxBlockOffset, proxBaseBlockOffset);
    Arrays.fill(payloadLength, 0);
  }

  /** Returns the freq pointer of the doc to which the last call of 
   * {@link MultiLevelSkipListReader#skipTo(int)} has skipped.  */
  public long getFreqBlockPointer() {
    return lastFreqBlockPointer;
  }

  public long getFreqBlockOffset() {
    return lastFreqBlockOffset;
  }

  /** Returns the prox pointer of the doc to which the last call of 
   * {@link MultiLevelSkipListReader#skipTo(int)} has skipped.  */
  public long getProxBlockPointer() {
    return lastProxBlockPointer;
  }

  public long getProxBlockOffset() {
    return lastProxBlockOffset;
  }
  
  /** Returns the payload length of the payload stored just before 
   * the doc to which the last call of {@link MultiLevelSkipListReader#skipTo(int)} 
   * has skipped.  */
  public int getPayloadLength() {
    return lastPayloadLength;
  }
  
  @Override
  protected void seekChild(int level) throws IOException {
    super.seekChild(level);
    freqBlockPointer[level] = lastFreqBlockPointer;
    freqBlockOffset[level] = lastFreqBlockOffset;
    proxBlockPointer[level] = lastProxBlockPointer;
    proxBlockOffset[level] = lastProxBlockOffset;
    payloadLength[level] = lastPayloadLength;
//    System.err.println( "seekChild: level=" + level + ", lastFreqBlockPointer="+lastFreqBlockPointer + ", lastFreqBlockOffset=" + lastFreqBlockOffset);
  }
  
  @Override
  protected void setLastSkipData(int level) {
    super.setLastSkipData(level);
    lastFreqBlockPointer = freqBlockPointer[level];
    lastFreqBlockOffset = freqBlockOffset[level];
    lastProxBlockPointer = proxBlockPointer[level];
    lastProxBlockOffset = proxBlockOffset[level];
    lastPayloadLength = payloadLength[level];
//    System.err.println( "setLastSkipData: level=" + level + ", lastFreqBlockPointer="+lastFreqBlockPointer + ", lastFreqBlockOffset=" + lastFreqBlockOffset);
  }


  @Override
  protected int readSkipData(int level, IndexInput skipStream) throws IOException {
    int delta;
    if (currentFieldStoresPayloads) {
      // the current field stores payloads.
      // if the doc delta is odd then we have
      // to read the current payload length
      // because it differs from the length of the
      // previous payload
      delta = skipStream.readVInt();
      if ((delta & 1) != 0) {
        payloadLength[level] = skipStream.readVInt();
      }
      delta >>>= 1;
    } else {
      delta = skipStream.readVInt();
    }
//    freqPointer[level] += skipStream.readVInt();
//    proxPointer[level] += skipStream.readVInt();

    long saved = freqBlockPointer[level];
    freqBlockPointer[level] += skipStream.readVInt();
    if( saved == freqBlockPointer[level] )
    {
	freqBlockOffset[level] += skipStream.readVInt();
    }
    else
    {
	freqBlockOffset[level] = skipStream.readVInt();
    }
//    System.err.println( "Saved: " + saved + " freqBlockPointer["+level+"]=" + freqBlockPointer[level] + ", freqBlockOffset=" + freqBlockOffset[level] );

    saved = proxBlockPointer[level];
    proxBlockPointer[level] += skipStream.readVInt();
    if( saved == proxBlockPointer[level] )
    {
	proxBlockOffset[level] += skipStream.readVInt();
    }
    else
    {
	proxBlockOffset[level] = skipStream.readVInt();
    }
//    System.err.println( "Saved: " + saved + " proxBlockPointer[level]=" + proxBlockPointer[level] + ", proxBlockOffset=" + proxBlockOffset[level] );

    return delta;
  }

    @Override
    public int skipTo(int target) throws IOException {
        return super.skipTo(target);
    }
}
