package org.apache.lucene.index.codecs.yandex2;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.Closeable;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;

import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.index.FieldsEnum;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.SegmentInfo;
import org.apache.lucene.index.TermState;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.index.codecs.FieldsProducer;
import org.apache.lucene.index.codecs.PostingsReaderBase;
import org.apache.lucene.index.codecs.TermsIndexReaderBase;
import org.apache.lucene.index.codecs.TermsConsumer;
import org.apache.lucene.index.codecs.TermStats;
import org.apache.lucene.index.codecs.VariableGapTermsIndexReader;
import org.apache.lucene.index.codecs.bloom.FuzzySet; // javadocs
import org.apache.lucene.index.codecs.bloom.FuzzySet.ContainsResult; // javadocs
//import org.apache.lucene.index.codecs.bloom.NativeFuzzySet; // javadocs
//import org.apache.lucene.index.codecs.bloom.NativeFuzzySet.ContainsResult; // javadocs
import org.apache.lucene.index.codecs.yandex.YandexPostingsReader; // javadocs
import org.apache.lucene.index.codecs.yandex.YandexPostingsReader.YandexTermState; // javadocs
import org.apache.lucene.index.codecs.yandex.YandexPostingsWriter; // javadocs
import org.apache.lucene.store.ByteArrayDataInput;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.DeflateDecompressor;
import org.apache.lucene.store.Decompressor;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.ChainedBlockCompressedInputStream;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.CodecUtil;
import org.apache.lucene.util.CloseableThreadLocal;
import org.apache.lucene.util.DoubleBarrelLRUCache;

import ru.yandex.msearch.util.JavaAllocator;

import ru.yandex.util.string.StringUtils;

/** Handles a terms dict, but decouples all details of
 *  doc/freqs/positions reading to an instance of {@link
 *  PostingsReaderBase}.  This class is reusable for
 *  codecs that use a different format for
 *  docs/freqs/positions (though codecs are also free to
 *  make their own terms dict impl).
 *
 * <p>This class also interacts with an instance of {@link
 * TermsIndexReaderBase}, to abstract away the specific
 * implementation of the terms dict index. 
 * @lucene.experimental */

public class Yandex2TermsReader extends FieldsProducer {
    private static final boolean DEBUG = false;
    private static final boolean DEBUG2 = false;
    private static final Comparator<BytesRef> COMPARATOR =
        BytesRef.getUTF8SortedAsUnicodeComparator();
    private static final JavaAllocator ALLOCATOR =
        JavaAllocator.get("Yandex2TermsReaderBlockCache");
    private static final String TERMS_TAG = "terms";
    private static LongAdder bloomSuccess = new LongAdder();
    private static LongAdder bloomFail = new LongAdder();
    private static LongAdder seekExactTotal = new LongAdder();
    private final IndexInput in;
    private final Compressor compressor;
    private final Decompressor decompressor;

    private final PostingsReaderBase postingsReader;

    private final TreeMap<String,FieldReader> fields =
        new TreeMap<String,FieldReader>();
    private final HashMap<String,FieldReader> fieldsFast =
        new HashMap<String,FieldReader>();
    private final HashMap<String,FuzzySet> bloomsByFieldName;

    // Caches the most recently looked-up field + terms:
    private final DoubleBarrelLRUCache<FieldAndTerm,YandexTermState> termsCache;
    private static final LongAdder cacheSuccess = new LongAdder();
    private static final LongAdder cacheFail = new LongAdder();

    private TermsIndexReaderBase indexReader;
    private final String segment;

    // keeps the dirStart offset
    protected long dirOffset;

    static {
        Thread bloomStats = new Thread("BloomStats2") {
            public double round(final double value) {
                double round = value * 100;
                round = (double)((int) round);
                return round / 100;
            }

            @Override
            public void run() {
                long prevHits = 0;
                long prevMisses = 0;
                long prevSeeks = 0;
                Averager hits10sec = new Averager(2);
                Averager hits60sec = new Averager(12);
                Averager hits300sec = new Averager(60);
                Averager misses10sec = new Averager(2);
                Averager misses60sec = new Averager(12);
                Averager misses300sec = new Averager(60);
                String prevHitsStats = "";
                String prevMissesStats = "";
                while (true) {
                    final long currentSeeks = seekExactTotal.sum();
                    final long currentHits = bloomSuccess.sum();
                    final long currentMisses = bloomFail.sum();
                    final long hits = currentHits - prevHits;
                    final long misses = currentMisses - prevMisses;
                    final long seeks = currentSeeks - prevSeeks;
                    if (seeks != 0) {
                        prevHits = currentHits;
                        prevMisses = currentMisses;
                        prevSeeks = currentSeeks;
                        double hitRatio =
                            (hits * 100.0) /
                            (double)(seeks);
                        hits10sec.push(hitRatio);
                        hits60sec.push(hitRatio);
                        hits300sec.push(hitRatio);
                        double missRatio =
                            (misses * 100.0) /
                            (double)(seeks);
                        misses10sec.push(missRatio);
                        misses60sec.push(missRatio);
                        misses300sec.push(missRatio);
                    }
                    double totalHitRatio =
                        (currentHits * 100.0) / (double)(currentSeeks);
                    double totalMissRatio =
                        (currentMisses * 100.0) / (double)(currentSeeks);
                    String hitsStats = "BloomFilter2.hits: "
                            + "lifetimeRatio=" + round(totalHitRatio)
                            + ", 10secRatio=" + round(hits10sec.value())
                            + ", 60secRatio=" + round(hits60sec.value())
                            + ", 300secRatio=" + round(hits300sec.value());
                    String missesStats = "BloomFilter2.misses: "
                            + "lifetimeRatio=" + round(totalMissRatio)
                            + ", 10secRatio=" + round(misses10sec.value())
                            + ", 60secRatio=" + round(misses60sec.value())
                            + ", 300secRatio=" + round(misses300sec.value());
                    if (!hitsStats.equals(prevHitsStats)
                        || !missesStats.equals(prevMissesStats))
                    {
                        System.err.println(hitsStats);
                        System.err.println(missesStats);
                        prevHitsStats = hitsStats;
                        prevMissesStats = missesStats;
                    }
                    try {
                        Thread.sleep(5000);
                    } catch (InterruptedException ign) {
                    }
                }
            }
        };
        bloomStats.setDaemon(true);
        bloomStats.start();
    }

  static {
    Thread bloomStats = new Thread("Terms2CacheStats") {
        public double round(final double value) {
            double round = value * 100;
            round = (double)((int) round);
            return round / 100;
        }

        @Override
        public void run() {
            long prevHits = 0;
            long prevMisses = 0;
            Averager avg10sec = new Averager(2);
            Averager avg60sec = new Averager(12);
            Averager avg300sec = new Averager(60);
            String prevStats = "";
            while (true) {
                final long currentHits = cacheSuccess.sum();
                final long currentMisses = cacheFail.sum();
                final long hits = currentHits - prevHits;
                final long misses = currentMisses - prevMisses;
                if (hits != 0 || misses != 0) {
                    prevHits = currentHits;
                    prevMisses = currentMisses;
                    double diffRatio =
                        (hits * 100.0) /
                        (double)(hits + misses);
                    avg10sec.push(diffRatio);
                    avg60sec.push(diffRatio);
                    avg300sec.push(diffRatio);
                }
                double totalRatio =
                    (currentHits * 100.0) /
                    (double)(currentHits + currentMisses);
                String stats = "Terms2Cache: "
                        + "lifetimeRatio=" + round(totalRatio)
                        + ", 10secRatio=" + round(avg10sec.value())
                        + ", 60secRatio=" + round(avg60sec.value())
                        + ", 300secRatio=" + round(avg300sec.value());
                if (!stats.equals(prevStats)) {
                    System.err.println(stats);
                    prevStats = stats;
                }
                try {
                    Thread.sleep(5000);
                } catch (InterruptedException ign) {
                }
            }
        }
    };
    bloomStats.setDaemon(true);
    bloomStats.start();
  }

    private static class Averager {
        public double sum = 0;
        public double[] values;
        public int idx = 0;
        public int count = 0;

        public Averager(int depth) {
            this.values = new double[depth];
        }

        public void push(double value) {
            sum += value;
            sum -= values[idx];
            values[idx++] = value;
            if (idx >= values.length) {
                idx = 0;
            }
            if (count < values.length) {
                count++;
            }
        }

        public double value() {
            return sum / count;
        }
    }

    private static class FieldAndTerm extends DoubleBarrelLRUCache.CloneableKey {
        String field;
        BytesRef term;

        public FieldAndTerm() {
        }

        public FieldAndTerm(FieldAndTerm other) {
            field = other.field;
            term = new BytesRef(other.term);
        }

        @Override
        public boolean equals(Object _other) {
            FieldAndTerm other = (FieldAndTerm) _other;
            return other.field == field && term.bytesEquals(other.term);
        }

        @Override
        public Object clone() {
            return new FieldAndTerm(this);
        }

        @Override
        public int hashCode() {
            return field.hashCode() * 31 + term.hashCode();
        }
    }

    public Yandex2TermsReader(
        final Compressor compressor,
        final TermsIndexReaderBase indexReader,
        final Directory dir,
        final FieldInfos fieldInfos,
        final String segment,
        final PostingsReaderBase postingsReader,
        final int readBufferSize,
        int termsCacheSize,
        final String codecId,
        final Set<String> bloomSet,
        final Set<String> indexedFields)
        throws IOException
    {
        this.compressor = compressor;
        this.decompressor = compressor.decompressor();
        this.postingsReader = postingsReader;
        this.segment = segment;

        in = dir.openInput(
            IndexFileNames.segmentFileName(
                segment,
                codecId,
                Yandex2TermsWriter.TERMS_EXTENSION),
            readBufferSize);

        boolean success = false;
        int totalNumTerms = 0;
        try {
            bloomsByFieldName = new HashMap<>();
            final String bloomFileName = IndexFileNames.segmentFileName(
                segment,
                codecId,
                Yandex2TermsWriter.BLOOM_EXTENSION);
            if (dir.fileExists(bloomFileName)) {
                try (IndexInput bloomIn =
                        dir.openInput(bloomFileName, readBufferSize))
                {
                    CodecUtil.checkHeader(
                        bloomIn,
                        Yandex2TermsWriter.BLOOM_CODEC_NAME,
                        Yandex2TermsWriter.BLOOM_CODEC_VERSION,
                        Yandex2TermsWriter.BLOOM_CODEC_VERSION);
                    int numBlooms = bloomIn.readInt();
                    for (int i = 0; i < numBlooms; i++) {
                        int fieldNum = bloomIn.readInt();
                        final FieldInfo fieldInfo =
                            fieldInfos.fieldInfo(fieldNum);
                        FuzzySet bloom =
                            FuzzySet.deserialize(bloomIn, fieldInfo.name);
                        if (bloomSet.contains(fieldInfo.name)
                            && (indexedFields == null
                                || indexedFields.contains(fieldInfo.name))
                            )
                        {
                            bloomsByFieldName.put(fieldInfo.name, bloom);
                        } else {
                            bloom.close();
                        }
                    }
                }
            }

            readHeader(in);

            // Have PostingsReader init itself
            postingsReader.init(in);

            // Read per-field details
            seekDir(in, dirOffset);

            final int numFields = in.readVInt();

            this.indexReader = indexReader;

            for (int i = 0; i < numFields; i++) {
                final int field = in.readVInt();
                final long numTerms = in.readVLong();
                totalNumTerms += numTerms;
                final long termsStartPointer = in.readVLong();
                final int fullTermInterval = in.readVInt();
                final FieldInfo fieldInfo = fieldInfos.fieldInfo(field);
                final long sumTotalTermFreq = fieldInfo.omitTermFreqAndPositions ? -1 : in.readVLong();
                if (indexedFields == null || indexedFields.contains(fieldInfo.name)) {
                    final FieldReader fr = 
                        new FieldReader(
                            fieldInfo,
                            numTerms,
                            termsStartPointer,
                            sumTotalTermFreq,
                            fullTermInterval,
                            bloomsByFieldName.get(fieldInfo.name));
                    fields.put(fieldInfo.name, fr);
                    fieldsFast.put(fieldInfo.name, fr);
                }
            }

            success = true;
        } finally {
            if (!success) {
                in.close();
            }
        }

        //TODO: make this cache common for all segments
        if (termsCacheSize == -1) {
            termsCacheSize = totalNumTerms * 1024 / 10000000;
            if (termsCacheSize < 50) {
                termsCacheSize = 50;
            }
            if (termsCacheSize > 1024) {
                termsCacheSize = 1024;
            }
        }
        termsCache = new DoubleBarrelLRUCache<>(termsCacheSize);
    }

    private void debug(final String msg) {
        final String thread = Thread.currentThread().getName();
        if ((DEBUG || DEBUG2) && !thread.startsWith("StaticMerge")) {
            System.err.println(thread + " <" + segment + ">: " + msg);
        }
    }

    private String codecName() {
        if (compressor.id() != null) {
            return StringUtils.concat(
                Yandex2TermsWriter.CODEC_NAME,
                '_',
                compressor.id());
        } else {
            return Yandex2TermsWriter.CODEC_NAME;
        }
    }

    protected void readHeader(final IndexInput input) throws IOException {
        CodecUtil.checkHeader(
            in,
            codecName(),
            Yandex2TermsWriter.VERSION_START,
            Yandex2TermsWriter.VERSION_CURRENT);
        dirOffset = in.readLong();
    }

    protected void seekDir(final IndexInput input, final long dirOffset)
        throws IOException
    {
        input.seek(dirOffset);
    }

    @Override
    public void loadTermsIndex(final int indexDivisor) throws IOException {
        indexReader.loadTermsIndex(indexDivisor);
    }

    @Override
    public void close() throws IOException {
        try {
            for (FuzzySet bloom : bloomsByFieldName.values()) {
                bloom.close();
            }
            bloomsByFieldName.clear();
            try {
                if (indexReader != null) {
                    indexReader.close();
                }
            } finally {
                indexReader = null;
                if (in != null) {
                    in.close();
                }
            }
        } finally {
            try {
                if (postingsReader != null) {
                    postingsReader.close();
                }
            } finally {
                for(FieldReader field : fieldsFast.values()) {
                    field.close();
                }
            }
        }
    }

    public static void files(
        final Directory dir,
        final SegmentInfo segmentInfo,
        final String id,
        final Collection<String> files)
        throws IOException
    {
        files.add(
            IndexFileNames.segmentFileName(
                segmentInfo.name,
                id,
                Yandex2TermsWriter.TERMS_EXTENSION));
        final String bloomFile =
            IndexFileNames.segmentFileName(
                segmentInfo.name,
                id,
                Yandex2TermsWriter.BLOOM_EXTENSION);
        if (dir.fileExists(bloomFile)) {
            files.add(bloomFile);
        }
    }

    public static void getExtensions(final Collection<String> extensions) {
        extensions.add(Yandex2TermsWriter.TERMS_EXTENSION);
        extensions.add(Yandex2TermsWriter.BLOOM_EXTENSION);
    }

    @Override
    public FieldsEnum iterator() {
        return new TermFieldsEnum();
    }

    @Override
    public Terms terms(String field) throws IOException {
        return fieldsFast.get(field);
    }

    private class TermFieldsEnum extends FieldsEnum {
        final Iterator<FieldReader> it;
        FieldReader current;

        TermFieldsEnum() {
            it = fields.values().iterator();
        }

        @Override
        public String next() {
            if (it.hasNext()) {
                current = it.next();
                return current.fieldInfo.name;
            } else {
                current = null;
                return null;
            }
        }

        @Override
        public TermsEnum terms(final boolean buffered) throws IOException {
            return current.iterator(buffered);
        }
    }

    private class FieldReader extends Terms implements Closeable {
        final long numTerms;
        final FieldInfo fieldInfo;
        final long termsStartPointer;
        final long sumTotalTermFreq;
        final int fullTermInterval;
        final FuzzySet bloomFilter;
        final ConcurrentLinkedQueue<TermsEnum> cachedEnums =
            new ConcurrentLinkedQueue<TermsEnum>();
        final int indexDivisor;

        FieldReader(
            final FieldInfo fieldInfo,
            final long numTerms,
            final long termsStartPointer,
            final long sumTotalTermFreq,
            final int fullTermInterval,
            final FuzzySet bloomFilter)
        {
            this.fieldInfo = fieldInfo;
            this.numTerms = numTerms;
            this.termsStartPointer = termsStartPointer;
            this.sumTotalTermFreq = sumTotalTermFreq;
            this.bloomFilter = bloomFilter;
            this.fullTermInterval = fullTermInterval;
            this.indexDivisor =
                ((VariableGapTermsIndexReader) indexReader)
                    .getDivisor(fieldInfo);
        }

        @Override
        public Comparator<BytesRef> getComparator() {
            return BytesRef.getUTF8SortedAsUnicodeComparator();
        }

        @Override
        public void close() {
            cachedEnums.clear();
            super.close();
        }

        @Override
        public TermsEnum iterator(final boolean buffered) throws IOException {
            return new SegmentTermsEnum(buffered, false);
        }

        @Override
        public TermsEnum reverseIterator() throws IOException {
            return new ReverseSegmentTermsEnum();
        }

        @Override
        public long getUniqueTermCount() {
            return numTerms;
        }

        @Override
        public TermsEnum getThreadTermsEnum(final boolean buffered)
            throws IOException
        {
            TermsEnum te = cachedEnums.poll();
            if (te != null) {
                ((SegmentTermsEnum)te).reset(buffered);
            } else {
                te = new SegmentTermsEnum(buffered, true);
            }
            return te;
        }

        @Override
        public long getSumTotalTermFreq() {
            return sumTotalTermFreq;
        }

        private final class SegmentTermsEnum extends TermsEnum {
            private final IndexInput in;
            private final YandexTermState state;
            private final boolean doOrd;
            private final FieldAndTerm cacheKey = new FieldAndTerm();
            private final TermsIndexReaderBase.FieldIndexEnum indexEnum;
            private final BytesRef currentTerm = new BytesRef();
            private boolean buffered;
            private boolean bufferLocked = false;
            private boolean reused = false;
            private final boolean pooled;
            private int[] fullTermsOffsets;
            private int fullTermsCount;
            private int termsDataOffset;
            private int indexDivisor;

            private boolean seekPending = false;

            ChainedBlockCompressedInputStream bytesReader = null;

            public SegmentTermsEnum(
                final boolean buffered,
                final boolean pooled)
                throws IOException
            {
                in = (IndexInput) Yandex2TermsReader.this.in.clone();
                indexEnum = indexReader.getFieldEnum(fieldInfo);
                doOrd = indexReader.supportsOrd();
                cacheKey.field = fieldInfo.name;
                state = (YandexTermState) postingsReader.newTermState();
                state.totalTermFreq = -1;
                state.ord = -1;
                state.blockFirstTerm = new BytesRef();
                state.blockFilePointer = termsStartPointer;

                this.buffered = buffered;
                this.pooled = pooled;
                if (FieldReader.this.indexDivisor < 0) {
                    this.indexDivisor =
                        ((VariableGapTermsIndexReader) indexReader)
                            .getDivisor(fieldInfo);
                } else {
                    this.indexDivisor = FieldReader.this.indexDivisor;
                }
            }

            @Override
            public void close() {
                if (!pooled) {
                    return;
                }
                if (bytesReader != null) {
                    try {
                        bytesReader.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                    bytesReader = null;
                }
                cachedEnums.offer(this);
            }

            public void reset(final boolean buffered) throws IOException {
                if (indexDivisor < 0) {
                    indexDivisor =
                        ((VariableGapTermsIndexReader) indexReader)
                            .getDivisor(fieldInfo);
                }
                state.blockFilePointer = termsStartPointer;
                state.termPointer = 0;
                state.totalTermFreq = -1;
                state.ord = -1;
                state.termCount = 0;
                state.blockTermCount = 0;
                this.buffered = buffered;
                if (bytesReader != null) {
                    bytesReader.useCache(buffered);
                }
                seekPending = true;
                reused = true;
            }

            @Override
            public Comparator<BytesRef> getComparator() {
                return BytesRef.getUTF8SortedAsUnicodeComparator();
            }

            private ChainedBlockCompressedInputStream bytesReader()
                throws IOException
            {
                if (bytesReader != null) {
                    return bytesReader;
                }
                bytesReader = new ChainedBlockCompressedInputStream(
                        in,
                        decompressor,
                        ALLOCATOR,
                        TERMS_TAG);
                bytesReader.useCache(buffered);
                postingsReader.readTermsBlock(bytesReader, fieldInfo, state);
                return bytesReader;
            }

            private void initBytesReader()
                throws IOException
            {
                if (bytesReader == null) {
                    bytesReader = new ChainedBlockCompressedInputStream(
                            in,
                            decompressor,
                            ALLOCATOR,
                            TERMS_TAG);
                    bytesReader.useCache(buffered);
                    postingsReader.readTermsBlock(bytesReader, fieldInfo, state);
                    if (DEBUG) {
                        debug("LAZY bytesreader init: "
                            + "filePointer: " + state.blockFilePointer
                            + " blockPointer: " + state.termPointer);
                    }
                    bytesReader.seek(
                        state.blockFilePointer,
                        state.termPointer);
                    seekPending = false;
                } else {
                    if (seekPending) {
                        if (DEBUG) {
                            debug("Seeking pending: " + state);
                        }
                        bytesReader.seek(
                            state.blockFilePointer,
                            state.termPointer);
                    }
                }
                seekPending = false;
            }

            @Override
            public int seekExactEx(
                final BytesRef text,
                final boolean useCache) throws IOException
            {
                if (bloomFilter != null) {
                    seekExactTotal.add(1);
                    if (bloomFilter.contains(text) == ContainsResult.NO) {
                        if (DEBUG) {
                            debug("seekExact: bloom reject");
                        }
                        bloomSuccess.add(1);
                        return -1;
                    }
                }
                final boolean found = seek(text, useCache) == SeekStatus.FOUND;
                if (!found && bloomFilter != null) {
                    bloomFail.add(1);
                }
                if (found) {
                    return 1;
                } else {
                    return 0;
                }
            }

            public boolean seekExact(
                final BytesRef text,
                final boolean useCache)
                throws IOException
            {
                if (bloomFilter != null) {
                    seekExactTotal.add(1);
                    if (bloomFilter.contains(text) == ContainsResult.NO) {
                        if (DEBUG) {
                            debug("seekExact: bloom reject");
                        }
                        bloomSuccess.add(1);
                        return false;
                    }
                }
                final boolean found = seek(text, useCache) == SeekStatus.FOUND;
                if (!found && bloomFilter != null) {
                    bloomFail.add(1);
                }
                return found;
            }

            @Override
            public SeekStatus seek(
                final BytesRef target,
                final boolean useCache)
                throws IOException
            {

                if (indexEnum == null) {
                    throw new IllegalStateException(
                        "terms index was not loaded");
                }

                if (useCache) {
                    cacheKey.term = target;
                    final YandexTermState cachedState =
                        termsCache.get(cacheKey);
                    if (cachedState != null) {
                        if (DEBUG) {
                            debug("seek cache hit: " + cachedState);
                        }
                        cacheSuccess.add(1);
                        seek(target, cachedState);
                        return state.seekStatus;
                    } else {
                        cacheFail.add(1);
                    }
                }
                seekPending = false;

                long fileOffset = indexEnum.seek(target);
                currentTerm.copy(indexEnum.term());

                if (fileOffset == 0) {
                    return SeekStatus.END;
                }

                if (COMPARATOR.compare(target, currentTerm) < 0) {
                    return SeekStatus.NOT_FOUND;
                }
                int blocksToScan = indexDivisor;
                SeekStatus status = SeekStatus.END;
                if (DEBUG) {
                    debug("BlocksToScan: " + blocksToScan);
                }
                state.blockFilePointer = fileOffset;
                state.termPointer = 0;
                bytesReader().seek(state.blockFilePointer, state.termPointer);
                while (blocksToScan-- > 0) {
                    status = seekBlock(target, useCache);
                    if (DEBUG) {
                        debug("Block status: " + status);
                    }
                    if (status == SeekStatus.END) {
                        // This is last block which we are trying to scan, but
                        // still no suitable term found at the end of block.
                        // Probably target term is a prefix for the first term
                        // in next block
                        ++blocksToScan;
                    } else {
                        if (status == null) {
                            status = SeekStatus.END;
                        }
                        break;
                    }
                    if (DEBUG) {
                        debug("BlocksToScan: " + blocksToScan);
                    }
                }
                return status;
            }

            private SeekStatus seekBlock(
                final BytesRef target,
                final boolean useCache)
                throws IOException
            {
                boolean locked = false;
                try {
                    locked = lockBuffer();
                    if (bytesReader.eof()) {
                        return null;
                    }
                    loadBlock();

                    int scans = 1;
                    //compare first block to fail fast
                    int cmp = COMPARATOR.compare(currentTerm, target);
                    if (cmp > 0) {
                        if (DEBUG2) {
                            debug("NOT_FOUND_FAST: "
                                + target.utf8ToString() + " <> "
                                + currentTerm.utf8ToString() + ", scans: "
                                + scans);
                        }
                        if (useCache) {
                            cacheTerm(currentTerm, SeekStatus.NOT_FOUND);
                        }
                        return SeekStatus.NOT_FOUND;
                    } else if (cmp == 0) {
                        if (DEBUG2) {
                            debug("FOUND: "
                                + target.utf8ToString() + " <> "
                                + currentTerm.utf8ToString() + ", scans: "
                                + scans);
                        }
                        if (useCache) {
                            cacheTerm(currentTerm, SeekStatus.FOUND);
                        }
                        return SeekStatus.FOUND;
                    }
                    //////


                    //binary search subblock
                    int low = 0;
                    int high = fullTermsCount - 1;
                    int mid = low;
                    while (low <= high) {
                        mid = (low + high) >>> 1;
                        int fullTermOffset = fullTermsOffsets[mid] + termsDataOffset;
                        loadFullTerm(fullTermOffset, mid * fullTermInterval);
                        if (DEBUG) {
                            debug("MID: " + mid
                                + ", term: " + currentTerm.utf8ToString());
                        }
                        cmp = COMPARATOR.compare(currentTerm, target);
                        scans++;

                        if (cmp < 0) {
                            low = mid + 1;
                        } else if (cmp > 0) {
                            high = mid - 1;
                        } else {
                            //exact match
                            if (useCache) {
                                cacheTerm(currentTerm, SeekStatus.FOUND);
                            }
                            return SeekStatus.FOUND;
                        }
                    }
                    if (mid != high) {
                        if (DEBUG) {
                            debug("mid != high: " + mid + " != " + high
                                + " low: " + low);
                        }
                        mid = high;
                        loadFullTerm(
                            fullTermsOffsets[mid] + termsDataOffset,
                            mid * fullTermInterval);
                    } else {
                        if (DEBUG) {
                            debug("mid == high: " + mid + " == " + high);
                        }
                    }
                    //full scan next fullTermInterval elements
                    while (state.termCount < state.blockTermCount) {
                        loadDiffTerm();
                        scans++;
                        if (DEBUG) {
                            debug("next: " + currentTerm.utf8ToString());
                        }
                        cmp = COMPARATOR.compare(currentTerm, target);
                        if (cmp > 0) {
                            if (DEBUG2) {
                                debug("NOT_FOUND: "
                                    + target.utf8ToString() + " <> "
                                    + currentTerm.utf8ToString() + ", scans: "
                                    + scans);
                            }
                            if (useCache) {
                                cacheTerm(currentTerm, SeekStatus.NOT_FOUND);
                            }
                            return SeekStatus.NOT_FOUND;
                        } else if (cmp == 0) {
                            if (DEBUG2) {
                                debug("FOUND: "
                                    + target.utf8ToString() + " <> "
                                    + currentTerm.utf8ToString() + ", scans: "
                                    + scans);
                            }
                            if (useCache) {
                                cacheTerm(currentTerm, SeekStatus.FOUND);
                            }
                            return SeekStatus.FOUND;
                        }
                    }
                    if (DEBUG2) {
                        debug("SeekStatus.END: "
                            + target.utf8ToString() + ", scans=" + scans);
                    }
                    if (useCache) {
                        cacheTerm(currentTerm, SeekStatus.END);
                    }

                    if (mid == fullTermsCount - 1) {
                        // Nothing found in last subblock
                        return SeekStatus.END;
                    } else {
                        // Nothing found in mid subblock, while next subblock
                        // contains terms which are bigger than target term,
                        // so current block doesn't contain target term
                        return SeekStatus.NOT_FOUND;
                    }
                } finally {
                    if (locked) {
                        releaseBuffer();
                    }
                }
            }

            private String debugPrintState(final YandexTermState state) {
                BytesRef term = state.term;
                BytesRef blockFirstTerm = state.blockFirstTerm;
                return
                    " freqFP=" + state.freqBlockStart
//                    + " proxFP=" + state.proxBlockStart
//                    " skipOffset=" + state.skipBlockStart
                    + " term="
                        + (
                            term != null ?
                                "len:" + term.length
                                + ",arr.len:" + term.bytes.length
                                + ",off:" + term.offset
                                + ",s:" + (term.length + term.offset
                                    <= term.bytes.length ?
                                    term.utf8ToString() : "err")
                                : "null")
                    + " blockFirstTerm="
                        + (
                            blockFirstTerm != null ?
                                "len:" + blockFirstTerm.length
                                + ",arr.len:" + blockFirstTerm.bytes.length
                                + ",off:" + blockFirstTerm.offset
                                + ",s:" + (blockFirstTerm.length + blockFirstTerm.offset
                                    <= blockFirstTerm.bytes.length ?
                                    blockFirstTerm.utf8ToString() : "err")
                                : "null")
                    + " blockTermCount=" + state.blockTermCount
                    + " termPointer=" + state.termPointer
                    + " seekStatus=" + state.seekStatus
                    + " fullTerm=" + state.fullTerm
                    + " blockTermCount= " + state.blockTermCount;
            }


            private void cacheTerm(
                final BytesRef term,
                final SeekStatus seekStatus)
            {
                YandexTermState clonedState = null;
                try {
                    state.blockFilePointer =
                        bytesReader.getCurrentBlockFilePointer();
//                state.termPointer = bytesReader.getFilePointer();
                    state.seekStatus = seekStatus;
                    clonedState = (YandexTermState)state.clone();
                    clonedState.term = new BytesRef(term);
                    clonedState.blockFirstTerm = new BytesRef(state.blockFirstTerm);
                    termsCache.put(new FieldAndTerm(cacheKey), clonedState);
                } catch (Exception e) {
                    System.err.println("DEBUG: state: " + debugPrintState(state)
                        + ", clonedState: " + debugPrintState(clonedState));
                    e.printStackTrace();
                }
            }

            private final boolean lockBuffer() throws IOException {
                if (bytesReader != null && !bufferLocked) {
                    bytesReader.lockBuffer();
                    bufferLocked = true;
                    return true;
                } else {
                    return false;
                }
            }

            private final void releaseBuffer() throws IOException {
                if (bufferLocked) {
                    bufferLocked = false;
                    bytesReader.releaseBuffer();
                }
            }

            private void loadBlock() throws IOException {
                if (DEBUG) {
                    debug("LOADBLOCK");
                }
                state.blockFilePointer = bytesReader.nextFilePointer();
                final ChainedBlockCompressedInputStream bytesReader =
                    bytesReader();
                state.blockTermCount = bytesReader.readVIntUnlocked();
                fullTermsCount = bytesReader.readVIntUnlocked();
                if (DEBUG) {
                    debug("nextBlock: " + bytesReader.isNextBlock());
                }
                if (bytesReader.isNextBlock()) {
                    bytesReader.clearNextBlock();
                }
                if (DEBUG) {
                    debug("blockTermCount: " + state.blockTermCount);
                    debug("fullTermCount: " + fullTermsCount);
                }
                if (fullTermsOffsets == null
                    || fullTermsOffsets.length < fullTermsCount)
                {
                    fullTermsOffsets = new int[fullTermsCount << 1];
                }
                int offset = 0;
                for (int i = 0; i < fullTermsCount; i++) {
                    offset += bytesReader.readVIntUnlocked();
                    fullTermsOffsets[i] = offset;
                }
                termsDataOffset = (int) bytesReader.getFilePointer();
                state.termCount = 0;
                loadDiffTerm();
                state.blockFirstTerm.copy(currentTerm);
            }

            //this should be callend under locked bytesReader
            private void loadFullTerm(final int offset, final int termNum)
                throws IOException
            {
                bytesReader.seek(offset);
                loadFullTerm(termNum);
            }

            //this should be callend under locked bytesReader
            private void loadFullTerm(final int termNum)
                throws IOException
            {
                if (DEBUG) {
                    debug("loadFullTerm: fp: "
                        + bytesReader.getCurrentBlockFilePointer()
                        + ", fp2: " + bytesReader.nextFilePointer()
                        + ", bp: " + bytesReader.getFilePointer()
                        + ", bl: " + bytesReader.length());
                }
                final int prefix = bytesReader.readVIntUnlocked();
                final int suffix = bytesReader.readVIntUnlocked();
                if (DEBUG) {
                    debug("FullTerm: prefix: " + prefix + " suffix: "
                        + suffix + ", num: " + termNum);
                }
                currentTerm.length = prefix + suffix;
                if (currentTerm.bytes.length < currentTerm.length) {
                    currentTerm.grow(currentTerm.length);
                }
                System.arraycopy(
                    state.blockFirstTerm.bytes,
                    0,
                    currentTerm.bytes,
                    0,
                    prefix);
                bytesReader.readBytesUnlocked(
                    currentTerm.bytes,
                    prefix,
                    suffix);
                state.docFreq = bytesReader.readVIntUnlocked();
                if (!fieldInfo.omitTermFreqAndPositions) {
                    state.totalTermFreq =
                        state.docFreq + bytesReader.readVLongUnlocked();
                }
                postingsReader.readTermsBlock(
                    bytesReader,
                    fieldInfo,
                    state);
                state.fullTerm = true;
                postingsReader.nextTerm(fieldInfo, state);
                state.termCount = termNum + 1;
                state.termPointer = bytesReader.getFilePointer();
            }

            private void loadDiffTerm() throws IOException {
                final boolean locked = lockBuffer();
                try {
                    if ((state.termCount) % fullTermInterval == 0) {
                        loadFullTerm(state.termCount);
                    } else {
                        if (DEBUG) {
                            debug("loadDiffTerm: fp: "
                            + bytesReader.getCurrentBlockFilePointer()
                            + ", fp2: " + bytesReader.nextFilePointer()
                            + ", bp: " + bytesReader.getFilePointer()
                            + ", bl: " + bytesReader.length());
                        }
                        final int prefix = bytesReader.readVIntUnlocked();
                        final int suffix = bytesReader.readVIntUnlocked();
                        if (DEBUG) {
                            debug("nextBlock: " + bytesReader.isNextBlock());
                            debug("Diff prefix: " + prefix + " suffix: "
                                + suffix);
                        }
                        currentTerm.length = prefix + suffix;
                        if (currentTerm.bytes.length < currentTerm.length) {
                            currentTerm.grow(currentTerm.length);
                        }
                        bytesReader.readBytesUnlocked(
                            currentTerm.bytes,
                            prefix,
                            suffix);
                        state.docFreq = bytesReader.readVIntUnlocked();
                        if (!fieldInfo.omitTermFreqAndPositions) {
                            state.totalTermFreq =
                                state.docFreq + bytesReader.readVLongUnlocked();
                        }
                        postingsReader.readTermsBlock(
                            bytesReader,
                            fieldInfo,
                            state);
                        state.fullTerm = false;
                        postingsReader.nextTerm(fieldInfo, state);
                        state.termCount++;
                        state.termPointer = bytesReader.getFilePointer();
                    }
                } finally {
                    if (locked) {
                        releaseBuffer();
                    }
                }
            }

            @Override
            public BytesRef next() throws IOException {
                initBytesReader();
                if (DEBUG) {
                    debug("NEXT: " + bytesReader.getFilePointer()
                        + "/" + bytesReader.length());
                }
                final boolean locked = lockBuffer();
                try {
                    if (state.termCount >= state.blockTermCount) {
                        if (!bytesReader.eof()) {
                            loadBlock();
                        } else {
                            return null;
                        }
                    } else {
                        loadDiffTerm();
                    }
                    if (DEBUG) {
                        debug("TermPointer: " + state.termPointer);
                    }
                    return currentTerm;
                } finally {
                    if (locked) {
                        releaseBuffer();
                    }
                }
            }

            @Override
            public BytesRef term() {
                return currentTerm;
            }

            @Override
            public int docFreq() throws IOException {
                return state.docFreq;
            }

            @Override
            public long totalTermFreq() throws IOException {
                return state.totalTermFreq;
            }

            @Override
            public DocsEnum docs(
                final Bits skipDocs,
                final DocsEnum reuse) throws IOException
            {
                final DocsEnum docsEnum = postingsReader.docs(
                    fieldInfo,
                    state,
                    skipDocs,
                    reuse,
                    buffered);
                return docsEnum;
            }

            @Override
            public DocsAndPositionsEnum docsAndPositions(
                final Bits skipDocs,
                final DocsAndPositionsEnum reuse)
                throws IOException
            {
                if (fieldInfo.omitTermFreqAndPositions) {
                    return null;
                } else {
                    DocsAndPositionsEnum dpe = postingsReader.docsAndPositions(
                        fieldInfo,
                        state,
                        skipDocs,
                        reuse,
                        buffered);
                return dpe;
                }
            }

            @Override
            public void seek(
                final BytesRef target,
                final TermState otherState)
                throws IOException
            {
                if (state instanceof YandexTermState) {
                    YandexTermState savedState = (YandexTermState) otherState;
                    BytesRef blockFirstTermSave = state.blockFirstTerm;
                    state.copyFrom(savedState);
                    currentTerm.copy(savedState.term);
                    state.blockFirstTerm = blockFirstTermSave;
                    state.blockFirstTerm.copy(savedState.blockFirstTerm);
                    seekPending = true;
                } else {
                    seek(target);
                }
            }

            @Override
            public TermState termState() throws IOException {
//                ts.term = currentTerm;
                if (!seekPending) {
                    state.blockFilePointer =
                        bytesReader.getCurrentBlockFilePointer();
                }
                YandexTermState ts = state.clone();
                ts.blockFirstTerm = new BytesRef(state.blockFirstTerm);
                ts.term = new BytesRef(currentTerm);
                return ts;
            }

            @Override
            public SeekStatus seek(long ord) throws IOException {
                if (indexEnum == null) {
                    throw new IllegalStateException("terms index was not loaded");
                }

                if (ord >= numTerms) {
                    state.ord = numTerms-1;
                    return SeekStatus.END;
                }

                // TODO: if ord is in same terms block and
                // after current ord, we should avoid this seek just
                // like we do in the seek(BytesRef) case
                bytesReader.seek(indexEnum.seek(ord), 0);

                // Block must exist since ord < numTerms:

                seekPending = false;

                state.ord = indexEnum.ord()-1;
                assert state.ord >= -1: "ord=" + state.ord;
                currentTerm.copy(indexEnum.term());
                loadBlock();

                // Now, scan:
                int left = (int) (ord - state.ord);
                while (left > 0) {
                    final BytesRef currentTerm = next();
                    left--;
                }

                // always found
                return SeekStatus.FOUND;
            }

            @Override
            public long ord() {
                if (!doOrd) {
                    throw new UnsupportedOperationException();
                }
                return state.ord;
            }
        }

        private final class ReverseSegmentTermsEnum extends TermsEnum {
            private final IndexInput in;
            private YandexTermState state = null;
            private final TermsIndexReaderBase.FieldIndexEnum indexEnum;
            private BytesRef term = null;
            private final boolean useCache = true;
            private boolean bufferLocked = false;
            private boolean reused = false;
            private List<YandexTermState> blockTerms = null;
            private boolean end = false;

            ChainedBlockCompressedInputStream bytesReader = null;

            public ReverseSegmentTermsEnum() throws IOException {
                in = (IndexInput) Yandex2TermsReader.this.in.clone();
                in.seek(termsStartPointer);
                indexEnum = indexReader.getFieldEnum(fieldInfo);
                state = (YandexTermState)postingsReader.newTermState();
                state.totalTermFreq = -1;
                state.ord = -1;
            }

            @Override
            public void close() {
                if (bytesReader != null) {
                    try {
                        bytesReader.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                    bytesReader = null;
                }
            }

            public void reset() throws IOException {
                blockTerms.clear();
            }

            @Override
            public Comparator<BytesRef> getComparator() {
                return BytesRef.getUTF8SortedAsUnicodeComparator();
            }

            private void checkBytesReader() throws IOException {
                if (bytesReader != null) {
                    return;
                }
                bytesReader =
                    new ChainedBlockCompressedInputStream(
                        in,
                        decompressor,
                        ALLOCATOR,
                        TERMS_TAG);
            }

            public boolean seekExact(
                final BytesRef text,
                final boolean useCache)
                throws IOException
            {
                if (bloomFilter != null
                    && bloomFilter.contains(text) == ContainsResult.NO)
                {
                    bloomSuccess.add(1);
                    return false;
                }
                final boolean found = seek(text, useCache) == SeekStatus.FOUND;
                if (!found) {
                    bloomFail.add(1);
                }
                return found;
            }

            // TODO: we may want an alternate mode here which is
            // "if you are about to return NOT_FOUND I won't use
            // the terms data from that"; eg FuzzyTermsEnum will
            // (usually) just immediately call seek again if we
            // return NOT_FOUND so it's a waste for us to fill in
            // the term that was actually NOT_FOUND
            @Override
            public SeekStatus seek(
                final BytesRef target,
                final boolean useCache1)
                throws IOException
            {
                if (indexEnum == null) {
                    throw new IllegalStateException("terms index was not loaded");
                }
                end = false;
                long blockOffset = indexEnum.seek(target); //CEIL
                boolean skipPrev = true;
                if (blockOffset == 0) {
                    //scan to the latest block ever
                    if ((blockOffset = indexEnum.prev()) == 0) {
                        while ((blockOffset = indexEnum.prev()) == 0) {
                        }
                        skipPrev = blockOffset == termsStartPointer;
                    }
                }
                term = new BytesRef(indexEnum.term());
                final Comparator<BytesRef> termComp =
                    BytesRef.getUTF8SortedAsUnicodeComparator();

                if (skipPrev) {
                    //first prev() after seek() will point to the current block
                    // so skip it if iteration will be required later
                    final long skipOffset = indexEnum.prev();
                    if (skipOffset == termsStartPointer && indexDivisor <= 1) {
                        //skip one more index position
                        //skip first empty entry;
                         indexEnum.prev();
                    }
                }

                checkBytesReader();
                state = new YandexTermState();
                state.bytesReader = bytesReader;
                if (blockTerms == null) {
                    blockTerms = new ArrayList<>();
                } else {
                    blockTerms.clear();
                }
                boolean locked = lockBuffer();
                try {
                    bytesReader.seek(blockOffset, 0);
                    while (!bytesReader.eof()) {
                        if (bytesReader.isNextBlock()) {
                            bytesReader.clearNextBlock();
                            skipBlockIndex();
                        }
                        if ((state.termCount) % fullTermInterval == 0) {
                            state.fullTerm = true;
                        } else {
                            state.fullTerm = false;
                        }
                        final int prefix = bytesReader.readVIntUnlocked();
                        final int suffix = bytesReader.readVIntUnlocked();
                        term.length = prefix + suffix;
                        if (term.bytes.length < term.length) {
                            term.grow(term.length);
                        }
                        if (DEBUG) {
                            debug("Prefix: " + prefix + ", suffix: "
                                + suffix + ", termCount: " + state.termCount);
                        }
                        if (state.fullTerm && state.termCount != 0) {
                            System.arraycopy(
                                state.blockFirstTerm.bytes,
                                0,
                                term.bytes,
                                0,
                                prefix);
                        }
                        bytesReader.readBytesUnlocked(term.bytes, prefix, suffix);
                        if (DEBUG) {
                            debug("term: " + term.utf8ToString());
                        }

                        state.docFreq = bytesReader.readVIntUnlocked();
                        if (!fieldInfo.omitTermFreqAndPositions) {
                            state.totalTermFreq =
                                state.docFreq + bytesReader.readVLongUnlocked();
                        }
                        postingsReader.nextTerm(fieldInfo, state);

                        int cmp = termComp.compare(target, term);
                        if (cmp == 0) {
                            return SeekStatus.FOUND;
                        } else if (cmp < 0) {
                            if (blockTerms.size() > 0) {
                                state = blockTerms.remove(blockTerms.size() - 1);
                                term = state.term;
                                return SeekStatus.NOT_FOUND;
                            } else {
                                end = true;
                                return SeekStatus.END;
                            }
                        }

                        final YandexTermState clonedState = (YandexTermState) state.clone();
                        clonedState.term = new BytesRef(term);
                        blockTerms.add(clonedState);
                        if (state.termCount == 0) {
                            if (state.blockFirstTerm == null) {
                                state.blockFirstTerm = new BytesRef(term);
                            } else {
                                state.blockFirstTerm.copy(term);
                            }
                        }
                        state.termCount++;
                    }
                    if (blockTerms.size() > 0) {
                        state = blockTerms.remove(blockTerms.size() - 1);
                        term = state.term;
                        return SeekStatus.NOT_FOUND;
                    } else {
                        end = true;
                        return SeekStatus.END;
                    }
                } finally {
                    if (locked) {
                        releaseBuffer();
                    }
                }
            }

            private void skipBlockIndex() throws IOException {
                if (DEBUG) {
                    debug("LOADBLOCK");
                }
                state.blockFilePointer = bytesReader.nextFilePointer();
                state.blockTermCount = bytesReader.readVIntUnlocked();
                int fullTermsCount = bytesReader.readVIntUnlocked();
                if (DEBUG) {
                    debug("nextBlock: " + bytesReader.isNextBlock());
                }
                if (bytesReader.isNextBlock()) {
                    bytesReader.clearNextBlock();
                }
                if (DEBUG) {
                    debug("blockTermCount: " + state.blockTermCount);
                    debug("fullTermCount: " + fullTermsCount);
                }
                for (int i = 0; i < fullTermsCount; i++) {
                    bytesReader.readVIntUnlocked();
                }
                state.termCount = 0;
            }

            private final boolean lockBuffer() throws IOException {
                if (bytesReader != null && !bufferLocked) {
                    bytesReader.lockBuffer();
                    bufferLocked = true;
                    return true;
                } else {
                    return false;
                }
            }

            private final void releaseBuffer() throws IOException {
                if (bufferLocked) {
                    bufferLocked = false;
                    bytesReader.releaseBuffer();
                }
            }

            private void loadPrevBlock() throws IOException {
                if (blockTerms == null) {
                    indexEnum.seekEnd();
                    //skip the latest element because it pointing to the 0, not the last block start
                    final long offset = indexEnum.prev();
                    if (DEBUG) {
                        debug("loadPrevBlock prev1: " + offset);
                    }
                    if (offset != 0) {
                        indexEnum.seekEnd();
                    }
                    blockTerms = new ArrayList<>();
                }
                final long blockOffset = indexEnum.prev();
                if (DEBUG) {
                    debug("loadPrevBlock prev2: " + blockOffset);
                }
                if (blockOffset == -1) {
                    end = true;
                    return;
                }
                final BytesRef tempTerm = new BytesRef(indexEnum.term());
                if (blockOffset == termsStartPointer && indexDivisor <= 1) {
                    //skip first empty entry;
                    long skip = indexEnum.prev();
                    if (DEBUG) {
                        debug("loadPrevBlock prev3: " + skip);
                    }
                }
                checkBytesReader();
//                  System.err.println("loadPrevBlock: offset="
//                        + blockOffset + ", term=" + tempTerm.utf8ToString());
                bytesReader.seek(blockOffset, 0);
                boolean locked = lockBuffer();
                try {
                    final int blocks;
                    if (blockOffset == termsStartPointer && indexDivisor > 1) {
                        //there are to index pointers at the start which are poining
                        //to the first block
                        //so after index division we should read one block less
                        //if this is a first index pointer
                        blocks = indexDivisor - 1;
                    } else {
                        blocks = indexDivisor;
                    }
                    for (int b = 0; b < blocks; b++) {
                        final YandexTermState tempState = new YandexTermState();
                        tempState.bytesReader = bytesReader;
                        tempState.termCount = 0;
                        skipBlockIndex();
                        while (!bytesReader.blockEof()) {
                            if ((tempState.termCount) % fullTermInterval == 0) {
                                tempState.fullTerm = true;
                            } else {
                                tempState.fullTerm = false;
                            }
                            final int prefix = bytesReader.readVIntUnlocked();
                            final int suffix = bytesReader.readVIntUnlocked();
                            tempTerm.length = prefix + suffix;
                            if (tempTerm.bytes.length < tempTerm.length) {
                                tempTerm.grow(tempTerm.length);
                            }
                            if (tempState.fullTerm && tempState.termCount != 0) {
                                System.arraycopy(
                                    tempState.blockFirstTerm.bytes,
                                    0,
                                    tempTerm.bytes,
                                    0,
                                    prefix);
                            }
                            bytesReader.readBytesUnlocked(tempTerm.bytes, prefix, suffix);
//                                System.out.println("Adding " + blockTerms.size() + ": " + b + ": " + tempTerm.utf8ToString());
                            tempState.docFreq = bytesReader.readVIntUnlocked();
                            if (!fieldInfo.omitTermFreqAndPositions) {
                                tempState.totalTermFreq =
                                tempState.docFreq + bytesReader.readVLongUnlocked();
                            }
                            postingsReader.nextTerm(fieldInfo, tempState);
                            final YandexTermState clonedState = (YandexTermState) tempState.clone();
                            clonedState.term = new BytesRef(tempTerm);
                            blockTerms.add(clonedState);
                            if (tempState.termCount == 0) {
                                if (tempState.blockFirstTerm == null) {
                                    tempState.blockFirstTerm = new BytesRef(tempTerm);
                                } else {
                                    tempState.blockFirstTerm.copy(tempTerm);
                                }
                            }
                            tempState.termCount++;
                        }
                        if (blocks > 1 && bytesReader.eof()) {
                            break;
                        }
                        if (bytesReader.isNextBlock()) {
                            bytesReader.clearNextBlock();
                        }
                    }
                } finally {
                    if (locked) {
                        releaseBuffer();
                    }
                }
            }

            @Override
            public BytesRef next() throws IOException {
                if (end) {
                    return null;
                }

                if (blockTerms == null || blockTerms.size() == 0) {
                    loadPrevBlock();
                    return next();
                }

                state = blockTerms.remove(blockTerms.size() - 1);
                term = state.term;
                return term;
            }

          /* Decodes only the term bytes of the next term.  If caller then asks for
             metadata, ie docFreq, totalTermFreq or pulls a D/&PEnum, we then (lazily)
             decode all metadata up to the current term. */

            @Override
            public BytesRef term() {
                return term;
            }

            @Override
            public int docFreq() throws IOException {
                //System.out.println("BTR.docFreq");
                //System.out.println("  return " + state.docFreq);
                return state.docFreq;
            }

            @Override
            public long totalTermFreq() throws IOException {
                return state.totalTermFreq;
            }

            @Override
            public DocsEnum docs(Bits skipDocs, DocsEnum reuse) throws IOException {
                //System.out.println("BTR.docs this=" + this);
                //System.out.println("  state.docFreq=" + state.docFreq);
                final DocsEnum docsEnum = postingsReader.docs(fieldInfo, state, skipDocs, reuse, useCache);
                assert docsEnum != null;
                return docsEnum;
            }

            @Override
            public DocsAndPositionsEnum docsAndPositions(Bits skipDocs, DocsAndPositionsEnum reuse) throws IOException {
                //System.out.println("BTR.d&p this=" + this);
                if (fieldInfo.omitTermFreqAndPositions) {
                    return null;
                } else {
                    DocsAndPositionsEnum dpe = postingsReader.docsAndPositions(fieldInfo, state, skipDocs, reuse, useCache);
                    //System.out.println("  return d&pe=" + dpe);
                    return dpe;
                }
            }

            @Override
            public void seek(BytesRef target, TermState otherState) throws IOException {
                //System.out.println("BTR.seek termState target=" + target.utf8ToString() + " " + target + " this=" + this);
                seek(target);
            }

            @Override
            public TermState termState() throws IOException {
                //System.out.println("BTR.termState this=" + this);
                TermState ts = (TermState) state.clone();
                //System.out.println("  return ts=" + ts);
                return ts;
            }

            @Override
            public SeekStatus seek(long ord) throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override
            public long ord() {
                throw new UnsupportedOperationException();
            }
        }
    }
}
