package org.apache.lucene.search;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.util.ArrayUtil;
import java.io.IOException;
import java.util.Collection;
import java.util.Comparator;

/** Scorer for conjunctions, sets of queries, all of which are required. */
class ConjunctionScorer extends Scorer {
  
  private final Scorer[] scorers;
  private final int[] failes;
  private final float coord;
  private int lastDoc = -1;

  public ConjunctionScorer(Weight weight, float coord, Collection<Scorer> scorers) throws IOException {
    this(weight, coord, scorers.toArray(new Scorer[scorers.size()]));
  }

  public ConjunctionScorer(Weight weight, float coord, Scorer... scorers) throws IOException {
    super(weight);
    this.scorers = scorers;
    this.coord = coord;
    this.failes = new int[scorers.length];
    for (int i = 0; i < scorers.length; i++) {
//      System.err.println("SCORER[" + i + "]: freq: " + scorers[i].docFreq()
//        + ", scorer: " + scorers[i].toString());
      if (scorers[i].nextDoc() == NO_MORE_DOCS) {
        // If even one of the sub-scorers does not have any documents, this
        // scorer should not attempt to do any more work.
        lastDoc = NO_MORE_DOCS;
        return;
      }
    }

    ArrayUtil.mergeSort(
        scorers,
        new Comparator<Scorer>() {
            @Override
            public int compare(Scorer s1, Scorer s2) {
                try {
                    return Integer.compare(s2.docFreq(), s1.docFreq());
                } catch (IOException e) {
                    return 0;
                }
            }
        });

    for (int i = 0; i < scorers.length - 1; i++) {
        final Scorer s = scorers[i];
        final int doc = s.docID();
        final Scorer next = scorers[i + 1];
        final int nextDoc = next.docID();
        if (doc > nextDoc) {
            next.advance(doc);
        }
    }

    if (doNext() == NO_MORE_DOCS) {
      // The scorers did not agree on any document.
      lastDoc = NO_MORE_DOCS;
      return;
    }
  }

  private int doNext1() throws IOException {
    int first = 0;
    int doc = scorers[scorers.length - 1].docID();
    Scorer firstScorer;
    while ((firstScorer = scorers[first]).docID() < doc) {
      doc = firstScorer.advance(doc);
      first = first == scorers.length - 1 ? 0 : first + 1;
    }
    return doc;
  }

  private final int advanceScorer(final Scorer scorer, final int target)
    throws IOException
  {
    final int docID = scorer.docID();
    if (docID >= target) {
        return docID;
    }
    return scorer.advance(target);
  }

  private int doNext() throws IOException {
      if (scorers.length == 1) {
          return doNext1();
      }

    final Scorer lead1 = scorers[scorers.length - 1];
    int lead2Idx = scorers.length - 2;
    Scorer lead2 = scorers[lead2Idx];
    final int otherScorers = scorers.length - 3;
    int doc = lead1.docID();

    advanceHead:
    for (;;) {
        final int nextDoc = advanceScorer(lead2, doc);

        if (nextDoc != doc) {
            failes[lead2Idx]++;
            doc = advanceScorer(lead1, nextDoc);
            if (doc != nextDoc) {
                continue;
            }
        }

        for (int i = otherScorers; i >= 0; i--) {
            final Scorer scorer = scorers[i];
            if (scorer.docID() < doc) {
                final int nextDoc2 = advanceScorer(scorer, doc);
                if (nextDoc2 > doc) {
                    failes[i]++;
                    doc = advanceScorer(lead1, nextDoc2);
                    if (failes[i] > failes[lead2Idx]) {
                        int tmp = failes[i];
                        failes[i] = failes[lead2Idx];
                        failes[lead2Idx] = tmp;

                        lead2 = scorer;
                        scorers[i] = scorers[lead2Idx];
                        scorers[lead2Idx] = scorer;
                    }
                    continue advanceHead;
                }
            }
        }
        return doc;
    }
  }
  
  @Override
  public int advance(int target) throws IOException {
    if (lastDoc == NO_MORE_DOCS) {
      return lastDoc;
    } else if (scorers[(scorers.length - 1)].docID() < target) {
      scorers[(scorers.length - 1)].advance(target);
    }
    return lastDoc = doNext();
  }

  @Override
  public int docID() {
    return lastDoc;
  }
  
  @Override
  public int nextDoc() throws IOException {
    if (lastDoc == NO_MORE_DOCS) {
      return lastDoc;
    } else if (lastDoc == -1) {
      return lastDoc = scorers[scorers.length - 1].docID();
    }
    scorers[(scorers.length - 1)].nextDoc();
    return lastDoc = doNext();
  }
  
  @Override
  public float score() throws IOException {
    float sum = 0.0f;
    for (int i = 0; i < scorers.length; i++) {
      sum += scorers[i].score();
    }
    return sum * coord;
  }

  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder("ConjunctionScorer:[");
    for (Scorer s: scorers) {
        sb.append(s.toString());
        sb.append(',');
    }
    sb.setLength(sb.length() - 1);
    sb.append(']');
    return new String(sb);
  }
}
