package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import com.sun.management.GarbageCollectionNotificationInfo;
import com.sun.management.GcInfo;

import copyleft.com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMapLong;
import copyleft.com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMapLong.Builder;
import copyleft.com.googlecode.concurrentlinkedhashmap.EntryWeigher;
import copyleft.com.googlecode.concurrentlinkedhashmap.EvictionListener;

import java.io.FileDescriptor;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;

import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;

import java.util.ArrayList;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.IdentityHashMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import java.util.function.Consumer;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.locks.StampedLock;

import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;

import javax.management.openmbean.CompositeData;

import org.jctools.maps.NonBlockingHashMapLong;
import org.jctools.queues.MpmcArrayQueue;

import ru.yandex.cache.DBCache;
import ru.yandex.cache.CacheInfoProvider;

import ru.yandex.msearch.util.ConcurrentPositiveLongHashSet;

import ru.yandex.http.util.server.GCMonitor;

import ru.yandex.msearch.util.JavaAllocator;
import ru.yandex.msearch.util.Compress;

import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

import ru.yandex.unsafe.NativeMemory2;
import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;

import ru.yandex.util.timesource.TimeSource;

public abstract class BlockCompressedInputStreamBase extends IndexInput
    implements Cloneable
{
    public final static int PER_INPUT_HASHSET_SIZE = 16;
    private final static long CACHE_SIZE_HANDICAP = 128 * 1024 * 1024;
    private final static long SHRINKER_SAMPLING_INTERVAL = 100;
    private final static int SMOOTH_DEPTH = 1;
    private final static long ALLOCATOR_RELEASE_INTERVAL = 5000;
    private final static long ALLOCATOR_STATS_INTERVAL = 10000;
    private final static long DEFAULT_CACHE_SIZE = defaltCacheSize();
    private final static int CACHE_INITIAL_CAPACICTY =
        (int)(DEFAULT_CACHE_SIZE / 4096);
    private final static int DEFAULT_CACHE_CONCURRENCY =
        Runtime.getRuntime().availableProcessors() << 1;
    private static final int MAX_POOLED_BUFFER_SIZE = 1024 * 1024;
    private static final int BUFFER_SIZE_STEP_SHIFT = 8;
    private static final boolean QUIET_STDERR = Compress.QUIET_STDERR;

    private static final int VINT_VALUE_MASK = 0x7f;
    private static final long VLONG_VALUE_MASK = 0x7fL;

    private static final int VINT_SHIFT = 7;
    private static final int CLEANUP_QUEUE_WAIT_DELAY = 50;
    private static final int CLEANUP_QUEUE_SIZE = 10000;
    private static final int CLEANUP_THREADS =
        Math.max(1, Runtime.getRuntime().availableProcessors() >> 3);
    private static final MpmcArrayQueue<OutputBuffer> CLEANUP_QUEUE =
        new MpmcArrayQueue<>(CLEANUP_QUEUE_SIZE);

    private static final ThreadLocal<byte[]> threadLocalReadBuffer =
        new ThreadLocal<byte[]>();
    private static final ThreadLocal<NativeMemory2> tlsNativeBuff =
        new ThreadLocal<NativeMemory2>();
    private static final NativeMemoryAllocator TMP_ALLOCATOR =
        NativeMemoryAllocator.get("TempBuffer");
    private static final NativeMemoryAllocator tmpAllocator =
        NativeMemoryAllocator.get("DirectTempBuffer");
    private static final ConcurrentHashMap<Object, Integer> fileToIdMap =
        new ConcurrentHashMap<>();
    private static final AtomicInteger fileId = new AtomicInteger(1);
    private static final int FILE_ID_SHIFT = 40;
    private static final long FILE_POS_MASK = 0xFFFFFFFFFFL;
    private static final ConcurrentLinkedQueue<Integer> FREED_FILE_IDS =
        new ConcurrentLinkedQueue<>();
    private static final AtomicBoolean DYNAMIC_SHRINK_START =
        new AtomicBoolean(false);
    private static final ThreadLocal<NativeMemory2> threadLocalTempBuffer =
        new ThreadLocal<>();
    private static final ThreadLocal<NativeMemory2> threadLocalTempBuffer2 =
        new ThreadLocal<>();
    private static Shrinker shrinker;

    private static long defaltCacheSize() {
        String cacheSize = System.getProperty(
            "ru.yandex.lucene.default-cache-size",
            "2000000000");
        return Long.parseLong(cacheSize);
    }

    private static Consumer<Long> onInputBlockEvict =
        new Consumer<Long>() {
            @Override
            public void accept(final Long key) {
                removeInputBlock(key);
            }
        };

    private static CacheRemovalListener evictionListener =
        new CacheRemovalListener(onInputBlockEvict);

    private static long initialCompressedCacheCapacity;

    public static int maxFileId() {
        return fileId.get();
    }

    private static ConcurrentLinkedHashMapLong<OutputBuffer> blockCache =
        new Builder<OutputBuffer>()
            .initialCapacity(CACHE_INITIAL_CAPACICTY)
            .concurrencyLevel(DEFAULT_CACHE_CONCURRENCY << 2)
            .listener(evictionListener)
            .maximumWeightedCapacity(DEFAULT_CACHE_SIZE)
            .weigher(
                new EntryWeigher<Long, OutputBuffer>() {
                    @Override
                    public int weightOf(Long key, OutputBuffer value) {
                        if (value.out != null) {
                            value.weight = value.out.length + 50;
//                            return NativeMemory.unboxedSize(value.out) + 50;
                        } else {
                            value.weight = 50;
//                            return 50; //class size
                        }
                        return value.weight;
                    }
                })
            .build();
    private static NonBlockingHashMapLong<ConcurrentPositiveLongHashSet>
        indexInputBlocks = new NonBlockingHashMapLong<>();

    private static CacheStater rawStater =
        new CacheStater(
            "Raw",
            new CacheInfoProvider() {
                @Override
                public long capacity() {
                    return blockCache.capacity();
                }
                @Override
                public long used() {
                    return blockCache.weightedSize();
                }
                @Override
                public long count() {
                    return blockCache.size();
                }
            },
            5000);
    private static CompressedBlockCache compressedCache =
        new CompressedBlockCache(DEFAULT_CACHE_SIZE);
    private static DBCache ssdCache = null;

    static {
        Thread statsPrinter = new Thread("CacheStatsPrinter") {
            private long prevRelease;
            private long prevStats;
            @Override
            public void run() {
                while (true) {
                    if (TimeSource.INSTANCE.currentTimeMillis() - prevStats
                        > ALLOCATOR_STATS_INTERVAL)
                    {
                        System.err.println(compressedCache.tagsStats());
                        System.err.println("Allocator stats: " + Compress.allocatorStats());
                        prevStats = TimeSource.INSTANCE.currentTimeMillis();
                    }
                    if (TimeSource.INSTANCE.currentTimeMillis() - prevRelease
                        > ALLOCATOR_RELEASE_INTERVAL)
                    {
                        System.err.println("Calling ReleaseFreeMemory()");
                        Compress.releaseFreeMemory();
                        prevRelease = TimeSource.INSTANCE.currentTimeMillis();
                    }
                    try {
                        Thread.sleep(500);
                    } catch (InterruptedException ign) {
                    }
                }
            }};
        statsPrinter.setDaemon(true);
        statsPrinter.start();

        for (int i = 0; i < CLEANUP_THREADS; i++) {
            final int num = i;
            Thread cacheCleaner = new Thread("CacheCleaner-" + (num + 1)) {
                @Override
                public void run() {
                    //time shift threads
                    try {
                        Thread.sleep(
                            (CLEANUP_QUEUE_WAIT_DELAY / CLEANUP_THREADS) * num);
                    } catch (InterruptedException ign) {
                    }
                    while (true) {
                        try {
                            OutputBuffer out;
                            while ((out = CLEANUP_QUEUE.poll()) == null) {
                                Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
                            }
                            if (out.direct) {
                                out.free();
                            } else {
                                freeBlock(out);
                            }
                        } catch (Throwable t) {
                            t.printStackTrace();
                        }
                    }
                }
                private void freeBlock(final OutputBuffer out)
                    throws InterruptedException
                {
                    out.free();
                }
            };
            cacheCleaner.setDaemon(true);
            cacheCleaner.start();
        }
    }

    public static double cacheFill() {
        return Math.min(
            100,
            ((double) blockCache.weightedSize()
                / blockCache.capacity()) * 100.0);
    }

    public static double compressedCacheFill() {
        return compressedCache.cacheFill();
    }

    public static double ssdCacheFill() {
        if (ssdCache == null) {
            return 0;
        }
        return Math.min(
            100,
            ((double) ssdCache.used()
                / ssdCache.capacity()) * 100.0);
    }

    private static byte[] getReadBuffer(final int size) {
        byte[] buffer = threadLocalReadBuffer.get();
        if (buffer == null || buffer.length < size) {
            buffer = new byte[size];
            threadLocalReadBuffer.set(buffer);
        }
        return buffer;
    }

    private static void freeReadBuffer(byte[] buffer) {
    }

    private static final class CacheRemovalListener
        implements EvictionListener<Long,OutputBuffer>
    {
        private final Consumer<Long> onEvict;

        public CacheRemovalListener(final Consumer<Long> onEvict) {
            this.onEvict = onEvict;
        }

        @Override
        public void onEviction(
            final Long key,
            final OutputBuffer value)
        {
            if (value.direct) {
                value.free();
            } else {
                value.evicted = true;
                if (!value.forceEvict && key != null) {
                    try {
                        onEvict.accept(key);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                while (!CLEANUP_QUEUE.offer(value)) {
                    try {
                        Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        }
    }

    public static void freeCache() {
        blockCache.setCapacity(0);
        while (CLEANUP_QUEUE.size() > 0) {
            try {
                Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        compressedCache.freeCache();
    }

    public static void waitFree() {
        while (CLEANUP_QUEUE.size() > 0) {
            try {
                Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        compressedCache.waitFree();
    }

    public static void dropCache() {
        final long currentCapacity = blockCache.capacity();
        blockCache.setCapacity(0);
        blockCache.setCapacity(currentCapacity);
    }

    public static void dropCompressedCache() {
        compressedCache.dropCache();
    }

    public static void dynamicShrink(
        final boolean dynamic,
        final long minimalCacheSize,
        final long minimalMemoryFree)
    {
        if (dynamic) {
            if (DYNAMIC_SHRINK_START.compareAndSet(false, true)) {
                startShrinker(minimalCacheSize - 128, minimalMemoryFree);
            }
        } else {
            if (DYNAMIC_SHRINK_START.compareAndSet(true, false)) {
                stopShrinker();
            }
        }
    }

    public static void startShrinker(
        final long minimalCacheSize,
        final long minimalMemoryFree)
    {
        if (shrinker != null) {
            shrinker.exit();
        }
        shrinker = new Shrinker(minimalCacheSize, minimalMemoryFree);
        shrinker.start();
    }

    public static void stopShrinker() {
        if (shrinker != null) {
            shrinker.exit();
            shrinker = null;
        }
    }

    public static <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        rawStater.stats(statsConsumer);
        compressedCache.stats(statsConsumer);
    }

    public static long capacity() {
        return blockCache.capacity();
    }

    public static long compressedCacheCapacity() {
        return compressedCache.cacheCapacity();
    }

    public static long ssdCacheCapacity() {
        if (ssdCache != null) {
            return ssdCache.capacity();
        } else {
            return 0;
        }
    }

    public static void setCacheCapacity(final long capacity) {
        blockCache.setCapacity(capacity);
    }

    public static long initialCompressedCacheCapacity() {
        return initialCompressedCacheCapacity;
    }

    public static void setCompressedCacheCapacity(final long capacity) {
        initialCompressedCacheCapacity = capacity;
        compressedCache.setCacheCapacity(capacity);
    }

    public static void setSsdCache(final DBCache cache) {
        ssdCache = cache;
        compressedCache.setSsdCache(cache);
    }

    public static void ssdCacheEnable(final boolean enable) {
        compressedCache.ssdCacheEnable(enable);
    }

    public static void allocateInput(final IndexInput in) {
        long fileId = fileToId(in).longValue();
        indexInputBlocks.put(
            fileId,
            new ConcurrentPositiveLongHashSet(PER_INPUT_HASHSET_SIZE));
        compressedCache.allocateInput(fileId);
    }

    public static void removeFile(final String file) {
        if (!QUIET_STDERR) {
            System.err.println("Deleting file: " + file);
        }
        if (ssdCache != null) {
            try {
                ssdCache.removePrefix(
                    file + ':',
                    false);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public static void drainInputBlocks(final IndexInput in) {
        Integer fileIdBoxed = fileToId(in);
        long fileId = fileIdBoxed.longValue();
        ConcurrentPositiveLongHashSet blocks =
            indexInputBlocks.remove(fileId);
        if (blocks != null) {
            for (Long blockKey: blocks) {
                OutputBuffer ob = blockCache.get(blockKey);
                if (ob != null) {
                    ob.forceEvict = true;
                    if (blockCache.remove(blockKey, ob)) {
                        evictionListener.onEviction(blockKey, ob);
                    }
                }
            }
        } else {
            System.err.println("CACHE.drainInputBlocks null for: " + in);
        }
        compressedCache.drainInputBlocks(fileId);
        if (fileToIdMap.remove(in.getCacheKey()) == fileIdBoxed) {
            recycleFileId(fileIdBoxed);
        }
    }

    public static void removeInputBlock(final long key) {
        ConcurrentPositiveLongHashSet blocks =
            indexInputBlocks.get(fileIdFromKey(key));
        if (blocks != null) {
            if (!blocks.remove(key)) {
                System.err.println("CACHE.removeInputBlock: failed for " + key);
            }
        }
    }

    public static void printCacheInputs() {
        System.err.println("printCacheInputs {");
        HashSet uniqInputs = new HashSet<>();
        for (Long key: blockCache.keySet()) {
            uniqInputs.add(fileIdFromKey(key));
        }
        for (Object key: uniqInputs) {
            System.err.println("CacheInput: "
                + indexInputBlocks.containsKey(key)
                + ": " + key);
        }
        System.err.println("} printCacheInputs");
        System.err.println("CompressedCache printCacheInputs {");
        compressedCache.printCacheInputs();
        System.err.println("} printCacheInputs");
        System.err.println(compressedCache.tagsStats());
    }

    static final class OutputBuffer {
        public volatile boolean unpacked = false;
        public byte[] out = null;
        public int weight = 0;
        public int compressedSize = -1;
        public int plainSize = -1;
        public long nextBlockPointer = 0;
        public long thisBlockPointer = 0;
        public volatile boolean evicted = false;
        public volatile boolean puted = false;
        public volatile boolean direct = false;
        public volatile boolean forceEvict = false;
        private volatile boolean released = false;
        public volatile IOException exception = null;
        public JavaAllocator allocator = null;
//        public NativeMemoryAllocator compressedAllocator = null;

        public OutputBuffer(final JavaAllocator allocator) {
            this.allocator = allocator;
        }

        public int readVInt(int[] pos) {
            byte b = out[pos[0]++];
            int i = b & VINT_VALUE_MASK;
            for (int shift = VINT_SHIFT; b < 0; shift += VINT_SHIFT) {
                b = out[pos[0]++];
                i |= (b & VINT_VALUE_MASK) << shift;
            }
            return i;
        }

        public long readVLong(int[] pos) {
            byte b = out[pos[0]++];
            long i = b & VLONG_VALUE_MASK;
            for (int shift = VINT_SHIFT; b < 0; shift += VINT_SHIFT) {
                b = out[pos[0]++];
                i |= (b & VLONG_VALUE_MASK) << shift;
            }
            return i;
        }

        public byte readByte(int pos) {
            try {
                return out[pos];
            } catch (NullPointerException e) {
                throw new RuntimeException(toString(), e);
            }
        }

        public void readBytes(
            final int pos,
            final byte[] out,
            final int offset,
            final int len)
        {
            System.arraycopy(this.out, pos, out, offset, len);
        }

        public OutputBuffer emptyClone() {
            final OutputBuffer clone = new OutputBuffer(allocator);
            clone.unpacked = unpacked;
            clone.out = null;
            clone.weight = weight;
            clone.compressedSize = compressedSize;
            clone.plainSize = plainSize;
            clone.nextBlockPointer = nextBlockPointer;
            clone.thisBlockPointer = thisBlockPointer;
            clone.evicted = evicted;
            clone.puted = puted;
            clone.direct = direct;
            clone.released = released;
            return clone;
        }

        public synchronized void resize(final int size) {
            if (out != null) {
                if (out.length != size) {
                    out = allocator.realloc(out, size);
                }
            } else {
                out = allocator.alloc(size);
            }
        }

        public synchronized void free() {
            released = true;
            if (out != null) {
                allocator.free(out);
//                out = null;
            }
        }

        private String stackTrace(final Throwable t) {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            t.printStackTrace(pw);
            return sw.toString();
        }

        public void freeUnpacked() {
            if (out != null) {
                allocator.free(out);
//                out = null;
            }
        }

        public int size() {
            if (out == null) {
                return 0;
            } else {
                return out.length;
            }
        }

        public byte[] getNativeBuffer(final int size) {
            return allocator.alloc(size);
        }

        public void freeNativeBuffer(final byte[] buffer) {
            allocator.free(buffer);
        }

        public String toString() {
            return "OutputBuffer<" + System.identityHashCode(this)
                + ">: compressedSize=" + compressedSize
                + ", plainSize=" + plainSize
                + ", nextBlockPointer=" + nextBlockPointer
                + ", thisBlockPointer=" + thisBlockPointer
                + ", out=" + out
                + ", evicted=" + evicted
                + ", puted=" + puted
                + ", direct=" + direct
                + ", exception=" + exception
                + ", lock=" + super.toString()
                + ", released=" + released;
        }
    }

    final static class InputAndPos {
        public long pos;
        public IndexInput in;
        private Object cacheKey;
        private final int hashCode;

        private InputAndPos(final int hashCode) {
            super();
            this.hashCode = hashCode;
        }

        public InputAndPos(
            final IndexInput in,
            final long pos)
        {
            super();
            this.in = in;
            cacheKey = in.getCacheKey();
            this.pos = pos;
            this.hashCode = in.hashCode() * 31 + (int)(pos / 4);
        }

        public Object cacheKey() {
            return cacheKey;
        }

        @Override
        public boolean equals(final Object _other) {
            InputAndPos other = (InputAndPos) _other;
            return cacheKey.equals(other.cacheKey) && pos == other.pos;
        }

        @Override
        public int hashCode() {
            return hashCode;
        }

        @Override
        public String toString() {
            return cacheKey.toString() + ':' + pos;
        }
    }

    private byte[] readVIntBuffer = null;
    protected IndexInput otherInput;
    private IndexInput inputClone = null;

    private OutputBuffer directOutputBuffer = null;
    protected OutputBuffer currentOutputBuffer = null;

    protected int[] bufferPos = new int[] {0};
    protected int bufferLength = 0;

    private boolean bufferLocked = false;
    private long readStamp = 0;
    protected long nextFilePointer = 0;

    protected boolean cloned = false;
    protected final Decompressor decompressor;
    protected final String tag;
    protected final JavaAllocator allocator;

    protected BlockCompressedInputStreamBase(
        final IndexInput otherInput,
        final Decompressor decompressor,
        final JavaAllocator allocator,
        final String tag)
    {
        this.otherInput = otherInput;
        this.decompressor = decompressor;
        this.allocator = allocator;
        this.tag = tag;
    }

    public static void recycleFileId(final Integer id) {
        FREED_FILE_IDS.add(id);
    }

    public static int fileIdFromKey(final long key) {
        return (int) (key >>> FILE_ID_SHIFT);
    }

    public static Integer fileToId(final IndexInput in) {
        Object cacheKey = in.getCacheKey();
        Integer id = fileToIdMap.get(cacheKey);
        if (id == null) {
            boolean generatedId = false;
            Integer newId = FREED_FILE_IDS.poll();
            if (newId == null) {
                generatedId = true;
                newId = fileId.incrementAndGet();
            }
            id = fileToIdMap.putIfAbsent(cacheKey, newId);
            if (id == null) {
                id = newId;
            } else {
                if (!generatedId) {
                    recycleFileId(newId);
                }
            }
        }
        return id;
    }

    public static long longKey(final long fileId, long pos) {
        return (fileId << FILE_ID_SHIFT) | (pos & FILE_POS_MASK);
    }

    public abstract BlockCompressedInputStreamBase clone();
    protected abstract byte[] getNativeBuffer(final int size);
    protected abstract void freeNativeBuffer(final byte[] buffer);
    protected abstract void doLoadBlock(final boolean useCache)
        throws IOException;
    protected abstract boolean loadNextBlock(final boolean useCache)
        throws IOException;

    protected int doDecompressBlock(
        final byte[] compressed,
        final int compressedLen,
        final byte[] decompressed,
        final int decompressedLen)
    {
        long time = System.nanoTime();
        NativeMemory2 tmp = getTmpNativeBuf(decompressed.length);
        int ret = decompressor.decompress(compressed, compressedLen,
            tmp.address(), decompressedLen);
        time = System.nanoTime() - time;
        if (ret >= 0) {
            tmp.read(0, decompressed, 0, decompressedLen);
        }
        Compress.unpack(tag, compressedLen, time);
        return ret;
    }

    protected int doDecompressBlock(
        final int fd,
        final long fPos,
        final int compressedLen,
        final byte[] decompressed,
        final int decompressedLen)
        throws IOException
    {
        long time = System.nanoTime();
        NativeMemory2 tmp = getTmpNativeBuf(decompressed.length);
        int ret = ru.yandex.msearch.util.IOScheduler.instance().readOp(
            () -> ((ReadingDecompressor)decompressor).decompress(fd, fPos,
            compressedLen, tmp.address(), decompressedLen));
        if (ret >= 0) {
            tmp.read(0, decompressed, 0, decompressedLen);
        }
        time = System.nanoTime() - time;
        Compress.unpack(tag, compressedLen, time);
        return ret;
    }

    public void useCache(final boolean mayUseCache) {
        final boolean useCache;
        final Boolean forceUseCache = Compress.useCache();
        if (forceUseCache != null) {
            useCache = forceUseCache;
        } else {
            useCache = mayUseCache;
        }
        if (useCache) {
            freeDirectBuffer();
        } else {
            if (directOutputBuffer == null) {
                directOutputBuffer = new OutputBuffer(allocator);
                directOutputBuffer.direct = true;
            }
        }
    }

    public boolean useCache() {
//        return direct;
        return directOutputBuffer == null;
    }

    protected int readVInt(
        final IndexInput in)
        throws IOException
    {
        if (readVIntBuffer == null) {
            readVIntBuffer = new byte[5];
        }
        return readVInt(in, readVIntBuffer);
    }

    protected static int readVInt(
        final IndexInput in,
        final byte[] buffer)
        throws IOException
    {
        in.readBytes(buffer, 0, 1, false);
        byte b = buffer[0];
        int i = b & 0x7f;
        for (int shift = 7; b < 0; shift += 7) {
            in.readBytes(buffer, 0, 1, false);
            b = buffer[0];
            i |= (b & 0x7f) << shift;
        }
        return i;
    }

    public static NativeMemory2 getTmpNativeBuf(final int size) {
        NativeMemory2 buf = tlsNativeBuff.get();
        if (buf == null) {
            buf = TMP_ALLOCATOR.alloc(size == 0 ? 32 : size);
            tlsNativeBuff.set(buf);
        } else if (buf.size() < size) {
            buf = buf.realloc(size);
            tlsNativeBuff.set(buf);
        }
        return buf;
    }

    private static int readVInt(byte[] buffer, int maxPos) throws IOException {
        int pos = 0;
        if (maxPos == pos) {
            throw new EOFException("Read past eof: readVInt: pos="
                + pos + ", maxPos=" + maxPos);
        }
        byte b = buffer[pos++];
        int i = b & 0x7f;
        for (int shift = 7; b < 0; shift += 7) {
            if (maxPos == pos) {
                throw new EOFException("Read past eof: readVInt: pos="
                    + pos + ", maxPos=" + maxPos);
            }
            b = buffer[pos++];
            i |= (b & 0x7f) << shift;
        }
        return i;
    }

    public void copyFrom(final BlockCompressedInputStreamBase other)
        throws IOException
    {
        bufferPos[0] = other.bufferPos[0];
        bufferLength = other.bufferLength;
        currentOutputBuffer = other.currentOutputBuffer;
        nextFilePointer = other.nextFilePointer;
        if (bufferLocked && currentOutputBuffer != null) {
            bufferLocked = false;
            lockBuffer();
        }
    }

//    public final boolean lockBuffer(final StringBuilder trace)
    public final boolean lockBuffer()
        throws IOException
    {
        //locked buffer is a buffer with 2 or more references (one from us
        //and one from Cache
        if (bufferLocked) {
            return false;
        }
        if (currentOutputBuffer == null) {
            bufferLocked = true;
            return true;
        }
        while (true) {
            if (currentOutputBuffer.released) {
                nextFilePointer = currentOutputBuffer.thisBlockPointer;
                decompressBlock(
                    otherInput,
                    currentOutputBuffer.compressedSize,
                    currentOutputBuffer.plainSize,
                    useCache());
            } else if (!currentOutputBuffer.unpacked) {
                IOException exception = null;
                synchronized (currentOutputBuffer) {
                    if (!currentOutputBuffer.released
                        && !currentOutputBuffer.unpacked)
                    {
                        exception = currentOutputBuffer.exception;
                        if (exception == null) {
                            exception = new IOException(
                                "Weird, lack of synchronization? "
                                    + "buffer not unpacked "
                                    + currentOutputBuffer);
                        }
                    }
                }

                if (exception != null) {
                    throw exception;
                }
            } else {
                break;
            }
        }
        bufferLocked = true;
        return true;
    }

//    public final void releaseBuffer(final StringBuilder trace) throws IOException {
    public final void releaseBuffer() throws IOException {
        if (!bufferLocked) {
            new Exception("DEBUG6 Unlocking non locked: this=" + this
                + ", buffer=" + currentOutputBuffer).printStackTrace();
        }
        bufferLocked = false;
        if (currentOutputBuffer != null) {
            if (currentOutputBuffer.released) {
                new IOException("Buffer was released while in use:"
                    + currentOutputBuffer
                    + ", currentThread=" + Thread.currentThread())
                        .printStackTrace();
            }
        }
    }

    //useBuffer is actualy mean use guava cache or not
    protected final void loadBlock(final boolean useBuffer) throws IOException {
        final boolean relockBuffer = bufferLocked;
        if (bufferLocked) {
            releaseBuffer();
        }
        boolean success = false;
        try {
            doLoadBlock(useBuffer);
            success = true;
        } finally {
            if (relockBuffer) {
                if (success) {
                    lockBuffer();
                } else {
                    //outer code have finally {}
                    //section with releaseBuffer
                    //with should not fail
                    bufferLocked = true;
                    currentOutputBuffer = null;
                    bufferLength = 0;
                }
            }
        }
    }

    protected void decompressBlock(final IndexInput in, final boolean useBuffer)
        throws IOException
    {
        decompressBlock(in, -1, -1, useBuffer);
    }

    protected void decompressBlock(final IndexInput in, final int compressedSize,
        final int plainSize, final boolean useBuffer)
        throws IOException
    {
        if (useBuffer) {
            currentOutputBuffer = decompressBlockFromCache(in, compressedSize, plainSize);
        } else {
            currentOutputBuffer = decompressBlockDirect(in, compressedSize, plainSize);
        }
    }

    private OutputBuffer decompressBlockDirect(final IndexInput in,
        final int compressedSize, final int plainSize)
        throws IOException
    {
        if (inputClone == null) {
            inputClone = (IndexInput)in.clone();
        }
        if (directOutputBuffer == null) {
            directOutputBuffer = new OutputBuffer(allocator);
            directOutputBuffer.direct = true;
        }
        try {
            decompressBlock(directOutputBuffer, inputClone,
                nextFilePointer, compressedSize, plainSize);
            if (directOutputBuffer.out != null) {
                bufferLength = directOutputBuffer.plainSize;
            } else {
                if (directOutputBuffer.compressedSize == 0) {
                    bufferLength = 0;
                } else {
                    throw new IOException("Empty buffer: compressedSize="
                        + directOutputBuffer.compressedSize + ", inp.compressedSize="
                        + compressedSize);
                }
            }
            nextFilePointer = directOutputBuffer.nextBlockPointer;
            synchronized (directOutputBuffer) {
                directOutputBuffer.unpacked = true;
                directOutputBuffer.notifyAll();
            }
            return directOutputBuffer;
        } catch (Exception e) {
            String message = "Error trying to unpack block with compressor <"
                + decompressor + "> from file: "
                + in.getCacheKey();
            throw new IOException(message, e);
        }
    }

    private OutputBuffer prepareOutputBuffer(
        final OutputBuffer ret,
        final IndexInput in,
        final long inPos,
        int compressedSize,
        int plainSize)
        throws Exception
    {
        FileDescriptor fileDes = in.getFileDescriptor();
        final int res;
        if (fileDes != null && decompressor instanceof ReadingDecompressor) {
            final int fd =
                NIOFSDirectory.getFd(fileDes);
            return prepareOutputBufferNative(fd, ret, in, inPos, compressedSize,
                plainSize);
        } else {
            return prepareOutputBufferJava(ret, in, inPos, compressedSize,
                plainSize);
        }
    }

    private OutputBuffer prepareOutputBufferNative(
        final int fd,
        final OutputBuffer ret,
        final IndexInput in,
        final long inPos,
        int compressedSize,
        int plainSize)
        throws Exception
    {
        if (inPos < 0) {
            System.err.println("Negative fucking shit: " + inPos);
        }
        byte[] out = ret.out;
        in.seek(inPos);
        if (compressedSize == -1) {//guess
            long blockInfo =
                Compress.readBlockSizes(
                    fd,
                    inPos,
                    tag);
            compressedSize = (int)(blockInfo);
            plainSize = (int)(blockInfo >> 32);
            if (compressedSize == -100) {
                throw new EOFException("Read past eof: pos=" + inPos + ", len="
                    + in.length() + ", file=" + in.getCacheKey());
            }
            ret.thisBlockPointer = inPos + vIntSize(compressedSize)
                + vIntSize(plainSize);
        } else {
            ret.thisBlockPointer = inPos;
        }

        ret.compressedSize = compressedSize;
        ret.plainSize = plainSize;

        if (compressedSize == 0) {
            if (ret.out != null) {
                ret.freeNativeBuffer(ret.out);
            }
            ret.out = null;
            ret.unpacked = true;
            ret.plainSize = 0;
            return ret;
        }
        if (out == null) {
            out = ret.getNativeBuffer(plainSize);
            ret.out = out;
        }
        if (out.length < plainSize) {
            ret.resize(plainSize);
            out = ret.out;
        }
        ret.nextBlockPointer = ret.thisBlockPointer + compressedSize;
        return ret;
    }

    private static OutputBuffer prepareOutputBufferJava(
        final OutputBuffer ret,
        final IndexInput in,
        final long inPos,
        int compressedSize,
        int plainSize)
        throws Exception
    {
        byte[] out = ret.out;
        if (inPos < 0) {
            System.err.println("Negative fucking shit: " + inPos);
        }
        in.seek(inPos);
        if (compressedSize == -1) {//guess
            byte[] tmpBuffer = new byte[5];
            compressedSize = readVInt(in, tmpBuffer);
            plainSize = readVInt(in, tmpBuffer);
            ret.thisBlockPointer = in.getFilePointer();
        } else {
            ret.thisBlockPointer = inPos;
        }

        ret.compressedSize = compressedSize;
        ret.plainSize = plainSize;

        if (compressedSize == 0) {
            if (ret.out != null) {
                ret.freeNativeBuffer(ret.out);
            }
            ret.out = null;
            ret.unpacked = true;
            ret.plainSize = 0;
            return ret;
        }
        if (out == null) {
            out = ret.getNativeBuffer(plainSize);
            ret.out = out;
        }
        if (out.length < plainSize) {
            ret.resize(plainSize);
            out = ret.out;
        }
        ret.nextBlockPointer = ret.thisBlockPointer + compressedSize;
        return ret;
    }

    private OutputBuffer decompressBlock(
        final OutputBuffer ret,
        final IndexInput in,
        final long inPos,
        int compressedSize,
        int plainSize)
        throws Exception
    {
        FileDescriptor fileDes = in.getFileDescriptor();
        final int res;
        if (fileDes != null && decompressor instanceof ReadingDecompressor) {
            final int fd = NIOFSDirectory.getFd(fileDes);
            if (Compress.fadviseEnabled()) {
                Compress.fadviseOp(
                    fd,
                    inPos,
                    256 * 1024,
                    tag,
                    true);
            }
            return decompressBlockNative(fd, ret, in, inPos, compressedSize,
                plainSize);
        } else {
            return decompressBlockJava(ret, in, inPos, compressedSize,
                plainSize);
        }
    }

    private OutputBuffer decompressBlockJava(
        final OutputBuffer ret,
        final IndexInput in,
        final long inPos,
        int compressedSize,
        int plainSize)
        throws Exception
    {
        byte[] out = ret.out;
        byte[] readBuffer = null;
        try {
            if (inPos < 0) {
                System.err.println("Negative fucking shit: " + inPos);
            }
            in.seek(inPos);
            if (compressedSize == -1) {//guess
                byte[] tmpBuffer = new byte[5];
                compressedSize = readVInt(in, tmpBuffer);
                plainSize = readVInt(in, tmpBuffer);
                ret.thisBlockPointer = in.getFilePointer();
            } else {
                ret.thisBlockPointer = inPos;
            }

            ret.compressedSize = compressedSize;

            if (compressedSize == 0) {
                ret.out = null;
                ret.plainSize = 0;
                return ret;
            }
            if (out == null) {
                out = ret.getNativeBuffer(plainSize);
                ret.out = out;
            }
            if (out.length < plainSize) {
                ret.resize(plainSize);
                out = ret.out;
            }
            FileDescriptor fileDes = in.getFileDescriptor();
            final int res;
            if (fileDes != null
                && decompressor instanceof ReadingDecompressor)
            {
                final int fd = NIOFSDirectory.getFd(fileDes);
                ret.nextBlockPointer = ret.thisBlockPointer + compressedSize;
                res = doDecompressBlock(
                    fd,
                    ret.thisBlockPointer,
                    compressedSize,
                    out,
                    plainSize);
            } else {
                readBuffer = getReadBuffer(compressedSize);
                in.readBytes(readBuffer, 0, compressedSize);
                ret.nextBlockPointer = in.getFilePointer();
                res = doDecompressBlock(
                    readBuffer,
                    compressedSize,
                    out,
                    plainSize);
            }
            if (res < 0) {
                throw new IOException("Inflate error: " + res);
            }
            ret.plainSize = plainSize;
            return ret;
        } finally {
            if (readBuffer != null) {
                freeReadBuffer(readBuffer);
            }
        }
    }

    public static int vIntSize(int i) {
        int size = 1;
        while ((i & ~0x7F) != 0) {
            size++;
            i >>>= 7;
        }
        return size;
    }

    private OutputBuffer decompressBlockNative(
        final int fd,
        final OutputBuffer ret,
        final IndexInput in,
        final long inPos,
        int compressedSize,
        int plainSize)
        throws Exception
    {
        byte[] out = ret.out;
        byte[] readBuffer = null;
        try {
            if (inPos < 0) {
                System.err.println("Negative fucking shit: " + inPos);
            }
            in.seek(inPos);
            if (compressedSize == -1) {//guess
                long blockInfo =
                    Compress.readBlockSizes(
                        fd,
                        inPos,
                        tag);
                compressedSize = (int)(blockInfo);
                plainSize = (int)(blockInfo >> 32);
                if (compressedSize == -100) {
                    throw new EOFException("Read past eof: pos=" + inPos + ", len="
                        + in.length());
                }
                ret.thisBlockPointer = inPos + vIntSize(compressedSize)
                    + vIntSize(plainSize);
            } else {
                ret.thisBlockPointer = inPos;
            }

            ret.compressedSize = compressedSize;

            if (compressedSize == 0) {
                ret.out = null;
                ret.plainSize = 0;
                return ret;
            }
            if (out == null) {
                out = ret.getNativeBuffer(plainSize);
                ret.out = out;
            }
            if (out.length < plainSize) {
                ret.resize(plainSize);
                out = ret.out;
//                ret.out = out = ret.getNativeBuffer(plainSize);
            }
            ret.nextBlockPointer = ret.thisBlockPointer + compressedSize;

            NativeMemory2 compressed = getTemporaryBuffer(compressedSize);
            int red = ru.yandex.msearch.util.Compress.preadOp(
                fd,
                ret.thisBlockPointer,
                compressed.address(),
                compressedSize,
                tag,
                0);
            if (red != compressedSize) {
                throw new IOException("Read error: pread: "
                    + red + ", out.cs: " + compressedSize);
            }
            NativeMemory2 deflated = getTemporaryBuffer2(plainSize);
            int decompressed = decompressor.decompress(
                compressed.address(),
                compressedSize,
                deflated.address(),
                plainSize);
            if (decompressed != plainSize) {
                throw new IOException(
                    "Block decompress error: " +
                    decompressor.errorDescription(decompressed)
                    + ", cs: " + compressedSize
                    + ", ps: " + plainSize);
            }
            deflated.read(0, ret.out, 0, plainSize);

            ret.plainSize = plainSize;
            return ret;
        } finally {
            if (readBuffer != null) {
                freeReadBuffer(readBuffer);
            }
        }
    }

    private void freeDirectBuffer() {
        if (directOutputBuffer != null) {
            evictionListener.onEviction(null, directOutputBuffer);
            if (currentOutputBuffer == directOutputBuffer) {
                if (bufferLocked) {
                    System.err.println("Direct buffer is locked while being freed");
                }
                currentOutputBuffer = null;
            }
            directOutputBuffer = null;
        }
    }

    private void finishDecompressBlock(
        final OutputBuffer out,
        final IndexInput in)
        throws IOException
    {
        final long fileId = fileToId(in).longValue();
        final long iap = longKey(fileId, nextFilePointer);
        compressedCache.getAndDecompress(
            fileId,
            iap,
            nextFilePointer,
            in,
            out,
            decompressor,
            tag);
        if (out.out == null && out.compressedSize != 0) {
            throw new RuntimeException("out.out = null: " + out);
        }
    }

    private void unused1(
        final OutputBuffer out,
        final IndexInput in)
        throws IOException
    {
        FileDescriptor fileDes = in.getFileDescriptor();
        final int res;
        if (fileDes != null && decompressor instanceof ReadingDecompressor) {
            final int fd = NIOFSDirectory.getFd(fileDes);
            res = doDecompressBlock(
                fd,
                out.thisBlockPointer,
                out.compressedSize,
                out.out,
                out.plainSize);
        } else {
            byte[] readBuffer = getReadBuffer(out.compressedSize);
            try {
                in.seek(out.thisBlockPointer);
                in.readBytes(readBuffer, 0, out.compressedSize);
                res = doDecompressBlock(
                    readBuffer,
                    out.compressedSize,
                    out.out,
                    out.plainSize);
            } finally {
                if (readBuffer != null) {
                    freeReadBuffer(readBuffer);
                }
            }
        }
//        iap.in = null;
        if (res < 0) {
            throw new IOException("Inflate error: res=" + res
                + ", compressedSize=" + out.compressedSize
                + ", plainSize=" + out.plainSize
                + ", pos=" + out.thisBlockPointer
                + ", decompressor: " + decompressor
                + ", errorString: "
                + decompressor.errorDescription(res));
        }
    }

    private OutputBuffer decompressBlockFromCache(
        final IndexInput in,
        final int compressedSize,
        final int plainSize)
        throws IOException
    {
        freeDirectBuffer();
        final ConcurrentLinkedHashMapLong<OutputBuffer> cache = blockCache;
        final long fileId = fileToId(in).longValue();
        final long iap = longKey(fileId, nextFilePointer);
        OutputBuffer out = cache.get(iap);
        if (out == null) {
            rawStater.miss();
            OutputBuffer newBuffer = new OutputBuffer(allocator);
            synchronized (newBuffer) {
                out = cache.putIfAbsent(iap, newBuffer);
                if (out == null) {
                    out = newBuffer;
                    out.puted = true;
                    ConcurrentPositiveLongHashSet inputBlocks =
                        indexInputBlocks.get(fileId);
                    if (inputBlocks != null) {
                        inputBlocks.put(iap);
                    } else {
                        System.err.println("CACHE.get: inputBlocks == null");
                    }
                    out.compressedSize = compressedSize;
                    out.plainSize = plainSize;
                    try {
                        finishDecompressBlock(out, in);
                        out.unpacked = true;
                        blockCache.replace(iap, out);
                    } catch (IOException e) {
                        out.exception = e;
                        //exception releases lock
                        throw e;
                    }
                }
            }
        } else {
            rawStater.hit();
        }
        if (!out.unpacked) {
            IOException exception = null;
            synchronized (out) {
                if (!out.unpacked) {
                    exception = out.exception;
                    if (exception == null) {
                        exception = new IOException(
                            "Block is not unpacked, but no exception set, "
                                + ", lack of synchronization?");
                    }
                }
            }

            if (exception != null) {
                throw exception;
            }
            //wait for decompress if another thread is just inserted this block
        }
        if (out.out != null || out.released) {
            bufferLength = out.plainSize;
        } else {
            if (out.compressedSize == 0) {
                bufferLength = 0;
            } else {
                throw new IOException(
                    "Empty buffer: compressedSize="
                    + out.compressedSize + ", inp.compressedSize="
                    + compressedSize + ", out: " + out);
            }
        }
        nextFilePointer = out.nextBlockPointer;
        return out;
    }

    @Override
    public void close() throws IOException {
        otherInput.close();
        if (inputClone != null) {
            inputClone.close();
            inputClone = null;
        }
        freeDirectBuffer();
//        closed = true;
    }

    @Override
    public byte readByte() throws IOException {
//        if (!bufferLocked) {
//            new Exception("read without locked buffer: "
//                + currentOutputBuffer).printStackTrace();
//        }
        if (bufferPos[0] >= bufferLength) {
            if (!loadNextBlock(useCache())) {
                throw new EOFException("Read past EOF: file="
                    + otherInput.getCacheKey()
                    + ", bufferPos=" + bufferPos[0]
                    + ", bufferLength=" + bufferLength);
            }
        }
        final boolean locked = lockBuffer();
        try {
            return currentOutputBuffer.readByte(bufferPos[0]++);
        } finally {
            if (locked) {
                releaseBuffer();
            }
        }
    }

    @Override
    public byte readByteUnlocked() throws IOException {
//        if (!bufferLocked) {
//            new Exception("read without locked buffer: " + currentOutputBuffer).printStackTrace();
//        }
        if (bufferPos[0] >= bufferLength) {
            if (!loadNextBlock(useCache())) {
                throw new EOFException("Read past EOF: file="
                    + otherInput.getCacheKey()
                    + ", bufferPos=" + bufferPos[0]
                    + ", bufferLength=" + bufferLength);
            }
        }
        return currentOutputBuffer.readByte(bufferPos[0]++);
    }

    @Override
    public int readVInt() throws IOException {
        final boolean locked = lockBuffer();
        try {
            if (bufferPos[0] + 5 <= bufferLength) {
                return currentOutputBuffer.readVInt(bufferPos);
            } else {
                return readVIntSlow();
            }
        } finally {
            if (locked) {
                releaseBuffer();
            }
        }
    }

    public int readVIntUnlocked() throws IOException {
        return currentOutputBuffer.readVInt(bufferPos);
    }

    public int readVIntUnlockedSafe() throws IOException {
        if (bufferPos[0] + 5 <= bufferLength) {
            return currentOutputBuffer.readVInt(bufferPos);
        } else {
            return readVIntSlow();
        }
    }

    private int readVIntSlow() throws IOException {
        return super.readVInt();
    }

    @Override
    public long readVLong() throws IOException {
        final boolean locked = lockBuffer();
        try {
            if (bufferPos[0] + 9 <= bufferLength) {
                return currentOutputBuffer.readVLong(bufferPos);
            } else {
                return readVLongSlow();
            }
        } finally {
            if (locked) {
                releaseBuffer();
            }
        }
    }

    public long readVLongUnlocked() throws IOException {
        return currentOutputBuffer.readVLong(bufferPos);
    }

    private long readVLongSlow() throws IOException {
        return super.readVLong();
    }

    @Override
    public final void readBytes(
        final byte[] b,
        final int offset,
        final int len)
        throws IOException
    {
        readBytes(b, offset, len, useCache());
    }

    public final void readBytesUnlocked(
        final byte[] b,
        final int offset,
        final int len)
        throws IOException
    {
        currentOutputBuffer.readBytes(bufferPos[0], b, offset, len);
        bufferPos[0] += len;
    }

    @Override
    public void readBytes(
        final byte[] b,
        final int offset,
        final int len,
        final boolean useBuffer)
        throws IOException
    {
        int left = len;
        int outPos = offset;
//        if (!bufferLocked && !(this instanceof MutableBlockCompressedInputStream)) {
//            new Exception("read without locked buffer: " + currentOutputBuffer).printStackTrace();
//        }

        while (left > 0) {
            if (bufferPos[0] == bufferLength) {
                loadNextBlock(useBuffer);
            }
            int toCopy = Math.min(bufferLength - bufferPos[0], left);
            boolean locked = lockBuffer();
            try {
                currentOutputBuffer.readBytes(bufferPos[0], b, outPos, toCopy);
            } finally {
                if (locked) {
                    releaseBuffer();
                }
            }
            bufferPos[0] += toCopy;
            left -= toCopy;
            outPos += toCopy;
        }
    }

    @Override
    public void readBytesUnlockedSafe(
        final byte[] b,
        final int offset,
        final int len)
        throws IOException
    {
        int left = len;
        int outPos = offset;
//        if (!bufferLocked) {
//            new Exception("read without locked buffer: " + currentOutputBuffer).printStackTrace();
//        }

        while (left > 0) {
            if (bufferPos[0] == bufferLength) {
                loadNextBlock(useCache());
            }
            int toCopy = Math.min(bufferLength - bufferPos[0], left);
            currentOutputBuffer.readBytes(bufferPos[0], b, outPos, toCopy);
            bufferPos[0] += toCopy;
            left -= toCopy;
            outPos += toCopy;
        }
    }

    public Object clone(final BlockCompressedInputStreamBase cloned) {
        cloned.bufferPos[0] = bufferPos[0];
        cloned.bufferLength = bufferLength;
        cloned.currentOutputBuffer = currentOutputBuffer;
        if (bufferLocked) {
            new Exception("cloning with locked buffer: " + currentOutputBuffer)
                .printStackTrace();
            if (currentOutputBuffer != null) {
                cloned.bufferLocked = false;
                try {
                    cloned.lockBuffer();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        cloned.nextFilePointer = nextFilePointer;
        cloned.cloned = true;
        return cloned;
    }

    private NativeMemory2 getTemporaryBuffer(final int size) {
        NativeMemory2 tmpBuffer = threadLocalTempBuffer.get();
        if (tmpBuffer == null) {
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer.set(tmpBuffer);
        } else if (tmpBuffer.size() < (size << 1)) {
            tmpBuffer.free();
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer.set(tmpBuffer);
        }
        return tmpBuffer;
    }

    private NativeMemory2 getTemporaryBuffer2(final int size) {
        NativeMemory2 tmpBuffer = threadLocalTempBuffer2.get();
        if (tmpBuffer == null) {
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer2.set(tmpBuffer);
        } else if (tmpBuffer.size() < (size << 1)) {
            tmpBuffer.free();
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer2.set(tmpBuffer);
        }
        return tmpBuffer;
    }

    private static class Shrinker extends Thread {
        private volatile boolean stop = false;
        private final GCMonitor memMon;
        private final long memoryLimit;
        private final long minimalCacheSize;
        private final long minimalMemoryFree;
        private final ArrayDeque<Long> avgQueue = new ArrayDeque(SMOOTH_DEPTH);
        private long avgSum = 0;
        private long prevUsage = 0;
//        private GCMonitor gcmonitor;

        Shrinker(final long minimalCacheSize, final long minimalMemoryFree) {
            super("CacheShrinker");
            this.minimalCacheSize = minimalCacheSize;
            this.minimalMemoryFree = minimalMemoryFree;
            setDaemon(true);
            memMon = GCMonitor.instance();
            memoryLimit = memMon.memoryLimit();
        }

        @Override
        public void run() {
            if (memoryLimit == -1) {
                System.err.println("Memory limit is unknown, "
                    + "dynamic sizing is unavailable");
                return;
            }
            System.err.println("MemoryLimit: " + memoryLimit);
            while (!stop) {
                try {
                    long anonSize = anonymousMemoryUsage();
                    if (anonSize != prevUsage) {
                        prevUsage = anonSize;
                        long cacheUsed = compressedCache.used();
                        long usedMem = anonSize - cacheUsed;
                        long usageLimit = memoryLimit
                            - minimalMemoryFree - CACHE_SIZE_HANDICAP;
                        long cacheSize =
                            Math.max(usageLimit - usedMem, minimalCacheSize);
                        long diff = compressedCache.capacity() - cacheSize;
                        if (diff > CACHE_SIZE_HANDICAP) {
                            System.err.println("Shrinking compressed cache size"
                                + " to: " + cacheSize);
                            System.err.println("Shrinker: cacheUsed: "
                                + cacheUsed + ", anonSize: " + anonSize
                                + ", usedMem: " + usedMem
                                + ", usageLimit: " + usageLimit
                                + ", cacheSize: " + cacheSize
                                + ", diff=" + diff);
                            compressedCache.setCacheCapacity(cacheSize);
                        } else if (diff < -CACHE_SIZE_HANDICAP) {
                            System.err.println("Expanding compressed cache size"
                                + " to: " + cacheSize);
                            System.err.println("Shrinker: cacheUsed: "
                                + cacheUsed + ", anonSize: " + anonSize
                                + ", usedMem: " + usedMem
                                + ", usageLimit: " + usageLimit
                                + ", cacheSize: " + cacheSize
                                + ", diff=" + diff);
                            compressedCache.setCacheCapacity(cacheSize);
                        }
                    }
                    Thread.sleep(SHRINKER_SAMPLING_INTERVAL);
                } catch (Throwable t) {
                    t.printStackTrace();
                }
            }
        }

        private long anonymousMemoryUsage() {
            long mem = memMon.heapUsage();
            avgQueue.add(mem);
            avgSum += mem;
            if (avgQueue.size() > SMOOTH_DEPTH) {
                avgSum -= avgQueue.removeFirst();
            }
            return avgSum / avgQueue.size();
        }

        public void exit() {
            stop = true;
        }

    }
}
