package org.apache.lucene.store;

import ru.yandex.compress.NativeCompress;

public final class BrotliCompressor implements Compressor, ReadingDecompressor {
    private final int quality;

    public BrotliCompressor(final int level) {
        if (level <= 1) {
            quality = 0;
        } else if (level >= 9) {
            quality = 11;
        } else {
            quality = level;
        }
    }

    @Override
    public String id() {
        return "brotli";
    }

    @Override
    public int compress(
        final byte[] plain,
        final int plainLen,
        final byte[] compressed,
        final int maxCompressedLen)
    {
        return NativeCompress.brotliCompressByteArray(
            plain,
            plainLen,
            compressed,
            maxCompressedLen,
            quality);
    }

    @Override
    public int compress(
        final long plain,
        final int plainLen,
        final long compressed,
        final int maxCompressedLen)
    {
        return NativeCompress.brotliCompressRawArray(
            plain,
            plainLen,
            compressed,
            maxCompressedLen,
            quality);
    }

    @Override
    public BrotliCompressor decompressor() {
        return this;
    }

    @Override
    public int decompress(
        final byte[] compressed,
        final int compressedLen,
        final long decompressed,
        final int decompressedLen)
    {
        return NativeCompress.brotliDecompressByteArray(
            compressed,
            compressedLen,
            decompressed,
            decompressedLen);
    }

    @Override
    public int decompress(
        final long compressed,
        final int compressedLen,
        final long decompressed,
        final int decompressedLen)
    {
        return NativeCompress.brotliDecompressBytes(
            compressed,
            compressedLen,
            decompressed,
            decompressedLen);
    }

    @Override
    public int decompress(
        final int fd,
        final long fPos,
        final int compressedLen,
        final long decompressed,
        final int decompressedLen)
    {
        return NativeCompress.brotliReadAndDecompressByteArray(
            fd,
            fPos,
            compressedLen,
            decompressed,
            decompressedLen);
    }

    @Override
    public String errorDescription(int code) {
        return "Brotli error code " + code;
    }

    @Override
    public String toString() {
        return id();
    }
}
