package org.apache.lucene.store;

import java.io.IOException;

public class BufferedReader extends IndexInput {
    private byte[] buffer;
    private int bufferPosition;
    private int bufferLength;
    private int maxBufferSize;
    private IndexInput in;

    public BufferedReader() {
        this.in = null;
        this.buffer = null;
        this.bufferPosition = 0;
        this.bufferLength = 0;
    }

    public void setInput(
        final IndexInput in,
        final int maxBufferSize)
    {
        this.in = in;
        this.bufferPosition = 0;
        this.bufferLength = 0;
        this.maxBufferSize = maxBufferSize;
        if (this.buffer == null || this.buffer.length < maxBufferSize) {
            this.buffer = new byte[maxBufferSize];
        }
    }

    @Override
    public long length() {
        return in.length();
    }

    @Override
    public void seek(final long pos) throws IOException {
        if (pos < in.getFilePointer()) {
            int diff = (int)(in.getFilePointer() - pos);
            bufferPosition = bufferLength - diff;
        } else {
            in.seek(pos);
            this.bufferPosition = 0;
            this.bufferLength = 0;
        }
    }

    @Override
    public void close() {
    }

    @Override
    public long getFilePointer() {
        long fPos = in.getFilePointer();
        fPos -= bufferLength;
        fPos += bufferPosition;
//        System.err.println("BufferedReader.getFilePointer: " + fPos);
        return fPos;
    }

    public void clear() {
        in = null;
    }

    @Override
    public byte readByte() throws IOException {
        if (bufferPosition == bufferLength) {
            fillBuffer();
        }
        return buffer[bufferPosition++];
    }

    @Override
    public void readBytes(final byte[] out, final int offset, final int len)
        throws IOException
    {
        int left = len;
        int outPos = offset;
        while (left > 0) {
            if (bufferPosition == bufferLength) {
                fillBuffer();
            }
            int bufferLeft = bufferLength - bufferPosition;
            int toCopy = Math.min(left, bufferLeft);
            System.arraycopy(buffer, bufferPosition, out, outPos, toCopy);
            left -= toCopy;
            outPos += toCopy;
            bufferPosition += toCopy;
        }
    }

    private int fileLeft() {
        long left = in.length() - in.getFilePointer();
        if (left > (long)Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }
        return (int)left;
    }

    private void fillBuffer() throws IOException {
        int toRead = Math.min(maxBufferSize, fileLeft());
        in.readBytes(buffer, 0, toRead, true);
        bufferPosition = 0;
        bufferLength = toRead;
    }

  @Override
  public short readShort() throws IOException {
    if (2 <= (bufferLength-bufferPosition)) {
      return (short) (((buffer[bufferPosition++] & 0xFF) <<  8) |  (buffer[bufferPosition++] & 0xFF));
    } else {
      return super.readShort();
    }
  }

  @Override
  public int readInt() throws IOException {
    if (4 <= (bufferLength-bufferPosition)) {
      return ((buffer[bufferPosition++] & 0xFF) << 24) | ((buffer[bufferPosition++] & 0xFF) << 16)
        | ((buffer[bufferPosition++] & 0xFF) <<  8) |  (buffer[bufferPosition++] & 0xFF);
    } else {
      return super.readInt();
    }
  }

  @Override
  public long readLong() throws IOException {
    if (8 <= (bufferLength-bufferPosition)) {
      final int i1 = ((buffer[bufferPosition++] & 0xff) << 24) | ((buffer[bufferPosition++] & 0xff) << 16) |
        ((buffer[bufferPosition++] & 0xff) << 8) | (buffer[bufferPosition++] & 0xff);
      final int i2 = ((buffer[bufferPosition++] & 0xff) << 24) | ((buffer[bufferPosition++] & 0xff) << 16) |
        ((buffer[bufferPosition++] & 0xff) << 8) | (buffer[bufferPosition++] & 0xff);
      return (((long)i1) << 32) | (i2 & 0xFFFFFFFFL);
    } else {
      return super.readLong();
    }
  }

  @Override
  public int readVInt() throws IOException {
    if (5 <= (bufferLength-bufferPosition)) {
      byte b = buffer[bufferPosition++];
      int i = b & 0x7f;
      for (int shift = 7; b < 0; shift += 7) {
        b = buffer[bufferPosition++];
        i |= (b & 0x7f) << shift;
      }
      return i;
    } else {
      return super.readVInt();
    }
  }

  @Override
  public long readVLong() throws IOException {
    if (9 <= bufferLength-bufferPosition) {
      byte b = buffer[bufferPosition++];
      long i = b & 0x7fL;
      for (int shift = 7; b < 0; shift += 7) {
        b = buffer[bufferPosition++];
        i |= (b & 0x7fL) << shift;
      }
      return i;
    } else {
      return super.readVLong();
    }
  }

    @Override
    public Object getCacheKey() {
        return this;
    }

    @Override
    public boolean equals(Object o) {
        return false;
    }

}
