package org.apache.lucene.store;


import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;

import ru.yandex.cache.CacheInfoProvider;

import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class CacheStater implements Stater {
    private static final int[] INTERVALS = {
        60,
        600,
        3600};
    private static final Timer timer = new Timer("CacheStater", true);

    private final LongAdder hits = new LongAdder();
    private final LongAdder misses = new LongAdder();
    private final String name;
    private final int printInterval;
    private long prevPrintTime;
    private CacheInfoProvider sizeProvider;

    private long prevHits = 0;
    private long prevMisses = 0;
    private final Averager[] averagers = new Averager[INTERVALS.length];

    public CacheStater(
        final String name,
        final CacheInfoProvider sizeProvider,
        final int printInterval)
    {
        this.name = name;
        this.printInterval = printInterval;
        this.sizeProvider = sizeProvider;
        for (int i = 0; i < INTERVALS.length; i++) {
            averagers[i] = new Averager(INTERVALS[i]);
        }
        timer.scheduleAtFixedRate(new StatsPusher(), 1000, 1000);
        prevPrintTime = System.currentTimeMillis();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        for (int i = 0; i < INTERVALS.length; i++) {
            int interval = INTERVALS[i];
            Averager avg = averagers[i];
            statsConsumer.stat(
                name + "-cache-" + interval + "-hit-ratio_axxx",
                avg.value());
            statsConsumer.stat(
                name + "-cache-" + interval + "-hit-ratio_annn",
                avg.value());
            statsConsumer.stat(
                name + "-cache-" + interval + "-hit-ratio_avvv",
                avg.value());
        }
        long hits = this.hits.sum();
        long misses = this.misses.sum();
        double ratio = (hits * 100.0)
                / (double) (hits + misses);
        statsConsumer.stat(
            name + "-cache-total-hit-ratio_axxx",
            ratio);
        statsConsumer.stat(
            name + "-cache-total-hit-ratio_annn",
            ratio);
        statsConsumer.stat(
            name + "-cache-total-hit-ratio_avvv",
            ratio);
    }

    public void hit() {
        hits.add(1);
    }

    public void miss() {
        misses.add(1);
    }

    private void printStats() {
        StringBuilder sb = new StringBuilder();
        sb.append(name);
        sb.append("Cache stats: ");
        sb.append("capacity: ");
        sb.append(sizeProvider.capacity());
        sb.append(", used: ");
        sb.append(sizeProvider.used());
        sb.append(", fill: ");
        sb.append(
            round(
                (double) sizeProvider.used() / sizeProvider.capacity()
                    * 100.0));
        sb.append(" %, hit ratios: [");
        String sep = "";
        for (int i = 0; i < INTERVALS.length; i++) {
            sb.append(sep);
            int interval = INTERVALS[i];
            Averager avg = averagers[i];
            sb.append(interval);
            sb.append("s: ");
            sb.append(round(avg.value()));
            sep = " $, ";
        }
        sb.append(sep);
        sb.append("total: ");
        long hits = this.hits.sum();
        long misses = this.misses.sum();
        sb.append(
            round(
                (hits * 100.0)
                / (double) (hits + misses)));
        sb.append("], count: ");
        sb.append(sizeProvider.count());
        System.err.println(new String(sb));
    }

    private void pushStats() {
        long hits = this.hits.sum();
        long misses = this.misses.sum();
        long dHits = hits - prevHits;
        long dMisses = misses - prevMisses;
        prevHits = hits;
        prevMisses = misses;
        if (dHits != 0 || dMisses != 0) {
            double diffRatio =
                (dHits * 100.0) /
                (double) (dHits + dMisses);
            for (int i = 0; i < INTERVALS.length; i++) {
                averagers[i].push(diffRatio);
            }
        }
        long time = System.currentTimeMillis();
        if (time > prevPrintTime + printInterval) {
            prevPrintTime = time;
            printStats();
        }
    }

    private double round(final double value) {
        double round = value * 100;
        round = (double)((int) round);
        return round / 100;
    }

    private class StatsPusher extends TimerTask {
        @Override
        public void run() {
            pushStats();
        }
    }

    private static class Averager {
        public double sum = 0;
        public double[] values;
        public int idx = 0;
        public int count = 0;

        public Averager(int depth) {
            this.values = new double[depth];
        }

        public void push(double value) {
            sum += value;
            sum -= values[idx];
            values[idx++] = value;
            if (idx >= values.length) {
                idx = 0;
            }
            if (count < values.length) {
                count++;
            }
        }

        public double value() {
            return sum / count;
        }
    }
}
