package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import ru.yandex.msearch.util.JavaAllocator;

public class ChainedBlockCompressedInputStream
    extends BlockCompressedInputStreamBase
{
    private boolean isNextBlock = false;

    private int length;
    private long currentBlockStart = -1;
    private long nextBlockStart = -1;
    private boolean skipPending = false;
    private boolean eof = false;

    private long seekPending = -1;

    public ChainedBlockCompressedInputStream(
        final IndexInput in,
        final Decompressor decompressor,
        final JavaAllocator allocator,
        final String tag)
    {
        super(in, decompressor, allocator, tag);
        nextFilePointer = in.getFilePointer();
        length = 0;
        skipPending = false;
    }

    public void copyFrom(final ChainedBlockCompressedInputStream other)
        throws IOException
    {
        super.copyFrom(other);
        length = other.length;
        skipPending = other.skipPending;
        currentBlockStart = other.currentBlockStart;
        nextBlockStart = other.nextBlockStart;
        seekPending = other.nextFilePointer;
    }

    public int[] getPositionPtr() {
        return bufferPos;
    }

    @Override
    protected void doLoadBlock(final boolean useBuffer) throws IOException {
        if (seekPending != -1) {
            nextFilePointer = seekPending;
            seekPending = -1;
        }
        if (!skipPending) { //TODO: remember skipOffset
            if (currentBlockStart != nextFilePointer) {
                currentBlockStart = nextFilePointer;
                decompressBlock(otherInput, useBuffer);
                nextBlockStart = nextFilePointer;
                bufferPos[0] = 0;
            } else {
                skipPending = true;
            }
        } else {
            skipPending = false;
            if (nextBlockStart <= currentBlockStart) {
                throw new IllegalStateException(
                    "nextBlockStart is invalid: <= currentBlockStart: "
                    + nextBlockStart + " <= " + currentBlockStart);
            }
            nextFilePointer = nextBlockStart;
            doLoadBlock(useBuffer);
        }
//        System.err.println("NEXT BLOCK");
        isNextBlock = true;
    }

    @Override
    protected boolean loadNextBlock(final boolean useBuffer) throws IOException {
        loadBlock(useBuffer);
        return bufferLength != 0;
    }

    @Override
    public byte[] getNativeBuffer(final int size) {
        return allocator.alloc(size);
    }

    @Override
    public void freeNativeBuffer(final byte[] buffer) {
        allocator.free(buffer);
    }

    @Override
    public void close() throws IOException {
        super.close();
    }

    @Override
    public long length() {
        return bufferLength;
    }

    public final boolean eof() throws IOException {
        if (eof) {
            return true;
        }
        if (bufferPos[0] < bufferLength) {
            return false;
        }
        loadBlock(useCache());
        if (bufferLength > 0) {
            return false;
        }
        eof = true;
        return true;
    }

    public final boolean blockEof() {
        return bufferPos[0] >= bufferLength;
    }

    public final boolean isNextBlock() {
        return isNextBlock;
    }

    public final void clearNextBlock() {
        isNextBlock = false;
    }

    public final long getCurrentBlockFilePointer() {
        return currentBlockStart;
    }

    @Override
    public long getFilePointer() {
        return bufferPos[0];
    }

    public long nextFilePointer() {
        return nextFilePointer;
    }

    public BlockCompressedInputStreamBase clone() {
        ChainedBlockCompressedInputStream cloned =
            new ChainedBlockCompressedInputStream(
                (IndexInput)otherInput.clone(),
                decompressor,
                allocator,
                tag);
        super.clone(cloned);
        cloned.eof = eof;
        cloned.skipPending = skipPending;
        cloned.currentBlockStart = currentBlockStart;
        cloned.isNextBlock = isNextBlock;
        return cloned;
    }

    public final void seek(
        final long blockStart,
        final long blockOffset)
        throws IOException
    {
        eof = false;
        nextFilePointer = blockStart;
        skipPending = false;
        seekPending = -1;
        loadBlock(useCache());
        if (bufferLength != 0) {
            seek(blockOffset);
        }
    }

    @Override
    public final void seek(long pos) throws IOException {
        if (bufferPos[0] == 0 && bufferLength == 0) {
            loadBlock(useCache());
        }
        if (pos > bufferLength) {
            throw new IOException("Seek behind the end of buffer: pos=" + pos
                + ", length=" + bufferLength);
        }
        bufferPos[0] = (int)pos;
    }

    public boolean equals(Object _other) {
        return false;
    }

    public Object getCacheKey() {
        return this;
    }
}
