package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import copyleft.com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMapLong;
import copyleft.com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMapLong.Builder;
import copyleft.com.googlecode.concurrentlinkedhashmap.EntryWeigher;
import copyleft.com.googlecode.concurrentlinkedhashmap.EvictionListener;

import java.io.FileDescriptor;
import java.io.IOException;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.LongAdder;

import java.util.function.Consumer;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;

import org.apache.lucene.store.BlockCompressedInputStreamBase.InputAndPos;
import org.apache.lucene.store.BlockCompressedInputStreamBase.OutputBuffer;

import org.jctools.queues.MpmcArrayQueue;
import org.jctools.maps.NonBlockingHashMapLong;

import ru.yandex.base64.Base64Encoder;

import ru.yandex.cache.DBCache;
import ru.yandex.cache.CacheEntry;
import ru.yandex.cache.CacheInfoProvider;

import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.ConcurrentPositiveLongHashSet;

import ru.yandex.stater.StatsConsumer;

import ru.yandex.unsafe.NativeMemory2;
import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;

import ru.yandex.msearch.util.JavaAllocator;

public class CompressedBlockCache implements CacheInfoProvider {
    private static final int PREFETCH_BUFFER_SIZE = 4096;
    private static final NativeMemoryAllocator allocator =
        NativeMemoryAllocator.get("CompressedCache");
    private static final JavaAllocator BLOCK_ALLOCATOR =
        JavaAllocator.get("CompressedCache");
    private static final NativeMemoryAllocator tmpAllocator =
        NativeMemoryAllocator.get("CompressedCacheTemp");
    private static final int DEFAULT_CACHE_CONCURRENCY =
        Runtime.getRuntime().availableProcessors() << 1;
    private static final int CLEANUP_QUEUE_WAIT_DELAY = 50;
    private static final int CLEANUP_QUEUE_SIZE = 10000;
    private static final int CLEANUP_THREADS =
        Math.max(1, Runtime.getRuntime().availableProcessors() >> 3);
    private static final int REFINER_DELAY = 1;
    private static final int REFINER_BATCH_SIZE = 100;

    private static final MpmcArrayQueue<CompressedBlockPointer> CLEANUP_QUEUE =
        new MpmcArrayQueue<>(CLEANUP_QUEUE_SIZE);

    private final ThreadLocal<byte[]> threadLocalReadBuffer =
        new ThreadLocal<byte[]>();

    private final CacheRemovalListener evictionListener;

    private final CacheStater stater =
        new CacheStater("Compressed", this, 5000);

    private CacheStater ssdStater = null;

    private final ConcurrentLinkedHashMapLong<CompressedBlockPointer>
        blockCache;
    private final NonBlockingHashMapLong<ConcurrentPositiveLongHashSet>
        indexInputBlocks = new NonBlockingHashMapLong<>();
    private final ConcurrentMap<String, TagStats> tagsStats =
        new ConcurrentHashMap<String, TagStats>();
    private final ReferenceQueue<byte[]> referenceQueue =
        new ReferenceQueue<>();

    private final ThreadLocal<NativeMemory2> threadLocalTempBuffer =
        new ThreadLocal<>();
    private final ThreadLocal<NativeMemory2> threadLocalTempBuffer2 =
        new ThreadLocal<>();
    private boolean ssdCacheEnabled = true;


    private DBCache ssdCache = null;

    private double round(final double value) {
        double round = value * 100;
        round = (double)((int) round);
        return round / 100;
    }

    private static class Weigher
        implements EntryWeigher<Long, CompressedBlockPointer>
    {
        @Override
        public int weightOf(Long key, CompressedBlockPointer value) {
            return value.weight();
        }
    }

    private static class TagStats {
        private final LongAdder size = new LongAdder();
        private final LongAdder count = new LongAdder();
    }

    private static class Averager {
        public double sum = 0;
        public double[] values;
        public int idx = 0;
        public int count = 0;

        public Averager(int depth) {
            this.values = new double[depth];
        }

        public void push(double value) {
            sum += value;
            sum -= values[idx];
            values[idx++] = value;
            if (idx >= values.length) {
                idx = 0;
            }
            if (count < values.length) {
                count++;
            }
        }

        public double value() {
            return sum / count;
        }
    }

    private final class CacheRemovalListener
        implements EvictionListener<Long, CompressedBlockPointer>
    {
        public CacheRemovalListener() {
        }

        @Override
        public void onEviction(
            final Long key,
            final CompressedBlockPointer value)
        {
            if (key != -1) {
                removeInputBlock(key);
            }
            TagStats ts = tagsStats.get(value.tagKey);
            if (ts != null) {
                ts.size.add(-value.compressedSize);
                ts.count.add(-1);
            }
            while (!CLEANUP_QUEUE.offer(value)) {
                try {
                    Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    private static final class CompressedBlockPointer {
        private CompressedBlock blockRef;
        private int compressedSize = -1;
        private int decompressedSize;
        private long thisBlockPointer = 0;
        private long nextBlockPointer;
        private boolean red = false;
        private boolean evicted = false;
        private final String tagKey;
        private byte[] data;

        public CompressedBlockPointer(final String tagKey) {
            this.tagKey = tagKey;
        }

        public void allocate(final int size) {
            data = BLOCK_ALLOCATOR.alloc(size);
        }

        public int weight() {
            int weight = 200;
            byte[] data = this.data;
            if (data != null) {
                weight += data.length;
            } else if (blockRef != null) {
                data = blockRef.get();
                if (data != null) {
                    weight += data.length;
                }
            }
            return weight;
        }

        public void free() {
            byte[] data = this.data;
            if (data == null && blockRef != null) {
                data = blockRef.get();
            }
            if (data != null) {
                BLOCK_ALLOCATOR.free(data);
            }
            data = null;
            blockRef = null;
        }

        public void acquire() {
            if (blockRef != null) {
                data = blockRef.get();
            }
            if (data == null && blockRef != null) {
                System.err.println("GCed compressed block");
                red = false;
            }
        }

        public void release(
            final ReferenceQueue<byte[]> queue,
            final long key)
        {
            if (blockRef == null || blockRef.get() != data) {
                blockRef = new CompressedBlock(key, data, queue);
            }
            data = null;
        }

        public void write(
            final int pos,
            final NativeMemory2 mem,
            int offset,
            int size)
        {
            if (data == null) {
                throw new RuntimeException("writing to NULL buffer");
            }
            if (pos + size > data.length) {
                throw new RuntimeException(
                    "buffer size exceded: "
                        + "pos=" + pos
                        + ", offset=" + offset
                        + ", size=" + size
                        + ", data.len=" + data.length);
            }
            mem.read(offset, data, pos, size);
        }

        public void write(
            final int pos,
            final long mem,
            int size)
        {
            if (data == null) {
                throw new RuntimeException("writing to NULL buffer");
            }
            if (pos + size > data.length) {
                throw new RuntimeException(
                    "buffer size exceded: "
                        + "pos=" + pos
                        + ", size=" + size
                        + ", data.len=" + data.length);
            }
            NativeMemory2.unboxedRead(
                mem,
                data,
                pos,
                size);
        }

        public void read(
            final int pos,
            final NativeMemory2 mem,
            int size)
        {
            if (data == null) {
                throw new RuntimeException("reading from NULL buffer");
            }
            if (pos + size > data.length) {
                throw new RuntimeException("buffer size exceded: "
                    + data.length + " < " + (pos + size));
            }
            mem.write(0, data, pos, size);
        }
    }

    private static final class CompressedBlock extends SoftReference<byte[]> {
        private final long key;

        CompressedBlock(
            final long key,
            final byte[] data,
            final ReferenceQueue<byte[]> queue)
        {
            super(data, queue);
            this.key = key;
        }
    }

    public CompressedBlockCache(final long cacheSize) {
        evictionListener = new CacheRemovalListener();
        blockCache =
            new Builder<CompressedBlockPointer>()
                .initialCapacity((int) (cacheSize / 4096))
                .concurrencyLevel(DEFAULT_CACHE_CONCURRENCY << 2)
                .listener(evictionListener)
                .maximumWeightedCapacity(cacheSize)
                .weigher(new Weigher())
                .build();
        for (int i = 0; i < CLEANUP_THREADS; i++) {
            final int num = i;
            Thread cacheCleaner = new Thread("CBCacheCleaner-" + (num + 1)) {
                @Override
                public void run() {
                    //time shift threads
                    try {
                        Thread.sleep(
                            (CLEANUP_QUEUE_WAIT_DELAY / CLEANUP_THREADS) * num);
                    } catch (InterruptedException ign) {
                    }
                    while (true) {
                        try {
                            CompressedBlockPointer out;
                            while ((out = CLEANUP_QUEUE.poll()) == null) {
                                Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
                            }
//                            System.err.println("CC cleanup: " + out);
                            synchronized (out) {
                                out.evicted = true;
                                out.free();
                            }
                        } catch (Throwable t) {
                            t.printStackTrace();
                        }
                    }
                }
            };
            cacheCleaner.setDaemon(true);
            cacheCleaner.start();
        }
        Thread softRefCleaner = new Thread("CBSoftRefCleaner") {
            long prev = System.currentTimeMillis();
            int cleared = 0;

            private void printStats() {
                long time = System.currentTimeMillis();
                if((time - prev) > 1000 && cleared > 0) {
                    System.err.println("CBSoftRefCleaner: cleared: " + cleared);
                    cleared = 0;
                    prev = time;
                }
            }

            @Override
            public void run() {
                while (true) {
                    try {
                        Reference<? extends byte[]> ref;
                        while ((ref = referenceQueue.poll()) == null) {
                            printStats();
                            Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
                        }
                        cleared++;
                        printStats();
                        if (!(ref instanceof CompressedBlock)) {
                            continue;
                        }
                        CompressedBlock out = (CompressedBlock) ref;
//                        System.err.println("SoftRef cleanup: " + out);
                        CompressedBlockPointer pointer =
                            blockCache.get(out.key);
                        if (pointer != null) {
                            synchronized (pointer) {
                                if (pointer.data == null
                                    && pointer.blockRef == out)
                                {
                                    BLOCK_ALLOCATOR.free(
                                        pointer.compressedSize);
                                    pointer.evicted = true;
                                    blockCache.remove(out.key, pointer);
                                }
                            }
                        }
                    } catch (Throwable t) {
                        t.printStackTrace();
                    }
                }
            }};
        softRefCleaner.setDaemon(true);
        softRefCleaner.start();

        Thread softRefRefiner = new Thread("CBSoftRefRefiner") {
            @Override
            public void run() {
                while (true) {
                    try {
                        Thread.sleep(2000);
                        System.err.println("CBSoftRefRefiner: starting refine");
                        int cnt = 0;
                        int refined = 0;
                        long bytes = 0;
                        for (CompressedBlockPointer ptr: blockCache.values()) {
                            if (ptr.blockRef != null) {
                                byte[] data = ptr.blockRef.get();
                                if (data != null) {
                                    bytes += data.length;
                                }
                                refined++;
                            }
                            if (cnt++ > REFINER_BATCH_SIZE) {
                                Thread.sleep(REFINER_DELAY);
                                cnt = 0;
                            }
                        }
                        System.err.println(
                            "CBSoftRefRefiner: refine end: refined="
                                + refined + ", bytes=" + bytes);
                    } catch (Throwable t) {
                        t.printStackTrace();
                    }
                }
            }};
        softRefRefiner.setDaemon(true);
//        softRefRefiner.start();

    }

    public void setSsdCache(final DBCache ssdCache) {
        this.ssdCache = ssdCache;
        if (ssdCache != null) {
            this.ssdStater = new CacheStater("SSD", ssdCache, 5000);
        }
    }

    public void ssdCacheEnable(final boolean enable) {
        ssdCacheEnabled = enable;
    }

    public void freeCache() {
        blockCache.setCapacity(0);
        while (CLEANUP_QUEUE.size() > 0) {
            try {
                Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
//        System.err.println("toCompress.size() = " + toCompress.size());
    }

    public void waitFree() {
        while (CLEANUP_QUEUE.size() > 0) {
            try {
                Thread.sleep(CLEANUP_QUEUE_WAIT_DELAY);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
//        System.err.println("toCompress.size() = " + toCompress.size());
    }

    public static int fileIdFromKey(final long key) {
        return BlockCompressedInputStreamBase.fileIdFromKey(key);
    }

    public void printCacheInputs() {
        System.err.println("CC.size: " + blockCache.size());
        HashMap<Object, int[]> uniqInputs = new HashMap<>();
        for (Long key: blockCache.keySet()) {
            uniqInputs.computeIfAbsent(
                fileIdFromKey(key),
                (x) -> new int[1])[0]++;
        }
        for (Map.Entry<Object, int[]> entry: uniqInputs.entrySet()) {
            System.err.println("CacheInput: "
                + indexInputBlocks.containsKey(entry.getKey())
                + ": (C)" + entry.getKey() + ": " + entry.getValue()[0]);
        }
    }

    @Override
    public long capacity() {
        return blockCache.capacity();
    }

    @Override
    public long used() {
        return blockCache.weightedSize();
    }

    @Override
    public long count() {
        return blockCache.size();
    }

    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        stater.stats(statsConsumer);
        if (ssdStater != null) {
            ssdStater.stats(statsConsumer);
        }
    }

    public void dropCache() {
        final long currentCapacity = blockCache.capacity();
        freeCache();
        blockCache.setCapacity(currentCapacity);
    }

    public void setCacheCapacity(final long capacity) {
        blockCache.setCapacity(capacity);
    }

    public long cacheCapacity() {
        return blockCache.capacity();
    }

    public long freeCapacity() {
        return blockCache.capacity() - blockCache.weightedSize();
    }

    public void remove(final long key) {
        blockCache.remove(key);
    }

    long vIntsSize(final int a, final int b) {
        return (long) BlockCompressedInputStreamBase.vIntSize(a)
            + (long) BlockCompressedInputStreamBase.vIntSize(b);
    }

    public void getAndDecompress(
        final long fileId,
        final long key,
        final long pos,
        final IndexInput in,
        final OutputBuffer out,
        final Decompressor decompressor,
        final String tag)
        throws IOException
    {
        int retries = 0;
        while (true) {
            CompressedBlockPointer compressed = blockCache.get(key);
            if (compressed == null) {
                String tagKey = tag + '@' + ru.yandex.msearch.util
                    .IOScheduler.getThreadReadPrio();
                CompressedBlockPointer newBuffer =
                    new CompressedBlockPointer(tagKey.intern());
                compressed = blockCache.putIfAbsent(key, newBuffer);
                if (compressed == null) {
                    compressed = newBuffer;
                }
            }
            synchronized (compressed) {
                if (compressed.evicted) {
                    continue;
                }
                compressed.acquire();
                String ssdKey = null;
                long ssdTime = -1;
                if (!compressed.red || compressed.data == null) {
                    if (compressed.compressedSize == -1) {
                        compressed.compressedSize = out.compressedSize;
                        compressed.decompressedSize = out.plainSize;
                    }
                    boolean ssdRed = false;
                    if (ssdCache != null && ssdCacheEnabled) {
                        ssdKey = in.getCacheKey().toString() + ':' + pos;
                        ssdTime = System.nanoTime();
                        try (CacheEntry entry =
                            ssdCache.get(ssdKey, Compress.updateSsdCache()))
                        {
                            ssdTime = System.nanoTime() - ssdTime;
//                            System.err.println("entry: " + entry);
                            if (entry != null) {
//                                System.err.println(
//                                    "entry: cs: " + entry.compressedSize()
//                                    + ", ds: " + entry.decompressedSize());
                                compressed.compressedSize =
                                    entry.compressedSize();
                                compressed.decompressedSize =
                                    entry.decompressedSize();
                                if (entry.compressedSize() > 0) {
                                    compressed.allocate(entry.compressedSize());
//                                    compressed.address = allocator.allocUnboxed(
//                                        entry.compressedSize());
//                                    NativeMemory2.unboxedCopy(
//                                        entry.address(),
//                                        compressed.address,
//                                        entry.compressedSize());
                                    compressed.write(
                                        0,
                                        entry.address(),
                                        entry.compressedSize());
                                }
                                if (out.compressedSize == -1) {
                                    compressed.thisBlockPointer =
                                        pos + vIntsSize(
                                            compressed.compressedSize,
                                            compressed.decompressedSize);
                                } else {
                                    compressed.thisBlockPointer = pos;
                                }
                                compressed.nextBlockPointer =
                                    compressed.thisBlockPointer
                                    + (long) compressed.compressedSize;
                                ssdRed = true;
                                ssdStater.hit();
                                ru.yandex.msearch.util.Compress.ssdRead(
                                    tag,
                                    compressed.compressedSize,
                                    ssdTime);
/*
                                System.err.println("get ssdKey: " +
                                    ssdKey + ", cs: " + entry.compressedSize()
                                    + ", ps: " + entry.decompressedSize()
                                    + ", crc32: "
                                    + SqliteCache.crc32(
                                        compressed.address,
                                        compressed.compressedSize));
*/
                            }
                        } catch (IOException e) {
                            System.err.println("ssdCache.get error: "
                                + e.getMessage());
                            ssdStater.miss();
                        }
                    }
                    if (!ssdRed) {
                        readBlock(in, pos, compressed, tag);
                        if (ssdCache != null && ssdCacheEnabled) {
/*
                            System.err.println("put ssdKey: " + ssdKey
                                + ", cs: " + compressed.compressedSize
                                + ", ps: " + compressed.decompressedSize
                                + ", crc32: "
                                + SqliteCache.crc32(
                                    compressed.address,
                                    compressed.compressedSize));
*/
//                            if (ssdKey.equals("/u0/home/tabolin/mail/tabolin/5/_2m_0.tib:0")) {
//                                System.err.println("TIB0: cs: " + compressed.compressedSize
//                                    + ", ds: " + compressed.decompressedSize
//                                    + ", out.cs: " + out.compressedSize
//                                    + ", out.ps: " + out.plainSize);
//                            }
                            ssdStater.miss();
                            NativeMemory2 tmpBuf =
                                getTemporaryBuffer(compressed.compressedSize);
                            tmpBuf.write(
                                0,
                                compressed.data,
                                0,
                                compressed.compressedSize);
                            ssdCache.put(
                                ssdKey,
                                tmpBuf.address(),
                                compressed.compressedSize,
                                compressed.decompressedSize,
                                1,
                                false);
                            ru.yandex.msearch.util.Compress.ssdRead(
                                tag,
                                compressed.compressedSize,
                                ssdTime);
                        }
                    }
                    blockCache.replacel(key, compressed, compressed);
                }
                out.compressedSize = compressed.compressedSize;
                out.plainSize = compressed.decompressedSize;
                out.thisBlockPointer = compressed.thisBlockPointer;
                out.nextBlockPointer = compressed.nextBlockPointer;
                if (out.compressedSize > 0) {
                    try {
                        decompress(key, compressed, out, decompressor, tag);
                    } catch (IOException e) {
                        ssdKey = in.getCacheKey().toString() + ':' + pos;
                        Base64Encoder encoder = new Base64Encoder();
                        encoder.process(compressed.data);
                        System.err.println("Decompressd error on key: "
                            + ssdKey
                            + ", cs: " + out.compressedSize
                            + ", ds: " + out.plainSize
                            + ", dump: " + encoder.toString());
                        if (retries++ > 2) {
                            throw e;
                        } else {
                            compressed.red = false;
                            compressed.data = null;
                            continue;
                        }
                    }
                }
                if (!compressed.red) {
                    stater.miss();
                    ConcurrentPositiveLongHashSet inputBlocks =
                            indexInputBlocks.get(fileId);
                    if (inputBlocks != null) {
                        inputBlocks.put(key);
                    } else {
                        System.err.println("CACHE.get: inputBlocks == null");
                    }
                    TagStats ts = tagsStats.get(compressed.tagKey);
                    if (ts == null) {
                        TagStats newTs = new TagStats();
                        tagsStats.putIfAbsent(compressed.tagKey, newTs);
                        if (ts == null) {
                            ts = newTs;
                        }
                    }
                    ts.size.add(out.compressedSize);
                    ts.count.add(1);
                    compressed.red = true;
                } else {
                    stater.hit();
                }
            }
            break;
        }
    }

    private void readBlock(
        final IndexInput in,
        final long pos,
        final CompressedBlockPointer out,
        final String tag)
        throws IOException
    {
        final FileDescriptor fileDes = in.getFileDescriptor();
        final int fd = NIOFSDirectory.getFd(fileDes);
        if (out.compressedSize == -1) {
            long fadviseTime;
            if (ru.yandex.msearch.util.Compress.fadviseEnabled()) {
                fadviseTime = ru.yandex.msearch.util.Compress.fadviseOp(
                    fd,
                    pos,
                    512 * 1024,
                    tag,
                    false);
            } else {
                fadviseTime = 0;
            }
            NativeMemory2 tmpBuf = getTemporaryBuffer(PREFETCH_BUFFER_SIZE);
            int maxPrefetch =
                (int) Math.min((long) PREFETCH_BUFFER_SIZE, in.length() - pos);
//            System.err.println("pos: " + pos + ", maxPrefetch: " + maxPrefetch);
            int red = ru.yandex.msearch.util.Compress.preadOp(
                fd,
                pos,
                tmpBuf.address(),
                maxPrefetch,
                tag,
                fadviseTime);
            int[] ptr = new int[1];
            out.compressedSize = readVInt(tmpBuf, red, ptr);
            out.decompressedSize = readVInt(tmpBuf, red, ptr);
//            System.err.println(
//                "CompressedSize: " + out.compressedSize
//                + " decompressedSize: " + out.decompressedSize
//                + " ptr[0]: " + ptr[0]
//                + " red: " + red);
            out.thisBlockPointer = pos + ptr[0];
//            out.address = allocator.allocUnboxed(out.compressedSize);
            out.allocate(out.compressedSize);
            red -= ptr[0];
            int copy = Math.min(red, out.compressedSize);
            out.write(0, tmpBuf, ptr[0], copy);
            if (copy < out.compressedSize) {
                int readLeft = out.compressedSize - copy;
                tmpBuf = getTemporaryBuffer(readLeft);
//                red = ru.yandex.msearch.util.Compress.preadOp(
//                    fd,
//                    pos + ptr[0] + red,
//                    out.address + copy,
//                    out.compressedSize - copy,
//                    tag);
                red = ru.yandex.msearch.util.Compress.preadOp(
                    fd,
                    pos + ptr[0] + red,
                    tmpBuf.address(),
                    readLeft,
                    tag,
                    0);
                if (red != readLeft) {
                    throw new IOException("read error: ret2: "
                        + red + ", out.cs: " + out.compressedSize
                        + ", copy: " + copy + ", ptr: " + ptr[0]);
                }
                out.write(copy, tmpBuf, 0, red);
            }
        } else {
            long readPos = pos;
            if (out.thisBlockPointer != 0) {
                //garbage collected data, but pointers are valid
                readPos = out.thisBlockPointer;
            }
            long fadviseTime;
            if (ru.yandex.msearch.util.Compress.fadviseEnabled()) {
                fadviseTime = ru.yandex.msearch.util.Compress.fadviseOp(
                    fd,
                    readPos,
                    out.compressedSize * 8,
                    tag,
                    false);
            } else {
                fadviseTime = 0;
            }
            out.thisBlockPointer = readPos;
            out.allocate(out.compressedSize);//allocator.allocUnboxed(out.compressedSize);
            NativeMemory2 tmpBuf = getTemporaryBuffer(out.compressedSize);
            int red = ru.yandex.msearch.util.Compress.preadOp(
                fd,
                readPos,
                tmpBuf.address(),
                out.compressedSize,
                tag,
                fadviseTime);
            out.write(0, tmpBuf, 0, out.compressedSize);
            if (red < out.compressedSize) {
                throw new IOException("read error: " + red + " < "
                    + out.compressedSize);
            }
        }
        out.nextBlockPointer = out.thisBlockPointer + out.compressedSize;
    }

    private void decompress(
        final long key,
        final CompressedBlockPointer compressed,
        final OutputBuffer out,
        final Decompressor decompressor,
        final String tag)
        throws IOException
    {
        long time = System.nanoTime();
        out.resize(out.plainSize);
        NativeMemory2 tmpBuf = getTemporaryBuffer(out.out.length);
        NativeMemory2 tmpBuf2 =
            getTemporaryBuffer2(compressed.compressedSize);
        compressed.read(
            0,
            tmpBuf2,
            compressed.compressedSize);
        int ret = decompressor.decompress(
            tmpBuf2.address(),
            compressed.compressedSize,
            tmpBuf.address(),
            out.plainSize);
        time = System.nanoTime() - time;
        if (ret == out.plainSize) {
            tmpBuf.read(0, out.out, 0, out.out.length);
        }
        ru.yandex.msearch.util.Compress.unpack(
            tag,
            compressed.compressedSize,
            time);
        if (ret != out.plainSize) {
            throw new IOException(
                "Block decompress error: " +
                decompressor.errorDescription(ret)
                + ", cs: " + out.compressedSize
                + ", ps: " + out.plainSize);
        }
        compressed.release(referenceQueue, key);
    }

    private static int readVInt(final NativeMemory2 mem, int maxPtr, int[] ptr)
        throws IOException
    {
        byte b = readByte(mem, maxPtr, ptr);
        int i = b & 0x7F;
        for (int shift = 7; (b & 0x80) != 0; shift += 7) {
            b = readByte(mem, maxPtr, ptr);
            i |= (b & 0x7F) << shift;
        }
        return i;
    }

    private static byte readByte(final NativeMemory2 mem, int maxPtr, int[] ptr)
        throws IOException
    {
        if (ptr[0] >= maxPtr) {
            throw new EOFException("Read past eof: readVInt: pos="
                + ptr[0] + ", maxPos=" + maxPtr);
        }
        return mem.getByte(ptr[0]++);
    }

    private NativeMemory2 getTemporaryBuffer(final int size) {
        NativeMemory2 tmpBuffer = threadLocalTempBuffer.get();
        if (tmpBuffer == null) {
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer.set(tmpBuffer);
        } else if (tmpBuffer.size() < (size << 1)) {
            tmpBuffer.free();
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer.set(tmpBuffer);
        }
        return tmpBuffer;
    }

    private NativeMemory2 getTemporaryBuffer2(final int size) {
        NativeMemory2 tmpBuffer = threadLocalTempBuffer2.get();
        if (tmpBuffer == null) {
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer2.set(tmpBuffer);
        } else if (tmpBuffer.size() < (size << 1)) {
            tmpBuffer.free();
            tmpBuffer = tmpAllocator.alloc(size << 1);
            threadLocalTempBuffer2.set(tmpBuffer);
        }
        return tmpBuffer;
    }

    public void allocateInput(final long fileId) {
        indexInputBlocks.put(
            fileId,
            new ConcurrentPositiveLongHashSet(
                BlockCompressedInputStreamBase.PER_INPUT_HASHSET_SIZE));
    }

    public void drainInputBlocks(final long fileId) {
        ConcurrentPositiveLongHashSet blocks =
            indexInputBlocks.remove(fileId);
        if (blocks != null) {
            for (Long blockKey: blocks) {
                CompressedBlockPointer cb = blockCache.get(blockKey);
                if (cb != null) {
                    if (blockCache.remove(blockKey, cb)) {
                        evictionListener.onEviction(-1L, cb);
                    }
                }
            }
        } else {
            System.err.println("CACHE.drainInputBlocks null for: " + fileId);
        }
    }

    public void removeInputBlock(final long key) {
        ConcurrentPositiveLongHashSet blocks =
            indexInputBlocks.get(fileIdFromKey(key));
        if (blocks != null) {
            if (!blocks.remove(key)) {
                System.err.println("CACHE.removeInputBlock: failed for " + key);
            }
        }
    }

    public String tagsStats() {
        StringBuilder sb = new StringBuilder("CompressedCacheStats:\n");
        long totalSize = blockCache.weightedSize();
        for (Map.Entry<String, TagStats> entry: tagsStats.entrySet()) {
            String tag = entry.getKey();
            TagStats ts = entry.getValue();
            long size = ts.size.sum();
            sb.append('\t');
            sb.append(tag);
            sb.append(": ");
            sb.append(size);
            sb.append(" (");
            sb.append(ts.count.sum());
            sb.append(") ");
            sb.append(size * 100 / Math.max(totalSize, 1));
            sb.append("\n");
        }
        return new String(sb);
    }

    public double cacheFill() {
        return Math.min(
            100.0,
            ((double) blockCache.weightedSize() / blockCache.capacity()) * 100);
    }
}
