package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.io.File;
//import java.util.zip.Deflater;
import java.util.ArrayList;
import java.util.Arrays;

import org.apache.lucene.util.packed.PackedInts;
import ru.yandex.msearch.util.Compress;

import ru.yandex.unsafe.NativeMemory2;
import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;


/**
 * A memory-resident {@link IndexOutput} implementation.
 *
 * @lucene.internal
 */
public class DeflateOutputStream extends IndexOutput {
    private static final NativeMemoryAllocator allocator =
        NativeMemoryAllocator.get("CompressorTemp");

  static final int BUFFER_SIZE = 1024;

  private final Compressor compressor;
//  private Deflater deflater;

  private byte[] tailBuffer;
  private int tailBufferPos;
  
  private int tailBufferLength;
  
  public int finishSize;

  private byte[] deflateBuffer;
  
//  private int deflateBufferPosition;
//  private int deflateBufferLength;
  
  private int deflatedPos;
  
  private int confirmedPos;
  private int confirmedFree;
  
  private boolean nextFail;
  private int iterations;
  
//  private IndexOutput flushOutput;
  private boolean chainedMode = false;
  private int blockSize;
  private ArrayList<Integer> chainedDeflatedSize = null;
  private IndexOutput chainedOutput = null;
  private boolean backed = false;
  private boolean closeOut = false;

  private NativeMemory2 tempInputBuffer = null;
  private NativeMemory2 tempOutputBuffer = null;

  /** Construct an empty output buffer. */
  public DeflateOutputStream( int blockSize, final Compressor compressor ) {
    this.compressor = compressor;
//    deflater = new Deflater();
    this.blockSize = blockSize;
//    deflater.setLevel( 1 );
//    deflater.setStrategy( Deflater.FILTERED );
//    deflater.setStrategy( Deflater.HUFFMAN_ONLY );
    deflateBuffer = new byte[blockSize << 1];
//    deflateBufferPosition = 0;
//    deflateBufferLength = deflateBuffer.length;
    tailBufferPos = 0;
    tailBuffer = new byte[BUFFER_SIZE];
    tailBufferLength = tailBuffer.length;
    this.blockSize = blockSize;
    confirmedPos = 0;
    confirmedFree = blockSize;
    nextFail = false;
    iterations = 0;
  }
  
  public DeflateOutputStream( IndexOutput out, int blockSize, final boolean closeOut, final Compressor compressor ) throws IOException
  {
    this(blockSize, compressor);
    chainedOutput = out;
    this.closeOut = closeOut;
    backed = true;
    setChainedMode( true );
    out.seek( 8 );
  }
  
  public void setChainedMode( boolean chainedMode )
  {
    this.chainedMode = chainedMode;
//    this.flushOutput = flushOutput;
    if( chainedMode )
    {
	if( chainedDeflatedSize == null )
	{
	    chainedDeflatedSize = new ArrayList<Integer>();
	}
	if( chainedOutput == null ) chainedOutput = new RAMOutputStream();
	if( tailBufferLength < blockSize )
	{
	    growTailBuffer( blockSize - tailBufferLength );
	}
	if( !backed ) ((RAMOutputStream)chainedOutput).reset();
	tailBufferPos = 0;
    }
  }
  
  public void writeTo(IndexOutput out) throws IOException {
    out.writeBytes( tailBuffer, 0, tailBufferPos );
  }
  
  public void writeChainedTo( IndexOutput out ) throws IOException 
  {
    int lastBlockSize = blockSize;
    if( tailBufferPos > 0 )
    {
	lastBlockSize = tailBufferPos;
	chainedFlush();
    }
    out.writeVInt( chainedDeflatedSize.size() );
//    System.err.println( "writeChainedTo: chains count=" + chainedDeflatedSize.size() );
    for( int i = 0; i < chainedDeflatedSize.size(); i++ )
    {
//	System.err.println( "writeChaintTo: chain["+i+"]=" + chainedDeflatedSize.get(i) );
	out.writeVInt(chainedDeflatedSize.get(i));
    }
    out.writeVInt(lastBlockSize);
    ((RAMOutputStream)chainedOutput).writeTo( out );
  }
  
  private void finish() throws IOException
  {
    int lastBlockSize = blockSize;
    if( tailBufferPos > 0 )
    {
	lastBlockSize = tailBufferPos;
	chainedFlush();
    }
    long indexOffset = chainedOutput.getFilePointer();

    chainedOutput.writeVInt(blockSize);
    chainedOutput.writeVInt(lastBlockSize);

    long tmp = 8;
    ArrayList<Long> offsets = new ArrayList<Long>();
    offsets.add( tmp );
    for( int i = 0; i < chainedDeflatedSize.size(); i++ )
    {
	tmp += chainedDeflatedSize.get(i);
	offsets.add( tmp );
//	if( max < chainedDeflatedSize.get(i) ) max = chainedDeflatedSize.get(i);
    }
    PackedInts.Writer writer = PackedInts.getWriter( chainedOutput, offsets.size(), PackedInts.bitsRequired(offsets.get(offsets.size()-1)) );
    for( int i = 0; i < offsets.size(); i++ )
    {
	writer.add( offsets.get(i) );
    }
    writer.finish();

    chainedOutput.seek( 0 );
    chainedOutput.writeLong( indexOffset );

//    chainedOutput.writeTo( out );
  }
  
/*
  public void writeTo(IndexOutput out) throws IOException {
    flush();
    final long end = file.length;
    long pos = 0;
    int buffer = 0;
    while (pos < end) {
      int length = BUFFER_SIZE;
      long nextPos = pos + length;
      if (nextPos > end) {                        // at the last buffer
        length = (int)(end - pos);
      }
      out.writeBytes(file.getBuffer(buffer++), length);
      pos = nextPos;
    }
  }
*/
    public int deflateConfirmedTo( IndexOutput out ) throws IOException {
//	if( confirmedPos == 0 )
//	    if( 
	tailBufferPos = confirmedPos;
	return deflateTo(out);
    }

    private int compress(
        final byte[] input,
        final int inputLen,
        final byte[] output,
        final int outputLen)
    {
        if (tempInputBuffer == null) {
            tempInputBuffer = allocator.alloc(input.length);
        } else if (tempInputBuffer.size() < input.length) {
            tempInputBuffer.free();
            tempInputBuffer = allocator.alloc(input.length);
        }
        if (tempOutputBuffer == null) {
            tempOutputBuffer = allocator.alloc(output.length);
        } else if (tempOutputBuffer.size() < output.length) {
            tempOutputBuffer.free();
            tempOutputBuffer = allocator.alloc(output.length);
        }
        final int ret;
        tempInputBuffer.writeSlow(0, input, 0, inputLen);
        ret = compressor.compress(
            tempInputBuffer.address(),
            inputLen,
            tempOutputBuffer.address(),
            outputLen);
        tempOutputBuffer.readSlow(0, output, 0, outputLen);
        return ret;
    }

    public int deflateTo(IndexOutput out) throws IOException {
	int deflated = 0;
      try{
        if (deflateBuffer.length < tailBufferPos << 1) {
            deflateBuffer = new byte[tailBufferPos << 1];
        }
        int ret = compress(
            tailBuffer,
            tailBufferPos,
            deflateBuffer,
            deflateBuffer.length);
        if (ret < 0) {
            throw new IOException("Deflater error: ret = " + ret);
        }
        deflated = ret;
//        System.err.println("deflated=" + deflated);
        out.writeBytes(deflateBuffer, 0, deflated);
      } catch (IOException e) {
        System.err.println("deflated err tailBufferPos=" + tailBufferPos
            + ", deflatedBufferLength=" + deflateBuffer.length);
        e.printStackTrace();
        throw e;
      }
//	System.err.println( "tail<" + iterations + ">: " + tail + " : " + deflated + " / " + tailBufferPos + " / " + ((float)tailBufferPos / (float)deflated) );
//	for( int i = 0; i < blockSize - deflated; i++ ) out.writeByte( (byte)0 );
//	out.writeVInt( deflated );
//	out.writeVInt( tailBufferPos );
	return deflated;
    }

  /** Resets this to an empty file. */
  public void reset() {
    tailBufferPos = 0;
//    deflater.reset();
    confirmedPos = 0;
    confirmedFree = blockSize;
    nextFail = false;
    iterations = 0;
    if( chainedMode )
    {
	chainedMode = false;
	((RAMOutputStream)chainedOutput).reset();
	chainedDeflatedSize.clear();
    }
  }
  
//  public void checkPoint(  

  @Override
  public void close() throws IOException {
    flush();
    if( backed ) finish();
//    deflater.end();
    if (closeOut) {
        chainedOutput.close();
    }
    freeNativeBuffers();
  }

  private void freeNativeBuffers() {
    if (tempInputBuffer != null) {
      tempInputBuffer.free();
      tempInputBuffer = null;
    }
    if (tempOutputBuffer != null) {
      tempOutputBuffer.free();
      tempOutputBuffer = null;
    }
  }

  public void finalize() {
    freeNativeBuffers();
  }

  @Override
  public void seek(long pos) throws IOException {
    tailBufferPos = (int)pos;
    throw new RuntimeException( "Seek failed" );
  }

  @Override
  public long length() {
    if( chainedMode ) return chainedDeflatedSize.size() * blockSize + tailBufferPos;
    return tailBufferPos;
  }
  
  private void growTailBuffer( int delta )
  {
//    throw new RuntimeException( "Seek failed" );
//    System.err.println( "growTailBuffer: " + delta );
    byte[] newTailBuffer = new byte[tailBufferLength + delta];
    System.arraycopy( tailBuffer, 0, newTailBuffer, 0, tailBufferLength );
    tailBuffer = newTailBuffer;
    tailBufferLength = tailBuffer.length;
  }

  @Override
  public void writeByte(byte b) throws IOException {
//    System.err.println( "writeByte: " + tailBufferPos + "/" + tailBuffer.length );
//    assert tailBufferPos < tailBufferLength;

    if( chainedMode )
    {
	if( tailBufferPos == blockSize )
	{
	    chainedFlush();
	}
    }
    else
    {
	if( tailBufferPos == tailBufferLength )
	{
	    growTailBuffer( BUFFER_SIZE );
	}
    }


    tailBuffer[tailBufferPos++] = b;
  }
  
/*
  public boolean checkDeflatedSize()
  { 
    System.err.println( "checkSize: " + this + "bufferPos: " + tailBufferPos + " confirmedPos: " + confirmedPos + " confirmedFree: " + confirmedFree + " nextFail: " + nextFail + " iters: " + iterations );
    if( nextFail ) return false;
    if( tailBufferPos - confirmedPos < confirmedFree - ((blockSize / 100) + 1) - 12 )
    {
	confirmedFree -= tailBufferPos - confirmedPos;
	confirmedPos = tailBufferPos;
//	confirmedFree = blockSize - tailBufferPos;
	return true;
    }
    Deflater tmp = new Deflater();
    tmp.setInput( tailBuffer, 0, tailBufferPos );
    tmp.finish();
    int deflated = 0;
    while( !tmp.finished() )
    {
    int ret;
	ret = tmp.deflate( deflateBuffer, deflated, blockSize - deflated );
	if( ret > 0 ) deflated += ret;
	else break;
    }
    iterations++;
    if( !tmp.finished() )
    {
	System.err.println( "size not fit: " + deflated + "/" + tailBufferPos + " / " + confirmedPos + " / " + confirmedFree + " iters: " + iterations );
	return false;
    }
    confirmedPos = tailBufferPos;
    confirmedFree = blockSize - deflated;
    if( confirmedFree < 8 ) nextFail = true;
    return true;
  }
*/
  private void chainedFlush() throws IOException
  {
//            System.err.println("ChainedFlush: buf[0]=" + Integer.toHexString(tailBuffer[0])
//                 + ", buf[last]=" + Integer.toHexString(tailBuffer[tailBufferPos-1]));
	    int deflated = deflateTo( chainedOutput );
//	    System.err.println( "ChainedFlush: deflated=" + deflated + ", tailBufferPos=" + tailBufferPos );
	    chainedDeflatedSize.add( deflated );
	    tailBufferPos = 0;
//	    deflater.reset();
  }

  @Override
  public void writeBytes(byte[] b, int offset, int len) throws IOException {
    assert b != null;

    if( chainedMode )
    {
	if( len > blockSize - tailBufferPos )
	{
	int left = len;
	int offsetPos = offset;
//	    System.err.println( "len > blockSize - tailBufferPos=" + len + " > " + (blockSize - tailBufferPos) );
	    System.arraycopy( b, offsetPos, tailBuffer, tailBufferPos, blockSize - tailBufferPos );
//	    tailBufferPos += blockSize - tailBufferPos;
	    left -= blockSize - tailBufferPos;
	    offsetPos += blockSize - tailBufferPos;
//	    System.err.println( "left after first write: " + left + ", offsetPos = " + offsetPos );
	    tailBufferPos = blockSize;
	    chainedFlush();
	    while( (left / blockSize) > 0 )
	    {
		System.arraycopy( b, offsetPos, tailBuffer, 0, blockSize );
		tailBufferPos = blockSize;
		chainedFlush();
		left -= blockSize;
		offsetPos += blockSize;
	    }
	    if( left > 0 )
	    {
		System.arraycopy( b, offsetPos, tailBuffer, 0, left );
		tailBufferPos += left;
//		chainedFlush();
	    }
	}
	else
	{
	    System.arraycopy( b, offset, tailBuffer, tailBufferPos, len );
	    tailBufferPos += len;
	}
    }
    else
    {
	if( len > tailBufferLength - tailBufferPos )
	    growTailBuffer( ((len / BUFFER_SIZE) + 1)  * BUFFER_SIZE );

	System.arraycopy( b, offset, tailBuffer, tailBufferPos, len );
	tailBufferPos += len;
//	if( tailBufferPos == tailBufferLength )
//	    growTailBuffer( BUFFER_SIZE );
    }

//    for( int i = 0; i < len; i++ ) writeByte( b[offset+i] );    
  }

  @Override
  public void flush() throws IOException {
//    file.setLastModified(System.currentTimeMillis());
//    setFileLength();
  }

  @Override
  public long getFilePointer() {
    if( chainedMode ) return (long)chainedDeflatedSize.size() * (long)blockSize + (long)tailBufferPos;
    return tailBufferPos;
  }

  /** Returns byte usage of all buffers. */
  public long sizeInBytes() {
    return tailBuffer.length;
  }
/*  
  @Override
  public void copyBytes(DataInput input, long numBytes) throws IOException {
    assert numBytes >= 0: "numBytes=" + numBytes;

    while (numBytes > 0) {
      if (bufferPosition == bufferLength) {
        currentBufferIndex++;
        switchCurrentBuffer();
      }

      int toCopy = currentBuffer.length - bufferPosition;
      if (numBytes < toCopy) {
        toCopy = (int) numBytes;
      }
      input.readBytes(currentBuffer, bufferPosition, toCopy, false);
      numBytes -= toCopy;
      bufferPosition += toCopy;
    }

  }
*/
/*
    public static void main(String[] args) throws IOException {
        final String filePath = args[0];
        final String dirPath = filePath.substring(0, filePath.lastIndexOf('/'));
        final String name = filePath.substring(dirPath.length() + 1);
        Directory dir = new NIOFSDirectory(new File(dirPath));
        IndexInput ip = dir.openInput(name);
        final long indexOffset = ip.readLong();
        ip.seek(indexOffset);
        final int blockSize = ip.readVInt();
        final int lastBlockSize = ip.readVInt();
//        System.err.println("In
        System.err.println("blockSize=" + blockSize);
        System.err.println("lastBlockSize=" + lastBlockSize);
        final PackedInts.Reader index = PackedInts.getReader(ip);
        System.err.println("index.size()=" + index.size());
        System.err.println("lastOffset=" + index.get(index.size()-1));
        for (int i = 0; i < index.size(); i++) {
            System.err.println("i: " + i + ", pos=" + index.get(i));
        }
//        final long lastBlockOffset = 
    }
*/
}
