package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.lucene.util.ThreadInterruptedException;
import org.apache.lucene.util.Constants;

import ru.yandex.unsafe.NativeMemory2;

/**
 * <a name="subclasses"/>
 * Base class for Directory implementations that store index
 * files in the file system.  There are currently three core
 * subclasses:
 *
 * <ul>
 *
 *  <li> {@link SimpleFSDirectory} is a straightforward
 *       implementation using java.io.RandomAccessFile.
 *       However, it has poor concurrent performance
 *       (multiple threads will bottleneck) as it
 *       synchronizes when multiple threads read from the
 *       same file.
 *
 *  <li> {@link NIOFSDirectory} uses java.nio's
 *       FileChannel's positional io when reading to avoid
 *       synchronization when reading from the same file.
 *       Unfortunately, due to a Windows-only <a
 *       href="http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6265734">Sun
 *       JRE bug</a> this is a poor choice for Windows, but
 *       on all other platforms this is the preferred
 *       choice. Applications using {@link Thread#interrupt()} or
 *       {@link Future#cancel(boolean)} should use
 *       {@link SimpleFSDirectory} instead. See {@link NIOFSDirectory} java doc
 *       for details.
 *        
 *        
 *
 *  <li> {@link MMapDirectory} uses memory-mapped IO when
 *       reading. This is a good choice if you have plenty
 *       of virtual memory relative to your index size, eg
 *       if you are running on a 64 bit JRE, or you are
 *       running on a 32 bit JRE but your index sizes are
 *       small enough to fit into the virtual memory space.
 *       Java has currently the limitation of not being able to
 *       unmap files from user code. The files are unmapped, when GC
 *       releases the byte buffers. Due to
 *       <a href="http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4724038">
 *       this bug</a> in Sun's JRE, MMapDirectory's {@link IndexInput#close}
 *       is unable to close the underlying OS file handle. Only when
 *       GC finally collects the underlying objects, which could be
 *       quite some time later, will the file handle be closed.
 *       This will consume additional transient disk usage: on Windows,
 *       attempts to delete or overwrite the files will result in an
 *       exception; on other platforms, which typically have a &quot;delete on
 *       last close&quot; semantics, while such operations will succeed, the bytes
 *       are still consuming space on disk.  For many applications this
 *       limitation is not a problem (e.g. if you have plenty of disk space,
 *       and you don't rely on overwriting files on Windows) but it's still
 *       an important limitation to be aware of. This class supplies a
 *       (possibly dangerous) workaround mentioned in the bug report,
 *       which may fail on non-Sun JVMs.
 *       
 *       Applications using {@link Thread#interrupt()} or
 *       {@link Future#cancel(boolean)} should use
 *       {@link SimpleFSDirectory} instead. See {@link MMapDirectory}
 *       java doc for details.
 * </ul>
 *
 * Unfortunately, because of system peculiarities, there is
 * no single overall best implementation.  Therefore, we've
 * added the {@link #open} method, to allow Lucene to choose
 * the best FSDirectory implementation given your
 * environment, and the known limitations of each
 * implementation.  For users who have no reason to prefer a
 * specific implementation, it's best to simply use {@link
 * #open}.  For all others, you should instantiate the
 * desired implementation directly.
 *
 * <p>The locking implementation is by default {@link
 * NativeFSLockFactory}, but can be changed by
 * passing in a custom {@link LockFactory} instance.
 *
 * @see Directory
 */
public abstract class FSDirectory extends Directory {
  public static final long FSYNC_BYTES_INTERVAL =
    Long.parseLong(
      System.getProperty(
        "ru.yandex.lucene.fsync-bytes-interval",
        "8000000"));

  private final static MessageDigest DIGESTER;


    private static final boolean SUPPORT_O_DIRECT =
        Double.parseDouble(
            System.getProperty("java.specification.version")) > 10.0;

  static {
    try {
      DIGESTER = MessageDigest.getInstance("MD5");
    } catch (NoSuchAlgorithmException e) {
        throw new RuntimeException(e.toString(), e);
    }
  }

  /**
   * Default read chunk size.  This is a conditional default: on 32bit JVMs, it defaults to 100 MB.  On 64bit JVMs, it's
   * <code>Integer.MAX_VALUE</code>.
   *
   * @see #setReadChunkSize
   */
  public static final int DEFAULT_READ_CHUNK_SIZE = Constants.JRE_IS_64BIT ? Integer.MAX_VALUE : 100 * 1024 * 1024;
  private static final int HASH_SET_LIMIT = 1000;

  private final Object filesLock = new Object();
  private final boolean useODirect;
  private final long fsyncBytesInterval;

  protected final File directory; // The underlying filesystem directory
  protected final Set<String> staleFiles = Collections.synchronizedSet(new HashSet<>()); // Files written, but not yet sync'ed
  protected AtomicReference<FSIndexOutput[]> dirtyFiles =
    new AtomicReference<>(new FSIndexOutput[1]);
  protected final ConcurrentLinkedQueue<Integer> freeDirtyIds =
    new ConcurrentLinkedQueue<>();
  protected final AtomicInteger maxDirtyId = new AtomicInteger(0);
  protected final AtomicLong bytesLeftToSync = new AtomicLong(0);
  private Set<String> files = null;
  private int chunkSize = DEFAULT_READ_CHUNK_SIZE; // LUCENE-1566

  private int dirtyId() {
    Integer oldId = freeDirtyIds.poll();
    if (oldId != null) {
        return oldId;
    }
    int id = maxDirtyId.getAndIncrement();
    FSIndexOutput[] dirtyFiles = this.dirtyFiles.get();
    if (id >= dirtyFiles.length) {
        FSIndexOutput[] newFiles =
            Arrays.copyOf(dirtyFiles, dirtyFiles.length << 1);
        if (this.dirtyFiles.compareAndSet(dirtyFiles, newFiles)) {
            for (int i = 0; i < dirtyFiles.length; i++) {
                FSIndexOutput out = dirtyFiles[i];
                if (out != null) {
                    newFiles[i] = out;
                }
            }
        }
    }
    return id;
  }

  // returns the canonical version of the directory, creating it if it doesn't exist.
  private static File getCanonicalPath(File file) throws IOException {
    return new File(file.getCanonicalPath());
  }

  /** Create a new FSDirectory for the named location (ctor for subclasses).
   * @param path the path of the directory
   * @param lockFactory the lock factory to use, or null for the default
   * ({@link NativeFSLockFactory});
   * @throws IOException
   */
  protected FSDirectory(
    File path,
    LockFactory lockFactory,
    final boolean useODirect)
    throws IOException
  {
    this.useODirect = useODirect;
    this.fsyncBytesInterval = FSYNC_BYTES_INTERVAL;
    bytesLeftToSync.set(fsyncBytesInterval);
    // new ctors use always NativeFSLockFactory as default:
    if (lockFactory == null) {
      lockFactory = new NativeFSLockFactory();
    }
    directory = getCanonicalPath(path);

    if (directory.exists() && !directory.isDirectory())
      throw new NoSuchDirectoryException("file '" + directory + "' exists but is not a directory");

    setLockFactory(lockFactory);
    //populate .files
    listAll();
  }

  /** Creates an FSDirectory instance, trying to pick the
   *  best implementation given the current environment.
   *  The directory returned uses the {@link NativeFSLockFactory}.
   *
   *  <p>Currently this returns {@link MMapDirectory} for most Solaris
   *  and Windows 64-bit JREs, {@link NIOFSDirectory} for other
   *  non-Windows JREs, and {@link SimpleFSDirectory} for other
   *  JREs on Windows. It is highly recommended that you consult the
   *  implementation's documentation for your platform before
   *  using this method.
   *
   * <p><b>NOTE</b>: this method may suddenly change which
   * implementation is returned from release to release, in
   * the event that higher performance defaults become
   * possible; if the precise implementation is important to
   * your application, please instantiate it directly,
   * instead. For optimal performance you should consider using
   * {@link MMapDirectory} on 64 bit JVMs.
   *
   * <p>See <a href="#subclasses">above</a> */
  public static FSDirectory open(File path) throws IOException {
    return open(path, null);
  }

  public static FSDirectory open(
    final File path,
    final boolean useODirect)
    throws IOException
  {
    return open(path, null, useODirect);
  }

  public static FSDirectory open(File path, LockFactory lockFactory) throws IOException {
    return open(path, lockFactory, false);
  }

  /** Just like {@link #open(File)}, but allows you to
   *  also specify a custom {@link LockFactory}. */
  public static FSDirectory open(
    final File path,
    final LockFactory lockFactory,
    final boolean useODirect)
    throws IOException
  {
    if ((Constants.WINDOWS || Constants.SUN_OS)
          && Constants.JRE_IS_64BIT && MMapDirectory.UNMAP_SUPPORTED) {
      return new MMapDirectory(path, lockFactory, useODirect);
    } else if (Constants.WINDOWS) {
      return new SimpleFSDirectory(path, lockFactory, useODirect);
    } else {
      return new NIOFSDirectory(path, lockFactory, useODirect);
    }
  }

  @Override
  public void setLockFactory(LockFactory lockFactory) throws IOException {
    super.setLockFactory(lockFactory);

    // for filesystem based LockFactory, delete the lockPrefix, if the locks are placed
    // in index dir. If no index dir is given, set ourselves
    if (lockFactory instanceof FSLockFactory) {
      final FSLockFactory lf = (FSLockFactory) lockFactory;
      final File dir = lf.getLockDir();
      // if the lock factory has no lockDir set, use the this directory as lockDir
      if (dir == null) {
        lf.setLockDir(directory);
        lf.setLockPrefix(null);
      } else if (dir.getCanonicalPath().equals(directory.getCanonicalPath())) {
        lf.setLockPrefix(null);
      }
    }

  }
  
  /** Lists all files (not subdirectories) in the
   *  directory.  This method never returns null (throws
   *  {@link IOException} instead).
   *
   *  @throws NoSuchDirectoryException if the directory
   *   does not exist, or does exist but is not a
   *   directory.
   *  @throws IOException if list() returns null */
  public static String[] listAll(File dir) throws IOException {
    if (!dir.exists())
      throw new NoSuchDirectoryException("directory '" + dir + "' does not exist");
    else if (!dir.isDirectory())
      throw new NoSuchDirectoryException("file '" + dir + "' exists but is not a directory");

    // Exclude subdirs
    String[] result = dir.list(new FilenameFilter() {
        public boolean accept(File dir, String file) {
          return !new File(dir, file).isDirectory();
        }
      });

    if (result == null)
      throw new IOException("directory '" + dir + "' exists and is a directory, but cannot be listed: list() returned null");

    return result;
  }

  public static String[] listAll(File dir, final String prefix) throws IOException {
    if (!dir.exists())
      throw new NoSuchDirectoryException("directory '" + dir + "' does not exist");
    else if (!dir.isDirectory())
      throw new NoSuchDirectoryException("file '" + dir + "' exists but is not a directory");

    // Exclude subdirs
    String[] result = dir.list(new FilenameFilter() {
        public boolean accept(File dir, String file) {
          File f = new File(dir, file);
          return !f.isDirectory() && file.startsWith(prefix);
        }
      });

    if (result == null)
      throw new IOException("directory '" + dir + "' exists and is a directory, but cannot be listed: list() returned null");

    return result;
  }

    private void updateFileList(final String file) {
        synchronized (filesLock) {
            if (files != null) {
                files.add(file);
            }
        }
    }

    public void move(
        final Directory to,
        final String src,
        final String dest)
        throws IOException
    {
        if (to instanceof FSDirectory) {
            final FSDirectory toFs = (FSDirectory)to;
            final File srcFile = new File(directory, src);
            final File dstFile = new File(toFs.directory, dest);
            if (!srcFile.renameTo(dstFile)) {
                super.move(to, src, dest);
            } else {
                toFs.updateFileList(dest);
            }
        } else {
            super.move(to, src, dest);
        }
        synchronized (filesLock) {
            if (files != null) {
                files.remove(src);
                checkListLimit();
            }
        }
    }

    private void checkListLimit() {
        int size = files.size();
        if (size > HASH_SET_LIMIT) {
            if (!(files instanceof SortedSet)) {
                files = new TreeSet<>(files);
            }
        } else if (size < (HASH_SET_LIMIT >> 1)) {
            if (files instanceof SortedSet) {
                files = new HashSet<>(files);
            }
        }
    }

  /** Lists all files (not subdirectories) in the
   * directory.
   * @see #listAll(File) */
  @Override
  public String[] listAll() throws IOException {
    ensureOpen();
    synchronized (filesLock) {
        if (files != null) {
            return files.toArray(new String[files.size()]);
        }
    }
    final String[] result = listAll(directory);
    final Set<String> files;
    if (result.length > HASH_SET_LIMIT) {
        files = new TreeSet<>();
    } else {
        files = new HashSet<>(result.length << 1);
    }
    for (String file : result) {
        files.add(file);
    }
    synchronized (filesLock) {
        if (this.files == null) {
            this.files = files;
            return result;
        } else {
            return this.files.toArray(new String[this.files.size()]);
        }
    }
  }

  @Override
  public String[] listAll(final String prefix) throws IOException {
    ensureOpen();
    synchronized (filesLock) {
        if (files != null) {
            ArrayList<String> matched = new ArrayList<>();
            if (files instanceof SortedSet) {
                SortedSet<String> tailSet = ((SortedSet) files).tailSet(prefix);
                for (String file: tailSet) {
                    if (file.startsWith(prefix)) {
                        matched.add(file);
                    } else {
                        break;
                    }
                }
            } else {
                for (String file: files) {
                    if (file.startsWith(prefix)) {
                        matched.add(file);
                    }
                }
            }
            return matched.toArray(new String[0]);
        }
    }
    final String[] result = listAll(directory, prefix);
    return result;
  }

  /** Returns true iff a file with the given name exists. */
  @Override
  public boolean fileExists(String name) {
    ensureOpen();
    File file = new File(directory, name);
    return file.exists();
  }

  /** Returns the time the named file was last modified. */
  @Override
  public long fileModified(String name) {
    ensureOpen();
    File file = new File(directory, name);
    return file.lastModified();
  }

  /** Returns the time the named file was last modified. */
  public static long fileModified(File directory, String name) {
    File file = new File(directory, name);
    return file.lastModified();
  }

  /** Set the modified time of an existing file to now. */
  @Override
  public void touchFile(String name) {
    ensureOpen();
    File file = new File(directory, name);
    file.setLastModified(System.currentTimeMillis());
    synchronized (filesLock) {
        if (files != null) {
            files.add(name);
            checkListLimit();
        }
    }
  }

  /** Returns the length in bytes of a file in the directory. */
  @Override
  public long fileLength(String name) throws IOException {
    ensureOpen();
    File file = new File(directory, name);
    final long len = file.length();
    if (len == 0 && !file.exists()) {
      throw new FileNotFoundException(name);
    } else {
      return len;
    }
  }

  /** Removes an existing file in the directory. */
  @Override
  public void deleteFile(String name) throws IOException {
    ensureOpen();
    File file = new File(directory, name);
    if (!file.delete())
      throw new IOException("Cannot delete " + file);
    BlockCompressedInputStreamBase.removeFile(file.getCanonicalPath());
    staleFiles.remove(name);
    synchronized (filesLock) {
        if (files != null) {
            files.remove(name);
            checkListLimit();
        }
    }
  }

  /** Creates an IndexOutput for the file with the given name. */
  @Override
  public IndexOutput createOutput(String name) throws IOException {
    ensureOpen();

    boolean success = true;
    try {
        return new FSIndexOutput(this, name, useODirect);
    } catch (IOException e) {
        success = false;
        throw e;
    } finally {
        if (success) {
            synchronized (filesLock) {
                if (files != null) {
                    files.add(name);
                    checkListLimit();
                }
            }
        }
    }
  }

  @Override
  public IndexOutput createOutput(String name, final int bufferSize) throws IOException {
    ensureOpen();

    boolean success = true;
    try {
        return new FSIndexOutput(this, name, bufferSize, useODirect);
    } catch (IOException e) {
        success = false;
        throw e;
    } finally {
        if (success) {
            synchronized (filesLock) {
                if (files != null) {
                    files.add(name);
                    checkListLimit();
                }
            }
        }
    }
  }

  protected void onIndexOutputClosed(FSIndexOutput io) {
    staleFiles.add(io.name);
    freeDirtyIds.add(io.dirtyId);
  }

  protected void onBytesWritten(final int len) {
    if (bytesLeftToSync.addAndGet(-len) < 0) {
        boolean fsync = false;
        synchronized(this) {
            if (bytesLeftToSync.get() < 0) {
                bytesLeftToSync.set(fsyncBytesInterval);
                fsync = true;
            }
        }
        if (fsync) {
            System.err.println("Fsyncing directory: " + directory.getName());
            try {
                sync();
            } catch (IOException e) {
                System.err.println(
                    "Fsyncing failed for directory: " + directory.getName());
            }
        }
    }
  }

  @Override
  public void sync(Collection<String> names) throws IOException {
    ensureOpen();
    Set<String> toSync = new HashSet<String>(names);
    toSync.retainAll(staleFiles);

    for (String name : toSync)
      fsync(name);

    staleFiles.removeAll(toSync);
  }

  public void sync() throws IOException {
    FSIndexOutput[] dirtyFiles = this.dirtyFiles.get();
    for (int i = 0; i < dirtyFiles.length; i++) {
        FSIndexOutput out = dirtyFiles[i];
        if (out != null) {
            out.sync();
        }
    }
  }

  // Inherit javadoc
  @Override
  public IndexInput openInput(String name) throws IOException {
    ensureOpen();
    return openInput(name, BufferedIndexInput.BUFFER_SIZE);
  }

  /**
   * So we can do some byte-to-hexchar conversion below
   */
  private static final char[] HEX_DIGITS =
  {'0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f'};

  @Override
  public String getLockID() {
    ensureOpen();
    String dirName;                               // name to be hashed
    try {
      dirName = directory.getCanonicalPath();
    } catch (IOException e) {
      throw new RuntimeException(e.toString(), e);
    }

    byte digest[];
    synchronized (DIGESTER) {
      digest = DIGESTER.digest(dirName.getBytes());
    }
    StringBuilder buf = new StringBuilder();
    buf.append("lucene-");
    for (int i = 0; i < digest.length; i++) {
      int b = digest[i];
      buf.append(HEX_DIGITS[(b >> 4) & 0xf]);
      buf.append(HEX_DIGITS[b & 0xf]);
    }

    return buf.toString();
  }

  /** Closes the store to future operations. */
  @Override
  public synchronized void close() {
    isOpen = false;
  }

  /** @return the underlying filesystem directory */
  public File getDirectory() {
    ensureOpen();
    return directory;
  }

  /** For debug output. */
  @Override
  public String toString() {
    return this.getClass().getName() + '@' + directory + " lockFactory=" + getLockFactory();
  }

  /**
   * Sets the maximum number of bytes read at once from the
   * underlying file during {@link IndexInput#readBytes}.
   * The default value is {@link #DEFAULT_READ_CHUNK_SIZE};
   *
   * <p> This was introduced due to <a
   * href="http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6478546">Sun
   * JVM Bug 6478546</a>, which throws an incorrect
   * OutOfMemoryError when attempting to read too many bytes
   * at once.  It only happens on 32bit JVMs with a large
   * maximum heap size.</p>
   *
   * <p>Changes to this value will not impact any
   * already-opened {@link IndexInput}s.  You should call
   * this before attempting to open an index on the
   * directory.</p>
   *
   * <p> <b>NOTE</b>: This value should be as large as
   * possible to reduce any possible performance impact.  If
   * you still encounter an incorrect OutOfMemoryError,
   * trying lowering the chunk size.</p>
   */
  public final void setReadChunkSize(int chunkSize) {
    // LUCENE-1566
    if (chunkSize <= 0) {
      throw new IllegalArgumentException("chunkSize must be positive");
    }
    if (!Constants.JRE_IS_64BIT) {
      this.chunkSize = chunkSize;
    }
  }

  /**
   * The maximum number of bytes to read at once from the
   * underlying file during {@link IndexInput#readBytes}.
   * @see #setReadChunkSize
   */
  public final int getReadChunkSize() {
    // LUCENE-1566
    return chunkSize;
  }

  protected class FSIndexOutput extends BufferedIndexOutput {
    private final FSDirectory parent;
    private final String name;
    private final RandomAccessFile file;
    private volatile boolean isOpen; // remember if the file is open, so that we don't try to close it more than once
    private final boolean useODirect;
    protected final int dirtyId;

    public FSIndexOutput(
        final FSDirectory parent,
        final String name,
        final boolean useODirect)
        throws IOException
    {
        this(
            parent,
            name,
            BufferedIndexOutput.BUFFER_SIZE,
            useODirect);
    }

    public FSIndexOutput(
        final FSDirectory parent,
        final String name,
        final int bufferSize,
        final boolean useODirect)
      throws IOException
    {
      super(bufferSize);
      this.parent = parent;
      this.name = name;
      this.useODirect = useODirect;
      this.dirtyId = dirtyId();
      String mode;
      if (useODirect) {
        mode = "rwd";
      } else {
        mode = "rw";
      }
      file = new RandomAccessFile(new File(parent.directory, name), mode);
      file.setLength(0);
      isOpen = true;
      int fd = NIOFSDirectory.getFd(file.getFD());
      ru.yandex.msearch.util.Compress.fadviseDontneed(fd);
//      if (useODirect) {
//        ru.yandex.msearch.util.Compress.setODirect(fd, true);
//      }
    }

    @Override
    public void writeBytes(
        final NativeMemory2 src,
        final int offset,
        final int length)
        throws IOException
    {
        flush();
        final long filePointer = file.getFilePointer();
        long ret = ru.yandex.msearch.util.Compress.pwrite(
            NIOFSDirectory.getFd(file.getFD()),
            filePointer,
            src.address() + offset,
            length,
            useODirect);
        if (ret < length) {
            throw new IOException("Write failed: pwrite error: " + ret);
        }
        file.seek(filePointer + length);
        seek(filePointer + length);
        dirtyFiles.get()[dirtyId] = this;
        onBytesWritten(length);
    }

    /** output methods: */
    @Override
    public void flushBuffer(byte[] b, int offset, int size) throws IOException {
        if (size > 0) {
            ru.yandex.msearch.util.IOScheduler.instance().writeOp(
                () -> {
                    file.write(b, offset, size); return null;
                });
            dirtyFiles.get()[dirtyId] = this;
            onBytesWritten(size);
        }
    }

    @Override
    public void close() throws IOException {
      // only close the file if it has not been closed yet
      if (isOpen) {
        try (Closeable file = this.file) {
          dirtyFiles.get()[dirtyId] = null;
          super.close();
        } finally {
          isOpen = false;
          parent.onIndexOutputClosed(this);
        }
      }
    }

    /** Random-access methods */
    @Override
    public void seek(long pos) throws IOException {
      super.seek(pos);
      file.seek(pos);
    }

    @Override
    public long length() throws IOException {
      return file.length();
    }

    @Override
    public void setLength(long length) throws IOException {
      file.setLength(length);
    }

    protected void sync() throws IOException {
        dirtyFiles.get()[dirtyId] = null;
        file.getFD().sync();
    }


  }

  protected void fsync(String name) throws IOException {
    File fullFile = new File(directory, name);
    boolean success = false;
    int retryCount = 0;
    IOException exc = null;
    while (!success && retryCount < 5) {
      retryCount++;
      RandomAccessFile file = null;
      try {
        try {
          file = new RandomAccessFile(fullFile, "rw");
          file.getFD().sync();
          success = true;
        } finally {
          if (file != null)
            file.close();
        }
      } catch (IOException ioe) {
        if (exc == null)
          exc = ioe;
        try {
          // Pause 5 msec
          Thread.sleep(5);
        } catch (InterruptedException ie) {
          throw new ThreadInterruptedException(ie);
        }
      }
    }
    if (!success)
      // Throw original exception
      throw exc;
  }
}
