package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.util.packed.PackedInts;

import ru.yandex.collection.ChunkedIntList;

import ru.yandex.unsafe.NativeMemory2;
import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;

public class IndexingCompressingOutputStream extends IndexOutput {
    private static final NativeMemoryAllocator allocator =
        NativeMemoryAllocator.get("FieldsCompressorTemp");

    private final ChunkedIntList compressedSizes = new ChunkedIntList();
    private final IndexOutput out;
    private final int blockSize;
    private final boolean closeOut;
    private final Compressor compressor;

    private long maxOffset = 8L;
    private int tailBufferPos = 0;
    private NativeMemory2 tempOutputBuffer = null;
    private NativeMemory2 tailBuffer;

    public IndexingCompressingOutputStream(
        final IndexOutput out,
        final int blockSize,
        final boolean closeOut,
        final Compressor compressor) throws IOException
    {
        this.out = out;
        this.blockSize = blockSize;
        this.closeOut = closeOut;
        this.compressor = compressor;
        out.seek(8);
        tailBuffer = allocator.alloc(blockSize);
    }

    private void finish() throws IOException {
        int lastBlockSize = blockSize;
        if (tailBufferPos > 0) {
            lastBlockSize = tailBufferPos;
            flushBlock();
        }
        long indexOffset = out.getFilePointer();

        out.writeVInt(blockSize);
        out.writeVInt(lastBlockSize);

        long tmp = 8L;
        int size = compressedSizes.size();
        PackedInts.Writer writer = PackedInts.getWriter(
            out,
            size + 1,
            PackedInts.bitsRequired(maxOffset));
        writer.add(tmp);
        for (int i = 0; i < size; i++) {
            tmp += compressedSizes.getInt(i);
            writer.add(tmp);
        }
        writer.finish();

        out.seek( 0 );
        out.writeLong(indexOffset);
    }

    public int currentBlockNumber() {
        return compressedSizes.size();
    }

    public int currentBlockSize() {
        return tailBufferPos;
    }

    private int compress(
        final NativeMemory2 input,
        final int inputLen,
        final NativeMemory2 output,
        final int outputLen)
    {
        return compressor.compress(
            input.address(),
            inputLen,
            output.address(),
            outputLen);
    }

    public int compressBlock() throws IOException {
        int outputLen = Math.max(
            blockSize << 1,
            compressor.zeroCompressOverhead());
        if (tempOutputBuffer == null) {
            tempOutputBuffer = allocator.alloc(outputLen);
        }
        int compressed = compress(
            tailBuffer,
            tailBufferPos,
            tempOutputBuffer,
            outputLen);
        if (compressed < 0) {
            System.err.println("Deflater error: ret = " + compressed);
            throw new IOException("Deflater error: ret = " + compressed);
        }
        out.writeBytes(tempOutputBuffer, 0, compressed);
//        System.err.println("compressed: " + tailBufferPos + " -> " + compressed);
        return compressed;
    }

    @Override
    public void close() throws IOException {
        finish();
        if (closeOut) {
            out.close();
        }
        freeNativeBuffers();
    }

    private void freeNativeBuffers() {
        if (tempOutputBuffer != null) {
            tempOutputBuffer.free();
            tempOutputBuffer = null;
        }
        if (tailBuffer != null) {
            tailBuffer.free();
            tailBuffer = null;
            tailBufferPos = 0;
        }
    }

    public void finalize() {
        freeNativeBuffers();
    }

    @Override
    public void seek(long pos) throws IOException {
        throw new UnsupportedOperationException("seek");
    }

    @Override
    public long length() {
        throw new UnsupportedOperationException("seek");
    }


    private void flushBlock() throws IOException {
        if (tailBufferPos > 0) {
            int compressed = compressBlock();
            compressedSizes.addInt(compressed);
            maxOffset += compressed;
            tailBufferPos = 0;
        }
    }

    @Override
    public void writeByte(byte b) throws IOException {
        tailBuffer.setByte(tailBufferPos++, b);
        if (tailBufferPos == blockSize) {
            flushBlock();
        }
    }

    @Override
    public void writeBytes(
        final byte[] b,
        final int offset,
        final int len) throws IOException
    {
        int remains = len;
        int pos = offset;
        while (remains > 0) {
            int bufferFree = blockSize - tailBufferPos;
            int toWrite = Math.min(bufferFree, remains);
            tailBuffer.writeSlow(tailBufferPos, b, pos, toWrite);
            remains -= toWrite;
            pos += toWrite;
            tailBufferPos += toWrite;
            if (tailBufferPos == blockSize) {
                flushBlock();
            }
        }
    }

    @Override
    public void flush() throws IOException {
        flushBlock();
        out.flush();
    }

    @Override
    public long getFilePointer() {
        return (long)compressedSizes.size() * (long)blockSize
            + (long)tailBufferPos;
    }
}
