package org.apache.lucene.store;

import ru.yandex.compress.NativeCompress;

public final class LZ4Compressor implements Compressor, ReadingDecompressor {
    private final int acceleration;

    public LZ4Compressor(final int level) {
        if (level <= 1) {
            acceleration = 65537;
        } else if (level >= 9) {
            acceleration = 1;
        } else {
            acceleration = (level - 1) << 13;
        }
    }

    @Override
    public String id() {
        return "lz4";
    }

    @Override
    public int compress(
        final byte[] plain,
        final int plainLen,
        final byte[] compressed,
        final int maxCompressedLen)
    {
        return NativeCompress.lz4CompressByteArray(
            plain,
            plainLen,
            compressed,
            maxCompressedLen,
            acceleration);
    }

    @Override
    public int compress(
        final long plain,
        final int plainLen,
        final long compressed,
        final int maxCompressedLen)
    {
        return NativeCompress.lz4CompressRawArray(
            plain,
            plainLen,
            compressed,
            maxCompressedLen,
            acceleration);
    }

    @Override
    public LZ4Compressor decompressor() {
        return this;
    }

    @Override
    public int decompress(
        final byte[] compressed,
        final int compressedLen,
        final long decompressed,
        final int decompressedLen)
    {
        return NativeCompress.lz4DecompressByteArray(
            compressed,
            compressedLen,
            decompressed,
            decompressedLen);
    }

    @Override
    public int decompress(
        final long compressed,
        final int compressedLen,
        final long decompressed,
        final int decompressedLen)
    {
        return NativeCompress.lz4DecompressBytes(
            compressed,
            compressedLen,
            decompressed,
            decompressedLen);
    }

    @Override
    public int decompress(
        final int fd,
        final long fPos,
        final int compressedLen,
        final long decompressed,
        final int decompressedLen)
    {
        return NativeCompress.lz4ReadAndDecompressByteArray(
            fd,
            fPos,
            compressedLen,
            decompressed,
            decompressedLen);
    }

    @Override
    public String errorDescription(int code) {
        return "LZ4 error code " + code;
    }

    @Override
    public String toString() {
        return id();
    }
}
