package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import ru.yandex.msearch.util.JavaAllocator;

public class MutableBlockCompressedInputStream
    extends BlockCompressedInputStreamBase
{
    private static final JavaAllocator INDEX_CACHE_ALLOCATOR =
        JavaAllocator.get("MutableInputIndexCache");
    private static final int BUFFERED_READER_BUFFER_SIZE = 1024;
    private static final ThreadLocal<BufferedReader> bufferedReaderLocal =
        new ThreadLocal<>();
    private RawCachingInputStream rawInput = null;
    private boolean isNextBlock = false;
    private long currentBlockStart = -1;
    private long nextBlockStart = -1;
    private boolean skipPending = false;

    private int[] indexedDeflatedSize = null;
    private long[] indexedBlockStarts = null;
    private boolean indexed = false;

    private int indexedBlockSize;
    private int indexedBlockCount = 0;
    private int indexedCurrentBlock = -1;
    private int indexedLastBlockSize = 0;
    private long seekPending = -1;

    public MutableBlockCompressedInputStream(
        final IndexInput in,
        final Decompressor decompressor,
        final JavaAllocator allocator,
        final String tag)
    {
        super(in, decompressor, allocator, tag);
        nextFilePointer = in.getFilePointer();
    }

    public MutableBlockCompressedInputStream(
        final IndexInput in,
        final int blockSize,
        final boolean indexed,
        final Decompressor decompressor,
        final JavaAllocator allocator,
        final String tag)
    {
        this(in, decompressor, allocator, tag);
        this.indexed = indexed;
        this.indexedBlockSize = blockSize;
    }

    public void setIndexedMode(final boolean indexedMode) throws IOException {
        this.indexed = indexedMode;
        if (indexed) {
            indexedDeflatedSize = null;
            indexedBlockStarts = null;
            indexedCurrentBlock = -1;
            if (rawInput == null) {
                rawInput = new RawCachingInputStream(
                    otherInput,
                    0,
                    BUFFERED_READER_BUFFER_SIZE,
                    allocator,
                    "MutableIndexFor" + tag);
                rawInput.useCache(useCache());
            }
        }
    }

    public void copyFrom(MutableBlockCompressedInputStream other) throws IOException {
        super.copyFrom(other);
        indexed = other.indexed;
        skipPending = other.skipPending;
        currentBlockStart = other.currentBlockStart;
        nextBlockStart = other.nextBlockStart;
        if (indexed) {
            indexedDeflatedSize = other.indexedDeflatedSize;
            indexedBlockStarts = other.indexedBlockStarts;
            indexedBlockSize = other.indexedBlockSize;
            indexedBlockCount = other.indexedBlockCount;
            indexedCurrentBlock = other.indexedCurrentBlock;
            indexedLastBlockSize = other.indexedLastBlockSize;
        }
        seekPending = other.nextFilePointer;
    }

    private final long getLoadBlockStart() {
//        System.err.println("getLoadBlockStart: " + indexedCurrentBlock
//            + ", : " + otherInput.getCacheKey()
//            + " ,: " + indexedBlockStarts[indexedCurrentBlock]);
        return indexedBlockStarts[indexedCurrentBlock];
    }

    private final int getLoadPlainSize() {
        if (indexedCurrentBlock == indexedBlockCount - 1) {
            return indexedLastBlockSize;
        }
        return indexedBlockSize;
    }

    private final int getLoadDeflatedSize() {
        return indexedDeflatedSize[indexedCurrentBlock];
    }

    private BufferedReader getBufferedReader() {
        BufferedReader reader = bufferedReaderLocal.get();
        if (reader == null) {
            reader = new BufferedReader();
            bufferedReaderLocal.set(reader);
        }
        return reader;
    }

    private void freeBufferedReader(final BufferedReader reader) {
        reader.clear();
    }

    @Override
    protected final void doLoadBlock(final boolean useBuffer) throws IOException {
        if (seekPending != -1) {
            nextFilePointer = seekPending;
            seekPending = -1;
        }
        if (indexed) {
            if (indexedDeflatedSize == null) { // first block
//                System.err.println("LOADING IndexedBlockSize: " + nextFilePointer);
                bufferPos[0] = 0;
                otherInput.seek(nextFilePointer);

                long dataOffset;
                {
                    rawInput.useCache(useCache());
                    rawInput.updateDataOffset(nextFilePointer);
                    rawInput.seek(0);

                    indexedBlockCount = rawInput.readVInt();
                    indexedDeflatedSize = new int[indexedBlockCount];
                    indexedBlockStarts = new long[indexedBlockCount + 1];
//                    System.err.println("indexedBlockCount: " + indexedBlockCount);
                    for (int i = 0; i < indexedBlockCount; i++) {
                        indexedDeflatedSize[i] = rawInput.readVInt();
                        indexedBlockStarts[i + 1] = indexedDeflatedSize[i];
//                        System.err.println("indexedDeflatedSize[i]: " + indexedDeflatedSize[i]);
//                        System.err.println("indexedBlockStarts[i + 1]: " + indexedBlockStarts[i + 1]);
                    }
                    indexedLastBlockSize = rawInput.readVInt();
                    dataOffset = nextFilePointer + rawInput.getFilePointer();
//                    System.err.println(
//                        "MBCI: "
//                        + otherInput.getCacheKey()
//                        + ":" + nextFilePointer
//                        + ", dataOffset: " + dataOffset);
                }
                indexedBlockStarts[0] = dataOffset;
                for (int i = 1; i < indexedBlockCount; i++) {
                    indexedBlockStarts[i] += indexedBlockStarts[i-1];
                }
                if (indexedCurrentBlock == -1) {
                    indexedCurrentBlock++;
                }
            }
            long loadBlockStart = getLoadBlockStart();
            if (currentBlockStart != loadBlockStart) {
                int plainSize = getLoadPlainSize();
                int deflatedSize = getLoadDeflatedSize();
                nextFilePointer = loadBlockStart;
                decompressBlock(otherInput, deflatedSize, plainSize, useBuffer);
                nextBlockStart = nextFilePointer;
                currentBlockStart = loadBlockStart;
            }
            bufferPos[0] = 0;
        } else {
            if (!skipPending) {//TODO: remember skipOffset
                if (currentBlockStart != nextFilePointer) {
                    currentBlockStart = nextFilePointer;
                    decompressBlock(otherInput, useBuffer);
                    nextBlockStart = nextFilePointer;
                    bufferPos[0] = 0;
                } else {
                    skipPending = true;
                }
            } else {
                skipPending = false;
                if (nextBlockStart <= currentBlockStart) {
                    throw new IllegalStateException(
                        "nextBlockStart is invalid: <= currentBlockStart: "
                        + nextBlockStart + " <= " + currentBlockStart);
                }
                nextFilePointer = nextBlockStart;
                doLoadBlock(useBuffer);
            }
        }
        isNextBlock = true;
    }

    @Override
    protected boolean loadNextBlock(boolean useCache) throws IOException {
        if (indexed) {
            indexedCurrentBlock++;
            loadBlock(useCache);
        } else {
            loadBlock(useCache);
        }
        return bufferLength != 0;
    }

    @Override
    public byte[] getNativeBuffer(final int size) {
        return allocator.alloc(size);
    }

    @Override
    public void freeNativeBuffer(final byte[] buffer) {
        allocator.free(buffer);
    }

    @Override
    public void close() throws IOException {
        super.close();
        indexedDeflatedSize = null;
        indexedBlockStarts = null;
    }

    @Override
    public long length() {
        if (indexed) {
            return (long)(indexedBlockCount - 1)
                * (long)indexedBlockSize + (long)indexedLastBlockSize;
        } else {
            return bufferLength;
        }
    }

    public final boolean eof() throws IOException {
        if (bufferPos[0] < bufferLength) {
            return false;
        }
        return loadNextBlock(useCache());
    }

    public final boolean isNextBlock() {
        return isNextBlock;
    }

    public final void clearNextBlock() {
        isNextBlock = false;
    }

    public final long getCurrentBlockFilePointer() {
        return currentBlockStart;
    }

    @Override
    public long getFilePointer() {
        if (indexed) {
            return (long)indexedCurrentBlock
                * (long)indexedBlockSize + (long)bufferPos[0];
        } else {
            return bufferPos[0];
        }
    }

    public BlockCompressedInputStreamBase clone() {
        MutableBlockCompressedInputStream cloned =
            new MutableBlockCompressedInputStream(
                (IndexInput)otherInput.clone(),
                decompressor,
                allocator,
                tag);
        super.clone(cloned);
        cloned.skipPending = skipPending;
        cloned.currentBlockStart = currentBlockStart;
        cloned.nextBlockStart = nextBlockStart;
        cloned.isNextBlock = isNextBlock;
        if (indexed) {
            cloned.indexed = true;
            cloned.indexedBlockCount = indexedBlockCount;
            cloned.indexedBlockSize = indexedBlockSize;
            cloned.indexedDeflatedSize = indexedDeflatedSize;
            cloned.indexedBlockStarts = indexedBlockStarts;
            cloned.indexedCurrentBlock = indexedCurrentBlock;
            cloned.indexedLastBlockSize = indexedLastBlockSize;
        }
        return cloned;
    }

    public final void seek(final long blockStart,
        final long blockOffset)
        throws IOException
    {
        nextFilePointer = blockStart;
        skipPending = false;
        seekPending = -1;
        loadBlock(useCache());
        if (bufferLength != 0) {
            seek(blockOffset);
        }
    }

    @Override
    public final void seek(final long pos)
        throws IOException
    {
        if (indexed) {
            if (pos > length()) {
                throw new IOException("Seek behind the end of file: pos="
                    + pos + ", length=" + length()
                    + ", file=" + otherInput.getCacheKey());
            }
            int newBlock = (int)((long)pos / (long)indexedBlockSize);
            if (newBlock >= indexedBlockCount) {
                System.err.println( "NewBlock > indexedBlockCount: pos=" + pos
                    + ", length=" + length() + ", newBlock=" + newBlock
                    + ", blockcount=" + indexedBlockCount);
            }
            if (indexedCurrentBlock != newBlock) {
                indexedCurrentBlock = newBlock;
                if (pos < length()) {
                    loadBlock(useCache());
                }
            }
            bufferPos[0] = (int)(pos % indexedBlockSize);
        } else {
            if (bufferPos[0] == 0 && bufferLength == 0) {
                loadBlock(useCache());
            }
            if (pos > bufferLength) {
                throw new IOException("Seek behind the end of buffer: pos="
                    + pos + ", length=" + length());
            }
            bufferPos[0] = (int)pos;
        }
    }

    public boolean equals(Object _other) {
        return false;
    }

    public Object getCacheKey() {
        return this;
    }

    @Override
    public void useCache(final boolean mayUseCache) {
        super.useCache(mayUseCache);
        if (rawInput != null) {
            rawInput.useCache(mayUseCache);
        }
    }
}
