package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;

import ru.yandex.unsafe.NativeMemory2;
import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;

public class NativeMemoryIndexOutput extends IndexOutput {
    private static final NativeMemoryAllocator allocator =
        NativeMemoryAllocator.get("NativeMemoryIndexOutput");
    private static final NativeMemoryAllocator tempAllocator =
        NativeMemoryAllocator.get("CompressorTemp");

    private NativeMemory2 tailBuffer;
    private NativeMemory2 tempOutputBuffer = null;
    private int tailBufferLength;
    private int tailBufferPos;

    private final int blockSize;

    public NativeMemoryIndexOutput(final int blockSize) {
        this.blockSize = blockSize;
        tailBuffer = allocator.alloc(blockSize);
        tailBufferLength = blockSize;
        tailBufferPos = 0;
    }

    @Override
    public void close() throws IOException {
        freeNativeBuffers();
    }

    private void freeNativeBuffers() {
        if (tailBuffer != null) {
            tailBuffer.free();
            tailBuffer = null;
            tailBufferPos = 0;
            tailBufferLength = 0;
        }
        if (tempOutputBuffer != null) {
            tempOutputBuffer.free();
            tempOutputBuffer = null;
        }
    }

    public void finalize() {
        freeNativeBuffers();
    }

    @Override
    public void seek(long pos) throws IOException {
        throw new UnsupportedOperationException("seek");
    }

    @Override
    public long length() {
        return (long) tailBufferPos;
    }


    @Override
    public void writeByte(byte b) throws IOException {
        if (tailBufferPos == tailBufferLength) {
            growBuffer();
        }
        tailBuffer.setByte(tailBufferPos++, b);
    }

    @Override
    public void writeBytes(
        final byte[] b,
        final int offset,
        final int len)
        throws IOException
    {
        int remains = len;
        int pos = offset;
        while (remains > 0) {
            if (tailBufferPos == tailBufferLength) {
                growBuffer();
            }
            int bufferFree = tailBufferLength - tailBufferPos;
            int toWrite = Math.min(bufferFree, remains);
            tailBuffer.writeSlow(tailBufferPos, b, pos, toWrite);
            remains -= toWrite;
            pos += toWrite;
            tailBufferPos += toWrite;
        }
    }

    @Override
    public void writeBytes(
        final NativeMemory2 b,
        final int offset,
        final int len)
        throws IOException
    {
        int remains = len;
        int pos = offset;
        while (remains > 0) {
            if (tailBufferPos == tailBufferLength) {
                growBuffer();
            }
            int bufferFree = tailBufferLength - tailBufferPos;
            int toWrite = Math.min(bufferFree, remains);
            tailBuffer.write(tailBufferPos, b, pos, toWrite);
            remains -= toWrite;
            pos += toWrite;
            tailBufferPos += toWrite;
        }
    }

    private void growBuffer() {
        NativeMemory2 newBuffer = allocator.alloc(tailBufferLength + blockSize);
        newBuffer.write(0, tailBuffer, 0, tailBufferLength);
        tailBuffer.free();
        tailBuffer = newBuffer;
        tailBufferLength = newBuffer.size();
    }

    @Override
    public void flush() throws IOException {
    }

    @Override
    public long getFilePointer() {
        return (long) tailBufferPos;
    }

    public NativeMemory2 buffer() {
        return tailBuffer;
    }

    private int compress(
        final Compressor compressor,
        final NativeMemory2 input,
        final int inputLen,
        final NativeMemory2 output,
        final int outputLen)
    {
        return compressor.compress(
            input.address(),
            inputLen,
            output.address(),
            outputLen);
    }

    public int compressBlock(
        final Compressor compressor,
        final IndexOutput out,
        final boolean writeHeader)
        throws IOException
    {
        int outputLen = tailBufferLength << 1;
        if (tempOutputBuffer == null) {
            tempOutputBuffer = tempAllocator.alloc(outputLen);
        } else if (tempOutputBuffer.size() < outputLen) {
            tempOutputBuffer.free();
            tempOutputBuffer = tempAllocator.alloc(outputLen);
        }
//        System.err.println("Compress: " + tailBufferPos);
        int compressed = compress(
            compressor,
            tailBuffer,
            tailBufferPos,
            tempOutputBuffer,
            outputLen);
        if (compressed < 0) {
            throw new IOException(
                "Compressor error: ret = " + compressed
                + ", " + compressor.errorDescription(compressed)
                + ", outputLen: " + outputLen
                + ", tempOutputBuffer.size(): " + tempOutputBuffer.size()
                + ", tailBuffer.size(): " + tailBuffer.size()
                + ", tailBufferLength: " + tailBufferLength
                + ", tailBufferPos: " + tailBufferPos);
        }
        if (writeHeader) {
            out.writeVInt(compressed);
            out.writeVInt(tailBufferPos);
        }
        out.writeBytes(tempOutputBuffer, 0, compressed);
        return compressed;
    }

    public void reset() {
        tailBufferPos = 0;
    }
}
