package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.util.packed.PackedInts;

import ru.yandex.msearch.util.JavaAllocator;

public class PackedIndexedCompressedInputStream
    extends BlockCompressedInputStreamBase
{
    private static final JavaAllocator PACKED_ALLOCATOR =
        JavaAllocator.get("PackedBlockIndex");
    private int indexedBlockSize;
    private int indexedLastBlockSize;
    private int indexedCurrentBlock = -1;
    private PackedInts.Reader index;
    private int indexedBlockCount;
    private long length;

    public PackedIndexedCompressedInputStream(
        final IndexInput in,
        final Decompressor decompressor,
        final JavaAllocator allocator,
        final String tag)
        throws IOException
    {
        super(in, decompressor, allocator, tag);
        long indexOffset = in.readLong();
        in.seek(indexOffset);

        this.indexedBlockSize = in.readVInt();
        this.indexedLastBlockSize = in.readVInt();
        index = PackedInts.getReader(in, PACKED_ALLOCATOR);
        indexedBlockCount = index.size() - 1;
        length = (long)(indexedBlockCount - 1) * (long)indexedBlockSize
            + (long)indexedLastBlockSize;
    }

    private PackedIndexedCompressedInputStream(
        final IndexInput in,
        final Decompressor decompressor,
        final JavaAllocator allocator,
        final String tag,
        final boolean unusedForClone)
    {
        super(in, decompressor, allocator, tag);
    }

    @Override
    public byte[] getNativeBuffer(final int size) {
        return allocator.alloc(size);
    }

    @Override
    public void freeNativeBuffer(final byte[] buffer) {
        allocator.free(buffer);
    }

    private final long getLoadBlockStart() {
        return index.get(indexedCurrentBlock);
    }

    private final int getLoadPlainSize() {
        if (indexedCurrentBlock == indexedBlockCount - 1) {
            return indexedLastBlockSize;
        }
        return indexedBlockSize;
    }

    private final int getLoadDeflatedSize() {
        int size;
        size = (int)(index.get(indexedCurrentBlock + 1)
            - index.get(indexedCurrentBlock));
        if (size < 0) {
            System.err.println("getLoadDeflatedSize: size=" + size
                + " / index[i+1]="
                + index.get(indexedCurrentBlock + 1) + " - index[i]="
                + index.get(indexedCurrentBlock) + " / i=" + indexedCurrentBlock
                + " / index.size()=" + index.size());
        }
        return size;
    }

    @Override
    protected void doLoadBlock(
        final boolean useBuffer)
        throws IOException
    {
        if (indexedCurrentBlock >= indexedBlockCount) {
            throw new IOException("Read behind the end of file <"
                + otherInput.getCacheKey() + ">: "
                + "pos=" + getFilePointer() + ", length=" + length);
        }
        long loadBlockStart = getLoadBlockStart();
        int plainSize = getLoadPlainSize();
        int deflatedSize = getLoadDeflatedSize();
        nextFilePointer = loadBlockStart;
        decompressBlock(otherInput, deflatedSize, plainSize, useBuffer);
        bufferPos[0] = 0;
    }

    protected final boolean loadNextBlock(final boolean useBuffer) throws IOException {
        indexedCurrentBlock++;
        loadBlock(useBuffer);
        return true;
    }

    @Override
    public void close() throws IOException {
        super.close();
        if (index != null && !cloned) {
            index.close();
        }
        index = null;
    }

    @Override
    public long length() {
        return length;
    }

    @Override
    public long getFilePointer() {
        final long pos = (long)indexedCurrentBlock * (long)indexedBlockSize
            + (long)bufferPos[0];
        return pos;
    }

    public PackedIndexedCompressedInputStream clone() {
        PackedIndexedCompressedInputStream cloned =
            new PackedIndexedCompressedInputStream(
                (IndexInput)otherInput.clone(),
                decompressor,
                allocator,
                tag,
                true);
        super.clone(cloned);
        cloned.length = length;
        cloned.indexedBlockCount = indexedBlockCount;
        cloned.indexedBlockSize = indexedBlockSize;
        cloned.indexedCurrentBlock = indexedCurrentBlock;
        cloned.indexedLastBlockSize = indexedLastBlockSize;
        cloned.index = index;
        return cloned;
    }

    public final void seekToBlock(final int block) throws IOException {
        if (block < 0 || block >= indexedBlockCount) {
            throw new IOException("Block <" + block + "> number is "
                + "out of bounds: 0 < " + indexedBlockCount);
        }
        if (block != indexedCurrentBlock) {
            indexedCurrentBlock = block;
            loadBlock(useCache());
        }
        bufferPos[0] = 0;
    }

    public final void seekBlock(final int offset) throws IOException {
        if (offset < 0 || offset >= bufferLength) {
            throw new IOException("seek offset <" + offset + "> is "
                + "out of bounds: 0 < " + bufferLength);
        }
        bufferPos[0] = offset;
    }

    @Override
    public final void seek(long pos) throws IOException {
        if (pos > length()) {
            throw new IOException("Seek behind the end of file: pos=" + pos +
                ", length=" + length + ", file=" + otherInput.getCacheKey());
        }
        int newBlock = (int)((long)pos / (long)indexedBlockSize);
        int offset = (int)(pos % indexedBlockSize);
        if (newBlock == indexedBlockCount && offset == 0) {
            newBlock--;
            offset = indexedLastBlockSize;
            //seek to the end of file
            //set position to the end of buffer
            //this will issue IOException on read
        }
        if (indexedCurrentBlock != newBlock || currentOutputBuffer == null) {
            indexedCurrentBlock = newBlock;
            loadBlock(useCache());
        }
        bufferPos[0] = offset;
    }

    public boolean equals(Object _other) {
        return false;
    }

    public Object getCacheKey() {
        return this;
    }
}
