package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.util.packed.PackedInts;

import ru.yandex.msearch.util.JavaAllocator;

public class RawCachingInputStream
    extends BlockCompressedInputStreamBase
{
    private static final Decompressor NULL_DECOMPRESSOR =
        new NullDecompressor();

    private final int indexedBlockSize;
    private int indexedLastBlockSize;
    private long indexedCurrentBlock = -1;
    private long dataOffset = 0;
    private int indexedBlockCount;
    private long length;

    public RawCachingInputStream(
        final IndexInput in,
        final int blockSize,
        final JavaAllocator allocator,
        final String tag)
        throws IOException
    {
        this(in, in.getFilePointer(), blockSize, allocator, tag);
    }

    public RawCachingInputStream(
        final IndexInput in,
        final long offset,
        final int blockSize,
        final JavaAllocator allocator,
        final String tag)
        throws IOException
    {
        super(in, NULL_DECOMPRESSOR, allocator, tag);

        this.indexedBlockSize = blockSize;

        dataOffset = 0;
        length = in.length();

        updateDataOffset(offset);
    }

    private RawCachingInputStream(final RawCachingInputStream other) {
        super(
            (IndexInput) other.otherInput.clone(),
            NULL_DECOMPRESSOR,
            other.allocator,
            other.tag);
        dataOffset = other.dataOffset;
        length = other.length;
        indexedBlockSize = other.indexedBlockSize;
        indexedLastBlockSize = other.indexedLastBlockSize;
        indexedBlockCount = other.indexedBlockCount;
    }

    @Override
    public byte[] getNativeBuffer(final int size) {
        return allocator.alloc(size);
    }

    @Override
    public void freeNativeBuffer(final byte[] buffer) {
        allocator.free(buffer);
    }

    private final long getLoadBlockStart() {
        return indexedCurrentBlock * indexedBlockSize + dataOffset;
    }

    private final int getLoadPlainSize() {
        if (indexedCurrentBlock == indexedBlockCount - 1) {
            return indexedLastBlockSize;
        }
        return indexedBlockSize;
    }

    private final int getLoadDeflatedSize() {
        return getLoadPlainSize();
    }

    @Override
    protected void doLoadBlock(
        final boolean useBuffer)
        throws IOException
    {
        if (indexedCurrentBlock >= indexedBlockCount) {
            throw new IOException("Read behind the end of file <"
                + otherInput.getCacheKey() + ">: "
                + "pos=" + getFilePointer() + ", length=" + length);
        }
        long loadBlockStart = getLoadBlockStart();
        int plainSize = getLoadPlainSize();
        int deflatedSize = getLoadDeflatedSize();
        nextFilePointer = loadBlockStart;
        decompressBlock(otherInput, deflatedSize, plainSize, useBuffer);
        bufferPos[0] = 0;
    }

    protected final boolean loadNextBlock(final boolean useBuffer) throws IOException {
        indexedCurrentBlock++;
        loadBlock(useBuffer);
        return true;
    }

    @Override
    public void close() throws IOException {
        super.close();
    }

    @Override
    public long length() {
        return length;
    }

    @Override
    public long getFilePointer() {
        final long pos = (long)indexedCurrentBlock * (long)indexedBlockSize
            + (long)bufferPos[0];
//        System.err.println("RawCachingInputStream.getFilePointer: " + pos);
        return pos;
    }

    public RawCachingInputStream clone() {
        RawCachingInputStream cloned =
            new RawCachingInputStream(this);
        super.clone(cloned);
        return cloned;
    }

    public final void seekToBlock(final int block) throws IOException {
        if (block < 0 || block >= indexedBlockCount) {
            throw new IOException("Block <" + block + "> number is "
                + "out of bounds: 0 < " + indexedBlockCount);
        }
        if (block != indexedCurrentBlock) {
            indexedCurrentBlock = block;
            loadBlock(useCache());
        }
        bufferPos[0] = 0;
    }

    public final void seekBlock(final int offset) throws IOException {
        if (offset < 0 || offset >= bufferLength) {
            throw new IOException("seek offset <" + offset + "> is "
                + "out of bounds: 0 < " + bufferLength);
        }
        bufferPos[0] = offset;
    }

    public final void updateDataOffset(final long offset) throws IOException {
        long diff = offset - dataOffset;
        dataOffset = offset;
        length -= diff;

        int lastBlockSize = (int) (length % indexedBlockSize);
        int prevBlockCount = this.indexedBlockCount;
        if (lastBlockSize == 0) {
            indexedLastBlockSize = indexedBlockSize;
            indexedBlockCount = (int) (length / indexedBlockSize);
        } else {
            indexedLastBlockSize = lastBlockSize;
            indexedBlockCount = (int) (length / indexedBlockSize  + 1);
        }

        if (prevBlockCount > 0 && prevBlockCount != this.indexedBlockCount) {
            loadBlock(useCache());
        }
    }

    @Override
    public final void seek(long pos) throws IOException {
        if (pos > length()) {
            throw new IOException("Seek behind the end of file: pos=" + pos +
                ", length=" + length + ", file=" + otherInput.getCacheKey());
        }
        int newBlock = (int)((long)pos / (long)indexedBlockSize);
        int offset = (int)(pos % indexedBlockSize);
        if (newBlock == indexedBlockCount && offset == 0) {
            newBlock--;
            offset = indexedLastBlockSize;
            //seek to the end of file
            //set position to the end of buffer
            //this will issue IOException on read
        }
        if (indexedCurrentBlock != newBlock || currentOutputBuffer == null) {
            indexedCurrentBlock = newBlock;
            loadBlock(useCache());
        }
        bufferPos[0] = offset;
    }

    public boolean equals(Object _other) {
        return false;
    }

    public Object getCacheKey() {
        return this;
    }
}
