package org.apache.lucene.util;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.Closeable;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

/** Java's builtin ThreadLocal has a serious flaw:
 *  it can take an arbitrarily long amount of time to
 *  dereference the things you had stored in it, even once the
 *  ThreadLocal instance itself is no longer referenced.
 *  This is because there is single, master map stored for
 *  each thread, which all ThreadLocals share, and that
 *  master map only periodically purges "stale" entries.
 *
 *  While not technically a memory leak, because eventually
 *  the memory will be reclaimed, it can take a long time
 *  and you can easily hit OutOfMemoryError because from the
 *  GC's standpoint the stale entries are not reclaimable.
 * 
 *  This class works around that, by only enrolling
 *  WeakReference values into the ThreadLocal, and
 *  separately holding a hard reference to each stored
 *  value.  When you call {@link #close}, these hard
 *  references are cleared and then GC is freely able to
 *  reclaim space by objects stored in it.
 *
 *  We can not rely on {@link ThreadLocal#remove()} as it
 *  only removes the value for the caller thread, whereas
 *  {@link #close} takes care of all
 *  threads.  You should not call {@link #close} until all
 *  threads are done using the instance.
 *
 * @lucene.internal
 */

import java.io.Closeable;
import java.io.IOException;

public class ThreadLocalLikeStorage<T> implements Closeable {
  private ConcurrentLinkedQueue<T> localCache =
    new ConcurrentLinkedQueue<T>();
  private final AtomicInteger cachedCount = new AtomicInteger(0);
  private final int maxCached;

  public ThreadLocalLikeStorage(final int maxCached) {
    this.maxCached = maxCached;
  }

  protected T initialValue() {
    return null;
  }

  public T get() {
    T t = localCache.poll();
    if (t != null) {
        cachedCount.decrementAndGet();
    } else {
        t = initialValue();
    }
    return t;
  }

  public void free(T object) {
    localCache.add(object);
    if (cachedCount.incrementAndGet() == maxCached) {
        T evicted = localCache.poll();
        cachedCount.decrementAndGet();
        if (evicted instanceof Closeable) {
            try {
                ((Closeable)evicted).close();
            } catch (IOException ign) {
                ign.printStackTrace();
            }
        }
    }
  }

  public void close() {
    for (T t : localCache) {
        if (t instanceof Closeable) {
            try {
                ((Closeable)t).close();
            } catch (IOException ign) {
                ign.printStackTrace();
            }
        }
    }
    localCache = null;
  }
}
