package org.apache.lucene.util.automaton;

//import java.lang.IllegalArgumentException;
import java.util.Arrays;
import java.util.ConcurrentModificationException;
import java.util.NoSuchElementException;
import java.util.TreeMap;

public class TriMap {

    class PrivateEntryIterator {
        int next;
        int lastReturned;
        int expectedModCount;

        public PrivateEntryIterator(int first) {
            expectedModCount = modCount;
            lastReturned = NULL;
            next = first;
        }

        public final boolean hasNext() {
            return next != NULL;
        }

        final int nextEntry() {
            int e = next;
            if (e == NULL) {
                throw new NoSuchElementException();
            }
            if (modCount != expectedModCount) {
                throw new ConcurrentModificationException();
            }
            next = successor(e);
//            System.err.println("Successor: " + e + "=" + next);
            lastReturned = e;
            return e;
        }

        final int prevEntry() {
            int e = next;
            if (e == NULL) {
                throw new NoSuchElementException();
            }
            if (modCount != expectedModCount) {
                throw new ConcurrentModificationException();
            }
            next = predecessor(e);
            lastReturned = e;
            return e;
        }
    }

    final class EntryIterator extends PrivateEntryIterator {
        EntryIterator(int first) {
            super(first);
        }

        public int next() {
            return nextEntry();
        }
    }

    final class KeyIterator extends PrivateEntryIterator {
        public KeyIterator(int first) {
            super(first);
        }

        public int next() {
            return keys[nextEntry()];
        }
    }

    public TriMap() {
    }

    private static final boolean BLACK = false;
    private static final boolean RED = true;
    private static final int NULL = -1;
    public static final long NOT_EXISTS = Long.MIN_VALUE;

    private static final int START_SIZE = 32;

    private int root = NULL;
    private int[] keys;
    private int[] values;
    private int[] left;
    private int[] right;
    private int[] parent;
    private boolean[] color;
    private int size = 0;
    private int modCount = 0;
    private int lastIdx = 0;

    public KeyIterator keyIterator() {
        return new KeyIterator(getFirstEntry());
    }

    private final int getFirstEntry() {
        int p = root;
        if (p != NULL) {
            while (left[p] != NULL) {
                p = left[p];
            }
        }
        return p;
    }

    public long get(int key) {
        int p = getEntry(key);
        if (p == NULL) {
            return NOT_EXISTS;
        }
        return values[p];
    }

    public long remove(int key) {
        int p = getEntry(key);
        if (p == NULL) {
            return NOT_EXISTS;
        }
        long oldValue = values[p];
        deleteEntry(p);
        return oldValue;
    }

    public int size() {
        return size;
    }

    public void clear() {
        modCount++;
        size = 0;
        root = NULL;
        clearArrays();
    }

    public int getEntry(final int key) {
        int p = root;
//        System.err.println("getEntry: " + key);
        int i = 0;
        while (p != NULL) {
            i++;
            if (key < keys[p]) {
                p = left[p];
            } else if (key > keys[p]) {
                p = right[p];
            } else {
//                System.err.println("finish: " + i);
                return p;
            }
        }
        return NULL;
    }

    private int predecessor(int t) {
        if (t == NULL) {
            return NULL;
        } else if (left[t] != NULL) {
            int p = left[t];
            while (right[p] != NULL) {
                p = right[p];
            }
            return p;
        } else {
            int p = parent[t];
            int ch = t;
            while (p != NULL && ch == left[p]) {
                ch = p;
                p = parent[p];
            }
            return p;
        }
    }

    private int successor(int t) {
        if (t == NULL) {
            return NULL;
        } else if (right[t] != NULL) {
            int p = right[t];
            while(left[p] != NULL) {
                p = left[p];
            }
            return p;
        } else {
            int p = parent[t];
            int ch = t;
            while (p != NULL && ch == right[p]) {
                ch = p;
                p = parent[p];
            }
            return p;
        }
    }

    private void allocateArrays(final int size) {
//        System.err.println("allocate: " + size);
        if (keys == null) keys = new int[size];
        else keys = Arrays.copyOf(keys, size);
        if (values == null) values = new int[size];
        else values = Arrays.copyOf(values, size);
        if (left == null) left = new int[size];
        else left = Arrays.copyOf(left, size);
        if (right == null) right = new int[size];
        else right = Arrays.copyOf(right, size);
        if (parent == null) parent = new int[size];
        else parent = Arrays.copyOf(parent, size);
        if (color == null) color = new boolean[size];
        else color = Arrays.copyOf(color, size);
    }

    private void clearArrays() {
        keys = null;
        values = null;
        left = null;
        right = null;
        parent = null;
        color = null;
    }

    private void shrinkArrays(int toSize) {
        if (toSize < START_SIZE) {
            return;
        }
        allocateArrays(toSize);
    }

    private int addEntry(final int key, final int value, final int parent) {
        if (keys == null) {
            allocateArrays(START_SIZE);
        } else if (lastIdx >= keys.length) {
            allocateArrays(keys.length * 2);
        }
        keys[lastIdx] = key;
        values[lastIdx] = value;
        left[lastIdx] = NULL;
        right[lastIdx] = NULL;
        this.parent[lastIdx] = parent;
        color[lastIdx] = BLACK;
        return lastIdx++;
    }

    public long put(int key, int value) {
        int t = root;
        if (t == NULL) {
            root = addEntry(key, value, NULL);
            size = 1;
            modCount++;
            return NOT_EXISTS;
        }
        int cmp;
        int parent;
        do {
            parent = t;
            if (key < keys[t]) {
                cmp = -1;
                t = left[t];
            } else if (key > keys[t]){
                cmp = 1;
                t = right[t];
            } else {
                int ret = values[t];
                values[t] = value;
                return ret;
            }
        } while(t != NULL);

        int e = addEntry(key, value, parent);
        if (cmp < 0) {
            left[parent] = e;
        } else {
            right[parent] = e;
        }
        fixAfterInsertion(e);
        size++;
        modCount++;
        return NOT_EXISTS;
    }

    public void fixAfterInsertion(int x) {
        color[x] = RED;

//        System.err.println("Fix start");
        while(x != NULL && x != root && color[parent[x]] == RED) {
//            System.err.println("x=" + x);
            if (parentOf(x) == leftOf(parentOf(parentOf(x)))) {
                int y = rightOf(parentOf(parentOf(x)));
                if (colorOf(y) == RED) {
                    setColor(parentOf(x), BLACK);
                    setColor(y, BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    x = parentOf(parentOf(x));
                } else {
                    if (x == rightOf(parentOf(x))) {
                        x = parentOf(x);
                        rotateLeft(x);
                    }
                    setColor(parentOf(x), BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    rotateRight(parentOf(parentOf(x)));
                }
            } else {
                int y = leftOf(parentOf(parentOf(x)));
                if (colorOf(y) == RED) {
                    setColor(parentOf(x), BLACK);
                    setColor(y, BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    x = parentOf(parentOf(x));
                } else {
                    if (x == leftOf(parentOf(x))) {
                        x = parentOf(x);
                        rotateRight(x);
                    }
                    setColor(parentOf(x), BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    rotateLeft(parentOf(parentOf(x)));
                }
            }
        }
//        System.err.println("Fix end");
        color[root] = BLACK;

    }

    private void deleteEntry(int p) {
        modCount++;
        size--;

        // If strictly internal, copy successor's element to p and then make p
        // point to successor.
        if (left[p] != NULL && right[p] != NULL) {
            int s = successor(p);
            keys[p] = keys[s];
            values[p] = values[s];
            p = s;
        } // p has 2 children

        // Start fixup at replacement node, if it exists.
        int replacement = (left[p] != NULL ? left[p] : right[p]);

        if (replacement != NULL) {
            // Link replacement to parent
            parent[replacement] = parent[p];
            if (parent[p] == NULL)
                root = replacement;
            else if (p == left[parent[p]])
                left[parent[p]]  = replacement;
            else
                right[parent[p]] = replacement;

            // Null out links so they are OK to use by fixAfterDeletion.
            left[p] = right[p] = parent[p] = NULL;

            // Fix replacement
            if (color[p] == BLACK)
                fixAfterDeletion(replacement);
            lastIdx--;
            moveEntry(lastIdx, p);
            if (lastIdx < keys.length / 4) {
                shrinkArrays(keys.length / 2);
            }
        } else if (parent[p] == NULL) { // return if we are the only node.
            root = NULL;
            lastIdx = 0;
            shrinkArrays(START_SIZE);
        } else { //  No children. Use self as phantom replacement and unlink.
            if (color[p] == BLACK)
                fixAfterDeletion(p);

            if (parent[p] != NULL) {
                if (p == left[parent[p]])
                    left[parent[p]] = NULL;
                else if (p == right[parent[p]])
                    right[parent[p]] = NULL;
                parent[p] = NULL;
            }
            lastIdx--;
            moveEntry(lastIdx, p);
            if (lastIdx < keys.length / 4) {
                shrinkArrays(keys.length / 2);
            }
        }
    }


    private void moveEntry(int from, int to) {
        if (to == from || to < 0) {
            return;
        }
        int p = parent[from];
        if (p != NULL) {
            if (left[p] == from) {
                left[p] = to;
            } else if (right[p] == from) {
                right[p] = to;
            } else {
                throw new IllegalStateException("Parent has no childs: " + p);
            }
        }
        keys[to] = keys[from];
        values[to] = values[from];
        left[to] = left[from];
        right[to] = right[from];
        parent[to] = p;
        color[to] = color[from];
        if (left[to] != NULL) {
            parent[left[to]] = to;
        }
        if (right[to] != NULL) {
            parent[right[to]] = to;
        }
        if (from == root) {
            root = to;
        }
    }

    /** From CLR */
    private void fixAfterDeletion(int x) {
//        System.err.println("fixAfterDeletion: size=" + size);
        int i = 0;
        while (x != root && colorOf(x) == BLACK) {
            i++;
            if (x == leftOf(parentOf(x))) {
                int sib = rightOf(parentOf(x));

                if (colorOf(sib) == RED) {
                    setColor(sib, BLACK);
                    setColor(parentOf(x), RED);
                    rotateLeft(parentOf(x));
                    sib = rightOf(parentOf(x));
                }

                if (colorOf(leftOf(sib))  == BLACK &&
                    colorOf(rightOf(sib)) == BLACK) {
                    setColor(sib, RED);
                    x = parentOf(x);
                } else {
                    if (colorOf(rightOf(sib)) == BLACK) {
                        setColor(leftOf(sib), BLACK);
                        setColor(sib, RED);
                        rotateRight(sib);
                        sib = rightOf(parentOf(x));
                    }
                    setColor(sib, colorOf(parentOf(x)));
                    setColor(parentOf(x), BLACK);
                    setColor(rightOf(sib), BLACK);
                    rotateLeft(parentOf(x));
                    x = root;
                }
            } else { // symmetric
                int sib = leftOf(parentOf(x));

                if (colorOf(sib) == RED) {
                    setColor(sib, BLACK);
                    setColor(parentOf(x), RED);
                    rotateRight(parentOf(x));
                    sib = leftOf(parentOf(x));
                }

                if (colorOf(rightOf(sib)) == BLACK &&
                    colorOf(leftOf(sib)) == BLACK) {
                    setColor(sib, RED);
                    x = parentOf(x);
                } else {
                    if (colorOf(leftOf(sib)) == BLACK) {
                        setColor(rightOf(sib), BLACK);
                        setColor(sib, RED);
                        rotateLeft(sib);
                        sib = leftOf(parentOf(x));
                    }
                    setColor(sib, colorOf(parentOf(x)));
                    setColor(parentOf(x), BLACK);
                    setColor(leftOf(sib), BLACK);
                    rotateRight(parentOf(x));
                    x = root;
                }
            }
        }
//        System.err.println("Fix iters: " + i);

        setColor(x, BLACK);
    }

    private boolean colorOf(int p) {
        if (p == NULL) {
            return BLACK;
        }
        return color[p];
    }

    private int parentOf(int p) {
        if (p == NULL) {
            return NULL;
        }
        return parent[p];
    }

    private void setColor(int p, boolean c) {
        if (p != NULL) {
            color[p] = c;
        }
    }

    private int leftOf(int p) {
        if (p == NULL) {
            return NULL;
        }
        return left[p];
    }

    private int rightOf(int p) {
        if (p == NULL) {
            return NULL;
        }
        return right[p];
    }

    private void rotateLeft(int p) {
//        System.err.println("rotateLeft");
        if (p != NULL) {
            int r = right[p];
            right[p] = left[r];
            if (left[r] != NULL) {
                parent[left[r]] = p;
            }
            parent[r] = parent[p];
            if (parent[p] == NULL) {
                root = r;
            } else if (left[parent[p]] == p) {
                left[parent[p]] = r;
            } else {
                right[parent[p]] = r;
            }
            left[r] = p;
            parent[p] = r;
        }
    }

    private void rotateRight(int p) {
//        System.err.println("rotateRight");
        if (p != NULL) {
            int l = left[p];
            left[p] = right[l];
            if (right[l] != NULL) {
                parent[right[l]] = p;
            }
            parent[l] = parent[p];
            if (parent[p] == NULL) {
                root = l;
            } else if (right[parent[p]] == p) {
                right[parent[p]] = l;
            } else {
                left[parent[p]] = l;
            }
            right[l] = p;
            parent[p] = l;
        }
    }

    public static void main(String[] args) {
//        System.err.println("put1");
//        m.put(10,20);
//        System.err.println("put2");
//        m.put(5,30);
//        System.err.println("put3");
//        m.put(15,55);
//        System.err.println("put end");
//        System.err.println("5=" + m.get(5) + ", 10=" + m.get(10) + ", 15=" + m.get(15));
//        Runtime runtime = Runtime.getRuntime();
        for(int t = 0; t < 20; t++) {
            TriMap m = new TriMap();
            TreeMap<Integer,Integer> tm = new TreeMap<Integer,Integer>();
            long start = System.currentTimeMillis();
            for (int i = 3000000; i >= 0; i--) {
                m.put(i, i);
            }
            long time = System.currentTimeMillis() - start;
            System.out.println("Tri add time=" + time);
            start = System.currentTimeMillis();
            for (int i = 3000000; i >= 0; i--) {
//                System.err.println("removed: " + m.remove(i) + ", i=" + i);
                long rem = m.remove((int)(Math.random()*3000000));
//                if (rem != i) {
//                    throw new IllegalStateException("m.remove(i) != i : " + rem + " <> " + i);
//                }
//                m.get(i);
            }
            time = System.currentTimeMillis() - start;
            System.out.println("Tri remove time=" + time);

            start = System.currentTimeMillis();
            for (int i = 3000000; i >= 0; i--) {
                tm.put(i, i);
            }
            time = System.currentTimeMillis() - start;
            System.out.println("Tree add time=" + time);
            start = System.currentTimeMillis();
            for (int i = 3000000; i >= 0; i--) {
                tm.remove(i);
            }
            time = System.currentTimeMillis() - start;
            System.out.println("Tree remove time=" + time);
        }
/*
        KeyIterator ki = m.keyIterator();
        int i = 0;
        while (ki.hasNext()) {
            int n = ki.next();
//            System.err.println("next: " + n);
            if (n != i) {
                System.err.println("missmatch: " + n + " <> " + i);
            }
            i++;
        }
*/
    }

}