package org.apache.lucene.util.packed_ondisk;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.Closeable;

import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.CodecUtil;
import org.apache.lucene.util.Constants;

import java.io.IOException;

/**
 * Simplistic compression for array of unsigned long values.
 * Each value is >= 0 and <= a specified maximum value.  The
 * values are stored as packed ints, with each value
 * consuming a fixed number of bits.
 *
 * @lucene.internal
 */

public class PackedInts {

  private final static String CODEC_NAME = "PackedInts";
  private final static int VERSION_START = 0;
  private final static int VERSION_CURRENT = VERSION_START;

  /**
   * A read-only random access array of positive integers.
   * @lucene.internal
   */
  public static interface Reader {
    /**
     * @param index the position of the wanted value.
     * @return the value at the stated index.
     */
    long get(int index) throws IOException;

    /**
     * @return the number of bits used to store any given value.
     *         Note: This does not imply that memory usage is
     *         {@code bitsPerValue * #values} as implementations are free to
     *         use non-space-optimal packing of bits.
     */
    int getBitsPerValue();

    /**
     * @return the number of values.
     */
    int size();

    void close() throws IOException;
    ReaderImpl clone();
  }

  /**
   * Run-once iterator interface, to decode previously saved PackedInts.
   */
  public static interface ReaderIterator extends Closeable {
    /** Returns next value */
    long next() throws IOException;
    /** Returns number of bits per value */
    int getBitsPerValue();
    /** Returns number of values */
    int size();
    /** Returns the current position */
    int ord();
    /** Skips to the given ordinal and returns its value.
     * @return the value at the given position
     * @throws IOException if reading the value throws an IOException*/
    long advance(int ord) throws IOException;
  }

  /**
   * A simple base for Readers that keeps track of valueCount and bitsPerValue.
   * @lucene.internal
   */
  public static abstract class ReaderImpl implements Reader, Cloneable {
    protected final int bitsPerValue;
    protected final int valueCount;
    private IndexInput values = null;
    private final long startPos;

    protected ReaderImpl(final IndexInput in, int valueCount, int bitsPerValue) {
      this.values = in;
      this.startPos = in.getFilePointer();
      this.bitsPerValue = bitsPerValue;
      assert bitsPerValue > 0 && bitsPerValue <= 64 : "bitsPerValue=" + bitsPerValue;
      this.valueCount = valueCount;
    }

    @Override
    public ReaderImpl clone() {
        try {
            ReaderImpl clone = (ReaderImpl)super.clone();
            clone.values = (IndexInput)values.clone();
            return clone;
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    public int getBitsPerValue() {
      return bitsPerValue;
    }

    public int size() {
      return valueCount;
    }

    public long getMaxValue() { // Convenience method
      return maxValue(bitsPerValue);
    }

    public long getLong(final int index) throws IOException {
        try {
            values.seek(startPos + (index << 3));
            return values.readLong();
        } catch (IOException e) {
            throw new IOException("IndexFileSize=" + values.length() + ", startPos=" + startPos + ", index=" + index + ", valueCount=" + valueCount, e);
        }
    }

    public int getInt(final int index) throws IOException {
        values.seek(startPos + (index << 2));
        return values.readInt();
    }

    public short getShort(final int index) throws IOException {
        values.seek(startPos + (index << 1));
        return values.readShort();
    }

    public byte getByte(final int index) throws IOException {
        values.seek(startPos + index);
        return values.readByte();
    }

    @Override
    public void close() throws IOException {
        if (values != null) {
            values.close();
            values = null;
        }
    }
  }

  /**
   * Retrieve PackedInt data from the IndexInput and return a packed int
   * structure based on it.
   * @param in positioned at the beginning of a stored packed int structure.
   * @return a read only random access capable array of positive integers.
   * @throws IOException if the structure could not be retrieved.
   * @lucene.internal
   */
  public static Reader getReaderNoCodecHeader(IndexInput in) throws IOException {
    return getReader(in, false);
  }

  public static Reader getReader(IndexInput in) throws IOException {
    return getReader(in, true);
  }

  public static Reader getReader(IndexInput in, boolean readCodecHeader) throws IOException {
    if( readCodecHeader ) CodecUtil.checkHeader(in, CODEC_NAME, VERSION_START, VERSION_START);
    final int bitsPerValue = in.readVInt();
    assert bitsPerValue > 0 && bitsPerValue <= 64: "bitsPerValue=" + bitsPerValue;
    final int valueCount = in.readVInt();
    System.err.println("getReader: in=" + in + ", valueCount=" + valueCount + ", pos=" + in.getFilePointer());
//    new Exception("GET DISK READER").printStackTrace();

    switch (bitsPerValue) {
    case 8:
      return new Direct8(in, valueCount);
    case 16:
      return new Direct16(in, valueCount);
    case 32:
      return new Direct32(in, valueCount);
    case 64:
      return new Direct64(in, valueCount);
    default:
      if (Constants.JRE_IS_64BIT || bitsPerValue >= 32) {
        return new Packed64(in, valueCount, bitsPerValue);
      } else {
        return new Packed32(in, valueCount, bitsPerValue);
      }
    }
  }

  /** Returns how many bits are required to hold values up
   *  to and including maxValue
   * @param maxValue the maximum value that should be representable.
   * @return the amount of bits needed to represent values from 0 to maxValue.
   * @lucene.internal
   */
  public static int bitsRequired(long maxValue) {
    // Very high long values does not translate well to double, so we do an
    // explicit check for the edge cases
    if (maxValue > 0x3FFFFFFFFFFFFFFFL) {
      return 63;
    } if (maxValue > 0x1FFFFFFFFFFFFFFFL) {
      return 62;
    }
    return Math.max(1, (int) Math.ceil(Math.log(1+maxValue)/Math.log(2.0)));
  }

  /**
   * Calculates the maximum unsigned long that can be expressed with the given
   * number of bits.
   * @param bitsPerValue the number of bits available for any given value.
   * @return the maximum value for the given bits.
   * @lucene.internal
   */
  public static long maxValue(int bitsPerValue) {
    return bitsPerValue == 64 ? Long.MAX_VALUE : ~(~0L << bitsPerValue);
  }

  /** Rounds bitsPerValue up to 8, 16, 32 or 64. */
  public static int getNextFixedSize(int bitsPerValue) {
    if (bitsPerValue <= 8) {
      return 8;
    } else if (bitsPerValue <= 16) {
      return 16;
    } else if (bitsPerValue <= 32) {
      return 32;
    } else {
      return 64;
    }
  }

  /** Possibly wastes some storage in exchange for faster lookups */
  public static int getRoundedFixedSize(int bitsPerValue) {
    if (bitsPerValue > 58 || (bitsPerValue < 32 && bitsPerValue > 29)) { // 10% space-waste is ok
      return getNextFixedSize(bitsPerValue);
    } else {
      return bitsPerValue;
    }
  }
}
