package ru.yandex.analyzer;

import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.CharTermAttribute;

import java.io.IOException;
import java.util.Locale;

/**
 * A {@link TokenFilter} that parses headers.
 */
public final class HeadersFilter extends TokenFilter {
    private static final char[] UNKNOWN = "unknown".toCharArray();

    private CharTermAttribute termAtt;

    public HeadersFilter(TokenStream in) {
        super(in);
        termAtt = addAttribute(CharTermAttribute.class);
    }

    private int ltrimPos(final char[] buf, final int off, final int end) {
        int pos = off;
        while (pos < end && Character.isWhitespace(buf[pos])) {
            ++pos;
        }
        return pos;
    }

    private int rtrimPos(final char[] buf, final int off, final int end) {
        int pos = end;
        while (pos > off && Character.isWhitespace(buf[pos - 1])) {
            --pos;
        }
        return pos;
    }

    @Override
    public final boolean incrementToken() throws IOException {
        while (input.incrementToken()) {
            int len = termAtt.length();
            if (len > 0) {
                char[] cbuf = termAtt.buffer();
                int sep = -1;
                for (int i = 0; i < len; ++i) {
                    if (cbuf[i] == ':') {
                        sep = i;
                        break;
                    }
                }
                if (sep == -1) {
                    String value = new String(cbuf, 0, len);
                    termAtt.setEmpty();
                    termAtt.append(UNKNOWN);
                    termAtt.append(':');
                    termAtt.append(' ');
                    termAtt.append(value.trim());
                } else {
                    boolean needSpace = sep + 1 < len && cbuf[sep + 1] != ' ';
                    if (needSpace) {
                        cbuf = termAtt.resizeBuffer(len + 1);
                    }
                    for (int i = 0; i < sep; ++i) {
                        cbuf[i] = Character.toLowerCase(cbuf[i]);
                    }
                    if (needSpace) {
                        String value =
                            new String(cbuf, sep + 1, len - sep - 1);
                        termAtt.setLength(sep + 1);
                        termAtt.append(' ');
                        termAtt.append(value.trim());
                    } else {
                        int ltrimPos = ltrimPos(cbuf, sep + 2, len);
                        int rtrimPos = rtrimPos(cbuf, ltrimPos, len);
                        int off = ltrimPos - sep - 2;
                        if (off == 0) {
                            termAtt.setLength(rtrimPos);
                        } else {
                            int valueLen = rtrimPos - ltrimPos;
                            for (int i = 0; i < valueLen; ++i) {
                                cbuf[i + sep + 2] = cbuf[i + sep + 2 + off];
                            }
                            termAtt.setLength(sep + 2 + rtrimPos - ltrimPos);
                        }
                    }
                }
                return true;
            }
        }
        return false;
    }
}
